/* wcs750.c:
Writable control store driver for 750.
Note: do not use pcs750.bin v99 with this, as it will give spurious
ACV on accessing physical address F00FE0. Use v98 instead.
*/

#if VAX750 && WCS750

#include "param.h"
#include "conf.h"
#include "user.h"
#include "kernel.h"
#include "mtpr.h"
#include "cpu.h"
#include "pte.h"
#include "vmmac.h"
#include "uio.h"

#include "wcs750.h"

/* forward. */
static void wcs750_map();
static void wcs750_unmap();
static int wcs750_testbank();

/* Some flags. */
/* 1 if we've got the first bank. */
static int wcs750_haswcs = 0;
/* 1 if we've got the second bank. */
static int wcs750_hasbank2 = 0;
/* 1 if we have a PCS and have taken control of it. */
static int wcs750_haspcs = 0;

/* 1 if the device is currently open. */
static int wcs750_isopen = 0;

/* This is defined for us by code in locore.s, as far as I can tell. */
extern unsigned long wcs750memory;
extern unsigned long pcs750memory;
extern unsigned long pcs750flagmemory;

/* Map the WCS physical memory into our virtual address space, using the 
PTEs defined in locore.s */

static void wcs750_map()
{
	struct pte *pte = &WCS750map[0];
	unsigned long base = btop(WCS_BASE);
	int i = 0;

	/* Map all three banks, including PCS */
	for(i = 0; i < ((WCS_PAGES) * 3); i ++, pte ++, base ++) {
		*(int *)pte = PG_V|PG_KW|base;
	}
	pte = &PCS750flagmap[0];
	*(int *)pte = PG_V|PG_KW|PCS_FLAGREG;
	mtpr(TBIA, 0);
}

static void wcs750_unmap()
{
	int i = 0;
	struct pte *pte = &WCS750map[0];

	/* All three banks */
	for(i = 0; i < ((WCS_PAGES) * 3); i ++, pte ++) {
		*(int *)pte = 0;
	}
	pte = &PCS750flagmap[0];
	*(int *)pte = 0;
	mtpr(TBIA, 0);
}


/* Enable the WCS. This doesn't attempt to write the magic to the SCB vector;
it just sets the enable flag on the board. */

static void wcs750_enable()
{
	unsigned long word;
	word = wcs750memory;
	wcs750memory = word | WCS_ENABLE_BIT;
}


/* Disable the WCS, ditto. */

static void wcs750_disable()
{
	unsigned long word;
	word = wcs750memory;
	wcs750memory = word & WCS_MASK;
}


/* Test a bank of the WCS. 0 for first bank, 1 for second.
Return zero on failure, non-zero on success. Could be a bit more 
exhaustive!
*/

static int wcs750_testbank(bank)
int bank;
{
	register unsigned long *addr;
	register int index;
	register unsigned long val;
	addr = &wcs750memory + (bank * WCS_LONGS);
	for(index = 0; index < WCS_LONGS; index ++, addr ++) {
		*addr = (index & WCS_MASK);
	}
	addr = &wcs750memory + (bank * WCS_LONGS);
	for(index = 0; index < WCS_LONGS; index ++, addr ++) {
		/* Dear Mr. Optimiser,
			Please fuck off and leave my code alone.
				Yours sincerely,
				Mr. Scruffy
		*/
		val = (*addr) & WCS_MASK; /* throw optimiser off scent */
		if(val != (index & WCS_MASK)) {
			printf("WCS750: bank %d, index %d, got %x, expecting %x\n",
				bank, index, val, (index & WCS_MASK));
			return 0;
		}
	}
	return 1;
}


void wcs750_init()
{
	union cpusid sid;
	u_int rev;

	wcs750_map();

	if(badaddr(&wcs750memory, 4)) {
		printf("WCS750: writable control store not present\n");
		return;
	}

	printf("WCS750: bank 0 found\n");
	wcs750_haswcs = 1;

	if(!wcs750_testbank(0)) {
		printf("WCS750: WARNING: bank 0 bad\n");
	}
	else {
		printf("WCS750: bank 0 tested OK\n");
	}

	if(badaddr(&wcs750memory + WCS_LONGS, 4)) {
		printf("WCS750: bank 1 not found\n");
		return;
	}
	printf("WCS750: bank 1 found\n");
	wcs750_hasbank2 = 1;

	if(!wcs750_testbank(1)) {
		printf("WCS750: WARNING: bank 1 bad\n");
	}
	else {
		printf("WCS750: bank 1 tested OK\n");
	}

	/* Have we got a PCS? */
	if(badaddr(&pcs750memory, 4)) {
		printf("WCS750: patchable control store not present\n");
		return;
	}

	printf("WCS750: PCS bank found\n");
	/* Get sysid */
	sid.cpusid = mfpr(SID);
	rev = sid.cpu750.cp_urev;
	if(rev > PCS_BASEREV) {
		printf("WCS750: patches loaded, PCS not available\n");
		return;
	}

	/* Otherwise, we take control of the PCS ourselves */
	wcs750_haspcs = 1;
	printf("WCS750: patches not loaded, testing PCS\n");
	if(!wcs750_testbank(2)) {
		printf("WCS750: WARNING: PCS bank bad\n");
	}
	else {
		printf("WCS750: PCS bank tested OK\n");
	}
}


int wcs750_open(dev, flag)
dev_t dev;
int flag;
{
	int unit;
	unit = minor(dev);
	if(unit != 0) {
		return ENXIO;
	}
	/* Ought to be some better way of signalling this. */
	if(wcs750_isopen) {
		return EBUSY;
	}
	wcs750_isopen = 1;
	return 0;
}

int wcs750_close(dev, flag)
dev_t dev;
int flag;
{
	if(wcs750_isopen)
		wcs750_isopen = 0;
	return 0;
}

/* A buffer for keeping one control word-worth of data in. */
static unsigned long buff[4];

int wcs750_write(dev, uio)
dev_t dev;
struct uio *uio;
{	
	unsigned long *ptr;
	int max;
	int error;
	int count;

	max = WCS_BYTES * (wcs750_hasbank2 ? 2 : 1);

	printf("uio_iovcnt: %d uio_offset: %ld uio_segflg: %d uio_resid: %d\n",
		uio->uio_iovcnt, uio->uio_offset, uio->uio_segflg, uio->uio_resid);

	if(minor(dev) != 0)
		return ENXIO;
	/* Check that amount to be written is a multiple of 16 bytes 
	 * (i.e. one complete control word) 
	 */

	if(uio -> uio_resid % 16 != 0) {
		return EINVAL;
	}

	/* Check that we're not going to write off the end of the WCS. */
	if(uio -> uio_offset + uio -> uio_resid > max) {
		return ENOSPC;
	}

	/* Now we know we can do the write. */ 
	while(uio -> uio_resid != 0) {
		ptr = &wcs750memory + uio -> uio_offset / 4;
		if(error = uiomove(buff, sizeof(buff), UIO_WRITE, uio))
			return error;
		for(count = 0; count < 4; count ++, ptr ++) {
			*ptr = buff[count] & WCS_MASK;
		}
	}

	return 0;
}

int wcs750_read(dev, uio)
dev_t dev;
struct uio *uio;
{
	unsigned long *ptr;
	int max;
	int error;
	int count;

	/* Max number of bytes we can read */
	max = WCS_BYTES * (wcs750_hasbank2 ? 2 : 1);

	if(minor(dev) != 0)
		return ENXIO;
	if(uio -> uio_resid % 16 != 0) {
		return EINVAL;
	}

	if(uio -> uio_offset + uio -> uio_resid > max) {
		return ENOSPC;
	}

	while(uio -> uio_resid != 0) {
		ptr = &wcs750memory + uio -> uio_offset/4;
		for(count = 0; count < 4; count ++, ptr ++) 
			buff[count] = *ptr & WCS_MASK;
		if(error = uiomove(buff, sizeof(buff), UIO_READ, uio))
			return error;
	}

	return 0;
}

int wcs750_ioctl(dev, cmd, data, flag)
dev_t dev;
caddr_t data;
{
return 0;
}



#endif /* VAX750 && WCS750 */
