/* RM80 formatting utility */

/* Notes
1) Geometry. The RM80 has 32 physical sectors per track in 16-bit mode,
including the spare sector per track. There are fourteen heads (seven
recording surfaces with two heads per surface). There are 561 tracks
per head. The last two cylinders are reserved for bad-block stuff.
2) Format. Each sector has a two-word header. 
The first word looks like this:
[0..9]: cylinder address;
12: FMT (set to indicate 16-bit format);
13: SSF (set to indicate sector is to be skipped);
14: UF (set to indicate that sector has not been badded by user);
15: MF (set to indicate that sector has not been badded by manufacturer)
Second word looks like this:
[0..4]: sector address (from 0 to 31);
[8..11]: track address (0 to 14)
*/

/* bytes per sector */
#define SECTSIZE 512

/* Total number of cylinders */
#define NCYLS 561

/* Tracks/cylinder */
#define NTRACKS 14

/* Sectors/track (including skip) */
#define NSECT 32

typedef struct _sector {
  u_short header1;
  u_short header2;
  unsigned long data[128];
} sector;

/* Flags in header1 */

/* 16-bit format flag */
#define HDR1_16BIT	0x1000

/* skip-sector flag */
#define HDR1_SSF	0x2000

/* user's bad-sector flag (note that sector is bad when flag is clear) */
#define HDR1_UF		0x4000

/* Manufacturer's bad-sector flag (sector bad when flag clear) */
#define HDR1_MF		0x8000

/* In header2, various masks and shifts */
#define HDR2_SECTMASK	0x1f
#define HDR2_TRACKSHIFT 8
#define HDR2_TRACKMASK	0xf


/* The track image that will get written to the disk */
sector wsector[NSECT];

/* During multi-pass formatting of a track with bad sectors,
we keep the number of the sector replaced by the skip sector here.
When we haven't yet used it, it's set to -1. */
int skip_sector = -1;

/* The track image read back */


/* int format_track(int fd, int cylinder, int track, unsigned short patt):
format the specified track of the specified cylinder, with the
specified data pattern. fd is the file descriptor of the open drive. 
Algorithm:
1) Construct the track image in an array of sector structures. 
2) Seek to the start of the cylinder. 
3) Write the whole lot out.
4) Read it back in, noting errors. 
5) If there are errors, write again with the first error skipped.
6) Read in again. Add any further errors to the bad sector table.
7) Write out again.
*/

int format_track(fd, cyl, track, patt)
int fd, cyl, track;
unsigned short patt;
{


/* Initialise the write track image. Each sector gets its header initialised
with the cylinder, track and sector numbers, and the MF and UF flags get
set, to indicate that the sector is good. We also set the global skip_sector
to -1, to indicate that no sector has been skipped yet. */

void buff_init(buff, cyl, track, pattern)
sector *buff;
int cyl, track;
unsigned long pattern;
{
  unsigned int sect = 0;
  unsigned int i = 0;
  for(sect = 0; sect < NSECT; sect ++) {
    /* set cylinder number and user/maker OK flags, and 16-bit format flag */
    buff[sect].header1 = cyl | (HDR1_UF | HDR1_MF | HDR1_16BIT);
    /* set track and sector in header2 */
    buff[sect].header2 = 
      (sect & HDR2_SECTMASK) | ((track & HDR2_TRACKMASK) << HDR2_TRACKSHIFT);
    
    /* Fill in the pattern */
    for(i = 0; i < 128; i ++)
      buff[sect].data[i] = pattern;
  }
  skip_sector = -1;
}

/* void buff_skip(sector *buff, int sector):
set the skip flag for the specified sector in the current track.
Also sets the global skip_sector to indicate the skipped sector.
*/

void buff_skip(buff, sect)
sector *buff;
int sect;
{
  buff[sect].header1 |= HDR1_SSF;
  skip_sector = sect;
}

/* void buff_mbad(sector *buff, int sector):
set the manufacturer's bad flag for the specified sector. Note that
this involves *clearing* the bit.
*/

void buff_mbad(buff, sect)
sector *buff;
int sector;
{
  buff[sect].header1 &= ~HDR1_MF;
}

/* void buff_ubad(sector *buff, int sector):
Likewise, but for the user's bad flag. */

void buff_ubad(buff, sect)
sector *buff;
int sector;
{
  buff[sect].header1 &= ~HDR1_UF;
}

/* void write_track(int fd, int cyl, int track, sector *image):
write the specified track of the specified cylinder to the disk.
The image of the track is in image. 
If the write routine  encounters an error, it returns with the
write partially done. If this happens, we have to re-do the
rest of the write. This makes things more complicated than expected.
*/

void write_track(fd, cyl, track, image)
int fd;
int cyl, track;
buff *image;
{
  int start_sect, new_start;
  int towrite, written;
  /* compute block at which track starts */
  start_sect = (cyl * NTRACKS * NSECT) + (track * NSECT);
  towrite = NSECT * sizeof(sector);
  while(1) {
    lseek(fd, start_sect * SECTSIZE, 0);
    ioctl(fd, SAIOHDR, (char *)0);
    written = write(fd, image, towrite);
    /* Did we manage to write the lot? */
    if(written == towrite) 
      break;
    else {
      new_start = iob[fd - 3].i_errblk;
      image += (start_block - new_start);
      towrite -= (new_start - start_block) * sizeof(sector);
      start_block = new_start;
      if(towrite < sizeof(sector)) 
        break;
    }
  }
}
