#include "../h/param.h"
#include "../h/dir.h"
#include "../h/conf.h"
#include "../h/user.h"
#include "../h/tty.h"
/* Driver for modified DR-11 C */
/* this driver sends words from sender to receiver */

/* Multiplex driver, knows three kinds of lines :
        'real' tty      channel has full tty control e.g.
                        echo,tabs,raw etc.
                        Stops sending after every n characters and waits for a
                        GO from the other side.
                        The decision to pass characters to the user is
                        left to 'canon'.
                        Multiple open's on same line are possible.
        'sending' tty   sends characters to other side, no echo etc.
                        Sends everything received inmediatly to the user.
        'raw' tty       sends 8-bit bytes to other side, no echo.
                        waits with returning input to user until u_count
                        exhausted or a End Of Text is received.
                        Stops sending after passing a High water mark
                        continues after receiving a GO.

        All channels notify the other side of open and close actions.
        
        'open' waits until other side is also opened.

        Receiving a close or open from the other side
        while the channel is opened, inhibits further
        IO on the channel, until 'close' is called.
        On tty channels a SIGHUP signal is generated.

        The 'sending' tty's on one side, send to the 'real' tty's
        on the other side and vice-versa.
        The 'raw' channels are coupled one-to-one.

*/
/* #define      STAT    0               /* enable measurements */
#define SLACK   10      /* reserved space for echo's */

/* define values used as flags to mpout */

/* this structure needs to be defined to be able to process the include */


struct sizes {
        int     number ;        /* number of channels */
        struct tty (*tpbase)[] ;        /* ptr to lowest tp struct */
} ;

#include "mp.h"

#define NO_SLEEP 1      /* word must be on queue, can't afford to sleep */
#define CAN_WAIT 0      /* wait on full queue is allowed */

#define RREADY  0100000 /* receiver ready */
#define TREADY  0200    /* transmitter ready */
#define MPTINT  0100    /* transmitter INT ENABLE */
#define MPRINT  040     /* receiver INT ENABLE */


#define MPTIM   25              /* timer interval */

#define POLLCOUNT 32            /* tunable timeout count */

struct {
        /* register layout */
        int mpcsr ;             /* control and status */
        int mpoutb ;            /* output buffer */
        int mpinb  ;            /* input buffer */
};



/* types of channels, sent in bits 13 & 14 of words */
#define RTTY    0       /* type of 'login' tty's  */
#define STTY    1       /* type of 'counterparts' */
#define RAWC    2       /* type of 'raw' channels */

#ifdef MOREMAJOR
#define LINE    037     /* mask for line numbers */
#define NLINEBITS 5     /* # bits in LINE mask */
#else
#define LINE    07
#define NLINEBITS  3
#define DEVMASK 0340
#define DEVSHIFT 5
#endif
#define TYPE    03      /* mask for channel type */

#define HIWAT   60      /* arbitrarely chosen high water mark */
                        /* must be less the 128 (0200) */

#define MAXINCANQ 130   /* max # characters in canon queue */



#define CNTRL   0100000 /* control bit in words sent */

#define OPREQ   0       /* other side executed an open and
                           thinks i didn' t */
#define OPACK   1       /* other side executed an open and
                           thinks i also did */
#define CLOSE   2       /* other side executed a close */
                        /* and thinks i didn't */
#define GO      3       /* send more input please */
#define EOT     4       /* this was all i had for the moment */
#define ERR     5       /* unknown channel */
#define BLOCKED 6       /* Other side now waits for GO */


/* combination of bits in struct has special meaning */
#define WCLOSE  02      /*  WOPEN */
#define NEEDSGO 040000  /* Did receive a BLOCKED signal */

/* macro to make a word to put on line from rel. information */
#define tword(linedef,cntrl,c) (((linedef)->h_outline<<8)+(cntrl)+(c))


/* structure definitions */

#ifdef STAT
struct {
        int illiint ;           /* illegal R interrupt */
        int illoint ;           /* illegal T interrupt */
        long icount ;           /* # words received */
        long ocount ;           /* # words sent */
        long i_int ;            /* # R interrupts */
        long o_int ;            /* # T interrupts */
        int no_echo ;           /* # suppressed echo's */
        int un_chan ;           /* # accesses to strange channels */
        int horrible ;          /* # times character pool empty */
        int clclash ;           /* # times closes clashed */
        int opclash ;           /* # times opens clashed */
        int illopack ;          /* # unexpected OPACK */
        int maxiniq ;           /* max # in input queue */
        int maxinoq ;           /* max # in output queue */
        int overflow ;          /* overflow of canon queue */
} mpstat ;

int     mplxw ;
int     mplrw ;
#endif

struct queue {
        int *next_out ;         /* out pointer */
        int *next_in  ;         /* in  pointer */
        char nleft    ;         /* # words left in queue */
        char waitf    ;         /* waiting for queue to empty */
        int mpq[MPQSIZ] ;       /* queue buffer */
} ;


struct mpbooks {
        char mp_tready ; /* remembers whether a transm. int. was received */
        char mp_trbusy ;        /* signals transmission in progress */
        char mp_open ;  /* if true the first open is performed */
#ifdef MULTMP
#ifdef MOREMAJOR
        char    mp_maj ;        /* major device number of this book */
        char    *mp_addr ;              /* device address */
#endif
#endif

        struct queue mp_iq ;    /* words from DR-11 C */
        struct queue mp_oq ;    /* words to DR-11 C */
} ;

/* structure to facilitate information transfer between routines */
/* filled by mpgtp from minor device no */
struct handle {
        int  h_outline ;        /* line + type for output */
        struct tty *h_tp ;      /* tty pointer */
        char h_line ;           /* line # */
        char h_type ;           /* line type */
        struct mpbooks *h_book ;/* book for this line */
} ;

#define hibyte(x)       (int)((unsigned)x>>8)
#define lobyte(x)       (int)(x & 0377)

/* allocation of structures etc */
struct mpbooks mpbooks[NMP] ;

/* translation table for output line's */
char mptr[] {
        /* RTTY */ STTY,        /* STTY */ RTTY ,       /* RAW */ RAWC ,
} ;

/* to appease the compiler */
extern struct tty *mpgtp();
/* -------------------------------------------------------------- */
/* user-driver interface routines */
/* actual routines */

mpopen(dev,flag)
{
        struct handle linedef ;
        register struct tty *tp ;
        register struct mpbooks *book ;

        extern mpstart() ;

        if ( mpgtp(mpm(dev),&linedef,minor(dev))==0 ) {
                u.u_error=ENXIO ;
                return ;        /* unknown channel */
        }
        book=linedef.h_book ;
        /* enable reader */
        if ( !book->mp_open ) {
                /* initialize queue's */
                book->mp_open++ ;
                book->mp_oq.next_in=book->mp_oq.next_out=
                        book->mp_oq.mpq ;
                book->mp_iq.next_in=book->mp_iq.next_out=
                        book->mp_iq.mpq ;
                /* first open, force hardware in known state */
                if (MPADDR->mpinb) ; /* throw away the first */
                MPADDR->mpcsr=| MPRINT ;
        }
        tp=linedef.h_tp ;
#ifdef UCLA
        if ( flag==1 )
                tp->t_state=| WONLY ;
#endif
        spl5() ;
        if ( (tp->t_state&ISOPEN)==0 ) {
                /* file not open yet */
                ttyname(dev, tp);
                /* signal open to other */
                while ( (tp->t_state&(CARR_ON|TIMEOUT)) == 0 ) {
                        /* no open seen yet */
                        /* while the ISOPEN bit is off, the delct
                           field in the tty struct signals opens sent
                        */
                        if ( tp->t_delct== 0 ) {
                                /* no open sent yet */
                                /* send one */
                                tp->t_delct++ ;
                                mpout(book,tword(&linedef,CNTRL,OPREQ),CAN_WAIT) ;
                                mpsend(book) ;
                        }
                        /* wait for response */
                        sleep(tp,TTIPRI) ;
                }
                if ( tp->t_delct==0 ) {
                        /* open from other received, but nothing sent yet */
                        /* send open */
                        mpout(book,tword(&linedef,CNTRL,OPACK),CAN_WAIT) ;
                        mpsend(book) ;
                }
        }
        if (  ( (tp->t_state&ISOPEN) && linedef.h_type!=RTTY ) ||
              (tp->t_state&TIMEOUT ) ){
                /* more then one open only allowed on RTTY's */

                /* The TIMEOUT bit means a ERR was received on the channel */
                /* this bit can only be cleared by a OPREQ from the other */
                spl0();
                u.u_error=ENXIO ;
                return ;
        }
        /* RTTY's get as far as this after the other side has
           closed the connection but this side didn't */
        while((tp->t_state&CARR_ON)==0) sleep(tp,TTIPRI) ;
        if ( ( tp->t_state&ISOPEN ) == 0 ) {
                /* initialize tp structure */
                tp->t_state=| ISOPEN|NEEDSGO ;
                tp->t_oproc= mpstart ;
                tp->t_iproc= NULL;
                tp->t_flags= XTABS|ECHO|CRMOD|ODDP|EVENP ;
                ttychars(tp);
                tp->t_delct= 0 ;
                tp->t_col= 0 ;
        }
        ttyopen(dev, tp, flag);
        spl0();
}

mpwrite(dev)
{
        register struct tty *tp ;
        struct handle linedef ;
        register int c ;

        tp=mpgtp(mpm(dev),&linedef,minor(dev)) ;

        if (tp->t_state&WCLOSE ) {
                /* channel closed by other side */
                u.u_error=EIO ;
                return ;
        }
        while ( (c=cpass()) >=0 ) {
                if ( linedef.h_type==RTTY ) {
                        ttyoutput(c,tp) ;
                        while( (c=getc(&tp->t_outq))>=0 ){
                                if(c >= 0200 && (tp->t_flags&RAW) == 0) {
                                        if(c == 0200) {
                                                tp->t_xstate=| XPAGE1 ;
                                                mptoline(&linedef,CNTRL,EOT) ;
                                                mpsend(linedef.h_book);
                                                spl5();
                                                tp->t_state |= ASLEEP;
                                                sleep((caddr_t)&tp->t_outq, TTOPRI);
                                                spl0();
                                        }
                                } else {
                                        mptoline(&linedef,NULL,c) ;
                                }
                        }
                } else {
                        mptoline(&linedef,NULL,c) ;
                }
        }
        if ( linedef.h_type!=STTY ) {
                mptoline(&linedef,CNTRL,EOT) ;
        }
        mpsend(linedef.h_book);
}

mpread(dev)
{
        register struct tty *tp ;
        struct handle linedef ;
        register count ;

        tp=mpgtp(mpm(dev),&linedef,minor(dev)) ;



        if ( linedef.h_type==RTTY ) {
                if (tp->t_state&WCLOSE ) {
                        return ;
                }
                ttread(tp) ;
        } else {
                count=0 ;
                while(u.u_count) {
                        spl5() ;
                        while ( tp->t_rawq.c_cc ) {
                                count++ ;
                                if ( passc(getc(&tp->t_rawq))<0 ) break ;
                        }
                        /* There is nothing in the queue any more,
                                or the user's byte count is exhausted.
                           Now if the other side closed the connection,
                           exit whether or not any words have been copied
                           to the user */
                        if ( tp->t_state&WCLOSE) break ;
                        if(tp->t_delct||linedef.h_type==STTY||u.u_count==0){
                                tp->t_delct= 0 ;
                                /* delct signals other side finished a write */
                                /* but if all was passed don't give the user
                                   a zero length buffer */
                                if ( count!=0  ) break ;
                        }
                        if ( tp->t_state&NEEDSGO ) {
                                tp->t_state=& ~NEEDSGO ;
                                mpout(linedef.h_book,tword(&linedef,CNTRL,GO),CAN_WAIT) ;
                                mpsend(linedef.h_book);
                        }
                        tp->t_state=| ASLEEP ;
                        sleep(tp,TTIPRI) ;
                        spl0();
                }
                spl0();
        }
}

mpioctl(dev, cmd, adr, flag)
{
        struct handle linedef ;
        register struct tty *tp ;

        tp=mpgtp(mpm(dev),&linedef,minor(dev)) ;

        if ( linedef.h_type==RTTY ) {
                /* only on 'real' tty's */
                if(ttioccomm(cmd, tp, (caddr_t)adr, dev) == 0) {
                        u.u_error= ENOTTY;
                }
        }
}

mpclose(dev)
{
        register struct tty *tp ;
        struct handle linedef ;

        tp=mpgtp(mpm(dev),&linedef,minor(dev)) ;

        tp->t_pgrp= 0;
        if ( tp->t_state & CARR_ON ) {
                /* give close signal, if other didn't first */
                mpout(linedef.h_book,tword(&linedef,CNTRL,CLOSE),CAN_WAIT) ;
                mpsend(linedef.h_book);
        }
        flushtty(tp) ;
        /* reset state */
        tp->t_delct=0 ;
        tp->t_state=0 ;
        tp->t_xstate=0 ;
}

/* ------------------------------------------------------------- */
/* a diversity of character handlers */

/* put echo from outq on line, can't goto sleep and notifying
   in the low level routines that more output - not from a upper half -
   is available is hard
*/
mpstart(tp)
/* called only by ttyinput */
/* which is only called for RTTY type channels */
struct tty *tp ;
{
        register int c,device ;
        struct mpbooks *book ;
        struct handle linedef ;
        struct tty (*base)[] ;

#ifdef MULTMP
        for ( c=0 ; c<NMP ; c++ ) {
                base = mpsizes[c]->tpbase ;
                if ( tp>=base) {
                        device=tp-base ;
                        if ( device<=mpsizes[c]->number ) break ;
                }
        }
        if ( c>=NMP ) return ;
        book= &mpbooks[c] ;
#endif
#ifndef MULTMP
        device=tp-mpsizes->tpbase ;
        book= mpbooks ;
#endif
        mpgtp(book,&linedef,device) ;

        while( ( c=getc(&tp->t_outq) ) >=0 ) {
                /* use mpout with slack */
                if ( c<=0177 ) mpout(book,tword(&linedef,NULL,c),NO_SLEEP) ;
/*      UCLA mod for stopout not necessary, because this is only
        called for echoing the users characters
*/
        }
        mpout(book,tword(&linedef,CNTRL,EOT),NO_SLEEP) ;
        if(tp->t_state & ASLEEP) {
                tp->t_state &= ~ASLEEP;
                wakeup((caddr_t)&tp->t_outq);
        }
        mpsend(book) ;
}

/* put character on DR-11 C queue, goto sleep if transmitting beyond
   HIWAT
*/
mptoline(linedef,cntrl,c)
struct handle *linedef ;
{
        register struct tty *tp ;
        register struct mpbooks *book ;

        tp= linedef->h_tp ;
        book= linedef->h_book ;
        if ( (tp->t_state&(WCLOSE|CARR_ON))!=CARR_ON) {
                /* ignore output on closed lines */
                return ;
        }
        spl5() ;
        /* wait for ack if necessary */
        while(tp->t_state&BUSY) sleep(&tp->t_outq,TTOPRI);
        if ( tp->t_state&CARR_ON ) {
                mpout(book,tword(linedef,cntrl,c),CAN_WAIT) ;
                if ( linedef->h_type != STTY ) {
                        /* use the t_char field; no harm */
                        if ( ++(tp->t_char)>=HIWAT ) {
                                tp->t_state=| BUSY ;
                                mpout(book,tword(linedef,CNTRL,BLOCKED),CAN_WAIT) ;
                                mpsend(book) ;
                                tp->t_char=0 ;
                        }
                }
        }
        spl0();
}

/* ------------------------------------------------------------- */
/* input from other side handler */
mpin(abook)
struct mpbooks *abook ;
{
        /* pick character from queue */
        struct handle linedef ;
        register struct tty *tp ;
        register struct mpbooks *book ;
        register int word;

        book=abook ;
        while ( book->mp_iq.nleft!=0 ) {
                word= *book->mp_iq.next_out ;
                tp=mpgtp(book,&linedef,hibyte(word)) ;
                if (tp == NULL ) {
                        /* unknown channel */
#ifdef STAT
                        mpstat.un_chan++ ;
#endif
                        /* send error on same channel */
                        if ( ! (( word&(CNTRL|0377)) == CNTRL+ERR) ) {
                mpout(book,(word&( ~CNTRL & ~0377 ))+CNTRL+ERR,NO_SLEEP) ;
                                mpsend(book) ;
                        }
                } else
                if (word<0 ) mpinct(&linedef,word) ;    /* control */
                else
                if ( (tp->t_state&(ISOPEN|WCLOSE))==ISOPEN )
                if ( linedef.h_type==RTTY ) {
                        ttyinput(lobyte(word),tp) ;
                } else {
                        if ( putc(lobyte(word),&tp->t_rawq)<0 ) {
                                /* can't get rid of character */
#ifdef STAT
                                mpstat.horrible++ ;
#endif
                                /* horrible, timeout and retry */
                                timeout(mpin,NULL,25) ;
                /* mpin will not be called a second time because
                   the reader interrupt still is disabled */
                                return ;
                        }
                        if (tp->t_rawq.c_cc>MAXINCANQ ) {
                                /* queue overflow */
                                while(getc(&tp->t_rawq)>=0 ) ;
                                mpercl(&linedef,tp) ;
#ifdef STAT
                                mpstat.overflow++ ;
#endif
                        }
                }
                /* only one consumer, no race conditions */
                if ( ++book->mp_iq.next_out >= &book->mp_iq.mpq[MPQSIZ] ) {
                        book->mp_iq.next_out= &book->mp_iq.mpq[0] ;
                }

                book->mp_iq.nleft-- ;
        }
        /* because the queue is empty re-allow transmission */
        /* but don't allow DR-11 C to interrupt inmediatly */
        MPADDR->mpcsr=| MPRINT ;
}
mpinct(linedef,word)
struct handle *linedef ;
{
        register struct tty *tp ;

        tp=linedef->h_tp ;

        switch(lobyte(word)) {
        case OPREQ :
                /* open request from other side */
                if ( ( tp->t_state&CARR_ON ) == 0 ) {
                        /* nothing received yet */
                        if ( (tp->t_state&ISOPEN) == 0 ) {
                                /* file not opened, but
                                   open's could be hanging */
                                if ( tp->t_delct) {
                                        /* opens prob. are hanging */
                                        /* other side didn't see OPREQ */
                                        /* cause OPACK to be send */
                                        tp->t_delct=0 ;
                                }
                        } 
                        else {
                                /*
                                 file open but carr_off 
                                 this side must be in closing state 
                                just ignore the request.
                                */
                                if ( linedef->h_type != RTTY ) return ;
                        }
                        /* reset state totally, including TIMEOUT */
                        tp->t_state=CARR_ON ;
                        if ( linedef->h_type!=STTY ) tp->t_state=| BUSY ;
                } else {
                        /* the other side was down */
                        if ( tp->t_state&ISOPEN ) {
                                /* force close on file */
                                mpercl(linedef,tp) ;
                        }
                        /* ignore the OPREQ */
                }
                wakeup(tp) ;
                break ;
        case OPACK :
                /* response to an OPREQ */
                if ( (tp->t_state&CARR_ON) == 0 ) {
                        tp->t_state=| CARR_ON ;
                        if ( linedef->h_type!=STTY ) tp->t_state=| BUSY ;
                        /* nothing received so far */
                        if ( (tp->t_state&ISOPEN) == 0 ) {
                                /* channel not open, but probably a open
                                   is hanging */
                                if ( tp->t_delct ) {
                                        /* that prob. is almost one */
                                        /* the only way to have no
                                           process waiting, is to have it
                                           killed while waiting */
                                        wakeup(tp) ;
                                }
                                /* else {
                                        no open sent yet
                                        The other thought i sent a OPREQ
                                        but i didn't.
                                        Treat this as an OPREQ .
                                        That is the nicest to the other side
                                }
                                */
                        } else {
                                /* file is open, but carrier is off.
                                   This situation should be impossible
                                   because i didn't send an OPREQ.
                                   Close the file down
                                */
#ifdef STAT
                                mpstat.illopack++ ;
#endif
                                mpercl(linedef,tp) ;
                        }
                } else {
                        /* result of an OPEN clash, ignore */
#ifdef STAT
                        mpstat.opclash++ ;
#endif
                }
                break ;
        case CLOSE :
                /* other side did execute a close,
                   and did not receive a close signal from me */
                if ( (tp->t_state&ISOPEN) ) {
                        /* I really did not exec. an close,
                           force file in 'closed' status
                        */
                        mpercl(linedef,tp) ;
                } else {
                        /* closes clashed */
#ifdef STAT
                        mpstat.clclash++ ;
#endif
                }
                break ;
        case ERR :
                if ( tp->t_state&ISOPEN ) mpercl(linedef,tp) ;
                tp->t_state=| TIMEOUT ;
                tp->t_delct++ ;
                wakeup(tp) ;
                break ;
        case GO :
                tp->t_state=& ~BUSY ;
                wakeup(&tp->t_outq) ;
                break ;
        case EOT :
        case BLOCKED :
                if ( tp->t_state&CARR_ON ) {
                        if ( lobyte(word)==EOT ) {
                                tp->t_delct++ ;
                        } else {
                                tp->t_state=| NEEDSGO ;
                        }
                        if ( tp->t_state&ASLEEP ) {
                                tp->t_state=& ~ASLEEP ;
                                wakeup(tp) ;
                        }
                }
                break ;
        }
}

/* routine to force the file in close state */
/* no transfers or opens are possible until close has been called */
mpercl(linedef,tp)
struct tty *tp ;
struct handle *linedef ;
{
        register struct tty *atp ;

        atp=tp ;

        tp->t_state=& ~(CARR_ON|BUSY) ;
        if ( linedef->h_type==RTTY ) {
                /* HANGUP signal to tty's */
                signal(tp->t_pgrp,SIGHUP) ;
                flushtty(tp) ;
        } else {
                tp->t_delct++ ;
        }
        tp->t_state=| WCLOSE ;
        /* wake writers */
        wakeup(&tp->t_outq) ;
        /* and readers */
        wakeup(tp) ;
}

/* ------------------------------------------------------------- */
/* routine used in almost every other routine */
/* transforms minor device to line, type and tp pointer */
/* returns 0 if line not allocated */
struct tty *mpgtp(abook,linedef,channel)
struct handle *linedef ;
struct mpbooks *abook ;
{

        register int type,line ;
        register struct tty *tp ;
        int nbook ;

        nbook=abook-mpbooks ;
        if ( ( linedef->h_book=abook ) == NULL ) return ( NULL ) ;
        type= ( channel>>NLINEBITS ) & TYPE ;

        line= channel&LINE ;

        tp= NULL ;

        if ( type !=3 && line < mpsizes[nbook][type].number ) {
                /* 3 is the only unknown type */
                tp= & ( (*mpsizes[nbook][type].tpbase)[line]) ;
        }
        linedef->h_tp=tp ;
        linedef->h_line=line ;
        linedef->h_type=type ;
        linedef->h_outline=((mptr[type])<<NLINEBITS)+line ;
        return(tp) ;
}

#ifdef MULTMP
#ifdef MOREMAJOR
/* routine to map major device number onto book number.
   the first time around all character devices are checked
   whether they are mp devices, if so they are assigned books in
   the major device number order
*/
mpmap(dev)
{
        static int first ;
        register int ndev,nbook ;
        extern int nchrdev ;

        if ( first==0 ) {
                first++ ;
                nbook=0 ;
                for ( ndev=0 ; ndev<nchrdev ; ndev++ ) {
                        if ( cdevsw[ndev].d_open == &mpopen ) {
                                mpbooks[nbook].mp_maj = ndev ;
                                mpbooks[nbook].mp_addr = mpaddr[nbook] ;
                                if ( ++nbook>=NMP ) break ;
                        }
                }
        }
        for ( nbook=0 ; nbook<=NMP ; nbook++ ) {
                if ( mpbooks[nbook].mp_maj==major(dev) &&
                     mpbooks[nbook].mp_addr!=0 ) {
                /* &0377 is not necessary, both are chacter fields */
                        return(&mpbooks[nbook]) ;
                }
        }
        return(NULL) ;
}
#endif
#endif


/* ----------------------------------------------------------------- */
/* low level interface */

mpout(abook,word,slack)
struct mpbooks *abook ;
{
        /* routine to send one word over DR-11 C line */
        /* puts the word in a queue and start sending if necessary */
        register int ps ;
        register struct mpbooks *book ;

        book = abook ;

        ps= spl5() ;
        while(book->mp_oq.nleft >= MPQSIZ-SLACK) {
                /* slack flag :
                        if on the word must be put on the queue,
                        now or never, use the slack space */
                if (slack) {
                        if ( book->mp_oq.nleft>=MPQSIZ ) {
#ifdef STAT
                                mpstat.no_echo++ ;
#endif
                                splx(ps);
                                return ;
                        }
                        break ; /* leave the while loop */
                }
                /* full queue, wait for it to drain */
                book->mp_oq.waitf++ ;
                sleep( &book->mp_oq , TTOPRI ) ;
        }

        book->mp_oq.nleft++ ;
        *book->mp_oq.next_in = word ;

        if ( ++book->mp_oq.next_in >= &book->mp_oq.mpq[MPQSIZ] ) {
                book->mp_oq.next_in = &book->mp_oq.mpq[0] ;
        }
        /* start sending on full queue only */
        /* this to enhance polling efficiency */
        if(book->mp_oq.nleft==MPQSIZ-SLACK) mpsend(book);
        splx(ps);
}

mpsend(book)
struct mpbooks *book;
{
        register int ps;

        ps= spl5();
        if(!book->mp_trbusy)
                mprstart(book);
        splx(ps);
}

mprstart(abook)
struct mpbooks *abook ;
{
        /* start sending from queue to line */
        register int poll;
        register struct mpbooks *book ;

        book=abook ;
        book->mp_trbusy=1 ;
        /* only consumer don't worry about race conditions */
        
        while(book->mp_oq.nleft) {
                /* lower priority, interrupt might be pending */
                if ( !book->mp_tready ) {
                        /* poll for transmitter ready */
                        poll=POLLCOUNT ;
                        while( (MPADDR->mpcsr&TREADY)==0 && --poll ) ;
                        if ( (MPADDR->mpcsr & TREADY) == 0 ) {
                                /* previous transfer not finished */
                                /* enable T interrupt */
                                MPADDR->mpcsr=| MPTINT ;
                                return ;
                        }
                } else {
                        /* sending can continue, interrupt was seen */
                        book->mp_tready=0 ;
                }
#ifdef STAT
                mpstat.ocount++ ;
                mplxw= *book->mp_oq.next_out ;
#endif
                MPADDR->mpoutb= *book->mp_oq.next_out ;
                if ( ++book->mp_oq.next_out >= &book->mp_oq.mpq[MPQSIZ] ) {
                        /* buffer wrap-around */
                        book->mp_oq.next_out= &book->mp_oq.mpq[0] ;
                }
#ifdef STAT
                if ( book->mp_oq.nleft>mpstat.maxinoq ) {
                        mpstat.maxinoq=book->mp_oq.nleft ;
                }
#endif
                book->mp_oq.nleft-- ;
                if ( book->mp_oq.waitf) {
                        book->mp_oq.waitf=0 ;
                        wakeup( &book->mp_oq ) ;
                }
        }
        book->mp_trbusy=0 ;
}

/* ------------------------------------------------------------- */
/* interrupt routines */

mpxint(dev)
{
        /* transmitter interrupt */
        register struct mpbooks *book ;


        book = &mpbooks[dev] ;
#ifdef STAT
        mpstat.o_int++ ;
#endif

        /* disable further interrupts */
        MPADDR->mpcsr =& ~MPTINT ;
        book->mp_tready++ ;     /* int. received , DONE flag is zero (alas) */
        if ( !book->mp_trbusy ) {
#ifdef STAT
                mpstat.illoint++ ;
#endif
                return ;
        }
        mprstart(book) ;
}

mprint(dev)
{
        /* reader interrupts */
        /* store words in queue fetch words in polling mode */
        /* if polling times out set interrupt enable and exit */
        register int poll ;
        register struct mpbooks *book ;

#ifdef STAT
        mpstat.i_int++;
#endif

        book = &mpbooks[dev] ;
        /* disable further interrupts */
        MPADDR->mpcsr=& ~MPRINT ;

        if ( book->mp_iq.nleft ) {
#ifdef STAT
                mpstat.illiint++ ;
#endif
                return ;
        }

        do {
                *book->mp_iq.next_in = MPADDR->mpinb ;
#ifdef STAT
                mplrw= *book->mp_iq.next_in ;
                mpstat.icount++ ;
                if ( book->mp_iq.nleft>mpstat.maxiniq ) {
                        mpstat.maxiniq=book->mp_iq.nleft ;
                }
#endif
                book->mp_iq.nleft++ ;
                if ( ++book->mp_iq.next_in >= &book->mp_iq.mpq[MPQSIZ] ) {
                        book->mp_iq.next_in = &book->mp_iq.mpq[0] ;
                }
                /* buffer full ? */
                if ( book->mp_iq.nleft==MPQSIZ ) break ;
                poll= POLLCOUNT ;
                while( ( ( MPADDR->mpcsr&RREADY) == 0 ) && --poll ) ;
        } while (MPADDR->mpcsr&RREADY)  ;
        mpin(book) ;
        /* if the transmitter was ready after this routine the polling could
                be started again */
}
