/*
   Em - editor   UNIX verion 7.

   modified and extended by Th. A. Zoethout, KUN Nijmegen.
*/
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sgtty.h>
#include <setjmp.h>
#include "em.h"

long   lseek();
char   *mktemp();

/*      COMMAND INPUT         */

int   inpfid[INPFILES] = {STANDIN}; /* array of command filedescr */
#define inplim   &inpfid[INPFILES-2]
int   *cmdfid   = &inpfid[0];   /* pointer to current command filedescr */
int   rflag   = 0;      /* indicates recovery file input */
int   recfid   = -1;      /* recovery file id */
char   inpbuf[CBSIZE+1] = {NUL}; /* buffer for command input */
char   *inbufp   = &inpbuf[0];   /* pointer to next char in command input */
char   *globp = 0;      /* text with next global command */
int   peekc   = NUL;      /* look (peek) ahead command character */
int   lastc   = NUL;      /* last command character */

/*      BITMAPS FOR CHARACTERS         */

int   alfn[] = {   /* alfa numeric characters */
   0000000, 0000000,   /* control characters */
   0000000, 0001777,   /* digits */
   0177776, 0003777,   /* upper case */
   0177776, 0003777   /* lower case */
};
#define alfnum(cc)   (alfn[cc>>4] & (1<<(cc&017)))
int   prnt[] = {   /* printable characters */
   0000000, 0000000,   /* control characters */
   0177777, 0177777,   /* SP upto ? */
   0177777, 0177777,   /* @ upto _ */
   0177777, 0077777   /* ` upto ~ */
};
#define printb(cc)   (prnt[cc>>4] & (1<<(cc&017)))

/*      ENCRYPTION         */

int   zflag   = 0;      /* encrytion flag */
int   zfflag    = 0;      /* file encryption flag */
int   ztflag   = 0;      /* tempfile encryption flag */
char   key[KSIZE] = {NUL};   /* encryption key */
#define   keylim   &key[KSIZE-1]
char   crbuf[FBSIZE];      /* encryption buffer */
char   perm[768];      /* permutation buffer */
char   tperm[768];      /* permutation buffer */

/*      TEMPFILE I/O         */

char   tmpdir[FNSIZE] = "/tmp"; /* tempfile directory */
char   TMPTMP[] = "/eXXXXX";   /* template for tempfilename */
char   *tmpfile;      /* tempfilename */
int   tmpfid   = -1;      /* file descriptor of tmpfile */
char   ibuff[FBSIZE];      /* input block */
int   iblock   = -1;      /* input blocknumber */
char   obuff[FBSIZE];      /* output block */
int   oblock   = -1;      /* output blocknumber */
int   ichanged;      /* changes in inputblock */
int   tline;         /* tempfile address of first free line */
#define blocknr(p)   ((p >> 8) & 0377)
#define offset(p)   ((p << 1) & 0774)

/*      LINE AND LINEPOINTER BUFFER   */

union {            /* pointers for sbrk and brk */
   int integer;
   int *intp;
   char *charp;
} endcore, fendcore;
char   linbuf[LBSIZE];      /* line buffer input/output */
char   genbuf[LBSIZE];      /* general buffer */
#define   genlim   &genbuf[LBSIZE-1]
char   *linebp;
int   ninbuf;
char   *nextip;
int   *zero;         /* first line - 1 */
int   *dot;         /* current line */
int   *dol;         /* last line */
int   *addr1;         /* first address for command */
int   *addr2;         /* last address for command */
int   marks[KBSIZE];      /* pointers to marked lines */
#define marklim   &marks[KBSIZE-1]
int   kflag;         /* pointer existence flag */
int   subolda;      /* replaced line */
int   subnewa;      /* substituted line */
int   aline;         /* lines addd by append() */

/*   REGULAR EXPRESSIONS AND SUBSTITUTIONS   */

char   expbuf[EBSIZE+4];   /* regular expression buffer */
char   rhsbuf[SBSIZE];      /* substitution buffer */
#define rhslim   &rhsbuf[SBSIZE-1]
char   *loc1;         /* begin of matched expression */
char   *loc2;         /* end of matched expression */
char   *locs;          /* potential begin of star match */
char   *braslist[NBRA];   /* start of \( bracketed \) expression */
char   *braelist[NBRA];   /* end of \( bracketed \) expression */
int   nbra;         /* number of nested bracketed subexressions */
int   circfl;         /* begin of line match */

/*      EDIT- LOG- AND OTHER FILES   */

int   lflag   = 1;      /* indicates logging */
char   savfile[FNSIZE] = {NUL}; /* saved filename: f-command */
char   logfile[FNSIZE] = {NUL}; /* logfile name */
int   logfid   = -1;      /* logfile id */
char   genfile[FNSIZE] = {NUL}; /* general filename */
int   femfid   = -1;      /* editfile id */
int   standout = STANDOUT;   /* command output */
char   EMHUP[] = "em.hup";   /* dumped file after hangup */

/*      ERROR MESSAGES         */

char   Q[]   = "";      /* just an error */
char   S[]   = "?";      /* failed search */
char   T[]   = "TMP";   /* tempfile problem */
char   M[]   = "MEM";   /* memory overflow */
char   W[]   = "WRITE ERROR";
char   F[]   = "FORK";
char   P[]   = "PIPE";

/*      COMMAND CONTROL         */

char   EMPAR[]   = "EM";      /* shell parameter with em parameters */
char   prmpt   = PROMPT;   /* prompt character */
char   line[70];      /* interactive output line buffer */
char   *linp   = line;      /* output line pointer */
int   vflag   = 1;      /* controls silent mode */
int   tflag   = 1;      /* controls prompting */
int   iflag   = 0;      /* non-terminal input */
int   eflag   = 0;      /* controls response on interrupts */
int   aflag   = 0;      /* controls logfile on interrupts */
int   pflag   = 0;      /* suffix p or l (PRINT or LIST) */
int   ppflag   = 0;      /* controls putchr */
int   sflag   = 0;      /* command x or s, suffix g and n */
int   xflag   = 0;      /* controls confirmation of x cmd */
int   oflag   = 0;      /* controls open mode */
int   lbrkon   = 0;      /* line break in open mode */
int   thresh   = LBSIZE-40;   /* threshold for linebreaking */
char   *cgenp;         /* current position in open line */
char   *colump   = &linbuf[0];    /* current column on ttyscreen */
int   opchange = 0;      /* change in open mode */
int   gopeof   = 0;      /* last line from open mode */
int   cflag   = 0;      /* file change flag */
jmp_buf   savej   = 0;      /* saved environ for after error */
long   count;         /* general counter */

/*      UNIX UTILITIES         */

char   unixbuf[CBSIZE];      /* saved !! shell command */
char   pipebuf[CBSIZE];      /* saved || shell (pipe) command */
char   splbuf[CBSIZE];         /* buffer for spell command */
#define spllim   &splbuf[CBSIZE-1]
char   SHELCMD[] = "/bin/sh";      /* shell command name */
char   MAKKEYA[] = "/usr/lib/makekey";   /* encryption key */
char   MAKKEYB[] = "/lib/makekey";   /* emcryption key */
char   HELPDIR[] = "/usr/lib/emhelp/";   /* directory with em helpfiles */
char   HELPCMD[] = "commands";      /* helpfile with em commands */
char   HELPKEYS[] = "keys";      /* helpfile with open mode keys */
char   SPELCMD[] = "look ";      /* look command */
char   spelsfx[FNSIZE] = " ^ mc";   /* multi column pipe */

/*      SIGNAL AND TERMINAL CONTROL   */

int   (*oldintr)();      /* saved SIG_INT status */
int   (*oldhup)();      /* saved SIG_HUP status */
int   (*oldquit)();      /* saved SIG_QUIT status */
int   qflag   = 0;      /* QUIT signal control */
struct sgttyb   tty;      /* terminal setting buffer for open mode */
struct sgttyb   savtty;      /* saved terminal setting */
int   fast   = 0;      /* fast ( >= 1200 bps) or slow terminal */
int   tabs   = 0;      /* tab == ^I */
int   lines   = 24;      /* number of lines on screen */
int   width   = 79;      /* width of screen ( -1? ) */
int   scflag   = 1;      /* scope mode of terminal */

onintr   ()         /* interrupt (CTRLC) catch */
{
   signal(SIGINT, onintr);
   if( !eflag ) putchr(LF);
   if( aflag && lflag ) {
      write(logfid, ".\n", 2);
      eflag = 0;
   }
   lastc = LF;
   error(Q);
};

onhup   ()         /* hangup catch */
{
   signal(SIGINT, SIG_IGN);
   signal(SIGHUP, SIG_IGN);
   if( dol > zero ) {
      addr1 = zero+1;
      addr2 = dol;
      if( (femfid = creat(EMHUP, 0666)) >= 0 ) putfile();
   }
   quit();
};

quit   ()         /* quit (CTRL\) catch */
{
   unlink(tmpfile);
   if( lflag ) unlink(logfile);
   exit( 0 );
};

getterm   ()         /* get terminal settings */
{
   ioctl(0, TIOCGETA, &tty);
   ioctl(0, TIOCGETA, &savtty);
   tty.sg_flags |= RAW;
   tty.sg_flags &= ~CRMOD;
   tty.sg_flags &= ~ECHO;
   if( tty.sg_width != 0 ) {
      width = tty.sg_width;
      tty.sg_width = 0;
   }
   if( tty.sg_length != 0 ) {
      lines = tty.sg_length;
   }
   fast = ((tty.sg_ospeed >= B1200) ? 1 : 0);
   tabs = (tty.sg_flags & XTABS) == 0;
   scflag = (tty.sg_flags & SCOPE);
};

setraw   ()         /* set terminal in raw mode */
{
   if( ioctl(0, TIOCSETA, &tty) == -1 ) error(Q);
};

setcook   ()         /* set terminal in cooked mode */
{
   if( ioctl(0, TIOCSETA, &savtty) == -1 ) exit( 1 );
};

main   (argc, argv)
   int argc; char **argv;
{
   register char *d, *t;
   extern int onintr(), quit(), onhup();

   oldquit = signal(SIGQUIT, SIG_IGN);
   oldhup = signal(SIGHUP, SIG_IGN);
   oldintr = signal(SIGINT, SIG_IGN);
   if( signal(SIGTERM, SIG_IGN) == SIG_DFL ) {
      signal(SIGTERM, quit);
   }
   getterm();
   envparm();
   getparm(argc, argv);
   fendcore.charp = sbrk(0);   /* begin of lineaddresses */
   for( d = tmpdir; *d; d++ );
   for( t = TMPTMP; *d++ = *t++; );
   tmpfile = mktemp(tmpdir);      /* tempfilename */
   if( zflag && (zflag = getkey()) ) {   /* initialize encryption */
      zfflag = crinit(key, perm);
   }
   if( vflag ) puts("Editor", LF);
   if( rflag ) {
      if( *genfile == NUL || (recfid = open(genfile, 0)) < 0 ) {
         puts("no recovery file", LF);
         exit( 1 );
      } else {
         *++cmdfid = recfid;
      }
   } else if( *genfile != NUL ) {
      newfile(genfile);
      globp = "E";
   } else {
      newfile("");
   }
   if( oldintr == SIG_DFL ) {
      signal(SIGINT, onintr);
   }
   if( !qflag ) {
      signal(SIGQUIT, onintr);
   }
   if( oldhup == SIG_DFL ) {
      signal(SIGHUP, onhup);
   }
   if( setjmp(savej) == 0 ) eflag = 1;
   commands( 0 );
   quit();
};

envparm   ()
{
   register char delim, *ep, *pv;
   char **pp;
   int np;
   extern char *getenv();

   if( (ep = getenv(EMPAR)) == NUL ) return;
   pv = genbuf; *pv = NUL;
   pp = (char **)linbuf; *pp++ = pv++;
   for( np = 1; (delim = *ep++) != NUL; np++ ) {
      *pp++ = pv;
      while( *ep != delim && *ep != NUL ) *pv++ = *ep++;
      *pv++ = NUL;
   }
   *pp = NUL;
   if( np > 1 ) getparm(np, (char **)linbuf);
};

getparm   (parmc, parmv)         /* get parameter values */
   int parmc; char **parmv;
{
   register int p, pc;
   register char **pv;

   pc = parmc; pv = parmv;
   for( p = 1; p < pc && pv[p][0] == '-'; p++ ) {
      switch( pv[p][1] ) {
      case 'i':      /* input from duem (dued) */
         iflag = 1;
         break;
      case 'l':      /* no logfile */
         lflag = 0;
         break;
      case 'p':      /* redefine promptchar */
         if( pv[p][2] != NUL ) {
            prmpt = pv[p][2];
            tflag = 1;
         }
         break;
      case 'q':      /* cancel if QUIT */
         signal(SIGQUIT, SIG_DFL);
         qflag = 1;
         break;
      case 'r':      /* recover from logfile */
         rflag = 1;
         break;
      case 's':      /* silent mode, looks like 'ed' */
         vflag = 0;
         tflag = 0;
         break;
      case 't':      /* tempfile directory */
         cpystr(&(pv[p][2]), tmpdir);
         break;
      case 'v':      /* spell command */
         cpystr(&(pv[p][2]), spelsfx);
         break;
      case 'z':      /* set encryption flag */
         zflag = 1;
         break;
      }
   }
   if( p < pc ) {
      cpystr(pv[p], genfile);
   }
};

commands   (inglob)      /* command interpreter */
   int inglob;
{
   register int *a1, c, delim;
   int wflag;
   extern int getfile(), gettty(), shell(), help(), *address();

   for( ; ; ) {
      aflag = wflag = sflag = pflag = ppflag = oflag = 0;
      if( tflag && !inglob && !eflag ) {
         putchr(prmpt);
         putchr(EOF);
      }
      eflag = 0;
      addr1 = addr2 = 0;
      do {
         addr1 = addr2;
         if( (a1 = address()) == 0 ) {
            c = getchr();
            break;
         }
         addr2 = a1;
         if( (c = getchr()) == ';' ) {
            dot = a1;
         }
      } while( c == ';' || c == ',' );
      switch( c ) {
      case 'a':      /* append lines */
         suffix();
         oneaddr(dot, zero);
         dot = addr2;
         append(gettty, addr2);
         break;
      case 'b':      /* set line breaking */
         if( (c = getchr()) == '+' || c == '-' || c == LF ) {
            lbrkon = (c != '-');
            peekc = LF;
         } else {
            error(Q);
         }
         if( addr2 != 0 ) error(Q);
         suffix();
         continue;
      case 'c':      /* change lines */
         suffix();
         twoaddr(dot, dot, zero+1);
         delete(addr1, addr2);
         dot = addr1 - 1;
         append(gettty, addr1-1);
         break;
      case 'd':      /* delete lines */
         suffix();
         twoaddr(dot, dot, zero+1);
         delete(addr1, addr2);
         break;
      case 'E':      /* edit (without warning) new file */
         cflag = 0;
      case 'e':      /* edit new file */
         if( addr2 != 0 ) error(Q);
         if( filename(1) ) {
            if( !sure(c) ) continue;
            newfile( genfile );
         } else {
            if( !sure(c) ) continue;
            newfile( savfile );
         }
         opnfile(savfile, READ);
         append(getfile, zero);
         clsfile();
         cflag = 0;
         continue;
      case 'f':      /* (set) filename */
         if( addr2 != 0 ) error(Q);
         if( filename(1) ) {
            cpystr(genfile, savfile);
         } else {
            puts(savfile, LF);
         }
         continue;
      case 'g':      /* global commands */
      case 'v':      /* global except commands */
         if( inglob ) error(Q);
         twoaddr(zero+1, dol, zero+1);
         if( (delim = getchr()) == LF ) error(Q);
         comprex(delim);
         global( c == 'g' );
         continue;
      case 'h':      /* help command */
         process(help, (filename(1) ? genfile : HELPCMD));
         continue;
      case 'i':      /* insert lines */
         suffix();
         oneaddr(dot, zero+1);
         dot = addr2;
         append(gettty, addr2-1);
         continue;
      case 'J':      /* join lines */
      case 'j':      /* join lines, insert space */
         suffix();
         if( addr2 == 0 ) addr2 = dot;
         if( addr1 == 0 ) {
            addr1 = addr2;
            addr2++;
         }
         if( addr1 <= zero || addr1 > addr2 || addr2 > dol ) {
            error(Q);
         }
         join( c == 'j' );
         break;
      case 'k':      /* mark lines */
         if( (c = getchr()) < 'a' || 'z' < c ) error(Q);
         suffix();
         oneaddr(dot, zero+1);
         marks[c-'a'] = (*addr2 & ~01);
         kflag = 1;
         continue;
      case 'm':      /* move lines */
      case 't':      /* copy lines */
         if( (a1 = address()) == 0 ) error(Q);
         suffix();
         twoaddr(dot, dot, zero+1);
         move( a1, (c == 't') );
         break;
      case LF:      /* print next (current global) line */
         oneaddr( (inglob ? dot : dot+1), zero+1 );
         prlines( 0 );
         continue;
      case 'l':      /* list lines */
         pflag |= LIST;
      case 'p':      /* print lines */
      case 'P':      /* print lines */
         suffix();
         twoaddr(dot, dot, zero+1);
         prlines( 0 );
         continue;
      case 'O':      /* enter open mode, begin match */
         oflag = OBEGN;
      case 'o':      /* enter open mode, end match */
         if( iflag ) error(Q);
         if( (c = getchr()) == ':'
          ||  c == '+'
          ||  c == ';'
          ||  c == '-' ) {
            oflag |= ONEWL;
            switch( c ) {
            case ':':      /* as 'c' command */
               twoaddr(dot, dot, zero+1);
               delete(addr1, addr2);
               dot = addr1 = addr2 = addr1 - 1;
               break;
            case '+': case ';':   /* as 'a' command */
               oneaddr(dot, zero);
               dot = addr2;
               break;
            case '-':      /* as 'i' command */
               oneaddr(dot, zero+1);
               dot = addr2;
               addr1--; addr2--;
               break;
            }
         } else {         /* as 's' command */
            twoaddr(dot, dot, zero+1);
            if( c == LF ) {
               peekc = c;
            } else {
               oflag |= OEXPR;
               comprex( c );
            }
         }
         suffix();
         oflag |= OPMOD;
         opmod(inglob, oflag);
         break;
      case 'Q':      /* Quit without warning */
         cflag = 0;
      case 'q':      /* quit with warning */
         if( addr2 != 0 ) error(Q);
         if( filename(1) ) {
            if( lflag && link(logfile, genfile) < 0 )
               error(genfile);
         } else if( !sure(c) ) {
            continue;
         }
         quit();
      case 'r':      /* read file */
         oneaddr(dol, zero);
         if( filename(1) ) {
            if( *savfile == NUL ) cpystr(genfile, savfile);
         } else if( *savfile != NUL ) {
            cpystr(savfile, genfile);
         } else {
            error(Q);
         }
         opnfile(genfile, READ);
         append(getfile, addr2);
         clsfile();
         continue;
      case 'x':      /* substitute interactively */
         sflag = XCMD;
      case 's':      /* sustitute */
         if( (c = getchr()) == LF || c == SP ) error(Q);
         comprex(c);
         compsub(c);
         if( (peekc = getchr()) == 'g' ) {
            peekc = 0;
            sflag |= GCMD;
         }
         if( (peekc = getchr()) == 'n' ) {
            peekc = 0;
            sflag |= NCMD;
         }
         suffix();
         twoaddr(dot, dot, zero+1);
         substitute( inglob );
         break;
      case 'u':      /* undo last linechange */
         suffix();
         oneaddr(dot, zero+1);
         if( (*addr2 & ~01) != subnewa ) error(Q);
         *addr2 = repline( *addr2, subolda );
         dot = addr2;
         break;
      case 'W':      /* append to file */
         wflag = APPEND;
      case 'w':      /* write to file */
         wflag |= WRITE;
         if( filename(1) ) {
            if( *savfile == NUL ) cpystr(genfile, savfile);
         } else if( *savfile != NUL ) {
            cpystr(savfile, genfile);
         } else {
            error(Q);
         }
         twoaddr(zero+1, dol, zero+1);
         opnfile(genfile, wflag);
         putfile();
         clsfile();
         if( addr1 == zero + 1 && addr2 == dol ) cflag = 0;
         if( cmpstr(savfile, genfile) ) newlog();
         continue;
      case 'z':      /* input/output encryption */
         if( addr2 != 0 ) error(Q);
         suffix();
         if( pflag != 0 ) {
            ppflag = pflag;
            puts(key, LF);
         } else {
            puts("Entering encrypting mode", LF);
            zflag = getkey();
            zfflag = crinit(key, perm);
         }
         continue;
      case '"':      /* terminal page downwards */
         suffix();
         oneaddr(dot, zero);
         setpage(addr1+1, addr1+lines-2);
         prlines( 0 );
         continue;
      case '&':      /* terminal page upwards */
         suffix();
         oneaddr(dot, zero+1);
         setpage(addr2-(lines-3), addr2);
         prlines( 0 );
         continue;
      case '%':      /* terminal page centered */
         suffix();
         oneaddr(dot, zero+1);
         dot = addr2;
         setpage(dot-(lines/2-3), dot+(lines/2-3));
         prlines(1);
         continue;
      case '>':      /* switch prompting on */
         tflag = 1;
         suffix();
         continue;
      case '<':      /* switch prompting off */
         tflag = 0;
         suffix();
         continue;
      case '|':      /* pipe lines to shell */
         twoaddr(dot, dot, zero+1);
         pipeout();
         continue;
      case '=':      /* line number */
         suffix();
         oneaddr(dot, zero);
         count = ((addr2 - zero) & 077777);
         putnum(LF);
         continue;
      case '!':      /* shell command */
         if( addr2 != 0 ) error(Q);
         process(shell, !getshell(unixbuf, '!'));
         puts("!", LF);
         getterm();
         continue;
      case '@':      /* include commandfile */
         if( cmdfid == inplim ) error("@");
         oneaddr(dot, zero);
         if( !filename(0) ) {
            dot = addr2;
            if( rflag ) continue;
            *++cmdfid = STANDIN;
         } else if( rflag ) {
            dot = addr2;
            continue;
         } else if( (*++cmdfid = open(genfile, 0)) < 0 ) {
            cmdfid--;
            error(genfile);
         }
         dot = addr2;
         continue;
      case '#':      /* comment line */
         oneaddr(dot, zero);
         while( (c = getchr()) != LF && c != EOF );
         dot = addr2;
         continue;
      case EOF:      /* end of input */
         if( inglob || !isatty( *cmdfid ) ) return;
         continue;
      default:      /* not allowed */
         error(Q);
      }
      if( pflag ) {
         addr1 = addr2 = dot;
         if( dot != zero ) {
            prlines( 0 );
         } else {
            error(Q);
         }
      }
   }
};

int *address   ()      /* returns one address in command or 0 */
{
   register int *a1, minus, c;
   int n, relerr;

   minus = 0;
   a1 = 0;
   for( ; ; ) {
      if( '0' <= (c = getchr()) && c <= '9' ) {
         n = c - '0';
         while( '0' <= (c = getchr()) && c <= '9' ) {
            n *= 10;
            n += c - '0';
         }
         peekc = c;
         if( a1 == 0 ) a1 = zero;
         a1 += (minus >= 0 ? n : -n);
         minus = 0;
         continue;
      }
      relerr = 0;
      if( a1 || minus ) relerr++;
      switch( c ) {
      case SP:
      case HT:
         continue;
      case '+':
         minus++;
         if( a1 == 0 ) a1 = dot;
         continue;
      case '-':
      case '^':
         minus--;
         if( a1 == 0 ) a1 = dot;
         continue;
      case '?':
      case '/':
         comprex(c);
         a1 = dot;
         for( ; ; ) {
            if( c == '/' ) {
               if( ++a1 > dol ) a1 = zero;
            } else {
               if( --a1 < zero ) a1 = dol;
            }
            if( execute(0, a1) ) break;
            if( a1 == dot ) error(S); /* failed search */
         }
         break;
      case '$':
         a1 = dol;
         break;
      case '.':
         a1 = dot;
         break;
      case '\'':
         if( (c = getchr()) < 'a' || 'z' < c ) error(Q);
         for( a1 = zero; a1 <= dol; a1++ ) {
            if( marks[c-'a'] == (*a1 & ~01) ) break;
         }
         break;
      default:
         peekc = c;
         if( a1 == 0 ) return(0);
         a1 += minus;
         if( a1 < zero || a1 > dol ) error(Q);
         return(a1);
      }
      if( relerr ) error(Q);
   }
};

suffix    ()         /* command suffixes */
{
   register int c;

   if( (c = getchr()) == 'p' ) {
      pflag = PRINT;
      c = getchr();
   } else if( c == 'l' ) {
      pflag = LIST;
      c = getchr();
   }
   if( c == LF ) return;
   error(Q);
};

int sure   (cc)      /* edit/quit verification */
   int cc;
{
   register int a, c, ccc;

   if( !cflag || !vflag ) return( 1 );
   puts("sure? ", EOF);
   for( ccc = cc; c = getchr(); ) {
      if( (a = c) == LF ) return( 0 );
      while( c != LF ) c = getchr();
      if( a == ccc ) return( 1 );
      if( a == 'y' ) return( 1 );
      if( a == 'n' ) return( 0 );
      puts("answer y or n or ", EOF);
      putchr(ccc); putchr(SP); putchr(EOF);
   }
   return( 0 );
};

oneaddr   (a, bound)   /* commands a,i,k,\n,o,u,",&,%,= */
   int *a, *bound;
{
   if( addr2 == 0 ) addr2 = a;
   addr1 = addr2;
   if( bound <= addr2 && addr2 <= dol ) return;
   error(Q);
};

twoaddr   (a1, a2, bound)   /* commands c,d,g,v,m,t,l,0,p,P,s,x,w,W,| */
   int *a1, *a2, *bound;
{
   if( addr2 == 0 ) {
      addr1 = a1;
      addr2 = a2;
   } else if( addr1 == 0 ) {
      addr1 = addr2;
   } else if( addr1 > addr2 ) {
      error(Q);
   }
   if( bound <= addr1 && addr2 <= dol ) return;
   error(Q);
};

setpage   (a1, a2)         /* linelimits for display */
   int *a1, *a2;
{
   addr1 = a1; addr2 = a2;
   if( addr1 <= zero ) addr1 = zero+1;
   if( addr2 > dol ) {
      if( addr1 > dol ) error(Q);
      addr2 = dol;
   }
};

prlines   (split)
         /* display lines */
   int split;
{
   register int *a, w, sp;
   int ww;

   ppflag = pflag;
   ww = (fast ? width : 8);
   for( sp = split, a = addr1-1; ++a <= addr2; ) {
      getline(*a);
      if( sp && a == dot ) {
         ppflag = 0;
         for( w = ww; --w >= 0; ) putchr(SPLIT);
         putchr(LF);
         ppflag = pflag;
         puts(linbuf, LF);
         ppflag = 0;
         for( w = ww; --w >= 0; ) putchr(SPLIT);
         putchr(LF);
         ppflag = pflag;
      } else {
         puts(linbuf, LF);
      }
   }
   if( !sp ) dot = addr2;
};

int filename   (blank)         /* filename in command */
   int blank;
{
   register char *p;
   register int c;

   p = genfile;
   if( (c = getchr()) == LF || c == EOF ) {
      return( 0 );
   } else if( c == SP || c == HT || !blank ) {
      while( c == SP || c == HT ) c = getchr();
      while( c != LF ) {
         if( c == SP || c == HT || c == EOF ) error(Q);
         *p++ = c;
         c = getchr();
      }
      if( p == genfile ) error(Q);
      *p = 0;
      return( 1 );
   } else {
      error(Q);
   }
   return( 0 );
};

cpystr   (str1, str2)         /*  copy str1 to str2 */
   char *str1, *str2;
{
   register char *s1, *s2;

   for( s1 = str1, s2 = str2; *s2++ = *s1++; );
};

int cmpstr   (str1, str2)      /* compare strings */
   char *str1, *str2;
{
   register char *s1, *s2;

   for( s1 = str1, s2 = str2; *s1 == *s2++; )
      if( *s1++ == NUL ) return( 1 );
   return( 0 );
};

clrmarks   ()   /* unmark marked lines */
{
   register int *markp;

   for( markp = marks; markp < &marks[KBSIZE]; *markp++ = 0 );
   subnewa = kflag = 0;
};

newfile   (filen)      /* initialize new edit file */
   char *filen;
{
   endcore.charp = brk(fendcore.charp);
   dot = dol = zero = fendcore.intp;
   endcore.integer = fendcore.integer - 2;
   if( *filen ) cpystr(filen, savfile);
   if( (tmpfid >= 0 && close(tmpfid) < 0)
    || close(creat(tmpfile, 0600)) < 0
    || (tmpfid = open(tmpfile, 2)) < 0 ) {
      puts(tmpfile, EOF); puts("?", LF);
      exit( 1 );
   }
   iblock = oblock = -1;
   tline = 2;
   ichanged = 0;
   if( zflag ) ztflag = makekey(key, tperm);
   cflag = 0;
   clrmarks();
   if( lflag ) newlog();
};

newlog   ()      /* start new ligfile */
{
   register char *p, ppp, *q;
   int sout;
   struct stat statb;

   q = logfile;
   if( *q != NUL ) unlink(q);
   if( logfid >= 0 ) {
      close(logfid);
      logfid = -1;
   }
   if( *savfile != NUL ) {
      for( p = savfile; *p != NUL; p++ )
         if( *p == '/' ) {
            q = logfile;
         } else {
            *q++ = *p;
         }
      if( (q - logfile) > 8 ) q = logfile + 8;
      p = ".elog";
   } else {
      p = "elog";
   }
   while( *q++ = *p++ );
   *q-- = NUL;
   ppp = '1';
   while( stat(logfile, &statb) >= 0 ) *q = ppp++;
   if( (logfid = creat(logfile, 0666)) < 0 ) {
      lflag = 0;
      puts(logfile, EOF); puts("?", LF);
      return;
   }
   if( *savfile ) {
      sout = standout;
      standout = logfid;
      putchr('E'); putchr(SP);
      puts(savfile, LF);
      newmarks();
      if( (count = (dot - zero) & 077777) != 0 ) putnum(LF);
      standout = sout;
   }
};

opnfile   (file, rwa)      /* open file for READ, WRITE or APPEND */
   char *file; int rwa;      /* read, write, append */
{
   count = 0;
   if( rwa == READ ) {
      if( (femfid = open(file, 0)) < 0 )
         error(file);
      ninbuf = 0;
   } else if( rwa == APPEND+WRITE ) {
      if( (femfid = open(file, 1)) < 0 || lseek(femfid, 0L, 2) < 0 )
         if( (femfid = creat(file, 0666)) < 0 )
            error(file);
   } else if( rwa = WRITE ) {
      if( (femfid = creat(file, 0666)) < 0 )
         error(file);
   } else {
      error(Q);   /* never called */
   }
   return;
};

clsfile   ()      /* close file */
{
   if( close(femfid) < 0 ) error(genfile);
   femfid = -1;
   if( vflag ) putnum(LF);
};

error   (s)         /* error messages and recovery */
   char *s;
{
   register int c;

   if( !eflag ) { puts(s, EOF); puts("?", LF); };
   count = 0;
   if( globp ) {
      lastc = LF;
      globp = 0;
   }
   if( peekc = lastc ) while( (c = getchr()) != LF && c != EOF );
   if( femfid >= 0 && (femfid = close(femfid)) != -1 ) femfid = -1;
   longjmp(savej, 1);
};

char *getblock   (atl, iof)   /* get line from filebuffer */
   int atl, iof;      /* blockno + offset, READ or WRITE */
{
   register int bno, off;
   register int n;
   char *p1, *p2;

   if( (bno = blocknr(atl)) >= 255 ) {
      lastc = LF;
      error(T);
   }
   off = offset(atl);
   if( bno == iblock ) {
      ichanged = (iof == WRITE);
      return( ibuff + off );
   } else if( bno == oblock ) {
      return( obuff + off );
   } else if( iof == READ ) {
      if( ichanged) {
         if( ztflag ) crblock(tperm, ibuff, FBSIZE, (long)0);
         if( lseek(tmpfid, (long)iblock << 9, 0) < 0
          || write(tmpfid, ibuff, FBSIZE) != FBSIZE ) {
            error(T);
         }
      }
      ichanged = 0;
      iblock = bno;
      if( lseek(tmpfid, (long)iblock << 9, 0) < 0
       || read(tmpfid, ibuff, FBSIZE) != FBSIZE ) {
         error(T);
      }
      if( ztflag ) crblock(tperm, ibuff, FBSIZE, (long)0);
      return( ibuff + off );
   } else if( oblock >= 0 ) {
      if( ztflag ) {
         p1 = obuff; p2 = crbuf;
         for( n = FBSIZE; --n >= 0; *p1++ = *p2++ );
         crblock(tperm, crbuf, FBSIZE, (long)0);
         if( lseek(tmpfid, (long)oblock << 9, 0) < 0
          || write(tmpfid, crbuf, FBSIZE) != FBSIZE ) {
            error(T);
         }
      } else {
         if( lseek(tmpfid, (long)oblock << 9, 0) < 0
          || write(tmpfid, obuff, FBSIZE) != FBSIZE ) {
            error(T);
         }
      }
   }
   oblock = bno;
   return( obuff + off );
};

getline   (tl)      /* read line (tl) from tmpfile into linbuf */
   int tl;
{
   register char *bp, *lp;
   register int nl;
   extern char *getblock();

   lp = linbuf;
   bp = getblock(tl, READ);
   nl = FBSIZE - offset(tl);
   tl &= ~0377;
   while (*lp++ = *bp++)
      if( --nl == 0) {
         bp = getblock( (tl += 0400), READ );
         nl = FBSIZE;
      }
   return;
};

int putline   ()   /* writes line in linbuf to tmpfile */
{
   register char *bp, *lp;
   register int nl;
   int tl;
   extern char *getblock();

   cflag = 1;
   tl = tline;
   bp = getblock(tl, WRITE);
   nl = FBSIZE - offset(tl);
   tl &= ~0377;
   lp = linbuf;
   while( *bp = *lp++ ) {
      if( *bp++ == LF ) {
         *--bp = NUL;
         linebp = lp;
         break;
      } else if( --nl == 0 ) {
         bp = getblock( (tl += 0400), WRITE );
         nl = FBSIZE;
      }
   }
   nl = tline;
   tline += ((((lp - linbuf) + 03) >> 1) & 077776);
   return( nl );
};

int getchr   ()      /* command character */
{
   if( (lastc = peekc) != NUL ) {
      peekc = NUL;
      return( lastc );
   } else if( globp ) {
      if( (lastc = *globp++) != NUL ) return( lastc );
      globp = 0;
      return( EOF );
   }
   return( lastc = getch() );
};

int getch   ()      /* character from command input */
{
   register int c, n;

   if( *inbufp == NUL ) {
      inbufp = &inpbuf[0];
      *inbufp = NUL;
      eflag = 1;
      if( (n = read(*cmdfid, inpbuf, (isatty(*cmdfid) ? CBSIZE : 1))) <= 0 ) {
         eflag = 0;
         *inbufp = NUL;
         if( cmdfid == inpfid ) {
            return( EOF );
         } else if( xflag ) {
            return( getch() );
         } else if( *cmdfid == STANDIN ) {
            cmdfid--;
         } else if( close( *cmdfid-- ) < 0 ) {
            error(Q);
         }
         return( getch() );
      } else if( lflag ) {
         write(logfid, inpbuf, n);
      }
      inpbuf[n] = NUL;
      eflag = 0;
   }
   c = (*inbufp++ & ASCII);
   if( (oflag & OPMOD) == 0 && *cmdfid != STANDIN ) putchar( c );
   return( c );
};

putnum   (nl)         /* number to interactive output */
   int nl;
{
   register int r;

   r = count % 10;
   if( count /= 10 ) putnum(NUL);
   putchr( r + '0' );
   if( nl ) putchr(nl);
};

puts   (strp, nl)      /* string to interactive output */
   char *strp, nl;
{
   register char *sp;

   for( sp = strp; *sp; putchr( *sp++ ) );
   putchr(nl);
};

putchr   (ac)         /* character to interactive output */
   int ac;
{
   register char *lp;
   register int c;

   lp = linp;
   if( (c = ac) == EOF ) {
      if( lp > line ) {
         linp = line;
         write(standout, line, lp-line);
         return;
      }
   } else if( ppflag & LIST ) {
      if( c == HT ) {
         *lp++ = '-';
         c = '>';
      } else if( c == BS ) {
         *lp++ = '-';
         c = '<';
      } else if( c == LF ) {
         *lp++ = '$';
      } else if( (c < SP || c == DEL) ) {
         *lp++ = '\\';
         *lp++ = ((c >> 6) & 07) + '0';
         *lp++ = ((c >> 3) & 07) + '0';
         c     = ( c       & 07) + '0';
      }
   }
   if( (*lp++ = c) == LF || lp >= &line[64] ) {
      linp = line;
      write(standout, line, lp-line);
   } else {
      linp = lp;
   }
   return;
};

int gettty   ()      /* get lines in linbuf from command input */
{
   register int c;
   register char *p, *oldglobp;

   aflag = 1;
   oldglobp = globp;
   for( p = linbuf; (c = getchr()) != LF; ) {
      if( c == EOF ) {
         if( oldglobp ) peekc = c;
         return( c );
      } else if( (c &= ASCII) == NUL ) {
         continue;
      } else if( p >= &linbuf[LBSIZE-3] ) {
         error(Q);
      } else {
         *p++ = c;
      }
   }
   *p = NUL;
   if( linbuf[0] == '.' && linbuf[1] == NUL ) {
      aflag = 0;
      return( EOF );
   }
   return( 0 );
};

int getfile   ()      /* get lines in linbuf from file */
{
   register int c;
   register char *lp, *gp;

   for( lp = linbuf, gp = nextip; ; ) {
      if( --ninbuf < 0 ) {
         if( (ninbuf = read(femfid, genbuf, LBSIZE)-1) < 0 ) {
            if( lp == linbuf ) return(EOF);
            break;
         }
         if( zfflag ) {
            for( gp = genbuf; gp < &genbuf[ninbuf]; ) {
               if( *gp++ & 0200 ) {
                  crblock(perm, genbuf, ninbuf+1, count);
                  break;
               }
            }
         }
         gp = genbuf;
      }
      if( (c = (*gp++ & ASCII)) == NUL || c == LF ) {
         *lp = NUL;
         count++;
         break;
      } else if( lp >= &linbuf[LBSIZE-1] ) {
         ninbuf++;
         gp--;
         *lp = NUL;
         break;
      } else {
         *lp++ = c;
         count++;
      }
   }
   nextip = gp;
   return( 0 );
};

putfile   ()      /* write lines from addr1 to addr2 to file */
{
   register char *gp, *lp;
   register int nib;
   int *a1;

   if( addr2 == zero ) return;
   nib = LBSIZE;
   gp = genbuf;
   for( a1 = addr1; a1 <= addr2; ) {
      getline( *a1++ );
      for( lp = linbuf; ; ) {
         if( --nib < 0 ) {
            nib = gp - genbuf;
            if( zfflag ) crblock(perm, genbuf, nib, count-nib);
            if( write(femfid, genbuf, nib) != nib ) error(W);
            nib = LBSIZE - 1;
            gp = genbuf;
         }
         count++;
         if( (*gp++ = *lp++) == NUL ) {
            gp[-1] = LF;
            break;
         }
      }
   }
   nib = gp - genbuf;
   if( zfflag ) crblock(perm, genbuf, nib, count-nib);
   if( write(femfid, genbuf, nib) != nib ) {
      error(W);
   }
};

append   (f, a)         /* append lines from f after a */
   int (*f)(), *a;
{
   register int *a1, *a2, *rdot;
   int nline, tl;

   for( nline = 0, rdot = a; (*f)() == 0; ) {
      if( dol >= endcore.intp ) {
         if( /*NOSTRICT*/sbrk(MBSIZE) == -1 ) error(M);
         endcore.integer += MBSIZE;
      }
      tl = putline();
      nline++;
      dot = ++rdot; a1 = ++dol;
      for( a2 = a1 + 1; a1 > rdot; *--a2 = *--a1 );
      *rdot = tl;
   }
   aline = nline;
   return;
};

delete   (ad1, ad2)         /* delete lines from ad1 to ad2 */
   int *ad1, *ad2;
{
   register int *a1, *a2, *a3;

   for( (a1 = ad1, a2 = ad2+1, a3 = dol); a2 <= a3; *a1++ = *a2++ );
   dol -= (ad2 -ad1 + 1);
   dot = (ad1 > dol ? dol : ad1);
   cflag = 1;
};

move   (dest, copy)      /* move (copy) lines to dest */
   int *dest, copy;
{
   register int *adt, *ad1, *ad2;
   int getcopy();

   adt = dest;
   if( copy ) {
      ad1 = dol;
      append(getcopy, ad1++);
      ad2 = dol;
   } else {
      ad2 = addr2;
      for( ad1 = addr1; ad1 <= ad2; *ad1++ &= ~01 );
      ad1 = addr1;
   }
   ad2++;
   if( adt < ad1 ) {
      dot = adt + (ad2 - ad1);
      if( (++adt) == ad1 ) return;
      reverse(adt, ad1);
      reverse(ad1, ad2);
      reverse(adt, ad2);
   } else if( adt >= ad2 ) {
      dot = adt++;
      reverse(ad1, ad2);
      reverse(ad2, adt);
      reverse(ad1, adt);
   } else {
      error(Q);
   }
   cflag = 1;
};

reverse   (a1, a2)
   register int *a1, *a2;
{
   register int t;

   for( ; ; ) {
      t = *--a2;
      if( a2 <= a1 ) return;
      *a2 = *a1;
      *a1++ = t;
   }
};

int getcopy   ()
{
   if( addr1 > addr2 ) return( EOF );
   getline(*addr1++);
   return( 0 );
};

join   (space)
   int space;
{
   register char *gp, *lp;
   register int *a1;
   int newa;

   if( addr1 == addr2 ) return;
   for( gp = genbuf, a1 = addr1; a1 <= addr2; ) {
      getline( *a1++ );
      lp = linbuf;
      while( *gp = *lp++ ) {
         if( gp++ >= &genbuf[LBSIZE-3] ) error(Q);
      }
      if( a1 <= addr2 && space ) *gp++ = SP;
   }
   for( lp = linbuf, gp = genbuf; *lp++ = *gp++; );
   newa = putline();
   for( a1 = addr1; a1 <= addr2; a1++ ) cpymark((*a1) & ~01, newa);
   *addr1 = newa;
   delete(addr1+1, addr2);
   dot = addr1;
};

global   (gorv)
   int gorv;
{
   register int c, *a1;
   register char *gp;
   char globuf[GBSIZE];

   for( gp = globuf; (c = getchr()) != LF; ) {
      if( c == '\\' ) {
         if( (c = getchr()) == '\\' ) {
            *gp++ = c;
            c = getchr();
         }
         if( c != LF ) *gp++ = '\\';
      }
      if( c == EOF ) error(Q);
      *gp++ = c;
      if( gp >= &globuf[GBSIZE-3] ) error(Q);
   }
   *gp++ = LF; *gp = NUL;
   for (a1 = zero; a1 <= dol; a1++ ) {
      *a1 &= ~01;
      if( addr1 <= a1 && a1 <= addr2 && gorv == execute(0, a1) ) {
         *a1 |= 01;
      }
   }
   /* Special case: g/.../d (avoid n^2 algorithm)   */
   if( globuf[0] == 'd' && globuf[1] == LF && globuf[2] == NUL ) {
      gdelete();
   } else {
      for( a1 = zero; a1 <= dol; a1++ ) {
         if( *a1 & 01) {
            *a1 &= ~01;
            dot = a1;
            globp = globuf;
            commands( 1 );
            a1 = zero;
         }
      }
   }
};

gdelete   ()
{
   register int *a1, *a2, *a3;

   a3 = dol;
   for( a1 = zero + 1; (*a1 & 01) == 0; a1++) if( a1 >= a3 ) return;
   for( a2 = a1 + 1; a2 <= a3; ) {
      if( *a2 & 01 ) {
         a2++;
         dot = a1;
      } else {
         *a1++ = *a2++;
      }
   }
   dol = a1-1;
   if( dot > dol ) dot = dol;
   cflag = 1;
};

cpymark   (old, new)
   int old, new;
{
   register int *markp;

   if( kflag ) {
      for( markp = marks; markp <= marklim; markp++ ) {
         if( *markp == old ) *markp = new;
      }
   }
};

newmarks   ()
{
   register int *markp, *a;

   if( kflag ) {
      for( a = zero; ++a <= dol; ) {
         for( markp = marks; markp <= marklim; markp++ ) {
            if( *markp == (*a & ~01) ) {
               count = (a - zero) & 077777;
               putnum(EOF);
               putchr('k');
               putchr(markp-marks+'a');
               putchr(LF);
            }
         }
      }
   }
};

int repline   (oldline, newline)
   int oldline, newline;
{
   register int old, new;

   new = subnewa = newline;
   old = subolda = (oldline & ~01);
   cpymark(old, new);
   return( new );
};

comprex   (delim)
   int delim;
{
   register int eof, c;
   register char *ep;
   int cclcnt;
   char *lastep, bracket[NBRA], *bracketp;

   ep = expbuf;
   eof = delim;
   if( (c = getchr()) == eof || c == LF ) {
      if( *ep == NUL ) error(Q);
      if( c == LF ) peekc = c;
      return;
   }
   bracketp = bracket; nbra = 0;
   circfl = 0;
   if( c == '^' ) {
      c = getchr();
      circfl++;
   }
   peekc = c;
   for( lastep = 0; ep < &expbuf[EBSIZE]; ) {
      if( (c = getchr()) == eof ) {
         if( bracketp != bracket ) break;
         *ep++ = CEOF;
         return;
      }
      if( c != '*' ) lastep = ep;
      switch( c ) {
      case '\\':
         if( (c = getchr()) == LF ) {
            goto cerror;
         } else if( c == '(' ) {
            if( nbra >= NBRA ) goto cerror;
            *bracketp++ = nbra;
            *ep++ = CBRA;
            *ep++ = nbra++;
         } else if( c == ')' ) {
            if( bracketp <= bracket ) goto cerror;
            *ep++ = CKET;
            *ep++ = *--bracketp;
         } else if( '1' <= c && c < ('1' + NBRA) ) {
            *ep++ = CBACK;
            *ep++ = (c - '1');
         } else {
            *ep++ = CCHR;
            *ep++ = c;
         }
         continue;
      case '.':
         *ep++ = CDOT;
         continue;
      case LF:
         goto cerror;
      case '*':
         if( lastep == 0 || *lastep == CBRA || *lastep == CKET )
            goto defchar;
         *lastep |= STAR;
         continue;
      case '$':
         if( (peekc = getchr()) != eof )
            goto defchar;
         *ep++ = CDOL;
         continue;
      case '[':
         if( (c = getchr()) == '^' ) {
            *ep++ = NCCL;
            c = getchr();
         } else {
            *ep++ = CCL;
         }
         *ep++ = NUL;
         cclcnt = 1;
         do {
            if( c == LF ) goto cerror;
            if( c == '-' && ep[-1] != NUL ) {
               if( (c = getchr()) == ']' ) {
                  *ep++ = '-';
                  cclcnt++;
                  break;
               }
               while( ep[-1] < c ) {
                  *ep = ep[-1] + 1;
                  ep++;
                  cclcnt++;
                  if( ep >= &expbuf[EBSIZE] ) goto cerror;
               }
            }
            *ep++ = c;
            cclcnt++;
            if( ep >= &expbuf[EBSIZE] ) goto cerror;
         } while( (c = getchr()) != ']' );
         lastep[1] = cclcnt;
         continue;
      defchar:
      default:
         *ep++ = CCHR;
         *ep++ = c;
      }
   }
   cerror:
   expbuf[NUL] = 0;
   nbra = 0;
   error(Q);
};

compsub   (delim)
   int delim;
{
   register int seof, c;
   register char *p;

   p = rhsbuf;
   for( seof = delim; ; ) {
      if( (c = getchr()) == '\\' ) {
         c = (getchr() | 0200);
      } else if( c == LF ) {
         if( globp ) c |= 0200; else error(Q);
      } else if( c == seof ) {
         break;
      }
      *p++ = c;
      if( p >= &rhsbuf[SBSIZE] ) error(Q);
   }
   *p++ = NUL;
};

substitute   (inglob)
   int inglob;
{
   register int *a1;
   int getsub();

   if( (sflag & NCMD) ) count = 0;
   for( a1 = addr1; a1 <= addr2; a1++ ) {
      if( execute(0, a1) == 0 ) continue;
      inglob |= 1;
      if( !(sflag & XCMD) || confirmed() ) {
         dosub();
         count++;
      } else {
         loc2 = loc1 + 1;
      }
      if( (sflag & XCMD) || (sflag & GCMD) ) {
         while( *loc2 ) {
            if( execute(1, (int *)0) == 0 ) break;
            if( !(sflag & XCMD) || confirmed() ) {
               dosub();
               count++;
            } else {
               loc2 = loc1 + 1;
            }
         }
      }
      *a1 = repline( *a1, putline() );
      append(getsub, a1);
      dot = (a1 += aline);
      addr2 += aline;
   }
   if( sflag & NCMD ) {
      putnum(LF);
   } else if( !inglob ) {
      error(S);
   }
};

int confirmed   ()
{
   register char ch;

   if( !rflag && *cmdfid != STANDIN ) {
      xflag = 1;
      *++cmdfid = STANDIN;
   }
   for( ; ; ) {
      underline(linbuf, loc1, loc2);
      if( (ch = getch()) == LF ) break;
      while( getch() != LF );
      if( ch == CONFIRM ) break;
      puts("? '.' to confirm", LF);
   }
   if( xflag ) {
      cmdfid--;
      xflag = 0;
   }
   return( ch == CONFIRM );
};

underline   (lin, l1, l2)
   char *lin, *l1, *l2;
{
   register char *p, *l;
   register int col;
   int c;

   for( p = l = lin; p < l1; ) if( *p++ == LF ) l = p;
   if( fast || tabs ) {
      puts(l, LF);
      for( col = 0; l < l1; l++ ) {
         if( printb( *l ) ) {
            col++;
         } else if( *l == HT ) {
            col += (8 - (col % 8));
         } else if( *l == BS ) {
            col--;
         } else if( *l == CR ) {
            col = 0;
         }
      }
      for( p = genbuf, c = col; c >= 8; c -= 8 ) *p++ = HT;
      for( ; --c >= 0; ) *p++ = SP;
      for( ; l < l2; l++ ) {
         if( printb( *l ) ) {
            *p++ = SCORE;
            col++;
         } else if( *l == HT ) {
            for( col += (c = (8 - (col % 8))); --c >= 0; )
               *p++ = SCORE;
         } else if( *l == BS ) {
            *p++ = BS;
            col--;
         } else if( *l == CR ) {
            *p++ = CR;
            col = 0;
         }
      }
      *p = NUL;
   } else {
      p = genbuf;
      do {
         if( l == l1 ) *p++ = QUOTE;
         if( l == l2 ) *p++ = QUOTE;
      } while( *p++ = *l++ );
   }
   puts(genbuf, EOF);
};

dosub   ()
{
   register char *lp, *sp, *rp;
   char *le;
   int c;

   for( sp = genbuf, lp = linbuf; lp < loc1; *sp++ = *lp++ );
   for( rp = rhsbuf; c = (*rp++ & 0377); ) {
      if( c == '&' ) {
         for( lp = loc1; lp < loc2; *sp++ = *lp++ ) {
            if( sp > &genbuf[LBSIZE] ) error(Q);
         }
      } else if( (c & 0200) && (c &= ASCII) >= '1' && c < (nbra + '1') ) {
         lp = braslist[ c -= '1' ]; le = braelist[c];
         while( lp < le ) {
            *sp++ = *lp++;
            if( sp > &genbuf[LBSIZE] ) error(Q);
         }
      } else {
         *sp++ = (c & ASCII);
         if( sp >= &genbuf[LBSIZE] ) error(Q);
      }
   }
   lp = loc2;
   loc2 = sp - genbuf + linbuf;
   while( *sp++ = *lp++ ) if( sp >= &genbuf[LBSIZE] ) error(Q);
   for( lp = linbuf, sp = genbuf; *lp++ = *sp++; );
};

getsub   ()
{
   register char *p1, *p2;

   p1 = linbuf;
   if( (p2 = linebp) == 0)
      return(EOF);
   while (*p1++ = *p2++);
   linebp = 0;
   return(0);
}

execute   (gf, addr)
   int gf, *addr;
{
   register char *p1, *p2, c;

   if( !gf ) {
      if( addr == zero ) return( 0 );
      getline( *addr );
      p1 = linbuf;
      locs = 0;
   } else {
      if( circfl ) return( 0 );
      p1 = locs = loc2;
   }
   for( c = NBRA; --c >= 0; braslist[c] = braelist[c] = 0 );
   p2 = expbuf;
   if( circfl ) {         /* begin of line */
      loc1 = p1;
      return( advance(p1, p2) );
   } else if( *p2 == CCHR ) {    /* fast check for first character */
      c = p2[1];
      do {
         if( *p1 != c ) continue;
         if( advance(p1, p2) ) {
            loc1 = p1;
            return( 1 );
         }
      } while( *p1++ );
      return( 0 );
   } else {         /* regular algorithm */
      do {
         if( advance(p1, p2) ) {
            loc1 = p1;
            return( 1 );
         }
      } while( *p1++ );
      return( 0 );
   }
};

int advance   (linep, exprp)
   char *linep, *exprp;
{
   register char *lp, *ep, *curlp;
   int i;

   for( lp = linep, ep = exprp; ; ) {
      switch( *ep++ ) {
      case CCHR:
         if( *ep++ != *lp++ ) return( 0 );
         continue;
      case CDOT:
         if( *lp++ == NUL ) return( 0 );
         continue;
      case CDOL:
         if( *lp != 0 ) return( 0 );
         continue;
      case CEOF:
         loc2 = lp; return( 1 );
      case CCL:
         if( !cclass(ep, *lp++) ) return( 0 );
         ep += *ep;
         continue;
      case NCCL:
         if( cclass(ep, *lp++) ) return( 0 );
         ep += *ep;
         continue;
      case CBRA:
         braslist[*ep++] = lp;
         continue;
      case CKET:
         braelist[*ep++] = lp;
         continue;
      case CBACK:
         if( braelist[i = *ep++] == 0 ) error(Q);
         if( !backref(i, lp) ) return( 0 );
         lp += braelist[i] - braslist[i];
         continue;
      case CBACK|STAR:
         if( braelist[i = *ep++] == 0 ) error(Q);
         for( curlp = lp; backref(i, lp); lp += braelist[i] - braslist[i] );
         while( lp >= curlp ) {
            if( advance(lp, ep) ) return( 1 );
            lp -= braelist[i] - braslist[i];
         }
         continue;
      case CDOT|STAR:
         for( curlp = lp; *lp++; );
         goto star;
      case CCHR|STAR:
         for( curlp = lp; *lp++ == *ep; );
         ep++;
         goto star;
      case CCL|STAR:
         for( curlp = lp; cclass(ep, *lp++); );
         ep += *ep;
         goto star;
      case NCCL|STAR:
         for( curlp = lp; !cclass(ep, *lp++); );
         ep += *ep;
      star:
         do {
            if( --lp == locs ) break;
            if( advance(lp, ep) ) return( 1 );
         } while( lp > curlp );
         return( 0 );
      default:
         error(Q);
      }
   }
};

int backref   (index, linep)
   int index; char *linep;
{
   register char *ep, *sp, *lp;

   ep = braelist[index]; sp = braslist[index]; lp = linep;
   while( *sp++ == *lp++ ) if( sp >= ep ) return( 1 );
   return( 0 );
};

int cclass   (setp, chr)
   char *setp, chr;
{
   register char *s, c;
   register int n;

   if( (c = chr) == NUL ) return( 0 );
   for( s = setp, n = *s++; --n; ) if( *s++ == c ) return( 1 );
   return( 0 );
};

nxtcol   (curp, newp)
   char *curp, *newp;
{
   register char *colp, ch;
   register int curcol;

   for( colp = colump, curcol = *colp; curp < newp; ) {
      if( SP <= (ch = *curp++) && ch < DEL ) {
         *++colp = ++curcol;
      } else if( ch == HT ) {
         *++colp = (curcol += (8 - (curcol % 8)));
      } else if( ch == BS ) {
         *++colp = --curcol;
      } else if( ch == CR ) {
         *++colp = curcol = 0;
      } else {
         *++colp = curcol;
      }
   }
   if( newp < curp ) {
      colump = linbuf + (newp - genbuf);
   } else {
      colump = colp;
   }
   return;
};

char samecol   (curp, colp, curcol)
   char *curp, *colp; int curcol;
{
   register char *cp, *cl;
   register int cc;

   for( cp = curp, cl = colp, cc = curcol; genbuf <= cp; cl--, cp-- ) {
      if( *cl == cc && printb( *cp ) ) {
         return( *cp );
      }
   }
   return( NUL );
};

erase   (curp, newp)
   char *curp, *newp;
{
   register int curcol, newcol;
   register char ch;
   char *colp;

   for( colp = colump, curcol = *colp; newp <= --curp; ) {
      for( newcol = *--colp; newcol != curcol; ) {
         ch = samecol(curp, colp, curcol);
         putch( (ch == NUL ? SP : ch) );
         if( newcol < curcol ) {
            putch(BS); putch(BS);
            curcol--;
         } else {
            curcol++;
         }
      }
   }
   ch = samecol(curp, colp-1, *colp);
   putch( (ch == NUL ? SP : ch) ); putch(BS);
   colump = colp;
};

putchs   (start, end)
   char *start, *end;
{
   register char *sp, *ep, ch;

   for( sp = start, ep = end; sp < ep; sp++ ) {
      if( (SP <= (ch = *sp) && *sp < DEL)
       || ch == HT
       || ch == BS
       || ch == CR ) {
         putch(ch);
      }
   }
};

putch   (c)
   char c;
{
   if( write(standout, &c, 1) );
};

opmod   (inglob, oflag)
   int inglob, oflag;
{
   register int *a1;
   register char *lp, *gp;
   char *loc;
   extern int getopen();

   thresh = (lbrkon ? width-20 : LBSIZE-40);
   setraw();      /* terminal into raw mode*/
   if( (oflag & OEXPR) == 0 ) loc = linbuf;
   for( a1 = addr1; a1 <= addr2; a1++ ) {
      if( (oflag & OEXPR) != 0 ) {
         if( !execute(0, a1) ) continue;
         loc = ((oflag & OBEGN) != 0 ? loc1 : loc2);
      } else if( (oflag & ONEWL) == 0 ) {
         getline(*a1);
      } else {
         linbuf[0] = NUL;
      }
      inglob |= 01;
      for( lp = linbuf, gp = genbuf; *gp++ = *lp++; );
      loc += (genbuf-linbuf);
      gopeof = 0; cgenp = loc;
      if( (oflag & ONEWL) == 0 ) {
         if( getopen() == 0 && opchange ) {
            *a1 = repline( *a1, putline() );
            dot = a1;
         }
      }
      append(getopen, a1);
      a1 += aline;
      addr2 += aline;
   }
   setcook();      /* terminal in cook mode */
   if( inglob == 0 ) error(S);
};

int getopen   ()
{
   register char *cp, *gp, *np;
   char *bp;
   int lbflag, hsflag, rsflag, lbr, ch, lastch, c1, c2;
   extern int spell(), help(), getch();

   if( gopeof == 1 ) return( EOF );
   colump = &linbuf[0]; *colump = NUL;
   if( (np = cgenp) == genbuf ) {
      putch(BSL); putch(BS);
   } else {
      putchs(genbuf, np);
   }
   nxtcol(genbuf, np);
   ch = NUL; rsflag = hsflag = opchange = 0;
   lbr = lbrkon;
   for( ; ; ) {
      hsflag >>= 1; lbflag = 0; bp = (char *)0; cp = np;
      if( lbr
       && cp != genbuf
       && *(cp-1) != SP
       && *(cp-1) != HT
       && *colump >= thresh ) {
         lbflag = 1;
      }
      lastch = ch;
      switch( ch = getch() ) {
      case HELP:
         if( lastch == BSL ) goto escaped;
         if( rsflag ) { putch(']'); rsflag = 0; }
         setcook();
         process(help, HELPKEYS);
         setraw();
         if( fast ) goto retypen;
         hsflag = 2;
         continue;
      case MARGIN:
         if( lastch == BSL ) goto escaped;
         if( rsflag ) { putch(']'); rsflag = 0; }
         if( lbr = !lbr ) putch(BEL);
         continue;
      case SCOPEMOD:
         if( lastch == BSL ) goto escaped;
         if( scflag = !scflag && rsflag ) goto retype;
         continue;
      case INTERRUPT:
         if( lastch == BSL ) goto escaped;
         opchange = 0;
         setcook();
         error("\n");
      case SPELL:
         if( lastch == BSL ) goto escaped;
         if( rsflag ) { putch(']'); rsflag = 0; }
         if( np == genbuf ) goto oerror;
         for( ; --cp >= genbuf; ) {
            if( *cp != SP && *cp != HT ) break;
         }
         if( cp < genbuf || !alfnum( *cp ) ) goto oerror;
         gp = &rhsbuf[SBSIZE-1];
         for( *gp-- = NUL; alfnum( *cp ); *gp-- = *cp-- ) {
            if( gp < rhsbuf ) goto oerror;
         }
         putch(CR); putch(LF);
         putch('!'); putch('!'); putch(CR); putch(LF);
         setcook();
         process(spell, ++gp);
         setraw();
         putch('!'); putch('!'); putch(CR); putch(LF);
         if( fast ) goto retypen;
         hsflag = 2;
         continue;
      case CLOSE:
      case CLOSEA:
         if( lastch == BSL ) goto escaped;
         if( rsflag ) { putch(']'); rsflag = 0; }
         gopeof = 1;
         for( cp = linbuf, gp = genbuf; gp < np; *cp++ = *gp++ );
         while( *cp++ = *gp++ );
         if( ch == CLOSE ) putchs(np, --gp);
         putch(CR); putch(LF);
         return( 0 );
      case NEXTCHAR:
         if( lastch == BSL ) goto escaped;
         if( *np == NUL ) goto oerror;
         if( lbflag ) {
            if( *np == SP ) {
               bp = np;
               *np = LF;
            } else if( *np == HT ) {
               bp = np;
            }
            lbflag = 0;
         }
         np++;
         goto linebreak;
      case NEXTWORD:
         /* layout*, alfnum+ || layout* same+ */
         if( lastch == BSL ) goto escaped;
         if( *np == NUL ) goto oerror;
         for( ch = *np; ch != NUL; ch = *++np ) {
            if( lbflag ) {
               if( ch == SP ) {
                  bp = np;
                  *np = LF;
               } else if( ch == HT ) {
                  bp = np;
               }
               lbflag = 0;
            } else if( ch != SP && ch != HT ) {
               break;
            }
         }
         if( alfnum( ch ) ) {
            while( alfnum( *np ) ) np++;
         } else if( ch != NUL ) {
            while( *++np == ch );
         }
         ch = NUL;
         goto linebreak;
      case LINEEND:
         if( lastch == BSL ) goto escaped;
         if( *np == NUL ) goto oerror;
         while( *++np != NUL );
         goto forward;
      case BACKCHAR:
         if( lastch == BSL ) goto escaped;
         if( np == genbuf ) goto oerror;
         np--;
         goto backward;
      case BACKWORD:
          /* alfnum+, layout* || same+, layout* */
         if( lastch == BSL ) goto escaped;
         if( np == genbuf ) goto oerror;
         for( ch = *--np; np >= genbuf; ch = *--np ) {
            if( ch != SP && ch != HT ) break;
         }
         if( np < genbuf ) {
            ;
         } else if( alfnum( ch ) ) {
            while( alfnum( *np ) ) np--;
         } else {
            while( *--np == ch );
         }
         ch = NUL;
         np++;
         goto backward;
      case BACKBEGIN:
         if( lastch == BSL ) goto escaped;
         if( np == genbuf ) goto oerror;
         np = genbuf;
         goto backward;
      case RETYPE:
         if( lastch == BSL ) goto escaped;
         goto retype;
      case DELCHAR:
      case DELCHARA:
         if( lastch == BSL ) goto escaped;
         if( np == genbuf ) goto oerror;
         np--;
         goto delete;
      case DELWORD:
         if( lastch == BSL ) goto escaped;
         if( np == genbuf ) goto oerror;
         for( ch = *--np; np >= genbuf; ch = *--np ) {
            if( ch != SP && ch != HT ) break;
         }
         if( np < genbuf ) {
            ;
         } else if( alfnum( ch ) ) {
            while( alfnum( *np ) ) np--;
         } else {
            while( *--np == ch );
         }
         ch = NUL;
         np++;
         goto delete;
      case DELEND:
         if( lastch == BSL ) goto escaped;
         if( *np == NUL ) goto oerror;
         *np = NUL;
         goto delete;
      case DELBEGIN:
         if( lastch == BSL ) goto escaped;
         if( np == genbuf ) goto oerror;
         np = genbuf;
         goto delete;
      case SP:
         if( lbflag ) ch = LF;
      case HT:
         if( lbflag ) bp = np;
         lbflag = 0;
         goto insert;
      case CR:      /* carriage return */
         if( lastch == BSL ) goto escaped;
         ch = LF;
      case LF:      /* linefeed */
         /* treated as end of line, forced break */
         bp = np;
      default:      /* other character */
         goto insert;
      }
   escaped:
      if( rsflag ) { putch(']'); rsflag = 0; }
      *--cp = ch; colump--;
      putch( BS ); putch( SP );
      putch( (ch == CR ? CR : BS) );
      nxtcol(cp, np);
      opchange = 1;
      continue;
   insert:
      for( gp = cp, c1 = ch; c1 && gp < genlim; ) {
         c2 = *gp;
         *gp++ = c1;
         c1 = c2;
      };
      *gp = NUL;
      np++;
      opchange = 1;
   linebreak:
      if( !bp ) goto forward;
      opchange = 1;
      for( cp = linbuf, gp = genbuf; gp < bp; *cp++ = *gp++ );
      *cp = NUL;
      if( *gp == LF ) gp++;
      np -= (gp - genbuf);
      for( cp = genbuf; *cp++ = *gp++; );
      if( rsflag ) { putch(']'); rsflag = 0; }
      putch(CR); putch(LF);
      cgenp = np;
      return( 0 );
   forward:
      nxtcol(cp, np);
      if( rsflag ) { putch(']'); rsflag = 0; }
      putchs(cp, np);
      continue;
   retype:
      if( rsflag ) { putch(']'); rsflag = 0; }
      if( !hsflag ) { putch(CR); putch(LF); }
   retypen:
      putchs(genbuf, np);
      continue;
   backward:
      if( rsflag ) { putch(']'); rsflag = 0; }
      putch(BSL); putch(CR); putch(LF);
      putchs(genbuf, np);
      nxtcol(cp, np);
      continue;
   delete:
      if( !scflag ) {
         if( !rsflag ) putch('[');
         rsflag = 1;
         for( gp = cp; --gp >= np; ) {
            if( *gp < SP ) {
               putch('^'); putch( *gp | 0100 );
            } else if( *gp == DEL ) {
               putch('D'); putch('E'); putch('L');
            } else {
               putch( *gp );
            }
         }
         colump -= (cp - np);
      } else {
         erase(cp, np);
      }
      for( gp = np; *gp++ = *cp++; );
      opchange = 1;
      continue;
   oerror:
      if( rsflag ) { putch(']'); rsflag = 0; }
      putch(ERRCHAR);
      putch(BEL);
      putch(BS);
      ch = NUL;
      continue;
   }
};

help   (filen)
   char *filen;
{
   register char *p1, *p2;
   register int n;
   int fid;

   for( p1 = rhsbuf, p2 = HELPDIR; *p1 = *p2++; p1++ );
   for( p2 = filen; (p1 <= rhslim) && (*p1 = *p2++); p1++ );
   if( (fid = open(rhsbuf, READ)) < 0 ) {
      puts(rhsbuf, EOF);
      puts(" not found", LF);
      return;
   }
   while( (n = read(fid, rhsbuf, SBSIZE)) > 0
       && write(standout, rhsbuf, n) == n );
   if( close(fid) == -1 ) error(filen);
};

int getshell   (oldbuf, delim)
   char *oldbuf, delim;
{
   register int c;
   register char *ob, *sb;
   char *fb;
   int retype;

   if( (c = getchr()) == LF ) {      /* interactive shell */
      return( 0 );
   } else {
      if( c != delim ) {      /* new shell command */
         retype = 0;
         ob = oldbuf;
         do {
            *ob++ = c;
         } while( (c = getchr()) != LF );
         *ob = NUL;
      } else {         /* old shell command */
         retype = 1;
         c = getchr();
      }
      for( ob = oldbuf, sb = genbuf; *sb = *ob++; ) {
         if( *sb == '%' ) {
            retype = 1;
            for( fb = savfile; *sb++ = *fb++; );
            sb--;
         } else {
            sb++;
         }
      }
      while( c != LF ) {      /* suffix of shell command */
         if( c == '%' ) {
            retype = 1;
            for( fb = savfile; *sb++ = *fb++; );
            sb--;
         } else {
            *sb++ = c;
         }
         c = getchr();
      }
      *sb = NUL;
      if( retype ) {
         putchr(delim);
         puts(genbuf, LF);
      }
   }
   return( 1 );
};

shell   (shellonly)      /* shell command */
   int shellonly;
{
   if( shellonly ) {
      execl(SHELCMD, "sh", "-i", 0);
   } else {
      execl(SHELCMD, "sh", "-c", genbuf, 0);
   }
   return;
};

process   (proc, parm)
   int (*proc)(), parm;
{
   register int pid, rpid, (*savsig)();
   int retc;

   if( (pid = fork()) == 0 ) {
      signal(SIGHUP, oldhup);
      signal(SIGQUIT, oldquit);
      signal(SIGINT, SIG_DFL);
      (*proc)(parm);
      exit( 0100 );
   } else if( pid < 0 ) {
      error(F);
   } else {
      savsig = signal(SIGINT, SIG_IGN);
      while( (rpid = wait(&retc)) != pid && rpid != -1 );
      signal(SIGINT, savsig);
   }
};

spell   (word)
   char *word;
{
   register char *w, *s;

   for( w = SPELCMD, s = splbuf; *s = *w++; )
      if( ++s > spllim ) goto splerr;
   for( w = word; *s = *w++; )
      if( ++s > spllim ) goto splerr;
   for( w = spelsfx; *s = *w++; )
      if( ++s > spllim ) goto splerr;
   execl(SHELCMD, "sh", "-c", splbuf, 0);
   puts("Sorry, can't spell today", LF);
   return;
 splerr:
   puts("spellword too long", LF);
   return;
};

pipeout   ()         /* write lines to a pipe */
{
   register int *lad, pid, rpid;
   int pipefid[2], retc, saveout, shellpipe;

   shellpipe = !getshell(pipebuf, '|');
   if( pipe(pipefid) < 0 ) {
      error(P);
   } else if( (pid = fork()) == 0
      && dup2(pipefid[0], 0) != -1
      && close( pipefid[0] ) != -1
      && close( pipefid[1] ) != -1 ) {
      if( shellpipe ) {
         execl(SHELCMD, "sh", "-s", 0);
      } else {
         execl(SHELCMD, "sh", "-c", genbuf, 0);
      }
      exit( 0100 );
   } else if( pid < 0 ) {
      if( close(pipefid[0]) == 0, close(pipefid[1]) == 0 );
      error(F);
   } else {
      if( close(pipefid[0]) == -1 ) error(Q);
      signal(SIGPIPE, SIG_IGN);
      signal(SIGINT, SIG_IGN);
      signal(SIGHUP, SIG_IGN);
      saveout = standout;
      standout = pipefid[1];
      lad = addr1;
      do {
         getline( *lad++ );
         puts(linbuf, LF);
      } while( lad <= addr2 );
      standout = saveout;
      signal(SIGINT, onintr);
      signal(SIGHUP, onhup);
      dot = addr2;
      if( close(pipefid[1]) == -1 ) error(Q);
      while( (rpid = wait(&retc)) != pid && rpid != -1 );
   }
};

crblock   (permp, buf, nchar, startn)   /* en- de-crypt block */
   char *permp; char *buf; int nchar; long startn;
{
   register char *p1;
   register char *t1, *t2, *t3;
   int n1, n2;

   t1 = permp; t2 = &permp[256]; t3 = &permp[512];
   n1 = startn & 0377; n2 = (startn >> 8) & 0377;
   p1 = buf;
   while( nchar-- ) {
      *p1 = t2[(t3[(t1[(*p1+n1)&0377]+n2)&0377]-n2)&0377]-n1;
      if( (++n1) == 256 ) {
         n1 = 0;
         if( ++n2 == 256 ) n2 = 0;
      }
      p1++;
   }
};

int getkey   ()
{
   register char *p;
   register int c, (*savsig)();
   struct sgttyb b;

   if( gtty(0, &b) == -1 ) error("Input not tty");
   savsig = signal(SIGINT, SIG_IGN);
   b.sg_flags &= ~ECHO;
   stty(0, &b);
   puts("Key: ", EOF);
   for( p = key; (c = getchr()) != EOF && c != LF; ) {
      if( p < keylim ) *p++ = c;
   }
   *p = 0;
   b.sg_flags |= ECHO;
   stty(0, &b);
   signal(SIGINT, savsig);
   putch(LF);
   return( key[0] != 0 );
};

/*
 * Besides initializing the encryption machine, this routine
 * returns 0 if the key is null, and 1 if it is non-null.
 */
int crinit   (keyp, permp)
   char *keyp, *permp;
{
   register char *t1, *t2, *t3;
   register int i;
   int ic, k, temp, pipefid[2];
   unsigned random;
   char buf[13];
   long seed;

   if( *keyp == 0 ) return( 0 );
   t1 = permp; t2 = &permp[256]; t3 = &permp[512];
   strncpy(buf, keyp, 8);
   buf[8] = buf[0];
   buf[9] = buf[1];
   if( pipe(pipefid) < 0 ) pipefid[0] = pipefid[1] = -1;
   if( fork() == 0
    && dup2(pipefid[0], 0) != -1
    && dup2(pipefid[1], 1) != -1 ) {
      execl(MAKKEYA, "-", 0);
      execl(MAKKEYB, "-", 0);
      exit( 0100 );
   }
   if( write(pipefid[1], buf, 10) == -1
    || wait((int *)0) == -1
    || read(pipefid[0], buf, 13) != 13
    || close(pipefid[0]) == -1
    || close(pipefid[1]) == -1 ) {
      error("crypt: cannot generate key");
   }
   seed = 123;
   for( i=0; i<13; i++ ) seed = seed*buf[i] + i;
   for( i = 0; i < 256; i++ ) {
      t1[i] = i;
      t3[i] = 0;
   }
   for( i = 0; i < 256; i++ ) {
      seed = 5*seed + buf[i%13];
      random = seed % 65521;
      k = 256 - 1 - i;
      ic = (random&0377) % (k+1);
      random >>= 8;
      temp = t1[k];
      t1[k] = t1[ic];
      t1[ic] = temp;
      if( t3[k] != 0 ) continue;
      ic = (random&0377) % k;
      while( t3[ic] != 0 ) ic = (ic+1) % k;
      t3[k] = ic;
      t3[ic] = k;
   }
   for( i = 0; i < 256; i++ ) t2[t1[i]&0377] = i;
   return(1);
};

int makekey   (keyp, buffer)
   char *keyp, *buffer;
{
   register int i;
   register char *tk;
   long t;
   char tkey[KSIZE];
   extern long time();

   for( i = KSIZE, tk = tkey; --i >= 0; *tk++ = *keyp++ );
   t = time(0); t += getpid();
   for( i = 4; --i >= 0; tkey[i] ^= ((t >> (8 * i)) & 0377) );
   return( crinit(tkey, buffer) );
};
