#

/*      card reader driver for IBM 029 code (EBCDIC).
        George Rolf, Psychologisch Laboratorium,
        Nijmegen, The Netherlands.
        After several examples by others, but small and neat.

        The driver recognizes 65 different characters;
        all others are translated into arbitrary characters.

        Note the following deviations from 029 code:
        - letters will be translated to lower case.
        - the cent sign (12-2-8) is translated into a '['
        - the back quote (`, code 1-8) doesn't occur on the 029.
        - the back slash (\, code 0-2-8) is produced by pushing the
          button 0-2-8 on the 029.
        - the not sign (11-7-8) is translated into a ~ (tilde, octal 176).

        All but the cent sign will be handled correctly by the urc daemon,
        which also translates lower case to upper case.
        Each card is provided with a new-line character.
        Trailing blanks are discarded (compile-time option).
*/

#include "../h/conf.h"
#include "../h/param.h"
#include "../h/dir.h"
#include "../h/user.h"

#define C81         1   /* set to 0 if your reader reads 80 columns only
                           we modified ours to read 81 col's, i.e. including
                           marks punched in 81'th col by IBM's checkpunch */
#define RIDBLANKS   1   /* comment out if you wanna save a couple of bytes */

#define CRADDR  0177160

#define RDRENB  01
#define EJECT   02
#define IENABLE 0100
#define COLDNE  0200
#define OFFLINE 0400
#define BUSY    01000
#define ONLINE  02000
#define TIMING  04000
#define HOPCHECK        020000
#define CARDNE  040000
#define ERROR   01000000

#define CRPRIO  27

#define EOFCHAR -1      /* card punch 12-11-0-7-8-9 */
                        /* 0377 expanded to an 'int' */

#define BLANK   0       /* compressed, i.e. crdb2 code */

struct {
        int     crcsr;
        int     crdb1;
        int     crdb2;
};                      /* unibus registers */

struct  cr11 {
        char    crstate;
        char    crcount;
        char    *crpoint;
        char    crbuf[80+C81];
} cr11;

/* crstate codes */
#define CLOSED  0
#define IDLE    1
#define WONLINE 2
#define READING 3
#define RDONE   4
#define EBUF    5

crdopen()
{
        if(cr11.crstate == CLOSED) {
                cr11.crstate= IDLE;
        } else
                u.u_error= ENXIO;
}

crdclose()
{
        CRADDR->crcsr= 0;
        cr11.crstate= CLOSED;
}

crdint()
{
        register int  s;

        s= CRADDR->crcsr;
        switch(cr11.crstate) {
        case WONLINE:
                crstart();
                return;
        case READING:
                if(s & COLDNE) {
                        if(cr11.crcount < 80+C81)
                                cr11.crbuf[cr11.crcount++]= CRADDR->crdb2;
                }
                if(s & CARDNE)
                        if(((s&TIMING) == 0) && (cr11.crcount == 80+C81)) {
                                cr11.crstate= RDONE;
                                wakeup((caddr_t)&cr11);
                                return;
                        }
                if(s&ERROR) {
                        if (cr11.crcount > 0)
                                /* usually never happens */
                                printf("re-read last card\r\n");
                        crstart();
                }
        }
}

char    contab[] {
/*          1   2   3   4   5   6   7   8  1-8 2-8 3-8 4-8 5-8  6-8 7-8  9   */

/*     000 001 002 003 004 005 006 007 010 011 012 013 014 015  016 017 020  */
       ' ','1','2','3','4','5','6','7','8','`',':','#','@','\'','=','"','9',
/* 0   040 041 042 043 044 045 046 047 050 051 052  053 054 055 056 057 060  */
       '0','/','s','t','u','v','w','x','y',040,'\\',',','%','_','>','?','z',
/* 11  100 101 102 103 104 105 106 107 110 111 112 113 114 115 116 117 120  */
       '-','j','k','l','m','n','o','p','q',040,'!','$','*',')',';','~','r',
/* 12  200 201 202 203 204 205 206 207 210 211 212 213 214 215 216 217 220  */
       '&','a','b','c','d','e','f','g','h',040,'[','.','<','(','+','|','i'
};

crdread()
{
        register int  x, i;
        register int cardsp;
        /* cardsp: number of cards passed
           used to determine whether we have to retain an eof
        */

        cardsp = 0;
more:
        spl6();
        switch(cr11.crstate) {
        case IDLE:
                crstart();
        default:                        /* WONLINE or READING */
                sleep((caddr_t)&cr11, CRPRIO);
                if(cr11.crstate != RDONE) {
                        spl0();
                        return;
                }
        case RDONE:
                spl0();
                cr11.crpoint= &cr11.crbuf[0];
                if (cr11.crcount > 80) cr11.crcount = 80;
#ifdef RIDBLANKS
                for(i= cr11.crcount; i > 0; i--)
                        if(cr11.crbuf[i-1] != BLANK)
                                break;
                cr11.crcount= i;
#endif
                if(cr11.crbuf[0] == EOFCHAR) {
                        if (cardsp == 0)
                                cr11.crstate= IDLE;
                        /* else: we'll get here next time */
                        return;
                }
                cr11.crstate= EBUF;
        case EBUF:
                spl0();
                while(cr11.crcount) {
                        x= (*cr11.crpoint) & 0377;
                        if (x >= 0200)
                                x =- 040;
                        x = ((x&0340)/040)*021 + (x&037);
                        if (x < sizeof contab)
                                x = contab[x];
                        cr11.crcount--;
                        cr11.crpoint++;
                        if(passc(x) < 0) {
                                return;
                        }
                }
                cardsp++;
                cr11.crstate = IDLE;
                if(passc('\n') >= 0) {
                        goto more;
                }
        }
}

crstart()
{
        register int  s;

        /* supposed to be entered at high priority */
        if(CRADDR->crcsr & OFFLINE) {
                cr11.crstate= WONLINE;
                CRADDR->crcsr= IENABLE;
        }
        if( !(CRADDR->crcsr & OFFLINE)) {
                cr11.crstate= READING;
                cr11.crcount= 0;
                CRADDR->crcsr= IENABLE|RDRENB;
        }
}
