#include "param.h"
#include "systm.h"
#include "dir.h"
#include "signal.h"
#include "seg.h"
#include "ipm.h"
#include "sid.h"
#include "user.h"
#include "errno.h"
#include "inode.h"
#include "file.h"
#include "proc.h"
#include "clock.h"
#include "reboot.h"
#include "timeb.h"
#include "buf.h"

/*
 * Everything in this file is a routine implementing a system call.
 */

gtime()
{
	u.u_rtime = time;
}

/*
 * New time entry--return TOD with milliseconds, timezone,
 * DST flag
 */

int	timezone = TIMEZONE;	/* static so it can be changed with adb */

ftime()
{
	register struct a {
		struct timeb *tp;
	} *uap;
	struct timeb t;
	register unsigned ms;

	uap = (struct a *) u.u_ap;
	splhigh();
	t.time = time;
	ms = lbolt;
	spl0();
	ms &= 077;
	if (ms > HZ) {
		ms -= HZ;
		t.time++;
	}
	t.millitm = (1000 * ms) / HZ;
	t.timezone = timezone;
	t.dstflag = DSTFLAG;
	if (copyout((caddr_t)&t, (caddr_t)uap->tp, sizeof(t)) < 0)
		u.u_error = EFAULT;
}

stime()
{
	register unsigned i;
	register struct a {
		time_t	time;
	} *uap;

	uap = (struct a *)u.u_ap;
	if (suser()) {
		rtcset(uap->time);
		time = uap->time;
	}
}

setuid()
{
	register unsigned uid;
	register struct a {
		int	uid;
	} *uap;

	uap = (struct a *)u.u_ap;
	uid = uap->uid;
	if (uid >= MAXUID) {
		u.u_error = EINVAL;
		return;
	}
	if (u.u_ruid == uid || suser()) {
		u.u_uid = uid;
		u.u_procp->p_uid = uid;
		u.u_ruid = uid;
	}
}

setruid()
{
	register unsigned uid;
	register struct a {
		int	uid;
	} *uap;

	uap = (struct a *)u.u_ap;
	uid = uap->uid;
	if (uid >= MAXUID) {
		u.u_error = EINVAL;
		return;
	}
	if (u.u_ruid == uid || suser())
		u.u_ruid = uid;
}

getuid()
{

	u.u_rval1 = u.u_ruid;
	u.u_rval2 = u.u_uid;
}

setgid()
{
	register unsigned gid;
	register struct a {
		int	gid;
	} *uap;

	uap = (struct a *)u.u_ap;
	gid = uap->gid;
	if (gid >= MAXUID) {
		u.u_error = EINVAL;
		return;
	}
	if (u.u_rgid == gid || suser()) {
		u.u_gid = gid;
		u.u_rgid = gid;
	}
}

getgid()
{

	u.u_rval1 = u.u_rgid;
	u.u_rval2 = u.u_gid;
}

getpid()
{
	u.u_rval1 = u.u_procp->p_pid;
	u.u_rval2 = u.u_procp->p_ppid;
}

#define BASE 	(char *)(8*1024*1024)
#define END	(char *)(12*1024*1024)
setpgrp()
{

	register char *i; /* temp */
	register int	c; /* temp */
	register struct proc *p = u.u_procp;
	register struct a {
		int	flag;
		int	ra;
		int	pid;
		int	pgrp;
	} *uap;

	uap = (struct a *)u.u_ap;
	/* check validity of uap->pid && uap->pgrp */
	if ((uap->pid < 0 || uap->pid > MAXPID) || 
		(uap->flag && (uap->pgrp < 0 || uap->pgrp > MAXPID)))
		goto error;
	/* check to see if you don't have to search proc table */
	if (!uap->pid || uap->pid == p->p_pid) {
		uap->pid = p->p_pid;
		goto foundit;
	}
	for (p = proc; p < &proc[NPROC]; p++) {
		if (p->p_stat == NULL)
			continue;
		if (p->p_pid != uap->pid)
			continue;
foundit:
		if (! (u.u_uid == 0 || u.u_uid == p->p_uid ||
			u.u_ruid == p->p_uid || u.u_uid == p->p_suid ||
			u.u_ruid == p->p_suid ))  {
			if (uap->flag) {
				u.u_error = EPERM;
				u.u_rval1 = -1;
				return;
			}
		}
		if (uap->flag) 
			p->p_pgrp = uap->pgrp;
		break;
	}
	u.u_rval1 = p->p_pgrp;
	if (p >= &proc[NPROC]) {
error:
		u.u_rval1 = -1;
		u.u_error = ESRCH;
	}
}

sync()
{

	update();
}

nice()
{
	register n;
	register struct a {
		int	niceness;
	} *uap;

	uap = (struct a *)u.u_ap;
	n = uap->niceness;
	if ((n < 0 || n > 2*NZERO) && !suser())
		n = 0;
	n += u.u_procp->p_nice;
	if (n >= 2*NZERO)
		n = 2*NZERO -1;
	if (n < 0)
		n = 0;
	u.u_procp->p_nice = n;
	u.u_rval1 = n - NZERO;
}

/*
 * Unlink system call.
 * Hard to avoid races here, especially
 * in unlinking directories.
 */
unlink()
{
	register struct inode *ip, *pp;
	struct a {
		char	*fname;
	};

	pp = namei(uchar, 2);
	if (pp == NULL)
		return;
	/*
	 * Check for unlink(".")
	 * to avoid hanging on the iget
	 */
	if (pp->i_number == u.u_dent.d_ino) {
		ip = pp;
		ip->i_count++;
	} else
		ip = iget(pp->i_dev, u.u_dent.d_ino);
	if (ip == NULL)
		goto out1;
	if ((ip->i_mode&IFMT) == IFDIR && !suser())
		goto out;
	/*
	 * Don't unlink a mounted file.
	 */
	if (ip->i_dev != pp->i_dev) {
		u.u_error = EBUSY;
		goto out;
	}
	if (ip->i_flag&ITEXT)
		xrele(ip);	/* try once to free text */
	if (ip->i_flag&ITEXT && ip->i_nlink == 1) {
		u.u_error = ETXTBSY;
		goto out;
	}
	u.u_offset -= sizeof(struct direct);
	u.u_base = (caddr_t)&u.u_dent;
	u.u_count = sizeof(struct direct);
	u.u_dent.d_ino = 0;
	u.u_segflg = 1;
	writei(pp);
	if (u.u_error)
		goto out;
	ip->i_nlink--;
	ip->i_flag |= ICHG;
out:
	iput(ip);
out1:
	iput(pp);
}

chdir()
{
	chdirec(&u.u_cdir);
}

chroot()
{
	if (!suser())
		return;
	chdirec(&u.u_rdir);
}

chdirec(ipp)
register struct inode **ipp;
{
	register struct inode *ip;
	struct a {
		char	*fname;
	};

	ip = namei(uchar, 0);
	if (ip == NULL)
		return;
	if ((ip->i_mode&IFMT) != IFDIR) {
		u.u_error = ENOTDIR;
		goto bad;
	}
	if (access(ip, IEXEC))
		goto bad;
	prele(ip);
	if (*ipp) {
		plock(*ipp);
		iput(*ipp);
	}
	*ipp = ip;
	return;

bad:
	iput(ip);
}

chmod()
{
	register struct inode *ip;
	register struct a {
		char	*fname;
		int	fmode;
	} *uap;

	uap = (struct a *)u.u_ap;
	if ((ip = owner()) == NULL)
		return;
	ip->i_mode &= ~07777;
	if (u.u_uid) {
		uap->fmode &= ~ISVTX;
		if (u.u_gid != ip->i_gid)
			uap->fmode &= ~ISGID;
	}
	ip->i_mode |= uap->fmode&07777;
	ip->i_flag |= ICHG;
	if (ip->i_flag&ITEXT && (ip->i_mode&ISVTX)==0)
		xrele(ip);
	iput(ip);
}

chown()
{
	register struct inode *ip;
	register struct a {
		char	*fname;
		int	uid;
		int	gid;
	} *uap;

	uap = (struct a *)u.u_ap;
	if ((ip = owner()) == NULL)
		return;
	ip->i_uid = uap->uid;
	ip->i_gid = uap->gid;
	if (u.u_uid != 0)
		ip->i_mode &= ~(ISUID|ISGID);
	ip->i_flag |= ICHG;
	iput(ip);
}

ssig()
{
	register a;
	register struct proc *p;
	struct a {
		int	signo;
		int	fun;
	} *uap;

	uap = (struct a *)u.u_ap;
	a = uap->signo;
	if (a <= 0 || a > NSIG || a == SIGKILL) {
		u.u_error = EINVAL;
		return;
	}
	u.u_rval1 = u.u_signal[a-1];
	u.u_signal[a-1] = uap->fun;
	u.u_procp->p_sig &= ~(1L<<(a-1));
}

kill()
{
	register struct proc *p, *q;
	register arg;
	register struct a {
		int	pid;
		int	signo;
	} *uap;
	int f;

	uap = (struct a *)u.u_ap;
	if (uap->signo < 0 || uap->signo > NSIG) {
		u.u_error = EINVAL;
		return;
	}
	/* Prevent proc 1 (init) from being SIGKILLed */
	if (uap->signo == SIGKILL && uap->pid == 1) {
		u.u_error = EINVAL;
		return;
	}
	f = 0;
	arg = uap->pid;
	if (arg > 0)
		p = &proc[1];
	else
		p = &proc[2];
	q = u.u_procp;
	if (arg == 0 && q->p_pgrp == 0) {
		u.u_error = ESRCH;
		return;
	}
	for (; p < &proc[NPROC]; p++) {
		if (p->p_stat == NULL)
			continue;
		if (arg > 0 && p->p_pid != arg)
			continue;
		if (arg == 0 && p->p_pgrp != q->p_pgrp) 
			continue;
		if (arg < -1 && p->p_pgrp != -arg)
			continue;
		if (! (u.u_uid == 0 ||
			u.u_uid == p->p_uid ||
			u.u_ruid == p->p_uid ||
			u.u_uid == p->p_suid ||
			u.u_ruid == p->p_suid ))
			if (arg > 0) {
				u.u_error = EPERM;
				return;
			} else
				continue;
		f++;
		if (uap->signo) {
			psignal(p, uap->signo);
		} 
		if (arg > 0)
			break;
	}
	if (f == 0)
		u.u_error = ESRCH;
}

times()
{
	register struct a {
		time_t	(*times)[4];
	} *uap;

	uap = (struct a *)u.u_ap;
	if (copyout((caddr_t)&u.u_utime, (caddr_t)uap->times, sizeof(*uap->times)))
		u.u_error = EFAULT;
	splhigh();
	u.u_rtime = lbolt;
	spl0();
}

profil()
{
	register struct a {
		short	*bufbase;
		unsigned bufsize;
		unsigned pcoffset;
		unsigned pcscale;
	} *uap;

	uap = (struct a *)u.u_ap;
	u.u_prof.pr_base = uap->bufbase;
	u.u_prof.pr_size = uap->bufsize;
	u.u_prof.pr_off = uap->pcoffset;
	u.u_prof.pr_scale = uap->pcscale;
}

/*
 * alarm clock signal
 */
alarm()
{
	register struct proc *p;
	register c;
	register struct a {
		int	deltat;
	} *uap;

	uap = (struct a *)u.u_ap;
	p = u.u_procp;
	c = p->p_clktim;
	p->p_clktim = uap->deltat;
	u.u_rval1 = c;
}

/*
 * indefinite wait.
 * no one should wakeup(&u)
 */
pause()
{

	for(;;)
		sleep((caddr_t)&u, PSLEP);
}

/*
 * mode mask for creation of files
 */
umask()
{
	register struct a {
		int	mask;
	} *uap;
	register t;

	uap = (struct a *)u.u_ap;
	t = u.u_cmask;
	u.u_cmask = uap->mask & 0777;
	u.u_rval1 = t;
}

/*
 * Set IUPD and IACC times on file.
 */
utime()
{
	register struct a {
		char	*fname;
		time_t	*tptr;
	} *uap;
	register struct inode *ip;
	time_t tv[2];

	uap = (struct a *)u.u_ap;
	if (uap->tptr != NULL) {
		if (copyin((caddr_t)uap->tptr, (caddr_t)tv, sizeof(tv))) {
			u.u_error = EFAULT;
			return;
		}
	} else {
		tv[0] = time;
		tv[1] = time;
	}
	ip = namei(uchar, 0);
	if (ip == NULL)
		return;
	if (u.u_uid != ip->i_uid && u.u_uid != 0) {
		if (uap->tptr != NULL)
			u.u_error = EPERM;
		else
			access(ip, IWRITE);
	}
	if (!u.u_error) {
		ip->i_flag |= IACC|IUPD|ICHG;
		iupdat(ip, &tv[0], &tv[1]);
	}
	iput(ip);
}


reboot()		/* go back to the roms. */
{
	register struct a {
		int flags;
	} *uap;
        extern struct buf sdtab;

	uap = (struct a *)u.u_ap;
	if (! suser()) {
		u.u_error = EPERM;
		return;
	}

        if ((uap->flags & RB_NOSYNC) == 0) {
                update();
                printf("Syncing discs... ");
                while (sdtab.av_forw)
                        idle();
                printf("Done\n");
        }
	if (uap->flags & RB_PANIC)
		panic("reboot");
	pcc_reboot(uap->flags);
	/* never to be heard from again. */
}

/*
 * Lock user into core as much
 * as possible.  Swapping may still 
 * occur if core grows.
 */

syslock()
{
	register struct proc *p;
	register struct a {
		int flag;
	} *uap;

	uap = (struct a *) u.u_ap;
	if (suser()) {
		p = u.u_procp;
		p->p_flag &= ~SULOCK;
		if (uap->flag)
		p->p_flag |= SULOCK;
	}
}
