/* tcpdev.c: Pseudo-device for tcp connections */
/*
 * This implements a pseudo-device that requires a line discipline
 * (tcp_ld) to be pushed onto /dev/iptcp.
 */

#include "param.h"
#include "types.h"
#include "errno.h"
#include "in.h"
#include "inio.h"
/* maybe ip_var.h */
#include "stream.h"
#include "tcp_user.h"
#include "tcp.h"
#include "proc.h"
#include "inode.h"
#include "dir.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "signal.h"
#include "user.h"

#define	MSS	700		/* should be somewhere else */
extern	putq(), nullsys();
static	tcpdqopen();
static	tcpdput();
int	tcpdns(), tcpdups();
int	tcpdqclose();

struct qinit	tcpdqinit[] = {
/*up*/	{ putq, tcpdups, tcpdqopen, tcpdqclose, 1500, 64 },
/*dn*/	{ tcpdput, tcpdns, nullsys, nullsys, 1500, 64}
};

struct	tcb	tcb[NTCB];		/* the tcp control block table */
static	timer_running;			/* is the slow timer running? */

static
tcpdqopen(dev, q)	/* init connection to tcb */
	dev_t dev;
	register struct queue *q;
{
	register unit;
	register struct tcb *tp;
	int tcp_tick();

	if ((unit = minor(dev)) < 0 || unit > NTCB) {
		u.u_error = ENXIO;
		return;
	}
	splnet();
	tp = &tcb[unit];
	if (tp->upq != NULL) {
		u.u_error = ENXIO;
		return;
	}
	tp->upq = q;
	tp->dnq = OTHERQ(q);
	q->ptr = (char *)tp;
	OTHERQ(q)->ptr = (char *)tp;
	OTHERQ(q->next)->flag |= QBIGB;
	if (! timer_running) {
		timeout(tcp_tick, 0, 30);	/* 500ms timer chain */
		timer_running = 1;
	}
	spl0();
}

tcpdopen(dev, flag)	/* check for odd/even units */
	dev_t dev;
{
	register struct tcb *tp;

	splnet();
	if ((tp = &tcb[minor(dev)])->flags & TCP_ISOPEN && dev&01) {
		u.u_error = ENXIO;
		spl0();
		return;
	}
	tp->flags |= TCP_ISOPEN;
	tp->rcv_wnd = MAXRCVWND;		/* hack */
	spl0();
}

tcpdclose(dev, flag)		/* close  */
	dev_t dev;
{
	register struct tcb *tp;

	splnet();
	tp = &tcb[minor(dev)];
	switch (tp->state) {
	case CLOSED:
		free_tcb(tp);
		break;
	case LISTEN:
	case SYN_SENT:
		free_tcb(tp);
		break;
	case SYN_RECEIVED:
	case ESTABLISHED:
	case CLOSE_WAIT:
		tp->flags |= TCP_CLOSING;
		if (tp->dnq) {
			qenable(tp->dnq);
			sleep(tp, TCPPRI);
		}
		break;
	case FIN_WAIT_1:
	case FIN_WAIT_2:
		break;
	}
	spl0();
	tp->upq = tp->dnq = 0;
}

tcpdqclose(dev, q)		/* shut down an ip device */
	dev_t dev;
	register struct queue *q;
{
	register struct tcb *tp;

	tp = &tcb[minor(dev)];
	tp->upq = tp->dnq = NULL;
	tp->qstate = TCP_IDLE;
	return 0;
}

static
tcpdput(q, bp)		/* filter out all non-data blocks */
	register struct queue *q;
	register struct block *bp;
{
	register struct block *bp2;
	register n;

	if (bp->type == M_DATA) {
#ifdef notdef
		if ((bp2 = q->last) == NULL) {
			putq(q, bp);
			return;
		}
		n = min(bp->wptr - bp->rptr, MSS - (bp2->wptr - bp2->rptr));
		if (n == bp->wptr - bp->rptr && n <= bp->lim - bp->wptr) {
			bcopy(bp->rptr, bp2->wptr, n);
			bp2->wptr += n;
			freeb(bp);
			return;
		}
#endif
		putq(q, bp);
	} else if (bp->type == M_IOCTL) {
		bp->type = M_IOCNAK;
		bp->wptr = bp->rptr;
		qreply(q, bp);
	} else 
		freeb(bp);
}

tcpdns(q)
	register struct queue *q;
{
	register struct tcb *tp;
	register struct block *bp;
	register struct tcpuser *up;

	tp = (struct tcb *)q->ptr;
	for (;;) {
		switch (tp->qstate) {
		case TCP_IDLE:
			/* process a user request */
			if ((bp = getq(q)) == NULL)
				return;
			if (bp->type != M_DATA) {
				freeb(bp);
				return;
			}
			if (bp->wptr - bp->rptr < sizeof (struct tcpuser)) {
				/* illegal request */
				freeb(bp);
				return;
			}
			up = (struct tcpuser *)bp->rptr;
			switch (up->code) {
			case TCPC_CONNECT:
				tp->qstate = TCP_WOPEN;
				if (tcp_open(up, tp)) {
					putbq(q, bp); 	/* wait for connect */
					return;
				}
				qreply(q, bp);
				tp->qstate = TCP_IDLE;
				continue;
			case TCPC_LISTEN:
				if (tcp_open(up, tp))
					tp->qstate = TCP_STARTED;
				else
					tp->qstate = TCP_IDLE;
				qreply(q, bp);
				continue;
			default:
				up->code = TCPC_SORRY;
				qreply(q, bp);
				tp->qstate = TCP_IDLE;
				continue;
			}
		case TCP_WOPEN:
			/* TBD: check to see if we got hit by a RST */
			if ((bp = getq(q)) == NULL)
				return;
			up = (struct tcpuser *)bp->rptr;
			up->code = TCPC_OK;
			qreply(q, bp);
			tp->qstate = TCP_STARTED;
			break;
		case TCP_STARTED:
			if ((bp = q->first) == NULL) {
				tp = (struct tcb *)q->ptr;
				if (tp->flags & TCP_CLOSING)
					if (tp->flags & TCP_CLEXPIRED) {
						tcp_close(tp);
						wakeup(tp);
					} else
						tp->cl_timer = 4;	/* 2s */
				return;
			}
			switch (bp->type) {
			case M_DATA:
				/* 
				 * three cases:
				 *	tcb closed or closing
				 *	segment sent and timeout running
				 *	segment not sent or timeout expired
				 */
				switch (tp->state) {
				case CLOSED:
				case FIN_WAIT_1:
				case FIN_WAIT_2:
				case CLOSING:
				case CLOSE_WAIT:
				case LAST_ACK:
					freeb(getq(q));
					continue;
				}
				if (tp->snd_una != tp->snd_nxt)
				if (tp->rt_timer > 0)
					return;
				if (tp->snd_wnd > bp->wptr - bp->rptr) {
					struct block *obp;
					tp->tries++;
					obp = dupb(bp);
					tcp_send(tp, obp);
					tp->rt_timer = 2 * tp->rtt;
					tp->cl_timer = 0;
				} else
					tp->per_timer = 2*tp->rtt;
				return;
			}
			break;
		default:
			if ((bp = getq(q)) == NULL)
				return;
			freeb(bp);
		}
	}
}

tcpdups(q)		/* copy and manipulate the rcv_wnd */
	register struct queue *q;
{
	register struct block *bp;
	int was_zero;
	register struct tcb *tp;

	if ((tp = (struct tcb *)q->ptr) == NULL)
		panic ("tcpdups: no tcb pointer! q 0x%x\n", q);
	was_zero = tp->rcv_wnd == 0;
	while (q->first && (q->next->flag & QFULL) == 0) {
		bp = getq(q);
		if (bp->type == M_DATA)
			tp->rcv_wnd += bp->wptr - bp->rptr;
		(*q->next->qinfo->putp)(q->next, bp);
	}
	if (q->first && q->next->flag & QFULL)
		q->next->flag |= QWANTW;
	if (tp->rcv_wnd == MAXRCVWND)
		send_sac(tp, tp->snd_nxt, tp->rcv_nxt, ACK);
}

