/* (lgl-
 *	The information contained herein is a trade secret of Mark Williams
 *	Company, and  is confidential information.  It is provided  under a
 *	license agreement,  and may be  copied or disclosed  only under the
 *	terms of  that agreement.  Any  reproduction or disclosure  of this
 *	material without the express written authorization of Mark Williams
 *	Company or persuant to the license agreement is unlawful.
 *
 *	COHERENT Version 4.2.10
 *	Copyright (c) 1982 - 1994
 *	An unpublished work by Mark Williams Company, Chicago.
 *	All rights reserved.
 -lgl) */

/*
 * History:
 *
 * 1.0	Udo Munk	initial release created
 * 1.1	Udo Munk	thoughput improved, semaphore for the read routine
 *			added
 */

#include <sys/types.h>
#include <kernel/v_types.h>
#include <sys/coherent.h>
#include <sys/devices.h>
#include <sys/io.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <sys/sched.h>
#include <sys/cdrom.h>
#include "mcd.h"

#define VERSION "v1.1"

/*
 * in the final production release all debugs should be commented
 */
#define DEBUG
/* #define DEBUG_IRQ */
/* #define DEBUG_SUBQ */
/* #define DEBUG_PLAY */

#define status_ready() ((inb(MCD_SDBASE + STATUS_REG) & HW_STEN) == 0)
#define data_ready() ((inb(MCD_SDBASE + STATUS_REG) & HW_DTEN) == 0)

/* -------------------------------------------------------------------- */
/* Forward declarations							*/
/* -------------------------------------------------------------------- */

static void mcd_open(), mcd_close(), mcd_read(), mcd_ioctl();
static void mcd_load(), mcd_unload(), mcd_irq();
static void mcd_busy(), mcd_not_busy();
static void blk2msf();
static unsigned char bin2bcd();

/* -------------------------------------------------------------------- */
/* Variables								*/
/* -------------------------------------------------------------------- */

extern int MCD_SDBASE;				/* base address from Space.c */
extern int MCD_SDIRQ;				/* IRQ from Space.c */

static unsigned char drive_type;		/* drive type */
static unsigned char drive_version;		/* version of drive control program */

static unsigned char read_command;		/* read command, single or double speed */

static int initialized = 0;			/* drive initialized ? */
static int usage = 0;				/* number of opens */
static int error_status = 0;			/* error status on IRQ */

static int audio_status = CDROM_AUDIO_NO_STATUS;/* current audio status */

static int toc_read = 0;			/* has the table of contents been read ? */
static struct toc_header mcd_toc_header;	/* TOC header */
static struct toc mcd_toc[MAX_TRACKS];		/* TOC */
static struct play mcd_play;			/* structure for play command */

static char *mcd_event_wait = "mcd_wait";	/* event we are sleeping on */
static char *mcd_wait_text = "mcdwait";		/* text for ps while waiting */
static TIM mcd_timer;				/* the timer for waiting */

static int char_dev_busy = 0;			/* character device busy flag */
static char *mcd_event_busy = "mcd_busy";	/* event when we are busy */
static char *mcd_busy_text = "mcdbusy";		/* text for ps when busy */

static char buffer[2048];			/* buffer for one block */
static int cur_block = -1;			/* current block in buffer */

/* -------------------------------------------------------------------- */
/* CON structure for device						*/
/* -------------------------------------------------------------------- */

CON	mcdcon = {
	DFCHR,				/* Flags */
	MCD_MAJOR,			/* Major index */
	mcd_open,			/* Open */
	mcd_close,			/* Close */
	NULL,				/* Block */
	mcd_read,			/* Read */
	NULL,				/* Write */
	mcd_ioctl,			/* Ioctl */
	NULL,				/* Powerfail */
	NULL,				/* Timeout */
	mcd_load,			/* Load */
	mcd_unload			/* Unload */
};

/* -------------------------------------------------------------------- */
/* Device driver functions						*/
/* -------------------------------------------------------------------- */

/*
 * open device:
 *	- check mode and minor device number
 *	- check if disk in drive
 *	- read TOC from disk
 */
static void mcd_open(dev, mode)
dev_t dev;
int mode;
{
	int status;

	if (mode != IPR) {
		/* a CD-ROM is read only fool */
#ifdef DEBUG
		devmsg(dev, "mcd: cannot open with mode %x\n", mode);
#endif
		set_user_error(ENODEV);
		return;
	}

	if (minor(dev)) {
		/* we can only handle one drive with minor number 0 */
#ifdef DEBUG
		devmsg(dev, "mcd: minor device %d not supported\n", dev & 0xff);
#endif
		set_user_error(ENXIO);
		return;
	}

	if (!initialized) {
		/* no drive found */
		set_user_error(ENXIO);
		return;
	}

	/* get drive status */
	outb(MCD_SDBASE + DATA_REG, MCD_REQUEST_DRIVE_STAT);
	status = wait_status(TIMEOUT_STATUS);
	if ((status & MCD_STATUS_DISC) == 0) {
		devmsg(dev, "mcd: no disk in drive\n");
		set_user_error(EIO);
		return;
	}

	/* read TOC */
	if (read_toc() != 0) {
		set_user_error(EIO);
		return;
	}

	usage++;
}

/*
 * close device:
 *	- hm, not much to do...
 */
static void mcd_close(dev, mode)
dev_t dev;
int mode;
{
	if (usage > 0)
		usage--;

	/* save some power and device lifetime, if possible */
	if ((usage == 0) && (audio_status != CDROM_AUDIO_PLAY)) {
		outb(MCD_SDBASE + DATA_REG, MCD_STOP);
		wait_status(TIMEOUT_STATUS);
	}
}

/*
 * read from the character device
 */
static void mcd_read(dev, iop)
dev_t dev;
IO *iop;
{
	int st;
	unsigned int block;
	unsigned int nsect;

	/* set and check device busy semaphore */
	mcd_busy();

	/* compute the block number to read */
	block = iop->io_seek / 2048;

	/* compute number of blocks to read */
	nsect = iop->io_ioc / 2048;
	if ((nsect == 0) && (iop->io_ioc > 0))
		nsect++;

	/* read the requested blocks */
	while (nsect > 0) {

		/* if the requested block isn't in the buffer read it */
		if (block != cur_block) {
			blk2msf(block, &mcd_play.start);
			mcd_play.end.min = 0;
			mcd_play.end.sec = 0;
			mcd_play.end.frame = 1;
			do_play(read_command, &mcd_play);
			if ((st = wait_data(TIMEOUT_DATA)) != 0) {
#ifdef DEBUG
				devmsg(dev, "mcd: mcd_read(): do_play() failed, status=%x\n", st);
#endif
				set_user_error(EIO);
				mcd_not_busy();
				return;
			}
			cur_block = block;
			st = splhi();
			repinsb(MCD_SDBASE + DATA_REG, &buffer[0], 2048);
			splx(st);
		}

		/* the requested block is in memory now, copy it to user buffer */
		iowrite(iop, buffer, 2048);

		/* handle next block */
		block++;
		nsect--;
	}

	/* free semaphore for device busy and awake waiting processes */
	mcd_not_busy();
}

/*
 * all the ioctl() stuff for the audio support
 */
static void mcd_ioctl(dev, com, arg)
dev_t dev;
int com;
char *arg;
{
	int i;
	struct toc *tocp;
	struct toc loc_mcd_toc;
	struct cdrom_tochdr loc_hdr;
	struct cdrom_tocentry loc_entry;
	struct cdrom_subchnl loc_subch;
	struct cdrom_ti loc_ti;
	struct cdrom_msf loc_msf;
	struct cdrom_volctrl loc_volume;

	switch(com) {

	case CDROMSTART:		/* start the drive */
		/* nothing to do, drive automatically starts with read command */
		break;

	case CDROMSTOP:			/* stop the drive */
		outb(MCD_SDBASE + DATA_REG, MCD_STOP);
		if (wait_status(TIMEOUT_STATUS) < 0) {
			/* I have no idea what to do, if stopping the drive fails */
#ifdef DEBUG
			devmsg(dev, "mcd: ioctl(CDROMSTOP) timeout\n");
#endif
		}
		audio_status = CDROM_AUDIO_NO_STATUS;
		break;

	case CDROMPAUSE:		/* pause the drive */
		/*
		 * If drive isn't playing ignore the command and
		 * don't return an error code! This is necessary
		 * because cdplayer programs executes a pause command
		 * before a stop and eject command. This is necessary
		 * because some CD-ROM (Linux) drivers doesn't work
		 * correct else.
		 */
		if (audio_status != CDROM_AUDIO_PLAY)
			return;
		outb(MCD_SDBASE + DATA_REG, MCD_HOLD);
		if (wait_status(TIMEOUT_STATUS) < 0) {
			/* I have no idea what to do, if holding the drive fails */
#ifdef DEBUG
			devmsg(dev, "mcd: ioctl(CDROMPAUSE) timeout\n");
#endif
		}
		if (get_subq(&loc_mcd_toc) < 0) {
			/* when we cannot get the current position, we cannot resume later */
			audio_status = CDROM_AUDIO_NO_STATUS;
			return;
		}
		mcd_play.start = loc_mcd_toc.disk_time;
		audio_status = CDROM_AUDIO_PAUSED;
		break;

	case CDROMRESUME:		/* start a paused drive again */
		if (audio_status != CDROM_AUDIO_PAUSED) {
			set_user_error(EINVAL);
			return;
		}
		do_play(MCD_SEEK_READ, &mcd_play);
		if (wait_status(TIMEOUT_STATUS) < 0) {
			audio_status = CDROM_AUDIO_ERROR;
			set_user_error(EIO);
			return;
		}
		audio_status = CDROM_AUDIO_PLAY;
		break;

	case CDROMREADTOCHDR:		/* read the TOC header */
		loc_hdr.cdth_trk0 = mcd_toc_header.first;
		loc_hdr.cdth_trk1 = mcd_toc_header.last;
		kucopy(&loc_hdr, arg, sizeof(loc_hdr));
		break;

	case CDROMREADTOCENTRY:		/* read an entry in the TOC */
		ukcopy(arg, &loc_entry, sizeof(loc_entry));
		if (loc_entry.cdte_track == CDROM_LEADOUT)
			tocp = &mcd_toc[mcd_toc_header.last + 1];
		else if ((loc_entry.cdte_track > mcd_toc_header.last) ||
			 (loc_entry.cdte_track < mcd_toc_header.first)) {
				set_user_error(EINVAL);
				return;
			}
		else
			tocp = &mcd_toc[loc_entry.cdte_track];
		loc_entry.cdte_adr = tocp->ctrl_addr;
		loc_entry.cdte_ctrl = tocp->ctrl_addr >> 4;
		if (loc_entry.cdte_format == CDROM_LBA)
			loc_entry.cdte_addr.lba = msf2blk(&tocp->disk_time);
		else if (loc_entry.cdte_format == CDROM_MSF) {
			loc_entry.cdte_addr.msf.minute = bcd2bin(tocp->disk_time.min);
			loc_entry.cdte_addr.msf.second = bcd2bin(tocp->disk_time.sec);
			loc_entry.cdte_addr.msf.frame = bcd2bin(tocp->disk_time.frame);
		} else {
			set_user_error(EINVAL);
			return;
		}
		kucopy(&loc_entry, arg, sizeof(loc_entry));
		break;

	case CDROMSUBCHNL:		/* get subchannel info */
		ukcopy(arg, &loc_subch, sizeof(loc_subch));
		if (get_subq(&loc_mcd_toc) < 0) {
			set_user_error(EIO);
			return;
		}
		loc_subch.cdsc_audiostatus = audio_status;
		loc_subch.cdsc_adr = loc_mcd_toc.ctrl_addr;
		loc_subch.cdsc_ctrl = loc_mcd_toc.ctrl_addr >> 4;
		loc_subch.cdsc_trk = bcd2bin(loc_mcd_toc.track);
		loc_subch.cdsc_ind = bcd2bin(loc_mcd_toc.index);
		if (loc_subch.cdsc_format == CDROM_LBA) {
			loc_subch.cdsc_absaddr.lba = msf2blk(&loc_mcd_toc.disk_time);
			loc_subch.cdsc_reladdr.lba = msf2blk(&loc_mcd_toc.track_time);
		} else if (loc_subch.cdsc_format == CDROM_MSF) {
			loc_subch.cdsc_absaddr.msf.minute = bcd2bin(loc_mcd_toc.disk_time.min);
			loc_subch.cdsc_absaddr.msf.second = bcd2bin(loc_mcd_toc.disk_time.sec);
			loc_subch.cdsc_absaddr.msf.frame = bcd2bin(loc_mcd_toc.disk_time.frame);
			loc_subch.cdsc_reladdr.msf.minute = bcd2bin(loc_mcd_toc.track_time.min);
			loc_subch.cdsc_reladdr.msf.second = bcd2bin(loc_mcd_toc.track_time.sec);
			loc_subch.cdsc_reladdr.msf.frame = bcd2bin(loc_mcd_toc.track_time.frame);
		} else {
			set_user_error(EINVAL);
			return;
		}
		kucopy(&loc_subch, arg, sizeof(loc_subch));
		break;

	case CDROMPLAYTRKIND:		/* play a track */
		ukcopy(arg, &loc_ti, sizeof(loc_ti));
		if ((loc_ti.cdti_trk0 < mcd_toc_header.first) ||
		    (loc_ti.cdti_trk0 > mcd_toc_header.last) ||
		    (loc_ti.cdti_trk1 < loc_ti.cdti_trk0)) {
			set_user_error(EINVAL);
			return;
		}
		if (loc_ti.cdti_trk1 > mcd_toc_header.last)
			loc_ti.cdti_trk1 = mcd_toc_header.last;
		mcd_play.start = mcd_toc[loc_ti.cdti_trk0].disk_time;
		mcd_play.end = mcd_toc[loc_ti.cdti_trk1 + 1].disk_time;
		do_play(MCD_SEEK_READ, &mcd_play);
		if (wait_status(TIMEOUT_STATUS) < 0) {
			audio_status = CDROM_AUDIO_ERROR;
			set_user_error(EIO);
			return;
		}
		audio_status = CDROM_AUDIO_PLAY;
		break;

	case CDROMPLAYMSF:		/* play a starting MSF position */
		if (audio_status == CDROM_AUDIO_PLAY) {
			outb(MCD_SDBASE + DATA_REG, MCD_HOLD);
			wait_status(TIMEOUT_STATUS);
			audio_status = CDROM_AUDIO_NO_STATUS;
		}
		ukcopy(arg, &loc_msf, sizeof(loc_msf));
		mcd_play.start.min = bin2bcd(loc_msf.cdmsf_min0);
		mcd_play.start.sec = bin2bcd(loc_msf.cdmsf_sec0);
		mcd_play.start.frame = bin2bcd(loc_msf.cdmsf_frame0);
		mcd_play.end.min = bin2bcd(loc_msf.cdmsf_min1);
		mcd_play.end.sec = bin2bcd(loc_msf.cdmsf_sec1);
		mcd_play.end.frame = bin2bcd(loc_msf.cdmsf_frame1);
		do_play(MCD_SEEK_READ, &mcd_play);
		if (wait_status(TIMEOUT_STATUS) < 0) {
			audio_status = CDROM_AUDIO_ERROR;
			set_user_error(EIO);
			return;
		}
		audio_status = CDROM_AUDIO_PLAY;
		break;

	case CDROMVOLCTRL:		/* volume control */
		ukcopy(arg, &loc_volume, sizeof(loc_volume));
		outb(MCD_SDBASE + DATA_REG, MCD_SET_ATTENATOR);
		outb(MCD_SDBASE + DATA_REG, loc_volume.channel0);
		outb(MCD_SDBASE + DATA_REG, 0);
		outb(MCD_SDBASE + DATA_REG, loc_volume.channel1);
		outb(MCD_SDBASE + DATA_REG, 0);
		wait_status(TIMEOUT_STATUS);
		read_status(&i);
		read_status(&i);
		read_status(&i);
		read_status(&i);
		break;

	case CDROMEJECT:		/* eject drive */
		/* all drives can be stopped at least */
		outb(MCD_SDBASE + DATA_REG, MCD_STOP);
		wait_status(TIMEOUT_STATUS);
		/*
		 * the eject command only does work with the FX drives,
		 * if you have a LU005 talk to Mitsumi, I don't programm
		 * their firmware
		 */
		outb(MCD_SDBASE + DATA_REG, MCD_EJECT);
		wait_status(TIMEOUT_STATUS);
		break;
		
	default:
		set_user_error(EINVAL);
		break;
	}
}

/*
 * initialization of the device when the OS is booted
 */
static void mcd_load()
{
	char *type_string;
	int i;

	printf("Mitsumi CD-ROM driver %s\n", VERSION);

	/* reset the drive and wait some time */
	outb(MCD_SDBASE + RESET_REG, 0);
	busyWait(NULL, 150);

	/* request drive status */
	outb(MCD_SDBASE + DATA_REG, MCD_REQUEST_DRIVE_STAT);

	/* let's see if we can get a status from the drive */
	for (i = 0; i < TIMEOUT_STATUS; i++) {
		if (status_ready())
			break;
		busyWait(NULL, 1);
	}
	if (i >= TIMEOUT_STATUS) {
		/* if not print message and forget it */
		printf("No Mitsumi CD-ROM drive recognized\n");
		return;
	}

	/* clear out returned status */
	inb(MCD_SDBASE + DATA_REG);

	/* setup drive, no DMA for now */
	outb(MCD_SDBASE + DATA_REG, MCD_DRIVE_CONFIGURATION);
	outb(MCD_SDBASE + DATA_REG, 2);
	outb(MCD_SDBASE + DATA_REG, 0);
	for (i = 0; i < TIMEOUT_STATUS; i++) {
		if (status_ready())
			break;
		busyWait(NULL, 1);
	}
	if (i >= TIMEOUT_STATUS) {
		printf("cannot switch DMA off\n");
		return;
	}

	/* clear out returned status */
	inb(MCD_SDBASE + DATA_REG);

	/* setup drive, generate interrupts on error */
	outb(MCD_SDBASE + DATA_REG, MCD_DRIVE_CONFIGURATION);
	outb(MCD_SDBASE + DATA_REG, 0x10);
	outb(MCD_SDBASE + DATA_REG, 4);
	for (i = 0; i < TIMEOUT_STATUS; i++) {
		if (status_ready())
			break;
		busyWait(NULL, 1);
	}
	if (i >= TIMEOUT_STATUS) {
		printf("cannot activate IRQ\n");
		return;
	}

	/* clear out returned status */
	inb(MCD_SDBASE + DATA_REG);

	/* now let's see what we have here, request drive version */
	outb(MCD_SDBASE + DATA_REG, MCD_REQUEST_VERSION);
	for (i = 0; i < TIMEOUT_STATUS; i++) {
		if (status_ready())
			break;
		busyWait(NULL, 1);
	}
	if (i >= TIMEOUT_STATUS) {
		printf("no response for REQUEST_VERSION\n");
		return;
	}

	/* clear out returned status */
	inb(MCD_SDBASE + DATA_REG);

	read_status(&drive_type);
	read_status(&drive_version);

	/* drive type dependent initialization */
	switch (drive_type) {
	case TYPE_LU005:
		type_string = "LU005/FX001 single speed";
		read_command = MCD_SEEK_READ;
		break;
	case TYPE_FX001:
		type_string = "FX001D double speed";
		read_command = MCD_SEEK_READ_DOUBLE;
		break;
	default:
		type_string ="unknown";
		read_command = MCD_SEEK_READ;
		break;
	}

	/* write informations about the drive to the console */
	printf("Mitsumi CD-ROM %s drive revision %d.%d found\n", type_string,
		drive_version & 0xf, drive_version >> 4);

	/* install interrupt handler for the device */
	setivec(MCD_SDIRQ, mcd_irq);

	/* device is initialized now and can be used */
	initialized = 1;
}

/*
 * just for the case that we get loadable device drivers sometime again,
 * without testing I'm not sure if enough is done here
 */
static void mcd_unload()
{
	clrivec(MCD_SDIRQ);
	initialized = 0;
}

/* -------------------------------------------------------------------- */
/* Interrupt handling							*/
/* -------------------------------------------------------------------- */

/*
 * I don't know yet for which reasons interrupts are generated, so just
 * take the interrupt and clear the status register
 */
static void mcd_irq()
{
	int i;

	i = inb(MCD_SDBASE + STATUS_REG);
	if (!(i & HW_STEN)) {
		error_status = inb(MCD_SDBASE + DATA_REG) & 0xff;
#ifdef DEBUG_IRQ
		printf("mcd: IRQ, status = %x\n", error_status);
#endif
	}
}

/* -------------------------------------------------------------------- */
/* Timing functions							*/
/* -------------------------------------------------------------------- */

/*
 * sleep timeout, awake process again
 */
static void awake_mcd_driver(event)
char *event;
{
	wakeup(event);
}

/*
 * test and set semaphore for the character device
 */
static void mcd_busy()
{
	int i;

	i = sphi();
	while (char_dev_busy)
		x_sleep(mcd_event_busy, pridisk, slpriSigLjmp, mcd_busy_text);
	char_dev_busy++;
	spl(i);
}

/*
 * free the semaphore for the character device and awake waiting processes
 */
static void mcd_not_busy()
{
	char_dev_busy = 0;
	wakeup(mcd_event_busy);
}

/* -------------------------------------------------------------------- */
/* Support for device driver functions to access CD-ROM			*/
/* -------------------------------------------------------------------- */

/*
 * read the TOC from disk
 */
static int read_toc()
{
	/* work done if TOC is uptodate */
	if (toc_read)
		return(0);

	if (get_toc_header() < 0) {
#ifdef DEBUG
		printf("mcd: read_toc(): cannot read TOC header\n");
#endif
		return(EIO);
	}

	if (get_toc() < 0) {
#ifdef DEBUG
		printf("mcd: read_toc(): cannot read TOC\n");
#endif
		return(EIO);
	}

	toc_read = 1;
	return(0);
}

/*
 * get the TOC header
 */
static int get_toc_header()
{
	outb(MCD_SDBASE + DATA_REG, MCD_REQUEST_TOC_DATA);
	if (wait_status(TIMEOUT_STATUS) == -1)
		return(-1);

	if (read_status(&mcd_toc_header.first) < 0) return(-1);
	if (read_status(&mcd_toc_header.last) < 0) return(-1);

	mcd_toc_header.first = bcd2bin(mcd_toc_header.first);
	mcd_toc_header.last = bcd2bin(mcd_toc_header.last);

	if (read_status(&mcd_toc_header.leadout.min) < 0) return(-1);
	if (read_status(&mcd_toc_header.leadout.sec) < 0) return(-1);
	if (read_status(&mcd_toc_header.leadout.frame) < 0) return(-1);

	if (read_status(&mcd_toc_header.first_track.min) < 0) return(-1);
	if (read_status(&mcd_toc_header.first_track.sec) < 0) return(-1);
	if (read_status(&mcd_toc_header.first_track.frame) < 0) return(-1);

	return(0);
}

/*
 * read the TOC from disk
 */
static int get_toc()
{
	int i;
	int count, limit;
	struct toc tmp_toc;

	/* clear the TOC */
	for (i = 0; i < MAX_TRACKS; i++)
		mcd_toc[i].index = 0;

	count = mcd_toc_header.last + 3;

	/* hold the drive */
	outb(MCD_SDBASE + DATA_REG, MCD_HOLD);
	if (wait_status(TIMEOUT_STATUS) == -1) {
#ifdef DEBUG
		printf("mcd: hold drive timeout\n");
#endif
		return(-1);
	}

	/* change mode to read the TOC */
	outb(MCD_SDBASE + DATA_REG, MCD_MODE_SET);
	outb(MCD_SDBASE + DATA_REG, 5);
	if ((i = wait_status(TIMEOUT_STATUS)) == -1) {
#ifdef DEBUG
		printf("mcd: change mode -> TOC timeout\n");
#endif
		return(-1);
	}

	/* and now read the TOC */
	for (limit = MAX_TRACKS*3; limit > 0; limit--) {
		/* get current SUBQ */
		if (get_subq(&tmp_toc) < 0) {
#ifdef DEBUG
			printf("mcd: get_subq() failed\n");
#endif
			break;
		}

		i = bcd2bin(tmp_toc.index);

		if ((i > 0) && (i < MAX_TRACKS) && (tmp_toc.track == 0))
			if (mcd_toc[i].index == 0) {
				mcd_toc[i] = tmp_toc;
				count--;
			}
		if (count <= 0)
			break;
	}

	mcd_toc[mcd_toc_header.last + 1].disk_time = mcd_toc_header.leadout;

	/* change mode back to read data */
	outb(MCD_SDBASE + DATA_REG, MCD_MODE_SET);
	outb(MCD_SDBASE + DATA_REG, 1);
	if (wait_status(TIMEOUT_STATUS) == -1) {
#ifdef DEBUG
		printf("mcd: set mode -> data timeout\n");
#endif
		return(-1);
	}

	return((limit > 0) ? 0 : -1);
}

/*
 * get the current subchannel informations
 */
static int get_subq(tocp)
struct toc *tocp;
{
	int ignore;

	outb(MCD_SDBASE + DATA_REG, MCD_REQUEST_SUBQ_CODE);
	if (wait_status(TIMEOUT_STATUS) == -1) {
#ifdef DEBUG
		printf("mcd: get_subq(): timeout\n");
#endif
		return(-1);
	}

	if (read_status(&tocp->ctrl_addr) < 0) return(-1);
	if (read_status(&tocp->track) < 0) return(-1);
	if (read_status(&tocp->index) < 0) return(-1);
	if (read_status(&tocp->track_time.min) < 0) return(-1);
	if (read_status(&tocp->track_time.sec) < 0) return(-1);
	if (read_status(&tocp->track_time.frame) < 0) return(-1);
	if (read_status(&ignore) < 0) return(-1);
	if (read_status(&tocp->disk_time.min) < 0) return(-1);
	if (read_status(&tocp->disk_time.sec) < 0) return(-1);
	if (read_status(&tocp->disk_time.frame) < 0) return(-1);

#ifdef DEBUG_SUBQ
	printf("addr=%d, track=%d, index=%d, track time=%d:%d:%d, disk time=%d:%d:%d\n",
		tocp->ctrl_addr,
		tocp->track,
		tocp->index,
		tocp->track_time.min,
		tocp->track_time.sec,
		tocp->track_time.frame,
		tocp->disk_time.min,
		tocp->disk_time.sec,
		tocp->disk_time.frame);
#endif

	return(0);
}

/*
 * send a play or read data command
 */
static int do_play(cmd, pp)
int cmd;
struct play *pp;
{
	outb(MCD_SDBASE + DATA_REG, cmd);
	outb(MCD_SDBASE + DATA_REG, pp->start.min);
	outb(MCD_SDBASE + DATA_REG, pp->start.sec);
	outb(MCD_SDBASE + DATA_REG, pp->start.frame);
	outb(MCD_SDBASE + DATA_REG, pp->end.min);
	outb(MCD_SDBASE + DATA_REG, pp->end.sec);
	outb(MCD_SDBASE + DATA_REG, pp->end.frame);

#ifdef DEBUG_PLAY
	printf("start = %d:%d:%d, end = %d:%d:%d\n",
		pp->start.min,
		pp->start.sec,
		pp->start.frame,
		pp->end.min,
		pp->end.sec,
		pp->end.frame);
#endif
}

/*
 * wait until drive status is available and return it
 */
static int wait_status(tries)
int tries;
{
	int i;

	for (i = 0; i < tries; i++) {
		if (status_ready())
			break;
		if (i > 200)
			busyWait2(NULL, 1);
		else {
			timeout(&mcd_timer, 1, awake_mcd_driver, mcd_event_wait);
			x_sleep(mcd_event_wait, pridisk, slpriNoSig, mcd_wait_text);
		}
	}

	if (i >= tries)
		return(-1);

	i = inb(MCD_SDBASE + DATA_REG) & 0xff;

	/*
	 * check if disk was changed, if yes the TOC in memory
	 * and the current audio status are invalid
	 */
	if (i & MCD_STATUS_CHANGE) {
		toc_read = 0;
		audio_status = CDROM_AUDIO_NO_STATUS;
	}

	/* check if audio CD still is playing */
	if ((audio_status == CDROM_AUDIO_PLAY) && ((i & MCD_STATUS_AUDIO) == 0))
		audio_status = CDROM_AUDIO_COMPLETED;

	return(i);
}

/*
 * wait until data is available
 */
static int wait_data(tries)
int tries;
{
	int i;
	int status;

	for (i = 0; i < tries; i++) {
		if (data_ready())
			return(0);
		timeout(&mcd_timer, 1, awake_mcd_driver, mcd_event_wait);
		x_sleep(mcd_event_wait, pridisk, slpriNoSig, mcd_wait_text);
	}

	return(-1);
}

/*
 * read next status byte
 */
static int read_status(s)
char *s;
{
	int i;

	for (i = 0; i < 100; i++) {
		if (status_ready())
			break;
		busyWait2(NULL, 1);
	}

	if (i >= 100)
		return(-1);

	i = inb(MCD_SDBASE + DATA_REG) & 0xff;
	*s = (unsigned char) i;
	return(0);
}

/* -------------------------------------------------------------------- */
/* Conversion functions							*/
/* -------------------------------------------------------------------- */

/*
 * convert two BCD digits to an integer
 */
static int bcd2bin(bcd)
unsigned char bcd;
{
	return((bcd >> 4) * 10 + (bcd & 0xf));
}

/*
 * convert an integer to two BCD digits
 */
static unsigned char bin2bcd(bin)
int bin;
{
	unsigned char result;

	result = (bin / 10) << 4;
	result |= (bin % 10);

	return(result);
}

/*
 * convert from MSF format to a block number
 */
static int msf2blk(mp)
struct msf *mp;
{
	return (bcd2bin(mp->frame) + bcd2bin(mp->sec) * 75 +
		bcd2bin(mp->min) * 4500 - FIRST_BLK_OFFSET);
}

/*
 * convert from block number to MSF format
 */
static void blk2msf(block, mp)
int block;
struct msf *mp;
{
	block += FIRST_BLK_OFFSET;
	mp->min = bin2bcd(block / 4500);
	block %= 4500;
	mp->sec = bin2bcd(block / 75);
	mp->frame = bin2bcd(block % 75);
}
