#ifndef ARP_H
#define ARP_H
/* RWDOCS */
#include "iface.h"

/* Size of ARP hash table */
#define	ARPSIZE	17

/* Lifetime of a valid ARP entry */
#define	ARPLIFE		900	/* 15 minutes */
/* Lifetime of a pending ARP entry */
#define	PENDTIME	30	/* 30 seconds */

/* ARP definitions (see RFC 826) */

/* Address size definitions */
#define	IPALEN	4		/* Length in bytes of an IP address */
#define	MAXHWALEN	255	/* Maximum length of a hardware address */

/* ARP opcodes */
#define	ARP_REQUEST	1
#define	ARP_REPLY	2

/* Hardware types */
#define	ARP_NETROM	0	/* Fake for NET/ROM (never actually sent) */
#define	ARP_ETHER	1	/* Assigned to 10 megabit Ethernet */
#define	ARP_EETHER	2	/* Assigned to experimental Ethernet */
#define	ARP_AX25	3	/* Assigned to AX.25 Level 2 */
#define	ARP_PRONET	4	/* Assigned to PROnet token ring */
#define	ARP_CHAOS	5	/* Assigned to Chaosnet */
#define	ARP_BLANK	6	/* hate to see a blank - K5JB */
#define	ARP_ARCNET	7
#define	ARP_APPLETALK	8

#define NTYPES 9


extern char *arptypes[];

/* Table of hardware types known to ARP 
 *** formatted to int where possible to avoid
 * alignment problems. This stuff must be
 * cast when it is inserted or retreived from
 * wire packets.
 */
struct arp_type {
	unsigned hwalen;		/* Hardware length */
	unsigned iptype;		/* Hardware type field for IP */
	unsigned arptype;		/* Hardware type field for ARP */
	char *bdcst;		/* Hardware broadcast address */
	int (*format)();	/* Function that formats addresses */
	int (*scan)();		/* Reverse of format */
};
extern struct arp_type arp_type[];
#define	NULLATYPE	(struct arp_type *)0

/* Format of an ARP request or reply packet. From p. 3
 *
 * Ahhh, dos has its segmented bs, but iBSC and COFF
 * Unix have this allignment bs.
 *
 * This is the packet in the form used on the wires
 * it needs to have exactly the alignment specified
 * below. It should align properly as is on 32 bit
 * machines except that 256 bytes each will be allocated for
 * shwaddr and for thwaddr.
 */
struct arp {
	int32 hardware;			/* Hardware type */
	int32 protocol;			/* Protocol type */
/* --------------------*/
	unsigned char hwalen;			/* Hardware address length, bytes */
	unsigned char pralen;			/* Length of protocol address */
	int32 opcode;			/* ARP opcode (request/reply) */
/* --------------------*/
	char shwaddr[MAXHWALEN];	/* Sender hardware address field */
/* --------------------*/
	int32 sprotaddr;		/* Sender Protocol address field */
/* --------------------*/
	char thwaddr[MAXHWALEN];	/* Target hardware address field */
/* --------------------*/
	int32 tprotaddr;		/* Target protocol address field */
/* --------------------*/
};

/* Format of ARP table */
struct arp_tab {
	struct arp_tab *next;	/* Doubly-linked list pointers */
	struct arp_tab *prev;
	int32 ip_addr;		/* IP Address, host order */
	int hardware;		/* Hardware type */
	char *hw_addr;		/* Hardware address */
	int state;		/* (In)complete, changed to int from char RLW */
#define	ARP_PENDING	0
#define	ARP_VALID	1
	int pub;		/* Respond to requests for this entry? */
	struct timer timer;	/* Time until aging this entry */
	struct mbuf *pending;	/* Queue of datagrams awaiting resolution */
};

#define	NULLARP	(struct arp_tab *)0
extern struct arp_tab *arp_tab[];

struct arp_stat {
	unsigned recv;		/* Total number of ARP packets received */
	unsigned badtype;	/* Incoming requests for unsupported hardware */
	unsigned badlen;	/* Incoming length field(s) didn't match types */
	unsigned badaddr;	/* Bogus incoming addresses */
	unsigned inreq;		/* Incoming requests for us */
	unsigned replies;	/* Replies sent */
	unsigned outreq;	/* Outoging requests sent */
};

#ifdef GCC
	/* arp.c */
/* Initialize an entry in the ARP table
 * Called by the device driver at attach time
 */
int arp_init(unsigned int hwtype,int hwalen,int iptype,int arptype,char *bdcst,int (*format)(),int (*scan)() );

/* Resolve an IP address to a hardware address; if not found,
 * initiate query and return NULLCHAR.  If an address is returned, the
 * interface driver may send the packet; if NULLCHAR is returned,
 * res_arp() will have saved the packet on its pending queue,
 * so no further action (like freeing the packet) is necessary.
 */
char *res_arp(struct interface *interface,int hardware,int32 target,struct mbuf *bp);

/* Handle incoming ARP packets. This is almost a direct implementation of
 * the algorithm on page 5 of RFC 826, except for:
 * 1. Outgoing datagrams to unresolved addresses are kept on a queue
 *    pending a reply to our ARP request.
 * 2. The names of the fields in the ARP packet were made more mnemonic.
 * 3. Requests for IP addresses listed in our table as "published" are
 *    responded to, even if the address is not our own.
 */
void arp_input(struct interface *interface,struct mbuf *bp);

/* Add an IP-addr / hardware-addr pair to the ARP table */
struct arp_tab *arp_add(int32 ipaddr,int hardware,
	char *hw_addr,int hw_alen,int pub);

/* Remove an entry from the ARP table */
void arp_drop(struct arp_tab *ap);

/* Look up the given IP address in the ARP table */
struct arp_tab *arp_lookup(int hardware,int32 ipaddr);

/* Send an ARP request to resolve IP address target_ip */
void arp_output(struct interface *interface,int hardware,int32 target);

/* Hash a {hardware type, IP address} pair */
unsigned arp_hash(int hardware,int32 ipaddr);

/* Copy a host format arp structure into mbuf for transmission */
struct mbuf *htonarp(struct arp *arp);

/* Convert an incoming ARP packet into a host-format structure */
int ntoharp(struct arp *arp,struct mbuf **bpp);

	/* arpcmd.c */
int doarp(int argc,char **argv);
int doarpadd(int argc,char **argv);
int doarpdrop(int argc,char **argv);
int dumparp();

	/* arpdump.c */
int arp_dump(struct mbuf **bpp);

#else
struct arp_tab *arp_lookup(),*arp_add();
unsigned arp_hash();
void arp_output();

#endif /* GCC */
/* RWDOCE */
#endif /* ARP_H */
