#
/*
 *		C - symbolic tracer
 *
 *		Paul Klint
 *		Mathematisch Centrum
 *		Amsterdam, 1975
 */
/*
 *	required changes and bugs:
 *
 *	1)	externals are only known inside the file where they are
 *		used(there seems no easy solution,apart from looking
 *		in the object module loader table)
 *	2)	mixing input to tracer and traced program
 *	3)	when a syntax error occurs in a linecond,
 *		line-range is changed.
 *	4)	echo input rvalue
 *	5)	echo after assignment
 *	6)	echo tree-structure after stopcondition
 */

/*	Name changes for automatically compiled version */
/*	This bit added by Peter C Dec 1977	*/
#define XXTRACER 1
#ifdef XXTRACER
#define stat  xxstat
#define lvalue  xxlval
#define subvar  xxsubv
#define var  xxvar
#define subscript  xxsubs
#define rvalue  xxrval
#define constant  xxcons
#define getname  xxgetnm
#define idchar  xxidch
#define letter  xxlet
#define digit  xxdig
#define lookup1  xxlook1
#define lookup  xxlook
#define show  xxshow
#define linecond  xxlcond
#define nextchar  xxnxtch
#define nonblank  xxnblnk
#define err  xxerr
#define decref  xxdec
#define incref  xxinc
#define length  xxlen
#define tbline  xxtbl
#define help  xxhelp
#define wait  xxwait
#define value  xxval
#define symbuf  xxsymb
#define lastc  xxlc
#define peekc  xxpc
#define saver5  xxsr5
#define curr5  xxcur5
#define offset  xxoff
#define atomtype  xxattp
#define atom  xxat
#define errcnt  xxecnt
#define psrepold  xxpsro
#define psrep  xxpsr
#define psfrom  xxpsf
#define psto  xxpst
#define funame  xxfun
#define psfu  xxpfu
#define d0  xxd0
#define dx  xxdx
#define name  xxnm
#define select  xxsel
#define repshow  xxrsh
#define nbounds  xxnbnds
#define baseadr  xxbadr
#define equal  xxeq
#define subseq  xxsubq
#define condspace  xxcndsp
#define condpt  xxcndpt
#define treespace  xxtsp
#define treept  xxtpt
#define tree  xxt
#define stopcond  xxscnd
#define varcond  xxvcnd
#define varprim  xxvprm
#define evcond  xxecnd
#define signal_catched  xxctch
#define stackflag  xxstflg
#define intracer  xxintr
#define setsigs  xxsss
#define interrupt  xxitp
#define intint  xxii
#define intbus  xxib
#define intseg  xxis
#define instack  xxist
#define charcon  xxchcon
#define conchar  xxconch
#endif

/*
 *		tracer stop criteria
 */

#define	INITIALIZE	0	/* initialization flag */
#define	STEP		1	/* step through traced program */
#define	LINEWAIT	2	/* give control in specified line range */
#define VALWAIT		3	/* give control on variable condition */

/*
 *		signals catched
 */

#define	SIGINT	2
#define	SIGBUS	10
#define	SIGSEG	11


/*
 *		types
 */

#define	INT	0
#define	CHAR	1
#define	FLOAT	2
#define	DOUBLE	3
#define	STRUCT	4
#define LONG 6

#define TYPE	07
#define TYLEN	2
#define XTYPE	(03<<3)
#define	PTR	010
#define	FUNC	020
#define	ARRAY	030
#define	CONSTANT	-1

/*
 *		storage classes
 */

#define	EXTERN	12
#define	STATIC	13
#define	REG	14

/*
 *		parameters
 */

#define	NAMESIZE	9	/* size of name in symbol table */
#define	NCONDNODES	10
#define	NTNODES	10
#define	END_OF_TABLE	-1	/* end of symbol table mark */

/*
 *		operators
 */

#define	ARROW	128
#define	OR	129
#define	AND	130
#define	EQUAL	131
#define	NOTEQ	132
#define	LESS	133
#define	LESSEQ	134
#define	GREAT	135
#define	GREATEQ	136

/* 
 * 		data structures
 */

struct {
	int	integer;		/* extracts int from stack */
};

struct {
	char	ch;			/* extracts char from stack */
};

struct {
	float	fl;			/* extracts float from stack */
};

struct {
	double	db;			/* extracts double from stack */
};

struct {
	long	lg;			/* extracts long from stack */
};

struct	hshtab{				/* format of symbol table entry */
	int	htype;			/* type */
	char	ssp;			/* subscript length */
	char 	lenp;			/* structure length */
	int	hoffset;		/* stack offset or abs address */
	char	hname[NAMESIZE];	/* identifier */
	char	hclass;			/* storage class */
	int	*dim;			/* pointer to dimension array */
};


struct condnode{		/* holds one stop condition */
	int	c_condition;	/* common with t_op */
	int	c_instack;	/* lies adr in stack flag */
	char	*c_off[2];
	int	c_type[2];
	double	c_val[2];
};

struct tnode{			/* holds a treenode */
	int	t_op;		/* common with c_condition */
	struct tnode *t_left,*t_right;
};

/*
 *		variables
 */

int	wait	INITIALIZE;	/* stop condition */
int	intracer;		/* running inside tracer ? */
int	stackflag;		/* variable in stop condition not
				   addressable */
int	signal_catched;		/* has a signal been sent to this process */
double	value;			/* value of last expression parsed */
char	symbuf[NAMESIZE];	/* holds an identifier */
int	lastc;			/* last symbol read */
int	peekc;			/* look ahead symbol */
char	*saver5;		/* previous stack frame */
char	*curr5;			/* current stack frame */
int	offset;			/* address of last lvalue */
int	atomtype;		/* type of last expression */
int	errcnt;			/* error count */
int	nbounds;		/* number of bounds of last array */
int	psrep	1;		/* repeat count */
int	psrepold	1;	/* previous value of psrep */
int	psfrom	1;		/* start of line range */
int	psto	1000;		/* end of line range */
char	*funame;		/* name of current function */
char	*psfu	"main";		/* traced function */


struct hshtab *d0;		/* global symbol table */
struct hshtab *dx;		/* local symbol table */

struct hshtab *name;		/* entry for last lvalue */
struct hshtab *atom;		/* last entry used */


/*
 *		variables for stop condition
 */

struct condnode condspace[NCONDNODES];
struct condnode *condpt;

struct tnode treespace[NTNODES];
struct tnode *treept;
struct tnode *tree;		/* current condition tree */



#ifdef XXTRACER
xtracer(line){
#endif
#ifndef XXTRACER
tracer(line){
#endif
	struct hshtab *tb;
	int oldfout;		/* added Peter C - to avoid problems with output streams */
	extern fout;
	extern intint(),intbus(),intseg();

	curr5  = &line  - 2;		/* thanks to these tricky statements */
	saver5 = (curr5->integer);	/* the number of arguments to  the trace */
	funame = (saver5-8)->integer;	/* routine can be reduced to one ! */
	dx = (saver5-10)->integer;
	d0 = (saver5-12)->integer;

	switch(wait){

	case INITIALIZE:
		setsigs();

	case STEP:
		break;

	case LINEWAIT:
		if((line < psfrom) ||
		   (line > psto)   ||
		   (!equal(funame,psfu)))

			return;
		if(--psrep)
			return;
		break;

	case VALWAIT:
		stackflag = 0;
		if(evcond(tree) == 0 && stackflag == 0)
			return;
		break;

	default:
		err("'wait' illegal value");
		exit(1);
	}

	wait = STEP;
	oldfout = fout;		/* added to select channel 2 output */
	flush();
	fout = 2;
	printf("\n<<<%s: %d\n",funame,line);
	if(stackflag)
		printf("(adr) %o in stopcondition lies outside stack\n",stackflag);
	intracer = 1;
	setexit();
	if(signal_catched)
		signal_catched = 0;
	while(nextchar() != '\n'){
		errcnt = 0;
		if(lastc == '?'){
			help();
			goto skip;
			}
		else
		if(!stat()){
			if(errcnt == 0)
				err("statement syntax");

		skip:
			while(lastc!='\n')
				nextchar();
		}
	}
	printf(">>>");
	flush();
	fout = oldfout;
	intracer = 0;
}

/*
 *	Parse one tracer input statement.
 */

stat(){
	register int l1,t1;

	if(lastc == '/')
		return(linecond());
	else
	if(lastc == '('){
		treept = &treespace[0];
		condpt = &condspace[0];
		if(tree = stopcond()){
			wait = VALWAIT;
			return(1);
		}
		else
			return(0);
	}
	if(!lvalue())
		return(rvalue());	/* a constant or address was
					perhaps required */
	l1=offset;
	t1 = atomtype;
	if(lastc == '\n')
		return(show(l1,t1));
	else
		if(lastc == '/')
			return(repshow(l1,t1));
		else
			if(lastc != '=')
				return(err("'=' is missing"));
	nextchar();
	if(!rvalue())
		return(0);
	if(t1&XTYPE){
		if((t1&XTYPE) == PTR)
			l1->integer = value;
		else
			return(err("illegal assignment"));
	}
	else
		switch(t1){
	
		case CHAR:
			l1->ch = value;break;
		case INT:
			l1->integer = value; break;
		case FLOAT:
			l1->fl = value;break;
		case DOUBLE:
			l1->db = value;break;
		case LONG:
			l1->lg = value; break;
		case STRUCT:
			return(err("can't assign to a struct"));
		default:
			return(err("tracer error -- stat"));
		}
	show(l1,t1);
	return(1);
}

lvalue(){
	register int pt,t;
	int indir;
	register struct hshtab *nm;
	indir = 1;
	while(lastc == '*'){
		nextchar();
		indir++;
	}
	if(!var())
		return(0);
	nm = name;
	pt = baseadr(name) + offset;
	while(lastc == '.' || lastc == ARROW){
		if(lastc==ARROW){
			if((atomtype&XTYPE) != PTR)
				return(err("illegal pointer"));
			pt = pt->integer;
			lastc = '.';
		}
		if(!var())
			return(0);
		pt =+ offset;
	}
	t = atomtype;
	while(--indir && ((t&XTYPE) == PTR)){
		pt = pt->integer;
		t = decref(t);
	}
	if(indir)
		return(err("too many indirections"));
	offset = pt;
	atomtype = t;
	name = nm;
	return(1);
}

var(){
	if( letter(lastc)  || lastc=='.' ) 
		getname();
	else
		return(0);
	if(!lookup())
		return(err("'%s' not found",symbuf));
	offset = name->hoffset;
	atom = name;
	atomtype = atom->htype;
	if(atom->hclass == REG)
		offset = curr5 - saver5 - (10 - 2*offset);
	nbounds = 0;
	if(lastc != '[')
		return(1);
	else
		return(subvar());
}

subvar(){
	int t,b,i,l,indir;
	register int bnd,index;
	register char *pt;
	struct hshtab *nm;

	t = name->htype;
	nm = name;
	pt = offset;
	b = baseadr(nm);
	index = 0;
	while(subscript()){
		indir = ((t&XTYPE) == PTR) ? 1 : 0;
		t = decref(t);
		if(l = length(nm,++index))
			if(indir)

				pt = ((pt + b)->integer)
					+ (i = value) * l
					- b;
			else
				pt =+ (i = value) * l;
		else
			return(0);
	}
	name = atom =nm;
	offset = pt;
	atomtype = t;
	nbounds = index;
	return(1);
}

subscript(){
	if(lastc != '[')
		return(0);
	else	nextchar();
	if( !rvalue() || lastc != ']' || atomtype > CHAR)
		return(err("bad subscript"));
	else{
		nextchar();
		return(1);
	}
}

rvalue(){
	register int adr;

	if(conchar(lastc))
		return(constant());
	if(lastc == '&'){
		adr = 1;
		nextchar();
	}
	else	adr = 0;
	if(!lvalue())
		return(0);
	if(adr){
		atomtype = incref(atomtype);
		value = offset;
		return(1);
	}
	return(select(offset,atomtype));
}

instack(adr){		/* lies adr in stack ? */
	register int s;

	s = curr5;
	if(s < 0)
		return((adr<0) ? (s<adr) : 0);
	else
		return((adr>0) ? (s<adr) : 1);
}

/*
 *	Compute base address of symbol table entry "nm".
 *	Selects absolute addressing or stack addressing.
 */

baseadr(nm)
struct hshtab *nm;{
	return(((nm->hclass == EXTERN) ||
		(nm->hclass == STATIC)) ? 0 : saver5);
}

/*
 *	Extract a value of type "type" from  address "off".
 */

select(off,type){
	register int roff;

	roff = off;
	if(type>=PTR)
		value = ((type & 030) == PTR) ? roff->integer
					      : roff;
	else
	switch(type){

	case CHAR:
		value = roff->ch;break;

	case INT:
		value = roff->integer;break;

	case FLOAT:
		value = roff->fl;break;

	case DOUBLE:
		value = roff->db;break;
	case LONG:
		value = roff->lg; break;

	case STRUCT:
		return(err("value of a struct not defined"));

	default:
		return(err("tracer error -- select"));
	}
	return(1);
}

constant(){
	register int base,sign,s;

	if(lastc == '\'')
		return(charcon());
	if((lastc=='+') || (lastc=='-')){
		sign = (lastc=='+') ? 1 : -1;
		nextchar();
	}
	else	sign = 1;

	if(!digit(lastc))
		return(err("in constant"));
	s = lastc - '0';
	base = (lastc=='0') ? 8 : 10;
	while(digit(nextchar()))
		s = s * base + (lastc-'0');
	atomtype = INT;
	value = s * sign;
	return(1);
}

getname(){
	register int i;
	symbuf[0] = lastc;
	i=1;
	while(idchar(nextchar()))
		if(i<NAMESIZE-1)
			symbuf[i++] = lastc;

	symbuf[i] = '\0';
	return(1);
}

charcon(){
	register int c;

	c = getchar()&0177;
	if(nextchar() != '\'')
		return(0);
	nextchar();
	value = c;
	atomtype = CHAR;
	return(1);
}

conchar(c)
char c;{
	return( digit(c) || c == '+' || c == '-' || c == '\'');
}

idchar(c)
char c;{
	return( (letter(c) || digit(c) || c == '_') ? 1 : 0);
}

letter(c)
char c;{
	return( ((c>='a' && c<='z') || (c>='A' && c<='Z')) ? 1 : 0);
}

digit(c)
char c;{
	return((c >= '0' && c <= '9') ? 1 : 0);
}

equal(s1,s2)
char *s1,*s2;{
	register char *rs1,*rs2;
	rs1 = s1;
	rs2 = s2;
	while(*rs1++ == *rs2)
		if(*rs2++ == '\0') return(1);
	return(0);
}

lookup(){
	if(lookup1(dx))			/* found in local table !! */
		return(1);
	else
		return(lookup1(d0));	/* found in global table ?? */
}

lookup1(tab)
struct hshtab *tab;{
	register struct hshtab *tb;
	for(tb=tab;(tb->htype)!=END_OF_TABLE;tb++)
		if(equal(symbuf,tb->hname)){
			name = tb;
			return(1);
		}
	return(0);
}

/*
 *	Show a value of type "tt" at address "ll".
 */

show(ll,tt)
char *ll;{
	register char rch;
	register int rint;
	extern char *locv();

	printf(" ... ");
	if(tt>=PTR)
		printf("(adr) 0%o",((tt&XTYPE) == PTR) ? ll->integer : ll);
	else
	switch(tt){

	case CHAR:
		rch = ll->ch;
		printf("0%o   '%c'",rch,
			(rch>=' ' && rch<128) ? rch : '?');
		break;

	case INT:
		rint = ll->integer;
		printf("0%o   %d",rint,rint);break;

	case FLOAT:
		printf("%f",ll->fl);break;

	case DOUBLE:
		printf("%f",ll->db);break;

	case LONG:
		printf("%s",locv(ll->lg)); break;

	case STRUCT:
		return(err("can't show a struct"));

	default:
		return(err("tracer error -- show %d %d",ll,tt));
	}
	printf("\n");
	return(1);
}

/*
 *	Show successive array elements.
 *	Syntax:
 *		lvalue '/' sub ',' sub '/'
 *	Modified Peter C Dec 1977 to allow
 *		lvalue '/' rvalue ',' rvalue '/'
 *	because I got sick of typing un-necessary '['s and ']'s
 */

repshow(l,t)
char *l;{
	register int i,s;
	register char *l1;
	int vfrom,vto,t1,bnd;
	struct hshtab *nm;

	nm = atom;
	bnd = nbounds + 1;
	nextchar();
	if(lastc == '[')
	{
		if(!subscript() || lastc != ',')
			return(0);
	}
	else
	if(!rvalue() || atomtype > CHAR || lastc != ',')
		return(err("Bad first subscript"));
	vfrom = value;
	nextchar();
	if(lastc == '[')
	{
		if(!subscript() || lastc != '/')
			return(0);
	}
	else
	if(!rvalue() || atomtype > CHAR || lastc != '/')
		return(err("Bad second subscript"));
	nextchar();
	vto = value;
	if((s = length(nm,bnd)) == 0)
		return(0);
	t1 = decref(t);
	l1 = (((t&XTYPE) == PTR) ? (l->integer) : l) + s * vfrom;

	printf("\n");
	for(i=vfrom;i<=vto;i++){
		printf("[%d]",i);
		if(show(l1,t1) == 0)
			return(0);
		l1 =+ s;
	}
	return(1);
}

/*
 *	Parse a line-condition.
 *	Syntax:
 *		'/' [var] '/' [[rvalue] [',' rvalue] '/' [rvalue]]
 */


linecond(){
	wait = STEP;
	if(nextchar() != '/'){
		if(!var())
			return(0);
		if((name->htype&XTYPE) != FUNC)
			return(err("'%s' not a function",name->hname));
		psfu = name->hname;
	}
	if(lastc != '/') return(0);
	if(nextchar() != '\n'){
		if(lastc != '/'){
			if(lastc != ','){
				if(!rvalue()) return(0);
				psfrom = value;
			}
			if(lastc == ','){
				if(nextchar() != '/'){
					if(!rvalue()) return(0);
					psto = value;
				}
			}
			else
				psto = psfrom;

			if(psfrom<0 || psfrom > psto)
				return(err("%d -- %d is illegal",
					   psfrom,psto));
		}
		if(lastc != '/') return(0);

		if(nextchar() != '\n'){
			if(!rvalue()) return(0);
			if(value<=0)
				return(err("%f is illegal repeat",
					   value));
			psrepold = value;
		}
	}

	if(lastc != '\n') return(0);

	/* a complete line condition is parsed */

	wait = LINEWAIT;
	psrep = psrepold;
	printf("%s from %d to %d",
		psfu,psfrom,psto);
	printf(psrep==1 ? "\n" : "repeat %d\n",psrep);
	return(1);
}

/*
 *	Parse a whole stop-condition.
 *	Syntax:
 *		'(' relation_list ')'
 */

stopcond(){
	register struct tnode *rt;

	nextchar();
	if(!(rt = varcond()) || lastc != ')')
		return(0);
	nextchar();
	return(rt);
}

/*
 *	Parse a relation_list.
 */

varcond(){
	register struct tnode *rt;

	if(treept == &treespace[NTNODES])
		return(err("treespace overflow"));
	else
		rt = treept++;

	if(lastc == '('){
		if(!(rt->t_left = stopcond()))
			return(0);
	}
	else
		if(!(rt->t_left = varprim()))
			return(0);
	if(lastc == AND || lastc == OR){
		rt->t_op = lastc;
		nextchar();
		if(!(rt->t_right = varcond()))
			rt = 0;
		return(rt);
	}
	else{
		treept--;
		return(rt->t_left);
	}
}

/*
 *	Parse one relation.
 *	Syntax:
 *		lvalue rel_op rvalue
 */

varprim(){
	register struct condnode *rc;
	register int i;

	if(condpt == &condspace[NCONDNODES])
		return(err("condition space overflow"));
	else
		rc = condpt++;

	rc->c_instack = 0;
	for(i=0;i<2;i++){
		if(lvalue()){
			rc->c_type[i] = atomtype;
			rc->c_off[i] = offset;
			rc->c_instack =| (instack(offset) << i);
		} else {
			if(!rvalue())
				return(0);
			rc->c_type[i] = CONSTANT;
			rc->c_val[i] = value;
		}

		if(i == 0){
			if(lastc < EQUAL || lastc > GREATEQ )
				return(0);
			rc->c_condition = lastc;
			nextchar();
		}
	}

	return(rc);
}

/*
 *	Evaluate an activated stop-condition.
 *	"tpt" points to the corresponding treestructure.
 */
evcond(tpt)
struct tnode *tpt;{
	register struct tnode *rt;
	double v0,v1;
	register int rop,roff;
	int lcond,rcond,i;

	if((rt=tpt) == 0)
		return(err("tracer error -- evcond"));
	rop = rt->t_op;
	if(rop == OR || rop == AND){
		lcond = evcond(rt->t_left);
		rcond = evcond(rt->t_right);
		return((rop == AND) ? (lcond && rcond)
				    : (lcond || rcond));
	}

	for(i=0;i<2;i++){
		if(rt->c_type[i] != CONSTANT){
			roff = rt->c_off[i];
			if(((rt->c_instack >> i)&01) && !instack(roff))
				return(stackflag = roff);
			select(roff,rt->c_type[i]);
			rt->c_val[i]=value;
		}
	}
	v0 = rt->c_val[0];
	v1 = rt->c_val[1];

	switch(rop){

	case EQUAL:
		return(v0 == v1);

	case NOTEQ:
		return(v0 != v1);

	case LESS:
		return(v0 < v1);

	case LESSEQ:
		return(v0 <= v1);

	case GREAT:
		return(v0 > v1);

	case GREATEQ:
		return(v0 >= v1);
	}
	return(err("tracer error -- evcond(2)"));
}

nextchar(){
	if(peekc){
		lastc = peekc;
		peekc = 0;
	}
	else
		lastc = nonblank();

	switch(lastc){

	case '-':
		return(subseq('>','-',ARROW));

	case '=':
		return(subseq('=','=',EQUAL));

	case '!':
		return(subseq('=','!',NOTEQ));

	case '<':
		return(subseq('=',LESS,LESSEQ));

	case '>':
		return(subseq('=',GREAT,GREATEQ));

	case '&':
		return(subseq('&','&',AND));

	case '|':
		return(subseq('|','|',OR));
	}

	return(lastc);
}

subseq(c,a,b){
	if(!peekc)
		peekc = nonblank();
	if(peekc != c)
		return(lastc = a);
	peekc = 0;
	return(lastc = b);
}

nonblank(){
	register int c;
	while((c=getchar()) ==' ' || c == '\t');	/* remove parity bit and check */
	if(c != -1 && c != '\0' && c != '\4')			/* for EOT in case input is "raw" */
		return(lastc = c);
	else
		exit(1);
}


err(s,t1,t2,t3,t4)
char *s;{
	printf("\n\terror: ");
	printf(s,t1,t2,t3,t4);
	printf("\n");
	errcnt++;
	return(0);
}

decref(at)
{
	register t;

	t = at;
	if ((t & ~07) == 0) {
		err("Illegal indirection");
		return(t);
	}
	return((t>>2) & ~07 | t&07);
}

incref(t)
{
	return((t<<2)&~034 | (t&07) | PTR);
}

length(acs,bnd)
struct hshtab *acs;
{
	register t, n;
	register struct hshtab *cs;
	int i,t1;

	cs = acs;
	t = cs->htype;
	i = (cs->ssp & 0377) + 1;
	n = 1;

	while((t&XTYPE) && bnd--){
		if((t&XTYPE) == FUNC)
			goto bad;
		n = ((t&XTYPE) == ARRAY) ? (cs->dim[i++]) : 1;
		t = decref(t);
	}
	if(bnd>0)
bad:
		return(err("too many subscripts"));
	if ((t&~07)==FUNC)
		return(0);
	if((t&XTYPE) != ARRAY)
		if(t >= PTR)
			return(2);
		else
			n = 1;
	switch(t&07) {

	case INT:
		return(2*n);

	case CHAR:
		return(n);

	case FLOAT:
	case LONG:
		return(4*n);

	case DOUBLE:
		return(8*n);

	case STRUCT:
		return(n * cs->dim[cs->lenp & 0377]);
	}
	return(err("tracer error -- length"));
}

tbline(tb)
struct hshtab *tb;{
	register int i;
	register struct hshtab *rtb;

	rtb = tb;
	printf("%o,%o,%o,%o,%s,%o,%o\n",
		rtb->htype,
		rtb->ssp,
		rtb->lenp,
		rtb->hoffset,
		rtb->hname,
		rtb->hclass,
		rtb->dim);
}

help(){
	register struct hshtab *tb;
	int	i;

	switch(nextchar()){

	case 's':
		nextchar();
		constant();
		for(i= -value;i<value;i++)
			printf("\n%d\t0%o\t%d",i,(saver5+2*i)->integer,
				(saver5+2*i)->integer);
		break;

	case '0':
		tb = d0;
		goto printtable;

	case 'x':
		tb = dx;
	printtable:
		for(;tb->htype != END_OF_TABLE;tb++)
			tbline(tb);
		break;
	default:
		err("help");
	}
	return;
}


intint(){
	signal(SIGINT,intint);
	return(interrupt(SIGINT));
}

intbus(){
	signal(SIGBUS,intbus);
	return(interrupt(SIGBUS));
}

intseg(){
	signal(SIGSEG,intseg);
	return(interrupt(SIGSEG));
}

interrupt(sig){
	switch(sig){

	case SIGINT:
		printf("?\n"); break;

	case SIGBUS:
		printf("\n-- bus error --\n"); break;

	case SIGSEG:
		printf("\n-- segmentation violation --\n"); break;
	}

	signal_catched ++;
	flush();
	if(intracer)
		reset();
	else
		wait = STEP;
}

setsigs(){
	signal(SIGINT,intint);
	signal(SIGBUS,intbus);
	signal(SIGSEG,intseg);
}
