#
/*
 */

/*
 * DH-11 line printer driver.
 *
 * This driver is capable of driving most printers that can be
 * interfaced to the computer through an RS232 interface.
 * This driver was adapted from the Unix lp11 driver
 * and has been tested with a Printronix printer/plotter and
 * and a HyType II printer (homemade intelligent interface).
 *
 * This driver will handle a printer connected to a printer scanner
 * box, which works as follows:
 *	When you want to use the printer, set request-to-send.
 *	When you can use it, you'll get clear-to-send back.
 *	When you're done using it, drop request-to-send.
 * The printer must be able to accept several (8-16) characters after
 * dropping clear-to-send.  This is not the case with the Tally printer,
 * and so it can not be used with this driver.
 *
 *
 *
 * Written 08/11/79 by Bill Shannon
 * Based on lx.c driver.
 *
 * Modified 08/28/79 by Bill Shannon
 * Converted to Version 7
 *
 */

#include "../h/param.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/tty.h"

#define NLH	1		/* number of printers */


/*
 * Driver control flags
 */

#define	LHPRI	10
#define	EJLINE	60
#define	LENGTH	66		/* length of paper in lines */
#define	WIDTH	132		/* width of paper in characters */
#define	INDENT	8		/* number of chars to indent, if IND set */

/*
 *	User specified flags
 */
#define CAP	01		/* printer can't handle lower case */
#define	EJECT	02		/* printer knows about form feeds */
#define IND	010		/* indent all lines 8 spaces */
#define	BACKSP	020		/* device can backspace */
#define PRX	040		/* device is a Printronix printer/plotter */

/*
 *	Internal states
 */
#define OPEN	01		/* device is open */
#define PLOTM	04		/* plot mode flag */
#define ELONGM	010		/* elongated mode flag */
#define	LOST	020		/* don't know what line we're on */

#define	B9600	13		/* 9600 baud */

/*
 *	Special characters
 */
#define	FORM	014
#define PLOT	05		/* enter plot mode character */
#define ELONG	07		/* print elongated characters */

struct	lh {
	struct tty *l_tp;
	char	l_flags;
	char	l_state;
	char	l_mcc;		/* max chars actually printed */
	char	l_ccc;		/* current char count, saved white space, >= mcc */
	char	l_mlc;		/* max lines actually printed */
	char	l_clc;		/* current line count, saved white space, >= mlc */
	char	l_width;
	char	l_length;
	char	l_indent;
	char	l_pad;
} lh[NLH];

int iflags[NLH] {
	PRX | EJECT,		/* Printronix */
};

extern struct tty dh11[];
extern char dhmodem[];

int lhdh[NLH] {
	15,
};


lhopen(dev, flag)
{
	register struct lh *rlh;
	register d;
	register struct tty *tp;

	if ((d = minor(dev)) >= NLH || (rlh = &lh[d])->l_state & OPEN) {
		u.u_error = ENXIO;
		return;
	}
	rlh->l_flags = iflags[d];	/* set up initial flags */
	rlh->l_mlc = rlh->l_mcc = rlh->l_clc = rlh->l_ccc = 0;
	rlh->l_length = LENGTH;
	rlh->l_width = WIDTH;
	rlh->l_indent = INDENT;
	rlh->l_state = OPEN;
	rlh->l_tp = &dh11[lhdh[d]];
	dhmodem[lhdh[d]] = 0;
	dhopen(lhdh[d], 1);
	dhmodem[lhdh[d]] = 1;
	if (u.u_error)
		return;
	tp = rlh->l_tp;
	tp->t_state |= XCLUDE | MSYNC | HUPCLS | TTSTOP;
	tp->t_ispeed = tp->t_ospeed = B9600;
	dhparam(lhdh[d]);
	dmctl(lhdh[d], 7);	/* turn on RTS */
	lhoutput('\r', rlh);
}

lhclose(dev, flag)
{
	register struct lh *rlh;
	register struct tty *tp;
	extern dhclose();

	rlh = &lh[minor(dev)];
	tp = rlh->l_tp;
	lhcanon(FORM, rlh);		/* make sure we're at top of page */
	lhoutput(FORM, rlh);		/* and give us a blank page */
	ttstart(tp);
	wflushtty(tp);
	tp->t_state =& ~XCLUDE;
	/* wait for all output to drain, then call dhclose */
	timeout(dhclose, lhdh[minor(dev)], 2);
	rlh->l_state = 0;
}

lhwrite(dev)
{
	register struct tty *tp;
	register struct lh *rlh;
	register int c;

	rlh = &lh[minor(dev)];
	tp = rlh->l_tp;
	while ((c=cpass())>=0) {
		spl5();
		while (tp->t_outq.c_cc > TTHIWAT) {
			ttstart(tp);
			tp->t_state =| ASLEEP;
			sleep(&tp->t_outq, TTOPRI);
		}
		spl0();
		lhcanon(c, rlh);
	}
	ttstart(tp);
}

lhcanon(c, alh)
struct lh *alh;
{
	register c1, c2;
	register struct lh *rlh;

	rlh = alh;
	c1 = c;
	if (rlh->l_state&PLOTM) {
		if (c1 == '\n')  
			rlh->l_state =& ~PLOTM;
		lhoutput(c1, rlh);
	} else {
		/* removed 11-16-77   /WAS
		if(rlh->l_flags&CAP) {
			if(c1>='a' && c1<='z')
				c1 =+ 'A'-'a'; else
			switch(c1) {
	
			case '{':
				c2 = '(';
				goto esc;
	
			case '}':
				c2 = ')';
				goto esc;
	
			case '`':
				c2 = '\'';
				goto esc;
	
			case '|':
				c2 = '!';
				goto esc;
	
			case '~':
				c2 = '^';
	
			esc:
				lhcanon(c2, rlh);
				rlh->l_ccc--;
				c1 = '-';
			}
		}
		*/
	
		switch(c1) {
	
		case '\t':
			rlh->l_ccc = (rlh->l_ccc+8) & ~7;
			return;
	
		case FORM:
			rlh->l_clc = rlh->l_length - 1;
		case '\n':
			if (rlh->l_mcc != 0) {		/* at left margin ? */
				lhoutput('\r', rlh);	/* put us there */
				lhoutput('\n', rlh);	/* physically */
				rlh->l_mcc = 0;		/* and logically */
				rlh->l_mlc++;
				if (rlh->l_state & ELONGM) rlh->l_mlc++;
			}
			rlh->l_clc++;
			if (rlh->l_state&ELONGM) rlh->l_clc++;
			rlh->l_state =& ~(PLOTM | ELONGM);
			if (rlh->l_clc >= rlh->l_length) {
				/* past end of page */
				if ((rlh->l_mlc || rlh->l_state&LOST)  &&
				    /* page was not blank (or we're lost) */
				    (rlh->l_clc > rlh->l_mlc)) {
				    /* and we spaced over page boundary */
					if (rlh->l_flags & EJECT) {
						/* printer knows about FF */
						/* bring it to start of page */
						lhoutput(FORM, rlh);
					} else {
						/* must simulate FF with LF's */
						while (rlh->l_mlc++ < rlh->l_clc)
							lhoutput('\n', rlh);
					}
					rlh->l_state =& ~LOST;
				}
				rlh->l_mlc = rlh->l_clc = 0;
			}
			/* note fall through here */

		case '\r':
			rlh->l_ccc = 0;
			if(rlh->l_flags&IND)
				rlh->l_ccc = rlh->l_indent;
			return;
	
		case 010:
			if(rlh->l_ccc > 0) {
				rlh->l_ccc--;
				if ((rlh->l_flags&BACKSP) &&
				    /* and last char output was printable */
				    rlh->l_ccc == rlh->l_mcc - 1) {
					rlh->l_mcc--;
					lhoutput(010, rlh);
				}
			}
			return;
	
		case ' ':
			rlh->l_ccc++;
			return;

		case PLOT:
			if (rlh->l_flags&PRX) {
				rlh->l_state =| PLOTM | LOST;
			}
			goto print;

		case ELONG:
			if (rlh->l_flags & PRX) {
				rlh->l_state =| ELONGM;
				c1 = 010;
			}
			goto print;
	
		default:
		print:
			if(rlh->l_ccc < rlh->l_mcc) {
				lhoutput('\r', rlh);
				rlh->l_mcc = 0;
			}
			while (rlh->l_clc > rlh->l_mlc) {
				lhoutput('\n', rlh);
				rlh->l_mlc++;
			}
			if(rlh->l_ccc < (rlh->l_width & 0377)) {
				while(rlh->l_ccc > rlh->l_mcc) {
					lhoutput(' ', rlh);
					rlh->l_mcc++;
				}
				lhoutput(c1, rlh);
				if (c1 >= ' ') rlh->l_mcc++;
			}
			if (c1 >= ' ') rlh->l_ccc++;
		}
	}
}


lhoutput(c, alh)
struct lh *alh;
{
	register m1, m2;
	register struct lh *rlh;
	int c2;

	rlh = alh;
	if (rlh->l_state&PLOTM) {
		m1 = 1;
		m2 = 1<<5;
		c2 = 0100;
		while (m2) {
			if (c&m1) c2 =| m2;
			m1 =<< 1;
			m2 =>> 1;
		}
		c = c2;
	}
	ttyoutput(c, rlh->l_tp);
}

lhioctl(dev, com, addr, flag)
caddr_t addr;
{
	register struct lh *rlh;

	rlh = &lh[minor(dev)];
	switch (com) {
	case TIOCSETP:
		if (copyin(addr, (caddr_t)&rlh->l_flags, 6))
			u.u_error = EFAULT;
		break;

	case TIOCGETP:
		if (copyout((caddr_t)&rlh->l_flags, addr, 6))
			u.u_error = EFAULT;
		break;

	default:
		return(0);
	}
	return(1);
}
