/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)con.c	1.9 84/10/18";

/*
**	con [-quit-char] remote_tty[^quit-char]
**
**	Connect user to remote machine:- assumes "conn"
**	 or intelligent "getty" exists on any intermediate machine.
**	Uses "ttyconnect" system call.
**
**	SETUID ==> ROOT
*/

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO
#define	TERMIOCTL
#define	TTYCONNECT

#include	"global.h"

#include	"Passwd.h"
#include	"state.h"

#include	<errno.h>
#include	<signal.h>
#include	<setjmp.h>


/*
**	Parameters
*/

char *		devnet		= DEVNET();
char *		remote;

#define		QUIT		'y'
#define		QUIT_SEP	'^'
#define		ENV_SEP		':'

char		quit				= QUIT;		/* default escape char is ctrl/q */
char *		env;
char *		quitdescrpt			= "ctrl-";	/* escape prefix */

#define		TIMEOUT		20	/* give up line if no result after TIMEOUT secs */

/*
**	Miscellaneous
*/

char *		Name;		/* program invoked name */
char *		HomeNode;
NodeLink	LinkD;
int		Traceflag;
jmp_buf		alrm_jb;
jmp_buf		to_jb;
#if	SYSTEM < 3
struct sgttyb	orig_tty, new_tty;
#else	SYSTEM < 3
struct termio	orig_tty, new_tty;
#endif	SYSTEM < 3
int		done_stty;

int		alrmcatch(), sigcatch();
void		con(), execute(), finish();




int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	register char *	qp;
	Passwd		user;

	Name = *argv++;
	if ( (qp = strrchr(Name, '/')) != NULLSTR )
		Name = qp+1;
	
	if ( argc != 2 && argc != 3 )
	{
		Error("Usage: %s [-quit-char] remote-host[^quit-char][:env]", argv[0]);
		return 1;
	}

	if ( !GetUser(&user, getuid()) )
	{
		Error(user.P_error);
		return 1;
	}

	if ( !(user.P_flags & P_CANCON) )
	{
		Error("No permission.");
		return 1;
	}

	HomeNode = NodeName();

	if ( okuse() )
	{
		if ( argc == 3 && argv[0][0] == '-' )
		{
			quit = argv[0][1];
			argv++;
		}

		if ( (qp = strchr(argv[0], ENV_SEP)) != NULLSTR )
		{
			env = qp + 1;
			*qp = '\0';
		}

		if ( (qp = strrchr(argv[0], QUIT_SEP)) != NULLSTR )
		{
			quit = qp[1];
			*qp = '\0';
		}

		con(argv[0], user.P_uid, user.P_gid);
		return 0;
	}

	Error("illegal use!");
	return 1;
}


void
con(target, uid, gid)
	char *		target;
	int		uid;
	int		gid;
{
	register char *	port;
	register char *	link;
	int		fd;
	bool		end_to_end;
	bool		find_error;
	FILE *		remotefd;
	char		sobuf[BUFSIZ];
	extern		errno;

	if ( strcmp(target, HomeNode) == STREQUAL )
	{
		Error("This IS %s!\07", HomeNode);
		return;
	}

	if ( (link = FindAlias(target)) == NULLSTR )
		link = target;
	else
		target = link;

	if ( FindNode(link, pt_con, &LinkD) )
	{
		find_error = false;
		link = LinkD.nl_name;

		if ( LinkD.nl_connector != NULLSTR )
		{
			execute(LinkD.nl_connector, link, target);
			return;
		}
	}
	else
		find_error = true;

	remote = concat(devnet, target, "0", NULLSTR);

	if ( access(remote, 0) == SYSERROR )
	{
		free(remote);

		remote = concat(devnet, link, "0", NULLSTR);

		if ( access(remote, 0) == SYSERROR )
		{
			if ( find_error )
				Error("destination \"%s\" %s", target, LinkD.nl_name);
			else
				Syserror("link via \"%s\" to \"%s\" unavailable", link, target);

			return;
		}
	}
	else
		link = target;

	end_to_end = (bool)(link == target || strcmp(link, target) == 0);
	port = remote + strlen(remote) - 1;

	if ( setjmp(alrm_jb) )
	{
		Error("line hung");
		return;
	}

	signal(SIGALRM, alrmcatch);
	alarm(TIMEOUT);

	while ( (fd = open(remote, O_EXCL|O_RDWR)) == SYSERROR )
		if
		(
			(
				errno == ENXIO
#				ifdef	EOPENFAIL
				||
				errno == EOPENFAIL
#				endif	EOPENFAIL
			)
			&&
			*port != '9'
		)
		{
			(*port)++;
		}
		else
		{
			if ( errno == ENOENT && *port != '0' )
				Error("link to %s busy", target);
			else
				Syserror("can't open link \"%s\" to %s", link, target);

			return;
		}

	/*
	**	Put circuit in RAW mode, don't change speed, turn on XON/XOFF.
	**	(For SYSTEM xx, set VMIN to min, VTIME to max.)
	*/

	SetRaw(fd, 0, 1, 0, true);

	if ( (remotefd = fdopen(fd, "r+")) == NULL )
	{
		Syserror("fdopen failed on %s file %d", remote, fd);
		return;
	}

	if ( setjmp(to_jb) )
	{
		finish(1);
		return;
	}

	signal(SIGINT, sigcatch);
	signal(SIGQUIT, sigcatch);

#	if	SYSTEM < 3
	if ( gtty(0, &orig_tty) != SYSERROR )
	{
		done_stty = 1;
		new_tty = orig_tty;
		new_tty.sg_flags &= ~CRMOD;
		(void)stty(0, &new_tty);
	}
#	else	SYSTEM < 3
	if ( ioctl(0, TCGETA, &orig_tty) != SYSERROR )
	{
		done_stty = 1;
		new_tty = orig_tty;
		new_tty.c_oflag &= ~(OCRNL|ONOCR|ONLRET|ONLCR);
		new_tty.c_iflag &= ~(ICRNL|IGNCR|INLCR);
		(void)ioctl(0, TCSETA, &new_tty);
	}
#	endif	SYSTEM < 3

	chown(remote, uid, gid);	/* indicate current user of the log line */
	chmod(remote, 0600);		/* protect against other access */

	/*
	** logically connect local terminal to remote until quit char received
	*/

	setbuf(stdout, sobuf);
	fprintf(stdout, " %s -> %s ('%s%c'=quit)\r\n", HomeNode, link, quitdescrpt, quit);
	fflush(stdout);
	setbuf(stdout, NULLSTR);
	setbuf(remotefd, sobuf);

	alarm(0);
	sleep(2);	/* allow getty to start up */

	if ( quit != QUIT || env != NULL )
	{
		fprintf(remotefd, "@@:%s%c%c", target, QUIT_SEP, quit);
		if ( env != NULL )
			fprintf(remotefd, ":%s", env);
	}
	else
	if ( !end_to_end )
	{
		fprintf(remotefd, "@@:%s", target);
	}

	fprintf(remotefd, "\n");

	fflush(remotefd);

	alarm(0);

	if ( ttyconnect(0, fileno(remotefd), CONNECT, quit&037) == SYSERROR )
	{
		Syserror("connect to link \"%s\" failed", link);
		return;
	}

	alarm(TIMEOUT);

	fprintf(remotefd, "@@\004\004%c@@\004\004@", quit&037);
	fflush(remotefd);

	finish(0);
}



#ifndef	TTYCONNECT
int
ttyconnect(local, remote, ignore, quitc)
	int	local, remote, ignore;
	char	quitc;
{
	register int	saveflags;
	int		child;
	char		c;

	switch ( child = fork() )
	{
	 case 0:	/*
			**	Child copies remote to user
			*/
			for ( ;; )
				if ( read(remote, &c, 1) == 1 )
					(void)write(local, &c, 1);

	 case SYSERROR:
			Syserror("can't fork, try again");
			return SYSERROR;

	 default:	/*
			**	Parent copies user to remote
			**	and intercepts escape
			*/

			SetRaw(local, 0, 1, 0, false);

			for ( ;; )
				if ( read(local, &c, 1) == 1 )
					if ( (c&0177) == quitc )
						break;
					else
						(void)write(remote, &c, 1);

			(void)kill(child, SIGKILL);
			(void)alarm(2);
			(void)wait(&child);
#			if	SYSTEM < 3
			(void)stty(local, &orig_tty);
#			else	SYSTEM < 3
			(void)ioctl(local, TCSETA, &orig_tty);
#			endif	SYSTEM < 3
			done_stty = 0;
	}

	return 0;
}
#endif	TTYCONNECT



int
alrmcatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	longjmp(alrm_jb, 1);
	return 0;
}



int
sigcatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	longjmp(to_jb, 1);
	return 0;
}



void
finish(error)
	int	error;
{
	if ( remote != NULLSTR && remote[0] != '\0' )
		chown(remote, ACSNETUID, ACSNETGID);	/* change back to daemon */
	if (done_stty)
#		if	SYSTEM < 3
		(void)stty(0, &orig_tty);
#		else	SYSTEM < 3
		(void)ioctl(0, TCSETA, &orig_tty);
#		endif	SYSTEM < 3
	exit(error);
}



/*
**	Validate that con is not being used with any
**	standard input or output redirection
**	- this includes asynchronous use.
**	The test checks that file descriptors 0, 1, and 2
**	represent the same tty.
*/

int
okuse()
{
	register int	i;
	struct stat	sbuf[3];

	for ( i = 0 ; i < 3 ; i++ )
		if
		(
			fstat(i, &sbuf[i]) == SYSERROR			/* stat failed ??? */
			||
			(sbuf[i].st_mode & S_IFMT) != S_IFCHR		/* not a tty */
			||
			i
			&&
			(
				sbuf[i].st_dev != sbuf[i-1].st_dev
				||
				sbuf[i].st_ino != sbuf[i-1].st_ino	/* some sort of redirection */
		        )
		)
			return 0;

	return 1;
}



/*
**	Execute connector program.
*/

void
execute(prog, link, target)
	char *	prog;
	char *	link;
	char *	target;
{
	VarArgs	va;

	FIRSTARG(&va) = prog;
	NEXTARG(&va) = newstr("-y");
	LASTARG(&va)[1] = quit;
	NEXTARG(&va) = concat("-l", link, NULLSTR);
	NEXTARG(&va) = target;
	NEXTARG(&va) = NULLSTR;

	(void)execve(ARG(&va, 0), &ARG(&va, 0), (char **)0);
	Syserror("Can't execve \"%s\"", ARG(&va, 0));
}
