/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)ReRoute.c	1.2 84/08/12
*/

/*
**	Search a directory for N-N command files that can be re-routed,
**	or removed.
*/

#define	FILE_CONTROL

#include	"global.h"

#include	"command.h"
#include	"debug.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"route.h"

#include	<ndir.h>


/*
**	Structure to remember file names to unlink
*/

typedef struct UnlinkN *	UN_p;
typedef struct UnlinkN
{
	UN_p	un_next;
	char *	un_name;
}
		UnlinkN;

static char *	Errstring;
static char *	HeaderFile;
static long	MesgLength;
static long	Time_to_die;
static UN_p	Unfreelist;
static UN_p	Unlist;

extern char *	HomeNode;
extern char *	Name;
extern Time_t	Time;

static void	bad_com();
bool		read_com();
static bool	read_header();



void
ReRoute(dir)
	char *			dir;
{
	DIR *			dirp;
	register struct direct *direp;
	register UN_p		unp;
	char *			linkname;

	Trace2(1, "ReRoute \"%s\"", dir);

	if
	(
		(RouteBase == NULLSTR && !ReadRoute())
		||
		LinkCount <= 1
	)
		return;	/* Why bother? */

	if ( (dirp = opendir(dir)) == NULL )
	{
		Syserror("Can't read \"%s\"", dir);
		return;
	}

	if ( (linkname = strrchr(dir, '/')) == NULLSTR )
		linkname = "";
	else
		linkname++;

	while ( (direp = readdir(dirp)) != NULL )
	{
		register char *	fp;
		register int	fd;
		Time_t		mtime;

		/*
		**	Find valid command file.
		*/

		switch ( direp->d_name[0] )
		{
		default:
			continue;
		case SMALL_ID:
		case MEDIUM_ID:
		case LARGE_ID:
			break;
		}

		Trace2(2, "found \"%s\"", direp->d_name);

		fp = concat(dir, "/", direp->d_name, NULLSTR);

		if ( (fd = open(fp, O_READ)) == SYSERROR )
		{
			Errstring = concat("can't read \"", fp, "\"", NULLSTR);
			bad_com(dir, fp, direp->d_name);
			free(fp);
			continue;
		}

		Time_to_die = 0;
		MesgLength = 0;
		if ( HeaderFile != NULLSTR )
			free(HeaderFile);
		HeaderFile = NULLSTR;

		if
		(
			!ReadCom(fd, &mtime, read_com)
			||
			HeaderFile == NULLSTR
			||
			!read_header(linkname, mtime, fp, direp->d_name)
		)
			bad_com(dir, fp, direp->d_name);

		(void)close(fd);

		while ( (unp = Unlist) != (UN_p)0 )
		{
			free(unp->un_name);
			Unlist = unp->un_next;
			unp->un_next = Unfreelist;
			Unfreelist = unp;
		}

		free(fp);
	}

	closedir(dirp);

	while ( (unp = Unfreelist) != (UN_p)0 )
	{
		Unfreelist = unp->un_next;
		free((char *)unp);
	}
}



/*
**	If message has timed-out, remove it, else
**	read HeaderFile, and if re-routable, move comfile.
*/

static bool
read_header(linkname, mtime, comfile, file)
	char *		linkname;
	Time_t		mtime;
	char *		comfile;
	char *		file;		/* Last component of "comfile" */
{
	register int	fd;
	register UN_p	unp;
	char *		fn;
	HdrReason	hr;
	NodeLink	link_descr;

	Trace3(1, "read_header \"%s\", header \"%s\"", comfile, HeaderFile);

	if ( Time_to_die > 0 && Time > (mtime + Time_to_die) )
	{
		Trace4(1, "message timed out: time=%lu, mtime=%lu, ttd=%ld", Time, mtime, Time_to_die);

		(void)unlink(comfile);

		for ( unp = Unlist ; unp != (UN_p)0 ; unp = unp->un_next )
			(void)unlink(unp->un_name);

		return true;
	}

	if ( (fd = open(HeaderFile, O_READ)) == SYSERROR )
	{
		Errstring = concat("can't read \"", HeaderFile, "\"", NULLSTR);
		return false;
	}

	if ( (hr = ReadHeader(fd)) != hr_ok )
	{
		(void)close(fd);
		Errstring = concat("header ", HeaderReason(hr), " error", NULLSTR);
		return false;
	}

	(void)close(fd);

	if ( !ReRoutable() )
		return true;

	if ( !FindAddress(HdrDest, &link_descr) )
	{
		Errstring = concat("destination \"", HdrDest, "\" ", link_descr.nl_name, NULLSTR);
		return false;
	}

	/*
	**	Don't re-route if already on fastest route,
	**	or looping and wait time has not been exceeded.
	*/

	if
	(
		strcmp(linkname, link_descr.nl_name) == STREQUAL
		||
		(
			(Time - mtime) < MAX_REROUTE_WAIT
			&&
			InRoute(link_descr.nl_name)
		)
	)
		return true;

	fn = UniqueName
	     (
		concat(SPOOLDIR(), link_descr.nl_name, "/", file, NULLSTR),
		MesgLength,
		mtime
	     );

	while ( link(comfile, fn) == SYSERROR )
		Syserror("Can't move \"%s\" to \"%s\"", comfile, fn);

	(void)unlink(comfile);

	Trace3(1, "re-route \"%s\" to \"%s\"", comfile, fn);

	free(fn);

	return true;
}



/*
**	Process bad commands file.
*/

static void
bad_com(dir, comfile, file)
	char *		dir;
	char *		comfile;
	char *		file;		/* Last component of "comfile" */
{
	register char *	badname;
	VarArgs		va;

	Trace2(1, "bad_com \"%s\"", file);

	if ( access(comfile, 4) == 0 )
	{
		badname = concat(BADDIR(), file, NULLSTR);

		while ( link(comfile, badname) == SYSERROR )
			Syserror("Can't move \"%s\" to \"%s\"", comfile, badname);

		FIRSTARG(&va) = BADHANDLER;
		NEXTARG(&va) = concat("-d", badname, NULLSTR);
		NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);
		NEXTARG(&va) = concat("-i", Name, NULLSTR);
		NEXTARG(&va) = concat("-eBad command file in ", dir, " - ", Errstring, NULLSTR);

		if ( (file = Execute(&va)) != NULLSTR )
		{
			Error(file);
			free(file);
		}

		free(ARG(&va, 4));
		free(ARG(&va, 3));
		free(ARG(&va, 2));
		free(ARG(&va, 1));

		free(badname);

		free(Errstring);
		Errstring = NULLSTR;
	}

	(void)unlink(comfile);
}



/*
**	Function called from "ReadCom" to process a command.
**	Remember last file name (containing header).
*/

bool
read_com(name, base, range)
	char *	name;
	long	base;
	long	range;
{
	Trace4(2, "command \"%s %lu %lu\"", name, base, range);

	if ( range != 0 )
	{
		MesgLength += range;
		if ( HeaderFile != NULLSTR )
			free(HeaderFile);
		HeaderFile = name;
	}
	else
	if ( base == 0 )
	{
		register UN_p	unp;

		if ( (unp = Unfreelist) == (UN_p)0 )
			unp = Talloc(UnlinkN);
		else
			Unfreelist = unp->un_next;
		
		unp->un_next = Unlist;
		Unlist = unp;
		unp->un_name = name;
	}
	else
	{
		Time_to_die = base;
		free(name);
	}

	return true;
}
