/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)stater.c	1.14 84/10/03";

/*
**	Process state messages from network.
*/

#define	FILE_CONTROL
#define	STAT_CALL
#define	RECOVER
#define	LOCKING
#define	STDIO

#include	"global.h"

#include	"address.h"
#include	"debug.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"sub_proto.h"

/*
**	Parameters set from arguments.
*/

bool	Broadcast;		/* Message has been everywhere */
char *	HomeNode;		/* Name of this node */
char *	LinkNode;		/* Message arrived from this node */
char *	Message;		/* Message */
char *	Name;			/* Program invoked name */
char *	SourceAddress;		/* Message originated at this node */
Time_t	StartTime;		/* Start time of message at source */
int	Traceflag;		/* Global tracing control */

/*
**	Miscellaneous
*/

NodeLink LinkD;			/* Info. from FindNode() */
bool	NoUpdate;		/* True if reply only */
int	Pid;			/* Used by UniqueName() in Exec...() */
char	ProtoType[2];		/* Header protocol type */
bool	Reply;			/* State info. required */
bool	ReplyAll;		/* Full state info. required */
int	ReplyFd;		/* File descr. for state info. */
char *	ReplyMesg	= WORKDIR(state..message);
Time_t	Time;			/* Used by UniqueName() in Exec...() */

#define	Fprintf		(void)fprintf
#define	free(A)

void	copydata(), finish(), reply();
bool	state();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	char *		cp;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'B':
					Broadcast = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'c':
					Error("Can't accept command files");
					goto break2;

				case 'd':
					DataLength = atol(++*argv);
					break;

				case 'e':
					HdrEnv = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 's':
					SourceAddress = ++*argv;
					goto break2;

				case 't':
					StartTime = Time - atol(++*argv);
					goto break2;

				default:
					Trace2(1, "unrecognised flag '%c'", c);
					goto break2;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		if ( Message != NULLSTR )
			Error("only one message allowed");
		else
			Message = *argv;
	}

	if ( strcmp(HomeNode, SourceAddress) == STREQUAL )
		return 0;	/* All done by "changestate" */

	if ( (cp = GetEnv(ENV_RETURNED)) != NULLSTR )
	{
		free(cp);

		Error
		(
			"state message returned from %s",
			SourceAddress
		);
	}

	if
	(
		(cp = GetEnv(ENV_ENQ)) != NULLSTR
		||
		(cp = GetEnv(ENV_ENQALL)) != NULLSTR && (ReplyAll = true)
	)
	{
		free(cp);

		Reply = true;

		while ( (ReplyFd = creat(UniqueName(ReplyMesg, (long)0, Time), 0600)) == SYSERROR )
			Syserror("Can't creat \"%s\"", ReplyMesg);
	}

#	if	LOG_STATE
	if ( SourceAddress != NULLSTR )
	{
		struct stat	statb;

		cp = concat(STATEDIR(), SourceAddress, NULLSTR);

		if
		(
			StartTime > 0
			&&
			stat(cp, &statb) != SYSERROR
			&&
			statb.st_mtime >= StartTime
		)
		{
			Report2("state message for \"%s\" out of date", SourceAddress);

			if ( Reply )
				NoUpdate = true;
			else
				return 0;
		}
		else
		{
			(void)unlink(cp);
			(void)link(Message, cp);

			if ( StartTime > 0 )
				SMdate(cp, StartTime);
		}

		free(cp);
	}
#	endif	LOG_STATE

	if ( state() )
	{
		Recover(ert_return);
		ReadRoute();		/* Re-load changed routing tables */
		ReRoute(REROUTEDIR(.));
		Recover(ert_finish);
	}

	if ( Reply )
		reply();

	return 0;
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int	error;
{
	register int i;

	if ( Reply )
		(void)unlink(ReplyMesg);

	for ( i = 0 ; i < 20 ; i++ )
		UnLock(i);

	(void)exit(error);
}



/*
**	State change processing:
**		if up/down, run "state";
**		if statefile, pass to "state".
**
**	If there is no state change return false.
*/

bool
state()
{
	register char *		cp;
	register char *		orgnode;
	register FILE *		fd;
	char *			errs;
	bool			statein = false;
	char			downflag;
	char			stim[TIME_SIZE+1];
	VarArgs			va;

	if ( NoUpdate )
		orgnode = NULLSTR;
	else
	{
		if ( (orgnode = GetEnv(ENV_UP)) != NULLSTR )
			downflag = '-';
		else
		if ( (orgnode = GetEnv(ENV_DOWN)) != NULLSTR )
			downflag = '+';

		if ( (cp = GetEnv(ENV_STATE)) != NULLSTR )
		{
			free(cp);
			statein = true;
		}
	}

	/*
	**	Process the information into state and routing files.
	**	Also, while we are at it, update the statistics,
	**	and read the default command file.
	*/

	FIRSTARG(&va) = STATE;
	if ( !NoUpdate )
		if ( statein && DataLength > 0 )
			NEXTARG(&va) = "-IRSXc";
		else
			NEXTARG(&va) = "-RSXc";
	if ( Reply )
		NEXTARG(&va) = ReplyAll?"-Q":"-O";
	if ( orgnode != NULLSTR )
		NEXTARG(&va) = "-C";
	(void)sprintf(stim, "%lu", StartTime);
	NEXTARG(&va) = concat("-d", stim, NULLSTR);
	NEXTARG(&va) = concat("-f", SourceAddress, NULLSTR);
	NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);

#	ifdef	DEBUG
	if ( Traceflag )
		NEXTARG(&va) = Traceflag==1?"-T1":Traceflag==2?"-T2":"-T3";
#	endif	DEBUG

	fd = Reply ? ExPipeO(&va, ReplyFd) : ExecPipe(&va);

	if ( statein && DataLength > 0 )
		copydata(fileno(fd));

	if ( orgnode != NULLSTR )
	{
		Address *	address = SplitAddress(SourceAddress);

		Fprintf
		(
			fd,
			"add %s\nadd %s\nlink %s,%s %cdown,msg\n",
			address->ad_node,
			orgnode,
			address->ad_node,
			orgnode,
			downflag
		);

		FreeAddress(address);

		free(orgnode);
	}

	if ( (errs = Reply ? ExOPipeClose(fd) : ExPipeClose(fd)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}

	return (bool)(orgnode != NULLSTR);
}



/*
**	Copy data part of message to "ofd".
*/

void
copydata(ofd)
	int		ofd;
{
	register long	l;
	register int	i;
	register int	s;
	register char *	cp;
	register int	ifd;
	char		buf[BUFSIZ];

	while ( (ifd = open(Message, O_READ)) == SYSERROR )
		Syserror("Can't open \"%s\"", Message);
	
	for ( l = DataLength ; l > 0 ; )
	{
		if ( l > sizeof buf )
			s = sizeof buf;
		else
			s = l;
		
		if ( (i = read(ifd, buf, s)) > 0 )
		{
			l -= i;
			cp = buf;

			Trace3(2, "copydata \"%.*s\"", i, cp);

			while ( (s = write(ofd, cp, i)) != i )
			{
				if ( s == SYSERROR )
					Syserror("Can't write pipe");
				else
				{
					cp += s;
					i -= s;
				}
			}
		}
		else
			Syserror("Can't read \"%s\"", Message);
	}

	(void)close(ifd);
}



/*
**	Reply to a request for local statefile.
*/

void
reply()
{
	char *		errs;
	VarArgs		va;

	DataLength = lseek(ReplyFd, (long)0, 2);

	HdrEnv = MakeEnv(ENV_STATE, NULLSTR, ENV_NORET, NULLSTR, NULLSTR);

	HdrDest = SourceAddress;
	HdrSource = HomeNode;
	HdrHandler = Name;	/* Reply to oneself */

	ProtoType[0] = STATE_PROTO;
	HdrSubpt = ProtoType;

	while ( WriteHeader(ReplyFd, DataLength, 0) == SYSERROR )
		Syserror("Can't write \"%s\"", ReplyMesg);
	
	(void)close(ReplyFd);

	FIRSTARG(&va) = RECEIVER;
	NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);
#	ifdef	DEBUG
	if ( Traceflag )
		NEXTARG(&va) = Traceflag==1?"-T1":Traceflag==2?"-T2":"-T3";
#	endif	DEBUG
	NEXTARG(&va) = ReplyMesg;

	if ( (errs = Execute(&va)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}
}
