/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)getfile.c	1.20 85/08/20";

/*
**	Get a user's files spooled from network.
**
**	SETUID ==> ROOT
*/

char *	Usage	="\"%s [-[A][C][D][E][L][M][N][O][R][S][V][Y]] \\\n\
	[-delete] [-o<days>] [-u<user>] [filename ...]\"\n";

#define	FILE_CONTROL
#define	STAT_CALL
#define	RECOVER
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"debug.h"
#include	"ftheader.h"
#include	"header.h"
#include	"spool.h"

#include	<ndir.h>
#include	<signal.h>

/*
**	Default sort order is by modify time.
*/

typedef enum
{
	 sb_modify = 0, sb_delivery, sb_name, sb_to
}
	Sort_t;

/*
**	File names list.
*/

typedef struct FN_list *FN_p;

typedef struct FN_list
{
	FN_p	fn_next;
	char *	fn_name;
}
	FN_el;

/*
**	Parameters set from arguments.
*/

bool	AllUsers;		/* Select all messages */
bool	Delete;			/* Delete the lot */
FN_p	FileNames;		/* Particular files to be retrieved */
bool	CheckCRC;		/* Check CRC of data (if present) */
bool	List;			/* List all files found and exit */
char *	Name;			/* Program invoked name */
bool	NoSMdate;		/* Don't set modify date of retrieved files */
Time_t	Older;			/* Only list details of messages older than this */
bool	Reverse;		/* Reverse order of sort */
bool	Silent;			/* Don't prompt for user response, assume "yes" */
Sort_t	Sortflag;		/* Sort order */
bool	Stdout;			/* Copy named file to stdout */
int	Traceflag;		/* Global tracing control */
char *	User;			/* User's name for files to be retrieved (root only) */
bool	Verbose;		/* Lots of verbiage */
bool	Yes;			/* Assume answer "yes" to all queries */

/*
**	Message list
*/

typedef struct M_List *ML_p;

typedef struct M_List
{
	ML_p	m_next;
	Time_t	m_time;		/* Delivery time for message */
	long	m_length;	/* Length of data in message */
	char *	m_name;		/* File name of message */
	FthFD_p	m_files;	/* List of files in message */
	char *	m_from;		/* Sender */
	char *	m_source;	/* Source node */
	char *	m_to;		/* Recipients */
	char *	m_route;	/* Route */
	char *	m_error;	/* Possible RETURN reason */
	char *	m_deleted;	/* Boolean array for deleted files */
	int	m_nfiles;	/* Number of files in list */
}
	MesList;

ML_p	MessageList;		/* List of messages found for user */
int	NMessages;		/* Number of messages found */

#define	DELETE(mp,i)	mp->m_deleted[i/8] |= (1<<(i%8))
#define	DELETED(mp,i)	mp->m_deleted[i/8] & (1<<(i%8))

typedef struct
{
	ML_p	fl_mp;		/* Pointer to message containing file */
	FthFD_p	fl_fp;		/* Pointer to file descriptor */
	int	fl_in;		/* Index of file in message */
}
	*FL_p;

FL_p	FilesList;		/* Sorted array of file descriptors */
int	NFiles;			/* Total number of files found */

/*
**	File getting commands:
*/

typedef enum
{
	r_yes, r_no, r_delete, r_rename, r_basename
}
	Cmdtyp;

typedef struct
{
	char *	c_name;
	Cmdtyp	c_type;
}
	Command;

typedef Command * Cmd_p;

Command	Cmds[] =
{
	{"basename",	r_basename},
	{"delete",	r_delete},
	{"no",		r_no},
	{"rename",	r_rename},
	{"yes",		r_yes},
};

#define	CMDZ	(sizeof(Command))
#define	NCMDS	((sizeof Cmds)/CMDZ)

#define	MAXCOMLEN	7
#define	MINCOMLEN	1

/*
**	Miscellaneous
*/

char	Buf[BUFSIZ];		/* Standard buffer for i/o */
int	ComLen;			/* Length of user's command */
int	FilesMoved;		/* Count of files delivered to user */
char *	HomeNode;		/* Needed by header library */
bool	Isatty;			/* Fd 0 is a terminal */
int	Pid;			/* Needed somewhere */
bool	Returned;		/* True if message has been returned */
char	SoBuf[BUFSIZ];
Time_t	Time;			/* Timeof invokation */
Passwd	To;			/* Details of invoker */
ulong	Tt;			/* Travel time for message */
char	UidStr[UID_LENGTH+1];	/* Ascii value of uid */

#define	MAXNAMELEN	132
#define	Fprintf		(void)fprintf
#define	Fflush		(void)fflush
#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))
#define	DAY		(24L*60L*60L)

bool	checkperm(), findfiles(), printroute();
char *	getname();
void	copyfile(), deletefile(), finish(), getfilename(), interact(),
	movefiles(), readDeleted(), removemessages(), sortfiles(), writeDeleted();
int	compare(), delivery(), modify(), name(), readword(), sigcatch(), to();
Cmdtyp	response();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	register int	uid = getuid();

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Time = time((long *)0);

	Recover(ert_return);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					AllUsers = true;
					Sortflag = sb_to;
					continue;

				case 'C':
					CheckCRC = true;
					continue;

				case 'D':
					NoSMdate = true;
					continue;

				case 'E':
					Sortflag = sb_delivery;
					continue;

				case 'L':
					List = true;
					continue;

				case 'M':
					Sortflag = sb_modify;
					continue;

				case 'N':
					Sortflag = sb_name;
					continue;

				case 'O':
					Stdout = true;
					continue;

				case 'R':
					Reverse = true;
					continue;

				case 'S':
					Silent = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					continue;

				case 'Y':
					Yes = true;
					continue;

				case 'd':
					if ( strcmp(*argv, "delete") == STREQUAL )
					{
						Delete = true;
						goto break2;
					}
					Error("unrecognised argument \"%s\"", *argv);
					Mesg("Usage", Usage, Name);
					return 1;

				case 'o':
					Older = Time - (Time_t)(atol(++*argv) * DAY);
					break;

				case 'u':
					User = ++*argv;
					if ( uid )
					{
						Error("No permission for '%c' flag", c);
						return 2;
					}
					goto break2;

				default:
					Error("unrecognised flag '%c'", c);
					Mesg("Usage", Usage, Name);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			FN_p	fnp = Talloc(FN_el);

			fnp->fn_next = FileNames;
			FileNames = fnp;
			fnp->fn_name = *argv;
			Sortflag = sb_name;
		}
	}

	if ( Stdout && FileNames == (FN_p)0 )
	{
		Error("need filename to copy to stdout");
		return 1;
	}

	if ( Silent && !Yes && !Delete && !Stdout )
	{
		Error("Do what silently?");
		Mesg("Usage", Usage, Name);
		return 1;
	}

	if
	(
		Stdout && Yes
		||
		List && (Delete || Silent || Yes)
		||
		AllUsers && User != NULLSTR
	)
	{
		Error("Ambiguous arguments");
		return 1;
	}

	if ( User==NULLSTR ? !GetUser(&To, uid) : !GetUid(&To, User) )
	{
		Error("Passwd error for user \"%s\": %s", User==NULLSTR?"Phantom":User, To.P_error);
		return 1;
	}

	if ( AllUsers && (!(To.P_flags&P_SU) || uid && !List) )
	{
		Error("No permission for 'A' flag");
		return 1;
	}

	Recover(ert_finish);

	(void)UniqueUid(UidStr, To.P_uid);

	Trace2(1, "User id \"%s\"", UidStr);

	HomeNode = NodeName();

	if ( !findfiles() )
	{
		if ( Silent )
			return 0;

		Fprintf(stderr, "No files found.\n");
		return 1;
	}

	sortfiles();

	if ( !Silent && !Stdout )
	{
		setbuf(stdout, SoBuf);
		Fprintf(stdout, "%d file%s found.\n", NFiles, NFiles==1?"":"s");

		(void)signal(SIGINT, sigcatch);
		Isatty = (bool)isatty(0);

		interact();
	}
	else
		movefiles();

	finish(0);

	return 0;
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int	error;
{
	if ( FilesMoved )
	{
		FilesMoved = 0;	/* Avoid recursion! */
		removemessages();
	}

	(void)exit(error);
}



/*
**	Catch signals to clean up
*/

int
sigcatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	putc('\n', stdout);
	Fflush(stdout);
	finish(sig);
}



/*
**	Find messages spooled for this user and extract list of filenames.
*/

bool
findfiles()
{
	register char *			np;
	register ML_p			mp;
	register int			fd;
	register struct direct *	direp;
	register DIR *			dirp;
	struct stat			statb;

	if ( (dirp = opendir(FILESDIR())) == NULL )
	{
		Syserror("Can't read \"%s\"", FILESDIR());
		finish(1);
		return false;
	}

	Recover(ert_return);

	while ( (direp = readdir(dirp)) != NULL )
	{
		Trace2(2, "Found spooled message \"%s\"", direp->d_name);

		if
		(
			(AllUsers && !dots(direp->d_name) && direp->d_name[0] < ('+'+'@'))
			||
			strncmp(UidStr, direp->d_name, UID_LENGTH) == STREQUAL
		)
		{
			np = concat(FILESDIR(), direp->d_name, NULLSTR);

			if ( (fd = readmessage(np)) == SYSERROR )
			{
				free(np);
				continue;
			}

			(void)fstat(fd, &statb);
			(void)close(fd);

			Trace3(1, "Found %d files in message to \"%s\"", NFthFiles, FthTo);

			if ( Older != (Time_t)0 && statb.st_mtime >= Older )
				continue;

			if ( !AllUsers && !Returned && !InFthTo(To.P_name) )
			{
				FreeFthFiles();
				free(np);
				continue;
			}

			mp = Talloc(MesList);
			mp->m_next = MessageList;
			MessageList = mp;
			NMessages++;

			mp->m_source = newstr(HdrSource);
			mp->m_nfiles = NFthFiles;
			mp->m_time = statb.st_mtime;
			mp->m_length = FtDataLength;
			mp->m_name = np;
			mp->m_files = FthFiles;
			if ( Returned )
			{
				mp->m_to = newstr(FthFrom);
				mp->m_from = newstr(FthTo);
				mp->m_error = GetEnv(ENV_ERR1);
				mp->m_route = GetEnv(ENV_ROUTE);
			}
			else
			{
				mp->m_to = newstr(FthTo);
				mp->m_from = newstr(FthFrom);
				mp->m_error = NULLSTR;
				mp->m_route = newstr(HdrRoute);
			}

			FthFiles = (FthFD_p)0;
			NFiles += NFthFiles;
			NFthFiles = 0;

			readDeleted(mp);
		}
	}

	closedir(dirp);

	Recover(ert_finish);

	return NMessages > 0 ? true : false;
}


/*
**	Read a spooled message.
*/

int
readmessage(name)
	char *		name;
{
	register int	fd;
	FthReason	fthr;
	HdrReason	hr;
	char *		cp;

	Trace2(1, "Read message \"%s\"", name);

	if ( (fd = open(name, O_READ)) == SYSERROR )
	{
		Syserror("Can't read \"%s\"", name);
		return SYSERROR;
	}

	if ( (hr = ReadHeader(fd)) != hr_ok )
	{
		Error("Message \"%s\" header \"%s\" error", name, HeaderReason(hr));
		(void)close(fd);
		return SYSERROR;
	}

	if ( (cp = GetEnv(ENV_RETURNED)) != NULLSTR )
	{
		free(cp);
		Returned = true;
	}
	else
		Returned = false;

	if 
	(
		(fthr = ReadFtHeader(fd, DataLength, !Returned && CheckCRC)) != fth_ok
		||
		(fthr = GetFthFiles()) != fth_ok
	)
	{
		Error("Message \"%s\" file transfer header \"%s\" error", name, FTHREASON(fthr));
		(void)close(fd);
		return SYSERROR;
	}

	return fd;
}



/*
**	Sort all files found according to sorting preference
*/

void
sortfiles()
{
	register int		i;
	register FthFD_p	fp;
	register FL_p		flp;
	register ML_p		mp;
	int			(*cfuncp)();

	FilesList = flp = (FL_p)Malloc(sizeof(*flp) * NFiles);

	for ( mp = MessageList ; mp != (ML_p)0 ; mp = mp->m_next )
		for ( fp = mp->m_files, i = 0 ; fp != (FthFD_p)0 ; fp = fp->f_next, i++ )
		{
			if ( DELETED(mp, i) )
				continue;

			flp->fl_in = i;
			flp->fl_fp = fp;
			flp->fl_mp = mp;
			flp++;

			DODEBUG(if((flp-FilesList)>NFiles)Fatal1("bad NFiles"));
		}

	switch ( Sortflag )
	{
	case sb_delivery:	cfuncp = delivery;	break;
	case sb_modify:		cfuncp = modify;	break;
	case sb_name:		cfuncp = name;		break;
	case sb_to:		cfuncp = to;		break;
	}

	qsort((char *)FilesList, NFiles, sizeof(*flp), cfuncp);
}


/*
**	Comparison routines for qsort().
*/

int
delivery(flp1, flp2)
	char *		flp1;
	char *		flp2;
{
	register FL_p	fp1;
	register FL_p	fp2;

	if ( Reverse )
	{
		fp1 = (FL_p)flp2;
		fp2 = (FL_p)flp1;
	}
	else
	{
		fp1 = (FL_p)flp1;
		fp2 = (FL_p)flp2;
	}

	if ( fp1->fl_mp->m_time > fp2->fl_mp->m_time )
		return 1;

	if ( fp1->fl_mp->m_time < fp2->fl_mp->m_time )
		return -1;

	return 0;
}

int
modify(flp1, flp2)
	char *		flp1;
	char *		flp2;
{
	register FL_p	fp1;
	register FL_p	fp2;

	if ( Reverse )
	{
		fp1 = (FL_p)flp2;
		fp2 = (FL_p)flp1;
	}
	else
	{
		fp1 = (FL_p)flp1;
		fp2 = (FL_p)flp2;
	}

	if ( fp1->fl_fp->f_time > fp2->fl_fp->f_time )
		return 1;

	if ( fp1->fl_fp->f_time < fp2->fl_fp->f_time )
		return -1;

	return name(flp1, flp2);
}

int
name(flp1, flp2)
	char *		flp1;
	char *		flp2;
{
	register FL_p	fp1;
	register FL_p	fp2;
	register int	val;

	if ( Reverse )
	{
		fp1 = (FL_p)flp2;
		fp2 = (FL_p)flp1;
	}
	else
	{
		fp1 = (FL_p)flp1;
		fp2 = (FL_p)flp2;
	}

	if ( (val = strcmp(fp1->fl_fp->f_name, fp2->fl_fp->f_name)) == 0 )
		return delivery(flp1, flp2);
	else
		return val;
}

int
to(flp1, flp2)
	char *		flp1;
	char *		flp2;
{
	register FL_p	fp1;
	register FL_p	fp2;
	register int	val;

	if ( Reverse )
	{
		fp1 = (FL_p)flp2;
		fp2 = (FL_p)flp1;
	}
	else
	{
		fp1 = (FL_p)flp1;
		fp2 = (FL_p)flp2;
	}


	if ( (val = strcmp(fp1->fl_mp->m_to, fp2->fl_mp->m_to)) == 0 )
		return delivery(flp1, flp2);
	else
		return val;
}



/*
**	Interact with user to retrieve files.
*/

void
interact()
{
	register FthFD_p	fp;
	register ML_p		mp;
	register FL_p		flp;
	register int		i;
	char *			hdrfmt = "%-14s %-9s %-14s %10s  %-12s";
	char *			format = "%-14s %-9s %-14s %10ld  %-12.12s";
	char *			dvrfmt = "  %-12s\n";
	char *			dvrstr = "  %.12s";

	Fprintf(stdout, hdrfmt, "File", "From", "At", "Size", AllUsers?"To":"   Mtime");

	if ( List )
		Fprintf(stdout, dvrfmt, "  Delivered");
	else
		putc('\n', stdout);

	for ( flp = FilesList, i = NFiles ; --i >= 0 ; flp++ )
	{
		bool	nowrite;
		bool	exists;
		Time_t	mtime;

		fp = flp->fl_fp;
		mp = flp->fl_mp;

		if ( FileNames != (FN_p)0 )
		{
			register FN_p	fnp;

			for ( fnp = FileNames ; fnp != (FN_p)0 ; fnp = fnp->fn_next )
				if ( strcmp(fp->f_name, fnp->fn_name) == STREQUAL )
					break;

			if ( fnp == (FN_p)0 )
				continue;
		}

again:
		Fprintf
		(
			stdout,
			format,
			fp->f_name,
			mp->m_from,
			(mp->m_error!=NULLSTR)?"RETURNED":mp->m_source,
			fp->f_length,
			AllUsers?mp->m_to:ctime(&fp->f_time)+4
		);

		if ( List )
			Fprintf(stdout, dvrstr, ctime(&mp->m_time)+4);

		if ( Verbose || mp->m_error != NULLSTR )
		{
			Fprintf(stdout, "\n Sent to \"%s\"", mp->m_to);
			Tt = 0;
			ExRoute(mp->m_route, printroute);
			if ( (long)Tt > 0 )
				Fprintf(stdout, "\" in %lu secs.", Tt);
			if ( mp->m_error != NULLSTR )
				Fprintf(stdout, "\n Failure reason follows:-\n%s\n", mp->m_error);
		}

		if ( List )
		{
			putc('\n', stdout);
			continue;
		}

		if ( !checkperm(fp->f_name, &mtime, &exists) )
			nowrite = true;
		else
			nowrite = false;

		if ( exists || nowrite )
			putc('\n', stdout);

		if ( exists )
			Fprintf
			(
				stdout,
				" (%salready exists)",
				mtime > fp->f_time
					? "later version "
					: ""
			);

		if ( nowrite )
			Fprintf(stdout, " (no write permission)");

		Fprintf(stdout, " ? ");

		switch ( response(nowrite, fp->f_name, &mtime, &exists) )
		{
		case r_no:
			continue;

		case r_rename:
			fp->f_name = getname();
			goto again;

		case r_basename:
			{
				char *	cp;

				if ( (cp = strrchr(fp->f_name, '/')) != NULLSTR )
					fp->f_name = cp + 1;
				goto again;
			}

		case r_yes:
			copyfile(flp, exists);
			break;

		case r_delete:
			;
		}

		deletefile(flp);
	}

	if ( List )
		Fflush(stdout);
}



/*
**	Print out each node in route, and accumulate travel-time.
*/

bool
printroute(time, from, to)
	ulong	time;
	char *	from;
	char *	to;
{
	if ( Tt == (ulong)0 )
		Fprintf(stdout, " via \"");
	else
		putc(',', stdout);
	Tt += time;
	Fprintf(stdout, to);
	return true;
}



/*
**	Check write permissions on file or in parent directory.
**	Return true if write allowed.
*/

bool
checkperm(file, mtime, exists)
	char *		file;
	Time_t *	mtime;
	bool *		exists;
{
	register char *	cp;
	register char *	ep;
	bool		write_ok;
	struct stat	statb;

	if ( stat(file, &statb) != SYSERROR )
	{
		*exists = true;
		*mtime = statb.st_mtime;

		if ( access(file, 2) == 0 )
			return true;

		return false;
	}

	*exists = false;

	if ( (ep = strrchr(file, '/')) != NULLSTR )
	{
		cp = file;
		*ep = '\0';
	}
	else
		cp = ".";

	if ( access(cp, 2) == 0 )
		write_ok = true;
	else
		write_ok = false;

	if ( ep != NULLSTR )
		*ep = '/';
	
	return write_ok;
}



/*
**	Read a response from user
*/

Cmdtyp
response(not_yes, file, mtime, exists)
	bool		not_yes;
	char *		file;
	Time_t *	mtime;
	bool *		exists;
{
	register Cmd_p	cmdp;
	Command		cmd;
	char		com[MAXCOMLEN+60];

	cmd.c_name = com;

	for ( ;; )
	{
		if
		(
			(ComLen = readword(com, sizeof com, true)) >= MINCOMLEN
			&&
			ComLen <= MAXCOMLEN
			&&
			(cmdp = (Cmd_p)bsearch((char *)&cmd, (char *)Cmds, NCMDS, CMDZ, compare)) != (Cmd_p)0
		)
		{
			if ( !not_yes || cmdp->c_type != r_yes || checkperm(file, mtime, exists) )
				return cmdp->c_type;
		}

		if ( Yes || !Isatty )
			Error("illegal response \"%s\", use manual interaction", com);

		Fprintf(stdout, "(Respond: ");

		for ( cmdp = Cmds ; cmdp < &Cmds[NCMDS] ; cmdp++ )
		{
			if ( not_yes && cmdp->c_type == r_yes )
				continue;

			Fprintf(stdout, "%s\"%s\"", cmdp==Cmds?"":", ", cmdp->c_name);
		}

		Fprintf(stdout, ".) ? ");
	}
}


/*
**	Compare two commands
*/

int
compare(cmdp1, cmdp2)
	char *	cmdp1;
	char *	cmdp2;
{
	return strncmp(((Cmd_p)cmdp1)->c_name, ((Cmd_p)cmdp2)->c_name, ComLen);
}



/*
**	Read a line from user
*/

int
readword(word, size, command)
	char *		word;
	int		size;
	bool		command;
{
	register char *	cp;
	register int	n;
	register int	i;

	if ( !Yes && !Delete )
	{
		Fflush(stdout);

		n = --size;
		cp = word;

		for ( ;; )
		{
			if ( (i = read(0, cp, Isatty?n:1)) <= 0 )
				finish(1);

			if ( cp[i-1] == '\n' )
			{
				cp[--i] = '\0';
				break;
			}

			if ( cp < &word[size] )
			{
				if ( (n -= i) == 0 )
					n = 1;
				cp += i;
			}
		}

		while ( word[0] == ' ' || word[0] == '\t' )
		{
			(void)strcpy(word, &word[1]);
			i--;
		}

		if ( (n = cp - word + i) == 0 && command )
		{
			(void)strcpy(word, "no");
			n = strlen(word);
		}
	}
	else
	if ( command )
	{
		(void)strcpy(word, Yes?"yes":"delete");
		n = strlen(word);
	}
	else
	{
		word[0] = '\0';
		n = 0;
	}

	if ( Yes || Delete || !Isatty )
	{
		Fprintf(stdout, "%s\n", word);
		Fflush(stdout);
	}

	return n;
}



/*
**	Get a file name
*/

char *
getname()
{
	char	name[MAXNAMELEN+1];

	for ( ;; )
	{
		Fprintf(stdout, "New name ? ");

		if ( readword(name, sizeof name, false) > 0 )
			return newstr(name);
	}
}


/*
**	Copy file from message to file-system
*/

void
copyfile(flp, exists)
	FL_p			flp;
	bool			exists;
{
	register char *		cp;
	register int		n;
	register int		r;
	register long		l;
	register int		ofd;
	register FthFD_p	fp;
	register int		fd;

	for
	(
		l = 0, fp = flp->fl_mp->m_files ;
		fp != flp->fl_fp && fp != (FthFD_p)0 ;
		fp = fp->f_next
	)
		l += fp->f_length;

	DODEBUG(if(fp!=flp->fl_fp)Fatal1("bad copyfile"));

	Trace4(1, "copyfile \"%s\" size %ld mode 0%o", fp->f_name, fp->f_length, fp->f_mode);

	while ( (fd = open(flp->fl_mp->m_name, O_READ)) == SYSERROR )
		Syserror("Can't open \"%s\"", flp->fl_mp->m_name);

	while ( lseek(fd, l, 0) == SYSERROR )
		Syserror("Can't seek \"%s\"", flp->fl_mp->m_name);

	if ( !Stdout )
	{
		while ( (ofd = creat(fp->f_name, (fp->f_mode)&FTH_MODES)) == SYSERROR )
			Syserror("Can't creat \"%s\"", fp->f_name);
		
		if ( !exists )
			(void)chown(fp->f_name, To.P_uid, To.P_gid);
	}
	else
		ofd = 1;

	for
	(
		l = fp->f_length ;
		(r = l > sizeof Buf ? sizeof Buf : l) > 0
		&&
		(n = read(fd, Buf, r)) > 0 ;
	)
	{
		l -= n;
		cp = Buf;

		while ( (r = write(ofd, cp, n)) != n )
		{
			if ( r == SYSERROR )
			{
				Syserror("Can't write \"%s\"", fp->f_name);

				if ( !Stdout )
					(void)unlink(fp->f_name);
				
				finish(1);
			}
			else
			{
				cp += r;
				n -= r;
			}
		}
	}

	(void)close(fd);

	if ( !Stdout )
	{
		(void)close(ofd);

		if ( !NoSMdate && fp->f_time < Time )
			SMdate(fp->f_name, fp->f_time);
	}
}


/*
**	Delete any messages that have been emptied.
*/

void
removemessages()
{
	register ML_p	mp;

	for ( mp = MessageList ; mp != (ML_p)0 ; mp = mp->m_next )
		writeDeleted(mp);
}


/*
**	Move all/requested files.
*/

void
movefiles()
{
	register FL_p	flp;
	register FN_p	fnp;
	register int	i;
	Time_t		mtime;
	bool		exists;

	for ( flp = FilesList, i = NFiles ; --i >= 0 ; flp++ )
	{
		if ( (fnp = FileNames) != (FN_p)0 )
		{
			for ( ; fnp != (FN_p)0 ; fnp = fnp->fn_next )
				if ( strcmp(flp->fl_fp->f_name, fnp->fn_name) == STREQUAL )
				{
					if
					(
						i > 0
						&&
						strcmp(flp[1].fl_fp->f_name, fnp->fn_name) == STREQUAL
					)
						Error("more than 1 file called \"%s\"", fnp->fn_name);

					fnp->fn_name[0] = '\0';
					break;
				}

			if ( fnp == (FN_p)0 )
				continue;
		}

		if ( Stdout || !Delete )
		{
			if ( Stdout || checkperm(flp->fl_fp->f_name, &mtime, &exists) )
				copyfile(flp, exists);
			else
				Error("\"%s\": no permission", flp->fl_fp->f_name);
		}

		if ( !Stdout || Delete )
			deletefile(flp);
	}
}


/*
**	Read in deleted files boolean array (if it exists)
*/

void
readDeleted(mp)
	register ML_p	mp;
{
	register int	n;
	register int	fd;
	register char *	del_file = newstr(mp->m_name);
	register char *	cp = strrchr(del_file, '/');

	if ( cp == NULLSTR )
		cp = del_file;
	else
		cp++;

	/*
	**	In order to generate a unique name related to the message,
	**	we assume first character is part of uid number,
	**	and that it will always be less than '@'.
	**	This will be true (see "UniqueUid") for uids < 48,000.
	*/

	*cp += '@';

	n = (mp->m_nfiles+7)/8;

	mp->m_deleted = strclr(Malloc(n), n);

	if ( (fd = open(del_file, O_READ)) == SYSERROR )
	{
		free(del_file);
		return;
	}

	if ( read(fd, mp->m_deleted, n) == n )
		for ( n = mp->m_nfiles ; --n >= 0 ; )
			if ( DELETED(mp, n) )
				NFiles--;
	
	(void)close(fd);
	free(del_file);
}


/*
**	Mark a file deleted from message.
*/

void
deletefile(flp)
	FL_p	flp;
{
	DELETE(flp->fl_mp, flp->fl_in);
	FilesMoved++;
}


/*
**	If files have been deleted from message,
**	if there are undeleted files left, write deleted array,
**	otherwise delete message.
*/

void
writeDeleted(mp)
	register ML_p	mp;
{
	register int	i;
	register char *	del_file = newstr(mp->m_name);
	register char *	cp = strrchr(del_file, '/');
	register int	fd;
	bool		deleted = false;
	bool		undeleted = false;

	if ( cp == NULLSTR )
		cp = del_file;
	else
		cp++;
	
	*cp += '@';

	for ( i = mp->m_nfiles ; --i >= 0 ; )
		if ( DELETED(mp, i) )
			deleted = true;
		else
			undeleted = true;
	
	if ( !undeleted )
	{
		(void)unlink(mp->m_name);
		(void)unlink(del_file);
		free(del_file);
		return;
	}

	if ( !deleted )
	{
		free(del_file);
		return;
	}
	
	if ( (fd = creat(del_file, 0600)) == SYSERROR )
	{
		Syserror("Can't creat \"%s\"", del_file);
		free(del_file);
		return;
	}

	i = (mp->m_nfiles+7)/8;

	while ( write(fd, mp->m_deleted, i) != i )
	{
		Syserror("Can't write \"%s\"", del_file);
		(void)lseek(fd, (long)0, 0);
	}

	(void)close(fd);
	free(del_file);
}
