/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Domains.c	1.25 85/08/16
*/

/*
**	Domain handling routines.
*/

#include	"global.h"
#include	"debug.h"
#include	"state.h"

#include	"node.h"



/*
**	Add new domain to end of list, return Entry.
*/

Entry *
AddDomain(cp, hp, states)
	register char *		cp;
	register DomHead *	hp;
	States			states;
{
	register Domain *	dp;
	register Entry *	ep;

	ep = Enter(cp, DomainHash);

	for ( dp = hp->d_head ; dp != (Domain *)0 ; dp = dp->d_next )
		if ( dp->d_entry == ep )
			return ep;	/* A duplicate! */

	dp = Talloc(Domain);

	dp->d_entry = ep;
	dp->d_states = states;

	if ( (states & S_FOUND) && !(ep->e_states & S_FOUND) )
		DomainCount++;

	ep->e_states |= states;

	dp->d_previous = hp->d_last;
	hp->d_last = dp;

	*hp->d_tail = dp;
	hp->d_tail = &dp->d_next;
	hp->d_count++;

	return ep;
}



/*
**	Check that node is a member of first domain mentioned in hierarchy,
**	that the primary domain is also in the hierarchy,
**	and that there are at least two domains in the hierarchy.
*/

void
CheckHierarchy(ep, hp)
	Entry *			ep;
	register DomHead *	hp;
{
	register Domain *	dp;
	register Domain *	hdp;

	if ( hp->d_count < 2 )
	{
		if ( hp->d_count > 0 )
			Error
			(
				"Less than 2 domains in hierarchy for node \"%s\"",
				ep->e_name
			);

		return;
	}

	if ( (hdp = ep->e_node->n_domains.d_head) == (Domain *)0 )
	{
		Error
		(
			"Node \"%s\" not a member of any domain in hierarchy",
			ep->e_name
		);

		return;
	}

	/*
	**	Check primary domain is in hierarchy
	*/

	for ( dp = hp->d_head ; dp != (Domain *)0 ; dp = dp->d_next )
		if ( dp->d_entry == hdp->d_entry )
			break;

	if ( dp == (Domain *)0 )
		Error
		(
			"Primary domain \"%s\" not in hierarchy for node \"%s\"",
			hdp->d_entry->e_name,
			ep->e_name
		);

	/*
	**	Check membership of first domain in hierarchy
	*/

	do
		if ( hp->d_head->d_entry == hdp->d_entry )
			break;
	while
		( (hdp = hdp->d_next) != (Domain *)0 );

	if ( hdp == (Domain *)0 )
		Error
		(
			"node \"%s\" not a member of first domain \"%s\" in hierarchy",
			ep->e_name,
			hp->d_head->d_entry->e_name
		);

	/*
	**	Mark top level domain
	*/

	for ( dp = hp->d_head ; dp->d_next != (Domain *)0 ; dp = dp->d_next )
		dp->d_entry->e_states &= ~S_TOPDOM;

	hp->d_last->d_entry->e_states |= S_TOPDOM;
}



/*
**	Check source domains.
*/

void
CheckSourceDomains(source)
	Entry *			source;		/* Source node */
{
	register Domain *	dp;
	register Domain *	hdp;
	register Link **	lpp;
	register bool		first;
#	if	KEEP_FGN_SUB_D == 1
	register bool		remove;
#	endif	KEEP_FGN_SUB_D == 1

	/*
	**	Discard any foreign internal domains from node and links,
	**	using information from foreign domain hierarchy.
	**	(Know that there are 0 or >1 domains in each hierarchy.)
	*/

	if ( (hdp = DomHier.d_head) == (Domain *)0 )
		return;

	if ( (dp = ForeignHier.d_head) == (Domain *)0 )
		return;

	/*
	**	If either hierarchy is a subset of the other, complain.
	*/

	while ( hdp->d_entry == dp->d_entry )
	{
		if ( (dp = dp->d_next) == (Domain *)0 )
		{
			if ( hdp->d_next != (Domain *)0 )
				Warn("hierarchy \"%s\" for node \"%s\" sub-set of ours", source->e_node->n_hierarchy, source->e_name);
			goto out;
		}
		if ( (hdp = hdp->d_next) == (Domain *)0 )
		{
			Warn("our hierarchy sub-set of \"%s\" for node \"%s\"", source->e_node->n_hierarchy, source->e_name);
			goto out;
		}
	}

	dp = ForeignHier.d_last;
	hdp = DomHier.d_last;

	/*
	**	Discard identical part
	*/

	while ( hdp->d_entry == dp->d_entry )
	{
		if ( (dp = dp->d_previous) == (Domain *)0 )
			goto out;

		if ( (hdp = hdp->d_previous) == (Domain *)0 )
			break;
	}

	/*
	**	Remaining sub-domains should be removed
	*/
#	if	KEEP_FGN_SUB_D == 1
	/*
	**	where we belong to one of the same name.
	*/

	remove = false;
#	endif	KEEP_FGN_SUB_D == 1

	/*
	**	But, if hierarchy suspect, take out the lot.
	*/

	if ( dp->d_entry->e_states & S_OURDOM )
	{
		hdp = dp;
		goto in;
	}

	while ( (hdp = dp->d_previous) != (Domain *)0 )
	{
in:
#		if	KEEP_FGN_SUB_D == 1
		if ( !remove )
		{
			if ( !(hdp->d_entry->e_states & S_OURDOM) )
			{
				dp = hdp;
				continue;
			}

			remove = true;
		}
#		endif	KEEP_FGN_SUB_D == 1

		if ( (dp = hdp->d_previous) == (Domain *)0 )
		{
			if ( (ForeignHier.d_head = hdp->d_next) != (Domain *)0 )
				hdp->d_next->d_previous = hdp->d_previous;
		}
		else
		if ( (dp->d_next = hdp->d_next) != (Domain *)0 )
			hdp->d_next->d_previous = hdp->d_previous;

		/*
		**	Remove references from source.
		*/

		for
		(
			dp = source->e_node->n_domains.d_head, first = true ;
			dp != (Domain *)0 ;
			dp = dp->d_next, first = false
		)
			if ( dp->d_entry == hdp->d_entry )
			{
				if ( first )
				{
					Warn
					(
						"\"%s\" removed (has external sub-domain \"%s\" as its primary)",
						source->e_name,
						dp->d_entry->e_name
					);

					NodeCount--;
					source->e_states &= ~S_FOUND;
					break;
				}

				Trace3
				(
					1,
					"Discarding domain \"%s\" from node \"%s\"",
					dp->d_entry->e_name,
					source->e_name
				);

				if ( (dp->d_previous->d_next = dp->d_next) != (Domain *)0 )
					dp->d_next->d_previous = dp->d_previous;
				else
				{
					source->e_node->n_domains.d_last = dp->d_previous;
					source->e_node->n_domains.d_tail = &dp->d_previous->d_next;
				}

				if ( !(dp->d_states & S_OTHDOM) )
					dp->d_entry->e_states |= S_OTHDOM;
				source->e_node->n_domains.d_count--;

				free((char *)dp);
				break;
			}

		/*
		**	And from its links.
		**	(Remove any links whose primary domain is removed.)
		*/

		for ( lpp = &source->e_node->n_l_first ; (*lpp) != (Link *)0 ; )
		{
			register Entry *	ep;
			char *			cp1;
			char *			cp2;

			if
			(
				(ep = (*lpp)->l_entry) == Home
				||
				(dp = ep->e_node->n_domains.d_head) == (Domain *)0
				||
				(
					(cp1 = Home->e_node->n_hierarchy) != NULLSTR
					&&
					(cp2 = ep->e_node->n_hierarchy) != NULLSTR
					&&
					strccmp(cp1, cp2) == STREQUAL
				)
			)
			{
				lpp = &(*lpp)->l_next;
				continue;
			}

			first = true;

			do
			{
				if ( dp->d_entry == hdp->d_entry )
				{
					if ( first )
					{
						Trace4
						(
							1,
							"\"%s\" disconnected from \"%s\" (has external sub-domain \"%s\" as its primary)",
							source->e_name,
							ep->e_name,
							dp->d_entry->e_name
						);

						Unlink(source, lpp);
					}
					else
					{
						Trace3
						(
							1,
							"Discarding domain \"%s\" from node \"%s\"",
							dp->d_entry->e_name,
							ep->e_name
						);

						if ( (dp->d_previous->d_next = dp->d_next) != (Domain *)0 )
							dp->d_next->d_previous = dp->d_previous;
						else
						{
							ep->e_node->n_domains.d_last = dp->d_previous;
							ep->e_node->n_domains.d_tail = &dp->d_previous->d_next;
						}

						if ( !(dp->d_states & S_OTHDOM) )
							dp->d_entry->e_states |= S_OTHDOM;
						ep->e_node->n_domains.d_count--;
						
						free((char *)dp);
						lpp = &(*lpp)->l_next;
					}

					break;
				}

				first = false;
			}
			while
				( (dp = dp->d_next) != (Domain *)0 );

			if ( dp == (Domain *)0 )
				lpp = &(*lpp)->l_next;
		}

		if ( (dp = hdp->d_next) == (Domain *)0 )
			dp = hdp;
		else
			free((char *)hdp);
	}
out:;
#	if	KEEP_FGN_SUB_D != 1
	/*
	**	Discard any links not a member of one of Home's domains.
	*/

	for ( lpp = &source->e_node->n_l_first ; (*lpp) != (Link *)0 ; )
	{
		if ( (dp = (*lpp)->l_entry->e_node->n_domains.d_head) == (Domain *)0 )
		{
cont2:
			lpp = &(*lpp)->l_next;
			continue;
		}

		do
			if ( (dp->d_entry->e_states & (S_HIERDOM|S_OURDOM)) == (S_HIERDOM|S_OURDOM) )
				goto cont2;
		while
			( (dp = dp->d_next) != (Domain *)0 );

		Unlink(source, lpp);

		Trace3
		(
			1,
			"\"%s\" disconnected from \"%s\" (not a member of one of our domains)",
			source->e_name,
			(*lpp)->l_entry->e_name
		);
	}
#	endif	KEEP_FGN_SUB_D != 1
}



/*
**	Clear (all/unfound) domains from node.
*/

void
ClearDomains(ep, all)
	Entry *			ep;
	bool			all;
{
	register Domain **	dpp;
	register Domain *	dp;
	register Node *		np;

	np = ep->e_node;

	for ( dpp = &np->n_domains.d_head ; (dp = *dpp) != (Domain *)0 ; )
		if
		(
			all
			||
			!(dp->d_entry->e_states & S_FOUND)
			||
			((dp->d_states & S_OTHDOM) && !(dp->d_entry->e_states & S_OTHDOM))
		)
		{
			if ( !all && !(dp->d_states & S_OTHDOM) )
				Report3
				(
					"Node \"%s\" removed from domain \"%s\"",
					ep->e_name,
					dp->d_entry->e_name
				);

			if ( (*dpp = dp->d_next) == (Domain *)0 )
				np->n_domains.d_last = dp->d_previous;
			else
				dp->d_next->d_previous = dp->d_previous;

			if ( ep == Home )
				dp->d_entry->e_states &= ~S_OURDOM;

			if ( !(dp->d_states & S_OTHDOM) )
				dp->d_entry->e_states |= S_OTHDOM;
			np->n_domains.d_count--;
			free((char *)dp);
		}
		else
			dpp = &dp->d_next;

	np->n_domains.d_tail = dpp;
}



/*
**	Extract a string representing domain list from a domain header
*/

char *
DomString(dhp, c)
	DomHead *		dhp;
	char			c;
{
	register Domain *	dp;
	register char *		cp;
	register int		len = 0;
	char *			str;

	if ( (dp = dhp->d_head) == (Domain *)0 )
		return NULLSTR;

	do
	{
		len += strlen(dp->d_entry->e_name) + 1;
	}
		while ( (dp = dp->d_next) != (Domain *)0 );

	str = cp = Malloc(len);

	dp = dhp->d_head;
	do
	{
		if ( !(dp->d_states & S_OTHDOM) )
		{
			cp = strcpyend(cp, dp->d_entry->e_name);
			*cp++ = c;
		}
	}
		while ( (dp = dp->d_next) != (Domain *)0 );

	*--cp = '\0';

	return str;
}



/*
**	Check if named domain is in list.
*/

bool
InDomain(cp, hp)
	register char *		cp;
	register DomHead *	hp;
{
	register Domain *	dp;
	register Entry *	ep;

	if ( (ep = Lookup(cp, DomainHash)) == (Entry *)0 )
		return false;

	for ( dp = hp->d_head ; dp != (Domain *)0 ; dp = dp->d_next )
		if ( dp->d_entry == ep )
			return true;

	return false;
}
