/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Precv.c	1.8 84/11/01
*/

/*
**	Receive data from remote, and disassemble packets
*/

#include	"global.h"

#include	"Pconfig.h"
#include	"Packet.h"
#include	"Channel.h"
#include	"Pstats.h"
#include	"Proto.h"
#include	"Debug.h"


#if	DEBUG >= 1
short			BadHdrs;
#endif

static Packet		Pkt;
static short		Sequence;
static short		Channo;
static Chn_p		Chnp;

#define	HDR		Pkt.pkt_hdr[PKTHDR]
#define	SIZ		Pkt.pkt_hdr[PKTSIZE]
#define	NSIZ		Pkt.pkt_hdr[PKTNSIZE]
#define	DATASIZE	BYTE(SIZ)
#define	DATANSIZE	BYTE(NSIZ)
#define	CRCEXPECTED	(PprotoT!=0)
#define	CRCPRESENT	((HDR&(PKTCHK_M<<PKTCHK_S))!=0)
#define	CHANNEL		((HDR>>PKTCHN_S)&PKTCHN_M)
#define	SEQUENCE	((HDR>>PKTSEQ_S)&PKTSEQ_M)
#define	TYPE		((HDR>>PKTTYP_S)&PKTTYP_M)
#define	ACK		(PKTACKTYP<<PKTTYP_S)
#define	NAK		(PKTNAKTYP<<PKTTYP_S)
#define	DATA0		BYTE(Pkt.pkt_data[0])
#define	DATA1		BYTE(Pkt.pkt_data[1])
#define	DATA2		BYTE(Pkt.pkt_data[2])
#define	Nextseq		Chnp->chn_rseq

static bool
			Movenak(),
			Retry(),
			Skipseq();

static void
			FlushData(),
			Rack(),
			Rcontrol(),
			Rnak(),
			Reply(),
			BufferData();



void
Precv()
{
	register int	datacount;

	/*
	**	Read header, turn on timeout
	*/

	(*PRread)((char *)&Pkt, PKTHDRZ, true, false);	

	while
	(
		(datacount = DATASIZE) > PKTDATAZ
		||
		DATANSIZE != ((~datacount)&0xff)
		||
		(CRCPRESENT ^ CRCEXPECTED)
	)
	{
		PSTATS(PS_BADHDR);
#		if	DEBUG >= 1
		if ( BadHdrs++ == 0 )
			Logpkt(&Pkt, PLOGIN);
#		endif	DEBUG
		HDR = SIZ;	/* Shift bytes up, assumes that PKTHDRZ == 3! */
		SIZ = NSIZ;
		(*PRread)(&NSIZ, 1, false, false);
	}

	if ( CRCPRESENT )
		datacount += PKTCRCZ;

	/*
	**	Read data, turn off timeout
	*/

	(*PRread)(Pkt.pkt_data, datacount, false, true);

#	if	DEBUG >= 1
	if ( BadHdrs )
	{
		Tracehdrs(BadHdrs);
		Dumphist(PDESC(PS_BADHDR));
		BadHdrs = 0;
	}
#	endif	DEBUG

	Logpkt(&Pkt, PLOGIN);

	if ( (Channo = CHANNEL) >= Pnchans || Channo < Pfchan )
	{
		PSTATS(PS_BADCHAN);
		Dumphist(PDESC(PS_BADCHAN));
		return;
	}

	if ( CRCPRESENT && crc((char *)&Pkt, datacount+PKTHDRZ-PKTCRCZ) )
	{
		PSTATS(PS_BADCRC);
		Dumphist(PDESC(PS_BADCRC));
		return;
	}

	PSTATS(PS_RPKTS);

	Chnp = &Channels[Channo];

	Chnp->chn_artime = 0;

	Sequence = SEQUENCE;
	Plastidle = 0;

	switch ( TYPE )
	{
	case PKTCNTLTYP:
		if ( DATA0 & CONTROL )
		{
			Rcontrol();
			break;
		}
	case PKTDATATYP:
		if ( Chnp->chn_rstate == CHNS_RESET )
			break;

		Chnp->chn_rtime = 0;

		if ( Sequence == Nextseq )
		{
			do
			{
				Nextseq = (Sequence+1) & PKTSEQ_M;

				if ( TYPE == PKTDATATYP )
					BufferData();
				else
				{
					FlushData();
					(*PrecvControl)(Channo, Pkt.pkt_data, DATASIZE);
				}
			}
			while
				( Chnp->chn_rnak && Movenak() );

			Reply(ACK);
		}
		else
		if ( Retry() )
		{
			PSTATS(PS_RDUP);
			Sequence = (Nextseq-1) & PKTSEQ_M;
			Reply(ACK);
		}
		else
		if ( Skipseq() )
		{
			PSTATS(PS_SKPSEQ);
			Reply(NAK);
		}
		else
		{
			PSTATS(PS_OUTSEQ);
			Dumphist(PDESC(PS_OUTSEQ));
			Preset(Channo);	/* Bad news... */
		}
		break;

	case PKTACKTYP:
		if ( Chnp->chn_xstate != CHNS_RESET )
			Rack();
		break;

	case PKTNAKTYP:
		if ( Chnp->chn_xstate != CHNS_RESET )
			Rnak();
		break;
	}
}



/*
**	Buffer the data
*/

static void
BufferData()
{
	register int	size = DATASIZE;

	if ( size > Chnp->chn_rbfc )
		FlushData();

	bcopy(Pkt.pkt_data, Chnp->chn_rbfp, size);

	Chnp->chn_rbfp += size;
	Chnp->chn_rbfc -= size;
}



/*
**	Flush buffered data
*/

static void
FlushData()
{
	register int	size;

	if ( (size = PBufMax-Chnp->chn_rbfc) > 0 )
		(*PrecvData)(Channo, Chnp->chn_rbuf, size);

	Chnp->chn_rbfp = Chnp->chn_rbuf;
	Chnp->chn_rbfc = PBufMax;
}



/*
**	Flush all the buffers.
*/

void
Pflush()
{
	for
	(
		Channo = Pfchan, Chnp = &Channels[Pfchan] ;
		Channo < Pnchans ;
		Channo++, Chnp++
	)
		if ( Chnp->chn_rbfp >= Chnp->chn_rbuf )
			FlushData();
}



/*
**	Is this packet a valid re-transmission?
*/

static bool
Retry()
{
	register char *	lastp = &SeqTable[Nextseq+SEQMOD-1];
	register char *	nextp = lastp - (NPKTBUFS-1);

	do
		if ( *nextp == Sequence )
			return true;
	while
		( ++nextp <= lastp );

	return false;
}



/*
**	Is this a valid subsequent packet after a skipped sequence?
*/

static bool
Skipseq()
{
	register char *	lastp = &SeqTable[Nextseq+NPKTBUFS-1];
	register char *	nextp = &SeqTable[Nextseq+1];

	do
		if ( *nextp == Sequence )
		{
			/*
			**	It is indeed, hold it in an empty slot.
			*/

			register Pks_p	pksp;

			for
			(
				pksp = Chnp->chn_rpkts ;
				pksp < &Chnp->chn_rpkts[NPKTBUFS-1] ;
				pksp++
			)
				if ( pksp->pks_state == RPS_NULL )
				{
					pksp->pks_pkt = Pkt;
					pksp->pks_state = RPS_NAK;
					Chnp->chn_rnak++;
					return true;
				}
				else
				if ( ((pksp->pks_pkt.pkt_hdr[PKTHDR]>>PKTSEQ_S)&PKTSEQ_M) == Sequence )
					return true;

			/*
			**	Oops, no empty slot, should probably PANIC.
			*/

			return false;
		}
	while
		( ++nextp <= lastp );

	return false;
}



/*
**	Move any NAKed packets that are now in sequence into packet.
*/

static bool
Movenak()
{
	register Pks_p	pksp;

	for
	(
		pksp = Chnp->chn_rpkts ;
		pksp < &Chnp->chn_rpkts[NPKTBUFS-1] ;
		pksp++
	)
		if
		(
			pksp->pks_state == RPS_NAK
			&&
			((pksp->pks_pkt.pkt_hdr[PKTHDR]>>PKTSEQ_S)&PKTSEQ_M) == Nextseq
		)
		{
			pksp->pks_state = RPS_NULL;
			Pkt = pksp->pks_pkt;
			Sequence = Nextseq;
			Chnp->chn_rnak--;
			return true;
		}

	return false;
}



/*
**	Send a reply packet
*/

static void
Reply(type)
	int	type;
{
	PsendCpkt
	(
		type | (Channo<<PKTCHN_S) | (Sequence<<PKTSEQ_S),
		0,
		0
	);
}



/*
**	This and all lesser sequenced packets ok
*/

static void
Rack()
{
	register Pks_p	pksp = Chnp->chn_nxpkt;
	register char *	lastp = &SeqTable[Sequence+SEQMOD];
	register char *	nextp = lastp - (Pnbufs-1);
	register int	hit = 0;

	if ( DATASIZE == 0 )
	do
	{
		if ( *nextp == ((pksp->pks_pkt.pkt_hdr[PKTHDR] >> PKTSEQ_S) & PKTSEQ_M) )
		{
			if ( pksp->pks_state != XPS_NULL )
			{
				pksp->pks_state = XPS_NULL;
				hit++;
			}

			if ( ++pksp >= &Chnp->chn_xpkts[Pnbufs] )
				pksp = Chnp->chn_xpkts;

			Chnp->chn_nxpkt = pksp;
			(void)Psend(Channo);
		}
	}
	while
		( ++nextp <= lastp );

	if ( hit )
	{
#		if	PSTATISTICS == 1
		Pstats[PS_LOSTACK].ss_count += hit-1;
#		endif	PSTATISTICS
	}
	else
	{
		PSTATS(PS_BADACK);
		Dumphist(PDESC(PS_BADACK));
	}
}



/*
**	Mark this packet NAKed, and retransmit the previous packet.
*/

static void
Rnak()
{
	register Pks_p	pksp = Chnp->chn_nxpkt;
	register char *	lastp = &SeqTable[Sequence+SEQMOD];
	register char *	nextp = lastp - (Pnbufs-1);
	register int	hit = 0;

	if ( DATASIZE == 0 )
	do
	{
		if ( *nextp == ((pksp->pks_pkt.pkt_hdr[PKTHDR] >> PKTSEQ_S) & PKTSEQ_M) )
		{
			if ( pksp->pks_state == XPS_WAIT )
			{
				if ( nextp == lastp )
				{
					pksp->pks_state = XPS_NAK;
					break;
				}
				else
				if ( nextp == (lastp-1) )
				{
					pksp->pks_xtime = 0;
					(*PqPkt)((char *)&pksp->pks_pkt, pksp->pks_size);
					Logpkt(&pksp->pks_pkt, PLOGOUT);
					PSTATS(PS_NAKPKT);
				}

				hit++;
			}

			if ( ++pksp >= &Chnp->chn_xpkts[Pnbufs] )
				pksp = Chnp->chn_xpkts;
		}
	}
	while
		( ++nextp <= lastp );

	if ( !hit )
	{
		PSTATS(PS_BADNAK);
		Dumphist(PDESC(PS_BADNAK));
	}
}



/*
**	Process a control packet.
*/

static void
Rcontrol()
{
	register Pks_p	pksp;
	register int	i;
	register int	hit;

	switch ( DATA0 )
	{
	case TIMEOUT:
		if ( DATASIZE != 1 )
			break;

		if ( Chnp->chn_xstate == CHNS_RESET )
			return;

		/*
		**	Re-transmit any packets that have been waiting long enough;
		**	but if there are no waiting packets,
		**	call higher level function, and mark channel idle.
		*/

		for ( hit = 0, pksp = Chnp->chn_nxpkt, i = Pnbufs ; --i >= 0 ; )
		{
			if ( pksp->pks_state == XPS_WAIT )
			{
				if ( pksp->pks_xtime >= PRtimo )
				{
					pksp->pks_xtime = 0;
					(*PqPkt)((char *)&pksp->pks_pkt, pksp->pks_size);
					Logpkt(&pksp->pks_pkt, PLOGOUT);
					PSTATS(PS_TIMOPKT);
				}

				hit++;
			}

			if ( ++pksp >= &Chnp->chn_xpkts[Pnbufs] )
				pksp = Chnp->chn_xpkts;
		}

		if ( !hit )
		{
			Chnp->chn_xstate = CHNS_IDLE;
			(*PrTimeout)(Channo);
		}
		else
		if ( hit < Pnbufs )
			while ( Psend(Channo) > 0 );

		return;

	case XMT_RESET:
		if ( DATASIZE != 3 )
			break;

		if ( DATA1 < Pnbufs )
			Pnbufs = DATA1;

		if ( DATA2 < PXmax )
		{
			PXmax = DATA2;

			if ( PXmax < PXsize )
				PXsize = PXmax;
		}

		/*
		**	Send an acknowledgement
		*/

		PsendCpkt
		(
			(PKTCNTLTYP<<PKTTYP_S)|(Channo<<PKTCHN_S),
			1,
			ACK_RESET
		);

		PXreset(Channo);

		return;

	case ACK_RESET:
		if ( DATASIZE != 1 )
			break;

		PRreset(Channo);

		return;

	case REQ_RESET:
		if ( DATASIZE != 1 )
			break;

		if ( Chnp->chn_rstate != CHNS_RESET )
			Preset(Channo);

		return;

	case IDLE:
		Plastidle = 1;
		return;
	}

	PSTATS(PS_BADCNTL);
	Dumphist(PDESC(PS_BADCNTL));
}
