/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)children.c	1.25 89/10/30
*/

/*
**	Routines that handle child processes
*/

#define	STDIO

#include	"global.h"
#include	"command.h"
#include	"debug.h"
#include	"messagename.h"
#include	"spool.h"

#define	BM_DATA
#include	"bad.h"
#include	"daemon.h"
#include	"Stream.h"
#include	"driver.h"
#include	"AQ.h"

#include	<setjmp.h>
#include	<signal.h>
#if	V8 == 1 || BSD4 == 1 && BSD4V < 'c'
#include	<wait.h>
#endif
#if	BSD4 >= 2 || BSD4 == 1 && BSD4V >= 'c'
#include	<sys/wait.h>
#endif
#include	<errno.h>

extern jmp_buf	QTimeouts;
extern bool	UseFd1;


#define	MINWAIT		3	/* Minimum seconds between "wait"s */
#define	MAXWAIT		60	/* Maximum seconds between "wait"s */
#define	MAXPROCWAIT	(30*60)	/* Maximum wait time for a process */
#define	STATEWAIT	30	/* No need to be interested in the state-changer */

static int	BadPid;
static int	CPid;
static int	NSpid;
static Time_t	NSstarted;
static bool	NewWait;
static int	RoutePid;
static char *	Router	= ROUTER;
static Time_t	Started[NSTREAMS];
static bool	StateUpdated;

void		WaitError(), StatusError();
void		start_router(), start_handler();



/*
**	Terminate reception, pass message to handler.
*/

void
EndMessage(chan)
	AQarg		chan;
{
	register Str_p	strp = &inStreams[(int)chan];
	register int	i;
	static char *	routename;
	static int	id;

	Trace3(1, "EndMessage for channel %d state %d", (int)chan, strp->str_state);

	NNstate.allmessages++;	/* Let driver know things are progressing */
	(void)fflush(stderr);

	if ( RouteDir == NULLSTR )
	{
		start_handler(chan);
		return;
	}

	strp->str_fd = 0;

	if ( routename == NULLSTR )	
	{
		routename = concat(RouteDir, "/01234567890123", NULLSTR);
		for ( i = 0 ; i < NSTREAMS ; i++ )
		{
			MesgNumber += inStreams[i].str_messages;
			MesgNumber += outStreams[i].str_messages;
		}
	}

	while
	(
		access
		(
			EncodeName
			(
				routename,
				(strp->str_flags & STR_DUP) ? (char)MESG_DUP : (char)0,
				(char)id,	/* May be used to distinguish consecutive runs */
				(LastTime > strp->str_time) ? LastTime - strp->str_time : 0L
			),
			0
		)
		!= SYSERROR
	)
		if ( ++id > VC_MASK )
			Syserror("Can't creat \"%s\"", routename);

	while ( link(strp->str_fname, routename) == SYSERROR )
	{
		Syserror("Can't link \"%s\" to \"%s\"", strp->str_fname, routename);
		if ( BatchMode )
			finish(SYSERROR);
	}

	while ( unlink(strp->str_fname) == SYSERROR )
	{
		Syserror("Can't unlink \"%s\"", strp->str_fname);
		if ( BatchMode )
			finish(SYSERROR);
	}

	SndEOMA((int)chan);

	strp->str_messages++;
	strp->str_bytes += strp->str_posn;
	strp->str_flags = 0;

	Update(up_force);

	if ( RoutePid == 0 )
		start_router((AQarg)0);
}



/*
**	Start the message handler if state update not in progress, else delay.
*/

void
start_handler(chan)
	AQarg		chan;
{
	register Str_p	strp = &inStreams[(int)chan];
	register int	i;
	char *		args[8];
	char		elapsedtime[16];

	if ( !StateUpdated || NSpid != 0 )
	{
		qAction(start_handler, chan);
		return;
	}

	switch ( strp->str_fd = fork() )
	{
	case SYSERROR:
		Report1("EndMessage cannot fork");
		qAction(EndMessage, chan);	/* Try again later */
		return;
	
	case 0:
		closeall();

		i = 0;
		args[i++] = MessageHandler;

		if ( LastTime > strp->str_time )
		{
			(void)sprintf(elapsedtime, "-t%lu", LastTime - strp->str_time);
			args[i++] = elapsedtime;
		}

		args[i++] = concat("-l", LinkDir, NULLSTR);
		args[i++] = concat("-h", HomeNode, NULLSTR);

		if ( strp->str_flags & STR_DUP )
			args[i++] = "-D";

		args[i++] = strp->str_fname;
		args[i  ] = NULLSTR;

		for ( ;; )
		{
			execve(args[0], args, StripEnv());
			Syserror("cannot execve handler \"%s\"", args[0]);
			if ( BatchMode )
				finish(SYSERROR);
		}
	}
	
	Started[(int)chan] = LastTime;

	if ( !Waiting )
	{
		Waiting = true;
		qAction(WaitHandler, (AQarg)(LastTime+MINWAIT));
	}
	else
		NewWait = true;

	Trace2(1, "EndMessage started process %d", strp->str_fd);
}



/*
**	Wait for handler to terminate;
**	if found, send an EOM_ACCEPT.
*/

void
WaitHandler(delay)
	AQarg		delay;
{
	register Str_p	strp;
	register int	chan;
	register int	procs;
	int		status;

	Trace2(1, "WaitHandler %ld", (Time_t)delay - LastTime);

	if ( (Time_t)delay >= LastTime )
	{
		if ( NewWait )
		{
			NewWait = false;

			delay = (AQarg)(LastTime+MINWAIT);
		}

		qAction(WaitHandler, delay);
		Waiting = true;
		return;
	}

	NewWait = false;

	for
	(
		strp = &inStreams[Fstream], chan = Fstream, procs = 0 ;
		chan < Nstreams ;
		chan++, strp++
	)
		if ( strp->str_state == STR_ENDING && strp->str_fd > 0 )
			procs++;
	
	if ( NSpid != 0 )
		procs++;
	else
	if ( procs == 0 )
	{
		Waiting = false;
		return;
	}

	do
	{
		CPid = 0;

		if ( !setjmp(QTimeouts) )
		{
#			if V8 == 1 || BSD4 >= 1

			CPid = wait3(&status, WNOHANG, (char *)0);

#			else V8 == 1 || BSD4 >= 1
			ALARM_ON(MINSLEEP);

			CPid = wait(&status);
			
			ALARM_OFF();
#			endif V8 == 1 || BSD4 >= 1

			if ( CPid == SYSERROR )
			{
				char *	fmt;

				if ( (CPid = NSpid) > 0 )
				{
					fmt = "newstatehandler (pid %d) disappeared!";
					NSpid = 0;
				}
				else
				{
					fmt = "handler (pid %d) for channel %d disappeared!";

					for
					(
						strp = &inStreams[Fstream], chan = Fstream ;
						chan < Nstreams ;
						chan++, strp++
					)
						if ( strp->str_state == STR_ENDING && (CPid = strp->str_fd) > 0 )
						{
							strp->str_fd = 0;
							if ( access(strp->str_fname, 0) == SYSERROR )
							{
								Warn(fmt, CPid, chan);
								goto ok;
							}

							qAction(EndMessage, (AQarg)chan);
							break;
						}

					if ( chan == Nstreams && BadPid != 0 )
					{
						BadPid = 0;
						Waiting = false;
						return;
					}
				}

				Warn(fmt, CPid, chan);
				procs--;
				continue;;
			}
		}

		if ( CPid == 0 )
		{
			/*
			**	Timed out, try later
			*/

			for
			(
				strp = &inStreams[Fstream], chan = Fstream, procs = MAXWAIT ;
				chan < Nstreams ;
				chan++, strp++
			)
				if ( strp->str_state == STR_ENDING && strp->str_fd > 0 )
				{
					if ( (CPid = LastTime-Started[chan]) > MAXPROCWAIT )
					{
						Started[chan] = LastTime;
						WaitError(MessageHandler, CPid, strp->str_fd, chan);
					}

					if ( CPid < procs )
						procs = CPid;
				}
			
			if ( NSpid != 0 )
			{
				if ( (CPid = LastTime-NSstarted) > MAXPROCWAIT )
				{
					NSstarted = LastTime;
					WaitError(NewstateHandler, CPid, NSpid, 0);
				}

				if ( CPid < procs )
					procs = CPid;
			}

			if ( procs < MINWAIT || (!Transmitting && !Receiving) )
				procs = MINWAIT;

			if ( (Time_t)delay == 0 )
				return;
			qAction(WaitHandler, (AQarg)(LastTime+procs));
			Waiting = true;
			return;
		}

		Trace2(1, "WaitHandler found process %d", CPid);

		if ( CPid == NSpid )
		{
			NSpid = 0;

			if ( status )
				StatusError(NewstateHandler, "", status);
			else
				StateUpdated = true;

			procs--;
			continue;
		}

		if ( CPid == RoutePid )
		{
			RoutePid = 0;
			StatusError(Router, "", status);
			continue;
		}
		
		for
		(
			strp = &inStreams[Fstream], chan = Fstream ;
			chan < Nstreams ;
			chan++, strp++
		)
			if ( strp->str_state == STR_ENDING && CPid == strp->str_fd )
			{
				strp->str_fd = 0;

				if ( status )
				{
					qAction(EndMessage, (AQarg)chan);

					StatusError(MessageHandler, strp->str_fname, status);

					procs--;
					break;
				}

ok:
				SndEOMA(chan);

				strp->str_messages++;
				strp->str_bytes += strp->str_posn;
				strp->str_flags = 0;

				Update(up_force);
				
				procs--;
				break;
			}

		if ( chan == Nstreams && BadPid == 0 )
			Report2("Collected orphan %d", CPid);
	}
		while ( procs > 0 || BadPid != 0 );

	Waiting = false;
}



void
StatusError(prog, file, status)
	char *	prog;
	char *	file;
	int	status;
{
	Error
	(
		 "\"%s %s\" bad exit status:%d signal:%d%s"
		,prog
		,file
		,(status>>8)&0xff
		,status&0x7f
		,status&0x80?" (core dumped)":""
	);
}



void
WaitError(prog, wait, pid, chan)
	char *	prog;
	int	wait;
	int	pid;
	int	chan;
{
	Error
	(
		"overlong wait (%d secs.) for process %d, \"%s\" on channel %d",
		wait,
		pid,
		prog,
		chan
	);
}


/*
**	Run state update program
*/

bool
RunState(newstate)
	AQarg		newstate;
{
	register int	i;
	char *		args[7];

	if ( NSpid != 0 )
		return false;	/* Let's not get too excited here */
	
	(void)fflush(stderr);

	switch ( NSpid = fork() )
	{
	case SYSERROR:
		NSpid = 0;
		Report1("RunState cannot fork");
		return false;

	case 0:
		closeall();

		i = 0;
		args[i++] = NewstateHandler;
		args[i++] = (int)newstate==LINK_DOWN?"-D":"-U";
		args[i++] = concat("-h", HomeNode, NULLSTR);
		if ( BatchMode )
			args[i++] = "-N";
#		ifdef	DEBUGPROGS
		if ( Traceflag )
			args[i++] = Traceflag==1?"-T1":"-T2";
#		endif	DEBUGPROGS
		args[i++] = LinkDir;
		args[i  ] = NULLSTR;

		for ( ;; )
		{
			(void)execve(args[0], args, StripEnv());
			Syserror("can't execve \"%s\"", args[0]);
			if ( BatchMode )
				finish(SYSERROR);
		}
	}

	NSstarted = LastTime;

	if ( !Waiting )
	{
		Waiting = true;
		qAction(WaitHandler, (AQarg)(LastTime+STATEWAIT));
	}

	Trace2(1, "RunState started process %d", NSpid);

	return true;
}



/*
**	Move offending command file to bad directory,
**	and set bad message handler onto it.
**	Then reset the channel and queue NewMessage.
*/

void
BadMessage(chan, reason)
	AQarg		chan;
	BMReason	reason;
{
	register Str_p	strp = &outStreams[(int)chan];
	char *		mesg;

	Trace3(1, "BadMessage for channel %d state %d", (int)chan, strp->str_state);

	mesg = BMExplanations[(int)reason];

	Report4("bad message on chan %d \"%s\" - %s", chan, strp->str_id, mesg);

	if ( reason != bm_nodata && access(strp->str_id, 0) == 0 )
	{
		char *	badname = concat(BADDIR(), strp->str_id, NULLSTR);
		char *	args[7];

		while ( link(strp->str_id, badname) == SYSERROR && errno != EEXIST )
		{
			Syserror("cannot link \"%s\" to \"%s\"", strp->str_id, badname);
			if ( BatchMode )
				finish(SYSERROR);
		}

		(void)unlink(strp->str_id);

		Report3("bad message \"%s\" moved to \"%s\"", strp->str_id, badname);

		(void)fflush(stderr);

		switch ( BadPid = fork() )
		{
		case SYSERROR:
			BadPid = 0;
			Report1("BadMessage cannot fork");
			qAction(BadMessage, chan);		/* Try again */
			free(badname);
			return;

		case 0:
			closeall();

			args[0] = BadHandler;
			args[1] = concat("-d", badname, NULLSTR);
			args[2] = concat("-l", LinkDir, NULLSTR);
			args[3] = concat("-i", Name, NULLSTR);
			args[4] = concat("-h", HomeNode, NULLSTR);
			args[5] = concat("-e", mesg, NULLSTR);
			args[6] = NULLSTR;

			for ( ;; )
			{
				execve(BadHandler, args, StripEnv());
				Syserror("cannot execve handler \"%s\"", BadHandler);
				if ( BatchMode )
					finish(SYSERROR);
			}
		}

		free(badname);

		if ( !Waiting )
		{
			Waiting = true;
			qAction(WaitHandler, (AQarg)(LastTime+STATEWAIT));
		}

		Trace2(1, "BadMessage started process %d", BadPid);
	}
	else
		Report2("bad message \"%s\" non-existent", strp->str_id);

	strp->str_state = STR_ERROR;
}



/*
**	Start routing daemon
*/

void
start_router(arg)
	AQarg		arg;
{
	char *		args[6];
	char		pid[12];
	static bool	router_queued;

	if ( arg == 1 )
		router_queued = false;

	if ( !StateUpdated || NSpid != 0 )
	{
		if ( RoutePid == 0 && !router_queued )
		{
			qAction(start_router, (AQarg)1);
			router_queued = true;
		}
		return;
	}

	for ( ;; )
	{
		switch ( RoutePid = fork() )
		{
		case SYSERROR:
			RoutePid = 0;
			Syserror("Can't fork");
			if ( BatchMode )
				finish(SYSERROR);
			continue;
		
		case 0:
			closeall();

			args[0] = Router;
			args[1] = concat("-d", RouteDir, NULLSTR);
			args[2] = concat("-h", HomeNode, NULLSTR);
			args[3] = concat("-l", LinkDir, NULLSTR);
			(void)sprintf(pid, "-p%d", Pid);
			args[4] = pid;
			args[5] = NULLSTR;

			for ( ;; )
			{
				execve(Router, args, StripEnv());
				Syserror("cannot execve routing daemon \"%s\"", Router);
				if ( BatchMode )
					finish(SYSERROR);
			}
		}

		break;
	}
}



/*
**	Check that router was started if needed, else force start it if necessary.
*/

void
check_router()
{
	if ( RouteDir != NULLSTR && RoutePid == 0 && NNstate.allmessages > 0 )
	{
		StateUpdated = true;
		NSpid = 0;
		start_router((AQarg)0);
	}
}
