/*
**	Copyright (c) 1986 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "88/07/24 @(#)stats.c	1.16";

/*
**	Accumulate statistics.
*/

char *	Usage	= "[-[A][F][H][L][S][T[n]][W]] [-b<[[[[yy]mm]dd]hh]mm[.ss]>]\\\n\
	[-e<[[[[yy]mm]dd]hh]mm[.ss]>] [-h<homename>] [-m<minmsgs>]\\\n\
	[-r<routefile>] [-t{-|<statsfile>}] [site|link|handler ...]";

#define	RECOVER
#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"handlers.h"
#include	"header.h"
#include	"state.h"
#include	"stats.h"

#include	<ctype.h>
#include	<time.h>


/*
**	Parameters set from arguments.
*/

Time_t	StopTime;
char *	HomeNode;			/* Name of Home node */
int	HomeLength;			/* Length of "HomeNode" */
int	MinMessages = 2;		/* Min. messages for an entry to be printed */
char *	Name;				/* Program invoked name */
bool	NoAddresses, NoFileServices, NoHandlers, NoLinks, NoSites;
Time_t	StartTime;
char *	StatsFile = ACCSTATS;		/* Process accumulated stats by default */
int	Traceflag;			/* Trace level */
bool	Warnings;			/* For warnings rather than termination on errors */

/*
**	Type allocator
*/

#ifdef	Talloc
#undef	Talloc
#endif
#define	Talloc(type)	(type *)strclr(Malloc(sizeof(type)),sizeof(type))

/*
**	Hashing parameters
*/

#define	SITE_NAME_SIZE	32		/* Max. chars used in hash */

#if	CRC_HASH == 0
#define	HASH_SIZE	32
#define	HASH_MASK	(HASH_SIZE-1)	/* Only define this if HASH_SIZE is a power of 2 */
#define	HASH_BITS	5
#define	HASH(H,C)	H+=H^C
#define	HASHSTRING	"H += H ^ C"
#else	CRC_HASH == 0
#define	HASH_SIZE	127
#define	HASHSTRING	"CRC-16"
#endif	CRC_HASH == 0

/*
**	Statistics structures
*/

typedef struct
{
	ulong	messages,
		bytes;
}
	Data;				/* The unit of statistic */

typedef struct
{
	Data	from,
		passed_from,
		to,
		passed_to,
		broadcast;
}
	Site;				/* Per site info. */

typedef struct
{
	Data	from,
		to;
}
	Link;				/* Per link info. */

typedef struct
{
	Data	home,
		in,
		out;
}
	Mess;				/* Per handler info. */

/*
**	File services
*/

typedef enum
{
	List, ListVerbose, SendFile, Errors,
	Services
}
	FilS_t;

#define	MATCH_FS_CHAR	4
#define	FS_LIST		'\0'
#define	FS_LISTV	'V'
#define	FS_FILES	'F'

typedef struct
{
	Data	services[(int)Services];	/* File server requests */
}
	FilS;

/*
**	Common structures
*/

typedef union
{
	Site *	site;			/* Site */
	Link *	link;			/* Link */
	Mess *	mesg;			/* Handler */
	FilS *	fils;			/* File service request site */
}
	Type;

typedef struct Entry
	Entry;				/* Hash table entry */

struct Entry
{
	Entry *	great;
	Entry *	less;
	Type	type;			/* Pointer to stats */
	char *	name;
};

Entry *	Sites[HASH_SIZE];		/* All the sites */
Entry *	Links[HASH_SIZE];		/* All the links */
Entry *	Messages[HASH_SIZE];		/* All the message types */
Entry * FileServices[HASH_SIZE];	/* All the serviced sites */
Entry * Printonly[HASH_SIZE];		/* Selected names */
Entry *	Home;				/* Us */

static ulong	SiteCount, LinkCount, MesgCount, FSCount, PrintCount;

typedef enum
{
	site, link, message, print, fils
}
	Select;

/*
**	Address types
*/

typedef enum
{
	simple, broadcast, multicast, explicit, group, n_addr
}
	Addr_t;

typedef struct
{
	char *	name;
	Data	home,
		in,
		out;
}
	Addr;

Addr	Addresses[]	=
{
	{"Simple"},
	{"Broadcast"},
	{"Multicast"},
	{"Explicit"},
	{"Group"}
};

static ulong	AddrCount;

/*
**	Miscellaneous definitions.
*/

bool	AtHomeVal;
char *	Bozo		= "-RUBBISH-";	/* Must be valid sitename! */
char *	BozoFrom	= "-NO_FROM-";	/* Must be valid sitename! */
char *	BozoLink	= "-NO_LINK-";	/* Must be valid sitename! */
char *	BozoTo		= "-NO_DEST-";	/* Must be valid sitename! */
bool	Broadcast;
int	Count;
char *	Dest;
jmp_buf	EndJmp;
Time_t	FirstTim;
char **	From;
Time_t	Last_Tim;
char *	MesgHandler;
char	Id;				/* ID of current statistics record */
char *	MesgLink;
jmp_buf	NameErrJmp;
bool	Nodedone;
int	Pid;				/* Process id */
char *	Route;
char *	Routelink;
long	Size;
char *	Source;
Time_t	Tim;				/* Time of stat */
Time_t	Time;				/* Time of invocation */
char **	To;
ulong	Tt;


int	stats();
void	usage(), Pstats();
bool	dostat(), dostat1(), dostat2();
bool	StatsTo(), StatsLink();
static bool	AtHome();
Entry	*Enter(), *Lookup();
Time_t	getdate();

extern char	_sobuf[];

#ifdef	NO_FLOAT
#define	BPD(e,d)	(d==0?0:(((e)*100)+((d)>>1))/(d))	/* May get int overflow */
#else	NO_FLOAT
float	Oh		= 100.0;
float	Pf		= 0.5;
#define	BPD(e,d)	(d==0?(ulong)0:(ulong)(((float)(e)*Oh)/(float)(d)+Pf))
#endif	NO_FLOAT

#define	Fprintf		(void)fprintf
#define	SecInDay	(24*60*60L)
#define	PC(s1,fl)	(int)BPD((s1).fl, grandtotal.fl)
#define	SPC(s1,s2,fl)	(int)BPD((s1).fl+(s2).fl, grandtotal.fl)


int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	(void)umask(022);
	Pid = getpid();
	Time = time((long *)0);
	StopTime = Time + SecInDay;	/* This may take a while to run... */

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					NoAddresses = true;
					continue;

				case 'F':
					NoFileServices = true;
					continue;

				case 'H':
				case 'M':
					NoHandlers = true;
					continue;

				case 'L':
					NoLinks = true;
					continue;

				case 'S':
					NoSites = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) <= 0 )
						Traceflag = 1;
					break;

				case 'W':
					Warnings = true;
					continue;

				case 'b':
					if ( (StartTime = getdate(++*argv)) == 0 )
					{
						usage("bad date \"%s\"", *argv);
						exit(1);
					}
					goto break2;

				case 'e':
					if ( (StopTime = getdate(++*argv)) == 0 )
					{
						usage("bad date \"%s\"", *argv);
						exit(1);
					}
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'm':
					MinMessages = atoi(++*argv);
					goto break2;

				case 'r':
					SetRoute(++*argv);
					goto break2;

				case 't':
					StatsFile = ++*argv;
					goto break2;

				default:
					usage("unrecognised flag '%c'", c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
			(void)Enter(*argv, print);
	}

	if ( setjmp(NameErrJmp) )
	{
		Error("Bad home nodename \"%s\"", HomeNode);
		exit(1);
	}

	if ( HomeNode == NULLSTR )
		HomeNode = NodeName();
	HomeLength = strlen(HomeNode);
	
	Home = Enter(HomeNode, site);

	setbuf(stdout, _sobuf);

	exit(stats());
}



/*
**	Read in stats file, accumulate and print out stats.
*/

int
stats()
{
	register FILE *	fd;

	if ( StatsFile == NULLSTR || StatsFile[0] == '\0' )
	{
		Error("Missing filename for 't' flag");
		return 1;
	}

	if ( strcmp(StatsFile, "-") == STREQUAL )
		fd = stdin;
	else
	if ( (fd = fopen(StatsFile, "r")) == NULL )
	{
		Syserror("Can't open \"%s\"", StatsFile);
		return 1;
	}

	if ( !setjmp(EndJmp) )
		Rstats(&Id, dostat, fd);

	(void)fclose(fd);

	if ( StartTime < FirstTim )
		StartTime = FirstTim;

	if ( StopTime > Last_Tim )
		StopTime = Last_Tim;

	Pstats(stdout);

	return 0;
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

#if VARARGS
void
usage(va_alist)
	va_dcl
{
	va_list		ap;
	register char 	*s;

	va_start(ap);
	s = va_arg(ap, char *);
	VMesg("bad arguments", s, ap);
	va_end(ap);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}
#else
/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	Fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}
#endif



/*
**	Process statistics field from record
*/

bool
dostat(cp, index)
	register char *	cp;
	register int	index;
{
	static FilS_t	service;

	Trace3(3, "dostat \"%s\" %d", cp, index);

	if ( index == 0 )
	{
		switch ( Id )
		{
		case ST_RETMESG:
			To = &Source;
			From = &Dest;
			break;

		case ST_MESSAGE:
		case ST_INMESG:
		case ST_OUTMESG:
			From = &Source;
			To = &Dest;
			break;

		case ST_F_S_LOG:
			service = Errors;
			break;

		default:
			return false;
		}

		MesgLink = NULLSTR;
	}

	if ( Id == ST_F_S_LOG )
	{
		switch ( (FSlog_t)index )
		{
		case fl_time:
			goto do_time;

		case fl_size:
			Size = atol(cp);
			break;

		case fl_dest:
			if ( (Dest = strchr(cp, '@')) == NULLSTR )
				Dest = Bozo;
			else
				Dest++;
			break;

		case fl_service:
			switch ( cp[MATCH_FS_CHAR] )
			{
			case FS_LIST:	service = List;		break;
			case FS_LISTV:	service = ListVerbose;	break;
			case FS_FILES:	service = SendFile;	break;
			}
			break;

		case fl_names:
			if ( strncmp(cp, "ERROR, ", 7) == STREQUAL )
				service = Errors;
		}

		if ( (FSlog_t)index == fl_last )
		{
			register Data *	dp;

			if ( setjmp(NameErrJmp) )
				Dest = Bozo;
			dp = &(Enter(Dest, fils))->type.fils->services[(int)service];
			dp->messages++;
			dp->bytes += Size;
			return false;
		}

		return true;
	}

	switch ( (StMesg_t)index )
	{
	case sm_time:
do_time:	Tim = (Time_t)atol(cp);
		if ( Tim < FirstTim )
		{
			if ( Warnings && (FirstTim-Tim) > 30 )
				Warn("statistic %ld out of order", Tim);
			FirstTim = Tim;
		}
		else
		if ( FirstTim == 0 )
			FirstTim = Tim;
		if ( Tim < Last_Tim )
		{
			if ( Warnings && (Last_Tim-Tim) > 30 )
				Warn("statistic %ld out of order", Tim);
		}
		else
			Last_Tim = Tim;
		if ( Tim >= StopTime )
		{
			if ( Tim > (StopTime + SecInDay) )
				longjmp(EndJmp, 1);
			return false;
		}
		if ( Tim < StartTime )
			return false;
		break;

	case sm_size:
		Size = atol(cp);
		break;

	case sm_dest:
		Dest = cp;
		break;

	case sm_source:
		Source = cp;
		break;

	case sm_route:
		Route = cp;
		break;

	case sm_handler:
		MesgHandler = cp;
		break;

#	if	0
	case sm_tt:
		Tt = (ulong)atol(cp);
		break;
#	endif

	case sm_link:
		if ( *cp != '\0' )
			MesgLink = cp;
		break;
	}

	if ( (StMesg_t)index == sm_last )
	{
		AtHomeVal = (bool)(strncmp(*From, HomeNode, HomeLength) == STREQUAL);

		Broadcast = false;

		Trace7
		(
			2,
			"dostat '%c' tim %lu siz %ld dest \"%s\" src \"%s\" route \"%s\"",
			Id, Tim, Size, Dest, Source, Route
		);

		Count = 0;
#		if	0
		Tim -= Tt;
#		endif
		Nodedone = false;

		if ( MesgLink == NULLSTR )
		{
			if ( Routelink != NULLSTR )
			{
				free(Routelink);
				Routelink = NULLSTR;
			}

			(void)ExRoute(Route, dostat2);
			MesgLink = Routelink;
		}

		switch ( Id )
		{
		case ST_MESSAGE:
			if ( !NoSites )
				(void)ExDests(*To, dostat1);
			(void)StatsLink(true);
			break;

		case ST_RETMESG:
		case ST_INMESG:
			if ( AtHome(*To) )
				AtHomeVal = true;
			if ( !NoSites )
				(void)StatsFrom(*From);
			(void)StatsLink(true);
			break;

		case ST_OUTMESG:
			if ( !NoSites )
				(void)ExDests(*To, StatsTo);
			(void)StatsLink(false);
			break;
		}
	}

	return true;
}



bool
dostat1(dest)
	char *	dest;
{
	Trace2(3, "dostat1(\"%s\")", dest);

	if ( AtHome(dest) )
	{
		if ( Nodedone )
			return true;
		Nodedone = true;
	}

	if ( Count++ == 0 )
		return Stats(st_from, *From, dest);
	else
		return Stats(st_to, *From, dest);
}



bool
dostat2(tt, dest1, dest2)
	ulong	tt;
	char *	dest1;
	char *	dest2;
{
	Trace4(3, "dostat2(%ld, \"%s\", \"%s\")", tt, dest1, dest2);

	if ( Routelink != NULLSTR )
		free(Routelink);

	Routelink = newstr(dest1);

	return true;
}



/*
**	Return 'true' if 'HomeNode' is in destination list.
*/

static bool
AtHome(s)
	register char *	s;
{
	register char *	a;
	register int	c;
	char *		handler;	/* Not used */

	Trace2(3, "AtHome(\"%s\")", s);

	if ( s == NULLSTR )
		return false;

	switch ( c = *s++ )
	{
	case '\0':
		return false;

	case ATYP_EXPLICIT:
		if ( (s = strrchr(s, c)) == NULLSTR )
			return false;

		return AtHome(s+1);

	case ATYP_GROUP:
	case ATYP_MULTICAST:
		for ( ;; )
		{
			if ( (a = strchr(s, c)) != NULLSTR )
				*a = '\0';
	
			if ( AtHome(s) )
			{
				if ( a != NULLSTR )
					*a = c;
				return true;
			}

			if ( a != NULLSTR )
			{
				*a++ = c;
				s = a;
			}
			else
				return false;
		}

	case ATYP_BROADCAST:
		Trace1(3, "AtHome broadcast address");

		Broadcast = true;

		if
		(
			s[0] == '\0'
			||
			(
				s[0] == DOMAIN_SEP
				&&
				(
					(
						s[1] == ATYP_BROADCAST
						&&
						s[2] == '\0'
					)
					||
					HomeDomain(&s[1], &handler, EXCL_HIER)
				)
			)
		)
			return true;

		return false;
	}
	
	if ( HomeAddress(--s, &handler, INCL_HIER) )
		return true;

	return false;
}



/*
**	Print out bad statistic
*/

void
BadStat(name, n)
	char *	name;
	int	n;
{
	if ( !Warnings )
		return;

	if ( n == -1 )
		Warn
		(
			"bad site name \"%s\" in statistic %ld",
			name,
			Tim
		);
	else
		Warn
		(
			"bad character at position %d in site name \"%s\" in statistic %ld",
			n+1,
			name,
			Tim
		);
}



/*
**	Calculate hash for a site name.
*/

int
HashName(name)
	char *		name;
{
	register char *	cp = name;
	register int	hash;
	register int	c;
#	if	CRC_HASH == 1
	char		nn[SITE_NAME_SIZE];
	register char *	np = nn;
#	else	CRC_HASH == 1
	register int	n = 0;

	hash = 0;
#	endif	CRC_HASH == 1

	while ( c = *cp++ )
	{
		if ( !isalnum(c) && c != '_' && c != '-' &&c != '.' )
		{
#			if	CRC_HASH == 1
			BadStat(name, (np-nn));
#			else	CRC_HASH == 1
			BadStat(name, n);
#			endif	CRC_HASH == 1
			longjmp(NameErrJmp, 1);
			return 0;
		}

#		if	CRC_HASH == 1
		if ( np < &nn[SITE_NAME_SIZE] )
			*np++ = c|040;
#		else	CRC_HASH == 1
		n++;
		HASH(hash, c|040);
#		endif	CRC_HASH == 1
	}

#	if	CRC_HASH == 1
	if ( np == nn )
	{
		BadStat(name, -1);
		longjmp(NameErrJmp, 1);
		return 0;
	}

	hash = acrc((Crc_t)0, nn, np-nn);
#	else	CRC_HASH == 1
	if ( n == 0 )
	{
		BadStat(name, -1);
		longjmp(NameErrJmp, 1);
		return 0;
	}
#	endif	CRC_HASH == 1

	Trace3(3, "\"%s\" hash = %#x", name, hash);

#	if	CRC_HASH == 0
#	ifdef	HASH_MASK
	return (hash ^ (hash>>(16-HASH_BITS))) & HASH_MASK;
#	else	HASH_MASK
	return (int)(((unsigned)hash ^ ((unsigned)hash>>(16-HASH_BITS))) % (unsigned)HASH_SIZE);
#	endif	HASH_MASK

#	else	CRC_HASH == 0

#	ifdef	HASH_MASK
	return hash & HASH_MASK;
#	else	HASH_MASK
	return (int)((unsigned)hash % (unsigned)HASH_SIZE);
#	endif	HASH_MASK
#	endif	CRC_HASH == 0
}



/*
**	Look-up an entry in a table
*/

Entry *
Lookup(name, type)
	char *		name;
	Select		type;
{
	register char *	cp1;
	register char *	cp2;
	register int	c;
	register Entry*	ep;

	c = HashName(name);

	switch ( type )
	{
	case site:	ep = Sites[c];		break;
	case link:	ep = Links[c];		break;
	case message:	ep = Messages[c];	break;
	case print:	ep = Printonly[c];	break;
	case fils:	ep = FileServices[c];	break;
	}

	for
	(
		;
		ep != (Entry *)0 ;
		ep = (c&1) ? ep->great : ep->less
	)
	{
		for
		(
			cp1 = name, cp2 = ep->name ;
			(c = *cp1++) != 0 ;
		)
			if ( (c ^= *cp2++) && c != 040 )
				break;

		if ( c == 0 && *cp2 == 0 )
			return ep;
	}

	return (Entry *)0;
}



/*
**	Make an entry in a table
*/

Entry *
Enter(name, type)
	char *			name;
	Select			type;
{
	register char *		cp1;
	register char *		cp2;
	register int		c;
	register Entry **	epp;

#	if	DEBUG == 2
	if ( Traceflag >= 4 )
	{
		Traceflag--;
		if ( Lookup(name, type) == (Entry *)0 )
			Trace(4, "\"%s\" hash = %#x", name, HashName(name));
		Traceflag++;
	}
#	endif	DEBUG == 2

	c = HashName(name);

	switch ( type )
	{
	case site:	epp = &Sites[c];	break;
	case link:	epp = &Links[c];	break;
	case message:	epp = &Messages[c];	break;
	case print:	epp = &Printonly[c];	break;
	case fils:	epp = &FileServices[c];	break;
	}

	for
	(
		;
		*epp != (Entry *)0 ;
		epp = (c&1) ? &((*epp)->great) : &((*epp)->less)
	)
	{
#		if	DEBUG == 2
		if ( Traceflag >= 4 )
		{
			Traceflag--;
			if ( Lookup(name, type) == (Entry *)0 )
				Trace(4, "hash duplication \"%s\"", (*epp)->name);
			Traceflag++;
		}
#		endif	DEBUG == 2

		for
		(
			cp1 = name, cp2 = (*epp)->name ;
			(c = *cp1++) != 0 ;
		)
			if ( (c ^= *cp2++) && c != 040 )
				break;

		if ( c == 0 && *cp2 == 0 )
			return *epp;
	}

	*epp = Talloc(Entry);

	(*epp)->name = newstr(name);

	switch ( type )
	{
	case site:
		SiteCount++;
		(*epp)->type.site = Talloc(Site);
		break;

	case link:
		LinkCount++;
		(*epp)->type.link = Talloc(Link);
		break;

	case message:
		MesgCount++;
		(*epp)->type.mesg = Talloc(Mess);
		break;

	case print:
		PrintCount++;
		break;

	case fils:
		FSCount++;
		(*epp)->type.fils = Talloc(FilS);
	}

	return *epp;
}



/*
**	Process statistics into structures.
*/

bool
Stats(type, from, to)
	Stats_t		type;
	char *		from;
	char *		to;
{
	Entry *		fep;
	Entry *		tep;
	register Site *	fp;
	register Site *	tp;

	Trace4
	(
		2,
		"Stats typ %d from \"%s\" to \"%s\"",
		type, from, to
	);

	if ( setjmp(NameErrJmp) )
		from = Bozo;
	fep = Enter(from, site);
	fp = fep->type.site;

	if ( setjmp(NameErrJmp) )
		to = Bozo;
	tep = Enter(to, site);
	tp = tep->type.site;

	switch ( type )
	{
	case st_from:
		if ( Home == fep || Home == tep )
		{
			fp->from.bytes += Size;
			fp->from.messages++;
		}
		else
		{
			fp->passed_from.bytes += Size;
			fp->passed_from.messages++;
		}

		break;

	case st_to:
		if ( Home == fep )
		{
			tp->to.bytes += Size;
			tp->to.messages++;
		}
		else
		if ( Home == tep )
			break;
		else
		{
			tp->passed_to.bytes += Size;
			tp->passed_to.messages++;
		}

		break;
	}

	return true;
}



bool
StatsFrom(from)
	char *		from;
{
	register Entry *fep;
	register Site *	fp;

	Trace4
	(
		2,
		"StatsFrom(\"%s\")%s%s",
		from, AtHomeVal?" athome":"", Broadcast?" broadcast":""
	);

	if ( from == NULLSTR || *from == '\0' )
		from = BozoFrom;
	if ( setjmp(NameErrJmp) )
		from = Bozo;
	fep = Enter(from, site);
	fp = fep->type.site;

	if ( Home == fep || AtHomeVal )
	{
		fp->from.bytes += Size;
		fp->from.messages++;
	}
	else
	{
		fp->passed_from.bytes += Size;
		fp->passed_from.messages++;
	}

	if ( Broadcast )
	{
		fp->broadcast.bytes += Size;
		fp->broadcast.messages++;
	}

	return true;
}



bool
StatsTo(to)
	char *		to;
{
	register Entry *tep;
	register Site *	tp;

	Trace3
	(
		2,
		"StatsTo(\"%s\")%s",
		to, AtHomeVal?" athome":""
	);

	if ( to == NULLSTR || *to == '\0' )
		to = BozoTo;
	if ( setjmp(NameErrJmp) )
		to = Bozo;
	if
	(
		(tep = Enter(to, site)) == Home	/* Broadcast */
		&&
		(tep = Enter(MesgLink, site)) == Home
	)
		return true;
	tp = tep->type.site;

	if ( AtHomeVal )
	{
		tp->to.bytes += Size;
		tp->to.messages++;
	}
	else
	{
		tp->passed_to.bytes += Size;
		tp->passed_to.messages++;
	}

	return true;
}



bool
StatsLink(in)
	bool		in;
{
	Trace4(2, "StatsLink(%s) handler=\"%s\", link=\"%s\"", in?"in":"out", MesgHandler, MesgLink);

	if ( !NoAddresses )
	{
		register Addr *	ap;

		Trace4(3, "Address \"%s\"[%s] - '%c'", Dest, *To, (*To)[0]);

		switch ( (*To)[0] )
		{
		case ATYP_MULTICAST:	ap = &Addresses[(int)multicast];break;
		case ATYP_BROADCAST:	ap = &Addresses[(int)broadcast];break;
		case ATYP_EXPLICIT:	ap = &Addresses[(int)explicit];	break;
		case ATYP_GROUP:	ap = &Addresses[(int)group];	break;
		default:		ap = &Addresses[(int)simple];	break;
		}

		if ( in )
		{
			ap->in.bytes += Size;
			ap->in.messages++;
		}
		else
		{
			ap->out.bytes += Size;
			ap->out.messages++;
		}

		if ( AtHomeVal )
		{
			ap->home.bytes += Size;
			ap->home.messages++;
		}

		AddrCount++;
	}

	if ( !NoHandlers )
	{
		register Mess *	mp;

		if ( setjmp(NameErrJmp) )
			MesgHandler = Bozo;
		mp = Enter(MesgHandler, message)->type.mesg;
	
		if ( in )
		{
			mp->in.bytes += Size;
			mp->in.messages++;
		}
		else
		{
			mp->out.bytes += Size;
			mp->out.messages++;
		}

		if ( AtHomeVal )
		{
			mp->home.bytes += Size;
			mp->home.messages++;
		}
	}

	if ( !NoLinks )
	{
		register Link *	lp;

		if ( MesgLink == NULLSTR || *MesgLink == '\0' )
			MesgLink = BozoLink;
		if ( setjmp(NameErrJmp) )
			MesgLink = Bozo;
		lp = Enter(MesgLink, link)->type.link;
	
		if ( in )
		{
			lp->from.bytes += Size;
			lp->from.messages++;
		}
		else
		{
			lp->to.bytes += Size;
			lp->to.messages++;
		}
	}

	return true;
}



/*
**	Convert a date in [[[[yy]mm]dd]hh]mm[.ss] format to a number.
*/

extern struct tm *localtime();


Time_t
getdate(cp)
	char *			cp;
{
	register struct tm *	tvec;
	register long		bit;
	register int		t, y, d, h, m, s;
	long			timbuf;
	struct tm		tm;

	if ( strchr(cp, '/') != NULLSTR )
	{
		if
		(
			(d = gpair(&cp)) < 0
			||
			*cp++ != '/'
			||
			(t = gpair(&cp)) < 0
			||
			*cp++ != '/'
			||
			(y = gpair(&cp)) < 0
			||
			*cp != '\0'
		)
			goto bad;

		h = 0;
		m = 0;
		s = 0;
		goto setime;
	}

	tvec = localtime(&Time);

	if ( (y = gpair(&cp)) < 0 )
		goto bad;
	if ( (t = gpair(&cp)) < 0 )
		goto setmin;
	if ( (d = gpair(&cp)) < 0 )
		goto sethrs;
	if ( (h = gpair(&cp)) < 0 )
		goto setday;
	if ( (m = gpair(&cp)) < 0 )
		goto setmon;

setime:
	if ( *cp++ == '.' )
	{
		if ( (s = gpair(&cp)) < 0 || s > 59 )
			goto bad;
	}
	else
		s = 0;
	if ( t < 1 || t > 12 )
		goto bad;
	if ( d < 1 || d > 31 )
		goto bad;
	if ( h >= 24 )
	{
		if ( h > 24 )
			goto bad;
		h = 0;
		if ( ++d > 31 )
		{
			d = 1;
			if ( ++t > 12 )
			{
				t = 1;
				y++;
			}
		}
	}
	if ( m > 59 )
		goto bad;

	tm.tm_year = y;
	tm.tm_mon = t-1;
	tm.tm_mday = d;
	tm.tm_hour = h;
	tm.tm_min = m;
	tm.tm_sec = s;

	timbuf = 0;

	for ( bit = 1<<30 ; bit != 0 ; bit >>= 1 )
	{
		timbuf |= bit;
		tvec = localtime(&timbuf);
		if ( (t = tmcmp(tvec, &tm)) == 0 )
			break;	/* exact match, found it */
		if ( t > 0 )	/* tvec > tm, must remove that bit */
			timbuf &= ~bit;
	}

	if ( bit == 0 )		/* time in the time warp, or something */
		return 0;

	return timbuf;

setmin:
	m = y;
	h = tvec->tm_hour;
getday:
	d = tvec->tm_mday;
getmon:
	t = tvec->tm_mon+1;
getyrs:
	y = tvec->tm_year;
	goto setime;

sethrs:
	m = t;
	h = y;
	goto getday;

setday:
	m = d;
	h = t;
	d = y;
	goto getmon;

setmon:
	m = h;
	h = d;
	d = t;
	t = y;
	goto getyrs;

bad:
	return 0;
}


int
gpair(cbp)
	char **		cbp;
{
	register int	c, d;
	register char *	cp;

	cp = *cbp;
	if ( *cp == 0 )
		return -1;

	c = *cp++ -'0';

	if ( c < 0 || c > 9 )
		return -1;

	if ( *cp == 0 )
		return -1;

	if ( *cp == '/' )
	{
		d = c;
		c = 0;
	}
	else
	if ( (d = *cp++ -'0') < 0 || d > 9 )
		return -1;

	*cbp = cp;

	return c*10 + d;
}



int
tmcmp(t1, t2)
	register struct tm *	t1;
	register struct tm *	t2;
{
	register int		i;

	if ( i = (t1->tm_year - t2->tm_year) )
		return i;
	if ( i = (t1->tm_mon - t2->tm_mon) )
		return i;
	if ( i = (t1->tm_mday - t2->tm_mday) )
		return i;
	if ( i = (t1->tm_hour - t2->tm_hour) )
		return i;
	if ( i = (t1->tm_min - t2->tm_min) )
		return i;
	return t1->tm_sec - t2->tm_sec;
}



/*
**	Make a sorted list of entries from a hash table.
*/


Entry **	Listp;
Select		MType;

void	MLfromTree();
int	MLcompare();



Entry **
MakeList(countp, table, type)
	register ulong *	countp;
	Entry **		table;
	Select			type;
{
	register Entry **	epp;
	register Entry **	end;
	Entry **		list;

	Trace3(1, "MakeList(%d, %s)", *countp, table==Sites?"sites":table==Links?"links":table==Messages?"messages":"files");

	if ( *countp == 0 )
		return (Entry **)0;
	
	list = Listp = (Entry **)Malloc(*countp * sizeof(Entry *));

	MType = type;

	for ( epp = table, end = &epp[HASH_SIZE] ; epp < end ; epp++ )
		if ( *epp != (Entry *)0 )
			MLfromTree(*epp);

	if ( (*countp = Listp-list) > 1 )
		qsort((char *)list, *countp, sizeof(Entry *), MLcompare);

	return list;
}


void
MLfromTree(ep)
	register Entry *	ep;
{
	register ulong		mcount;

	if ( ep->great != (Entry *)0 )
		MLfromTree(ep->great);

	if ( ep->less != (Entry *)0 )
		MLfromTree(ep->less);

	if ( PrintCount == 0 && MinMessages > 0 )
	{
		mcount = 0;

		switch ( MType )
		{
		case site:	mcount = ep->type.site->from.messages +
					 ep->type.site->passed_from.messages +
					 ep->type.site->to.messages +
					 ep->type.site->passed_to.messages;
				break;
		case link:	mcount = ep->type.link->from.messages +
					 ep->type.link->to.messages;
				break;
		case message:	mcount = ep->type.mesg->in.messages +
					 ep->type.mesg->out.messages;
				break;
		case fils:	mcount = ep->type.fils->services[(int)List].messages +
					 ep->type.fils->services[(int)ListVerbose].messages +
					 ep->type.fils->services[(int)SendFile].messages +
					 ep->type.fils->services[(int)Errors].messages;
				break;
		}

		if ( mcount < MinMessages )
			return;
	}
	else
	if ( Lookup(ep->name, print) == (Entry *)0 )
		return;

	*Listp++ = ep;
}



int
MLcompare(epp1, epp2)
	char *	epp1;
	char *	epp2;
{
	return strccmp((*(Entry **)epp1)->name, (*(Entry **)epp2)->name);
}



/*
**	Print gathered statistics
*/

ulong	ElapsedDays;				/* in 100th days */
char *	Format		= "\t%ld\t%d\t%ld\t%d";

void	paddresses(), pfiles(), plinks(), pmessages(), psites();

void
Pstats(fd)
	FILE *	fd;
{
	char	dates[2][21];

	dates[0][20] = dates[1][20] = '\0';

	(void)strncpy(dates[0], ctime(&FirstTim)+4, 20);
	(void)strncpy(dates[1], ctime(&Last_Tim)+4, 20);
	Fprintf
	(
		fd,
		".PP\n.TS\nr .\nStatistics start at %s\nand end after %s\nduration: %ld.%02ld days\n.TE\n",
		dates[0],
		dates[1],
		BPD(Last_Tim - FirstTim, SecInDay) / 100,
		BPD(Last_Tim - FirstTim, SecInDay) % 100
	);

	if ( StartTime >= StopTime )
		ElapsedDays = 0;
	else
		ElapsedDays = BPD(StopTime - StartTime, SecInDay);

	if ( FirstTim != StartTime || Last_Tim != StopTime )
	{
		(void)strncpy(dates[0], ctime(&StartTime)+4, 20);
		(void)strncpy(dates[1], ctime(&StopTime)+4, 20);
		Fprintf
		(
			fd,
			".PP\n.TS\nr .\nSelection starts at %s\nand ends before %s\nduration: %ld.%02ld days\n.TE\n",
			dates[0],
			dates[1],
			ElapsedDays/100, ElapsedDays%100
		);
	}

	if ( !NoAddresses && AddrCount )
		paddresses(fd);

	if ( !NoHandlers && MesgCount )
		pmessages(fd);

	if ( !NoLinks && LinkCount )
		plinks(fd);

	if ( !NoSites && SiteCount > 1 )
		psites(fd);

	if ( !NoFileServices && FSCount)
		pfiles(fd);
}



/*
**	Print per-address type statistics
*/

void
paddresses(fd)
	register FILE *		fd;
{
	register int		i;
	register Addr *		ap;
	Data			hometotal, intotal, outtotal, grandtotal;

	hometotal.messages = intotal.messages = outtotal.messages = 0;
	hometotal.bytes = intotal.bytes = outtotal.bytes = 0;

	Fprintf(fd, ".PP\n.TS H\nbox ;\n");
	Fprintf(fd, "l2 || c1 s1 s1 s2 || c1 s1 s1 s2 | c1 s1 s1 s2 | c1 s1 s1 s2\n");
	Fprintf(fd, "^2 || r1 r1 r1 r2 || r1 r1 r1 r2 | r1 r1 r1 r2 | r1 r1 r1 r2\n");
	Fprintf(fd, "l2 || n1 n1 n1 n2 || n1 n1 n1 n2 | n1 n1 n1 n2 | n1 n1 n1 n2 .\n");
	Fprintf(fd, "Address Type\tTo/From Home\tIn\tOut\tTotal\n");
	Fprintf(fd, "\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\n=\n.TH\n");

	for ( i = (int)n_addr, ap = Addresses ; i-- ; ap++ )
	{
		hometotal.messages += ap->home.messages;
		hometotal.bytes += ap->home.bytes;
		intotal.messages += ap->in.messages;
		intotal.bytes += ap->in.bytes;
		outtotal.messages += ap->out.messages;
		outtotal.bytes += ap->out.bytes;
	}

	grandtotal.messages = intotal.messages + outtotal.messages;
	grandtotal.bytes = intotal.bytes + outtotal.bytes;

	for ( i = (int)n_addr, ap = Addresses ; i-- ; ap++ )
	{
		Fprintf
		(
			fd,
			ap->name
		);
		Fprintf
		(
			fd,
			Format,
			ap->home.messages,
			PC(ap->home, messages),
			ap->home.bytes,
			PC(ap->home, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			ap->in.messages,
			PC(ap->in, messages),
			ap->in.bytes,
			PC(ap->in, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			ap->out.messages,
			PC(ap->out, messages),
			ap->out.bytes,
			PC(ap->out, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			ap->out.messages+ap->in.messages,
			SPC(ap->out, ap->in, messages),
			ap->out.bytes+ap->in.bytes,
			SPC(ap->out, ap->in, bytes)
		);
		putc('\n', fd);
	}

	Fprintf
	(
		fd,
		"_\n%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t100\t%ld\t100\n.TE\n",
		(int)n_addr,
		hometotal.messages,
		PC(hometotal, messages),
		hometotal.bytes,
		PC(hometotal, bytes),
		intotal.messages,
		PC(intotal, messages),
		intotal.bytes,
		PC(intotal, bytes),
		outtotal.messages,
		PC(outtotal, messages),
		outtotal.bytes,
		PC(outtotal, bytes),
		grandtotal.messages,
		grandtotal.bytes
	);
}



/*
**	Print per-message statistics
*/

void
pmessages(fd)
	register FILE *		fd;
{
	register int		i;
	register Mess *		mp;
	register Entry *	ep;
	register Entry **	epp;
	Entry **		list;
	Data			hometotal, intotal, outtotal, grandtotal;

	hometotal.messages = intotal.messages = outtotal.messages = 0;
	hometotal.bytes = intotal.bytes = outtotal.bytes = 0;

	list = MakeList(&MesgCount, Messages, message);

	Fprintf(fd, ".PP\n.TS H\nbox ;\n");
	Fprintf(fd, "l2 || c1 s1 s1 s2 || c1 s1 s1 s2 | c1 s1 s1 s2 | c1 s1 s1 s2\n");
	Fprintf(fd, "^2 || r1 r1 r1 r2 || r1 r1 r1 r2 | r1 r1 r1 r2 | r1 r1 r1 r2\n");
	Fprintf(fd, "l2 || n1 n1 n1 n2 || n1 n1 n1 n2 | n1 n1 n1 n2 | n1 n1 n1 n2 .\n");
	Fprintf(fd, "Handler\tTo/From Home\tIn\tOut\tTotal\n");
	Fprintf(fd, "\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\n=\n.TH\n");

	for ( i = MesgCount, epp = list ; i-- ; )
	{
		mp = (*epp++)->type.mesg;

		hometotal.messages += mp->home.messages;
		hometotal.bytes += mp->home.bytes;
		intotal.messages += mp->in.messages;
		intotal.bytes += mp->in.bytes;
		outtotal.messages += mp->out.messages;
		outtotal.bytes += mp->out.bytes;
	}

	grandtotal.messages = intotal.messages + outtotal.messages;
	grandtotal.bytes = intotal.bytes + outtotal.bytes;

	for ( i = MesgCount, epp = list ; i-- ; )
	{
		register Handler *	handler;
		register char *		type;

		ep = *epp++;
		mp = ep->type.mesg;

		Fprintf
		(
			fd,
			ep->name
		);
		Fprintf
		(
			fd,
			Format,
			mp->home.messages,
			PC(mp->home, messages),
			mp->home.bytes,
			PC(mp->home, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			mp->in.messages,
			PC(mp->in, messages),
			mp->in.bytes,
			PC(mp->in, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			mp->out.messages,
			PC(mp->out, messages),
			mp->out.bytes,
			PC(mp->out, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			mp->out.messages+mp->in.messages,
			SPC(mp->in, mp->out, messages),
			mp->out.bytes+mp->in.bytes,
			SPC(mp->in, mp->out, bytes)
		);
		putc('\n', fd);
	}

	Fprintf
	(
		fd,
		"_\n%ld\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t100\t%ld\t100\n.TE\n",
		MesgCount,
		hometotal.messages,
		PC(hometotal, messages),
		hometotal.bytes,
		PC(hometotal, bytes),
		intotal.messages,
		PC(intotal, messages),
		intotal.bytes,
		PC(intotal, bytes),
		outtotal.messages,
		PC(outtotal, messages),
		outtotal.bytes,
		PC(outtotal, bytes),
		grandtotal.messages,
		grandtotal.bytes
	);

	free((char *)list);
}



/*
**	Print per-link statistics
*/

void
plinks(fd)
	register FILE *		fd;
{
	register int		i;
	register Link *		lp;
	register Entry *	ep;
	register Entry **	epp;
	Entry **		list;
	Data			fromtotal, tototal, grandtotal;

	fromtotal.messages = tototal.messages = 0;
	fromtotal.bytes = tototal.bytes = 0;

	list = MakeList(&LinkCount, Links, link);

	Fprintf(fd, ".PP\n.TS H\nbox ;\nl | c s s s | c s s s | c s s s | c\n^ | r r r r | r r r r | r r r r | ^\nl | n n n n | n n n n | n n n n | n .\n");
	Fprintf(fd, "Link\tFrom\tTo\tTotal\tBytes/day\n");
	Fprintf(fd, "\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\tMessages\t%%\tBytes\t%%\n=\n.TH\n");

	for ( i = LinkCount, epp = list ; i-- ; )
	{
		lp = (*epp++)->type.link;

		fromtotal.messages += lp->from.messages;
		fromtotal.bytes += lp->from.bytes;
		tototal.messages += lp->to.messages;
		tototal.bytes += lp->to.bytes;
	}

	grandtotal.messages = fromtotal.messages + tototal.messages;
	grandtotal.bytes = fromtotal.bytes + tototal.bytes;

	for ( i = LinkCount, epp = list ; i-- ; )
	{
		ep = *epp++;
		lp = ep->type.link;

		Fprintf(fd, "%s", ep->name);
		Fprintf
		(
			fd,
			Format,
			lp->from.messages,
			PC(lp->from, messages),
			lp->from.bytes,
			PC(lp->from, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			lp->to.messages,
			PC(lp->to, messages),
			lp->to.bytes,
			PC(lp->to, bytes)
		);
		Fprintf
		(
			fd,
			Format,
			lp->from.messages+lp->to.messages,
			SPC(lp->from, lp->to, messages),
			lp->from.bytes+lp->to.bytes,
			SPC(lp->from, lp->to, bytes)
		);
		Fprintf(fd, "\t%ld\n", BPD(lp->from.bytes+lp->to.bytes, ElapsedDays));
	}

	Fprintf
	(
		fd,
		"_\n%ld\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t%d\t%ld\t100\t%ld\t100\t%ld\n.TE\n",
		LinkCount,
		fromtotal.messages,
		PC(fromtotal, messages),
		fromtotal.bytes,
		PC(fromtotal, bytes),
		tototal.messages,
		PC(tototal, messages),
		tototal.bytes,
		PC(tototal, bytes),
		grandtotal.messages,
		grandtotal.bytes,
		BPD(grandtotal.bytes, ElapsedDays)
	);

	free((char *)list);
}



/*
**	Print per-site statistics
*/

void
psites(fd)
	register FILE *		fd;
{

	register int		i;
	register Site *		sp;
	register Entry *	ep;
	register Entry **	epp;
	Entry **		list;
	Data			fromtotal, pfromtotal, tototal, ptototal, broadcast, grandtotal;

	fromtotal.messages = pfromtotal.messages = tototal.messages = ptototal.messages = broadcast.messages = 0;
	fromtotal.bytes = pfromtotal.bytes = tototal.bytes = ptototal.bytes = broadcast.bytes = 0;

	list = MakeList(&SiteCount, Sites, site);

	Fprintf(fd, ".PP\n.TS H\nbox ;\n");
	Fprintf(fd, "l2 | c1 s2 | c1 s2 | c1 s2 | c1 s2 || c1 s2 || c1 s2\n");
	Fprintf(fd, "^2 | r1 r2 | r1 r2 | r1 r2 | r1 r2 || r1 r2 || r1 r2\n");
	Fprintf(fd, "l2 | n1 n2 | n1 n2 | n1 n2 | n1 n2 || n1 n2 || n1 n2 .\n");
	Fprintf(fd, "Site\tFrom\tTo\tPassed-from\tPassed-to\tBroadcast-from\tTotal\n");
	Fprintf(fd, "\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\n=\n.TH\n");

	for ( i = SiteCount, epp = list ; i-- ; )
	{
		sp = (*epp++)->type.site;

		fromtotal.messages += sp->from.messages;
		fromtotal.bytes += sp->from.bytes;
		tototal.messages += sp->to.messages;
		tototal.bytes += sp->to.bytes;
		pfromtotal.messages += sp->passed_from.messages;
		pfromtotal.bytes += sp->passed_from.bytes;
		ptototal.messages += sp->passed_to.messages;
		ptototal.bytes += sp->passed_to.bytes;
		broadcast.messages += sp->broadcast.messages;
		broadcast.bytes += sp->broadcast.bytes;
	}

	grandtotal.messages = fromtotal.messages + tototal.messages + pfromtotal.messages + ptototal.messages;
	grandtotal.bytes = fromtotal.bytes + tototal.bytes + pfromtotal.bytes + ptototal.bytes;

	for ( i = SiteCount, epp = list ; i-- ; )
	{
		static char *	format = "\t%ld\t%ld\t%ld\t%ld";

		ep = *epp++;
		sp = ep->type.site;

		if ( ep->name[0] == '.' )
			Fprintf(fd, "\\&");

		Fprintf(fd, "%s", ep->name);
		Fprintf
		(
			fd,
			format,
			sp->from.messages,
			sp->from.bytes,
			sp->to.messages,
			sp->to.bytes
		);
		Fprintf
		(
			fd,
			format,
			sp->passed_from.messages,
			sp->passed_from.bytes,
			sp->passed_to.messages,
			sp->passed_to.bytes
		);
		Fprintf
		(
			fd,
			format,
			sp->broadcast.messages,
			sp->broadcast.bytes,
			sp->from.messages+sp->passed_from.messages+sp->to.messages+sp->passed_to.messages,
			sp->from.bytes+sp->passed_from.bytes+sp->to.bytes+sp->passed_to.bytes
		);
		putc('\n', fd);
	}

	Fprintf
	(
		fd,
		"_\n%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\n",
		SiteCount,
		fromtotal.messages,
		fromtotal.bytes,
		tototal.messages,
		tototal.bytes,
		pfromtotal.messages,
		pfromtotal.bytes,
		ptototal.messages,
		ptototal.bytes,
		broadcast.messages,
		broadcast.bytes,
		grandtotal.messages,
		grandtotal.bytes
	);

	Fprintf
	(
		fd,
		"_\n%%\t%d\t%d\t%d\t%d",
		PC(fromtotal, messages),
		PC(fromtotal, bytes),
		PC(tototal, messages),
		PC(tototal, bytes)
	);

	Fprintf
	(
		fd,
		"\t%d\t%d\t%d\t%d\t%d\t%d\t100\t100\n.TE\n",
		PC(pfromtotal, messages),
		PC(pfromtotal, bytes),
		PC(ptototal, messages),
		PC(ptototal, bytes),
		PC(broadcast, messages),
		PC(broadcast, bytes)
	);

	free((char *)list);
}



/*
**	Print per-site file service statistics
*/

void
pfiles(fd)
	register FILE *		fd;
{

	register int		i;
	register FilS *		sp;
	register Entry *	ep;
	register Entry **	epp;
	Entry **		list;
	Data			listtotal, listvtotal, filstotal, errtotal, grandtotal;

	listtotal.messages = listvtotal.messages = filstotal.messages = errtotal.messages  = 0;
	listtotal.bytes = listvtotal.bytes = filstotal.bytes = errtotal.bytes  = 0;

	list = MakeList(&FSCount, FileServices, fils);

	Fprintf(fd, ".PP\n.TS H\nbox ;\n");
	Fprintf(fd, "l2 | c1 s2 | c1 s2 | c1 s2 | c1 s2 | c1 s2 | c1 s2\n");
	Fprintf(fd, "^2 | r1 r2 | r1 r2 | r1 r2 | r1 r2 | r1 r2 | r1 r2\n");
	Fprintf(fd, "l2 | n1 n2 | n1 n2 | n1 n2 | n1 n2 | n1 n2 | n1 n2 .\n");
	Fprintf(fd, "Destination\tList\tListVerbose\tFetchFile\tErrors\tTotal\n");
	Fprintf(fd, "\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\tMessages\tBytes\n=\n.TH\n");

	for ( i = FSCount, epp = list ; i-- ; )
	{
		sp = (*epp++)->type.fils;

		listtotal.messages += sp->services[(int)List].messages;
		listtotal.bytes += sp->services[(int)List].bytes;
		listvtotal.messages += sp->services[(int)ListVerbose].messages;
		listvtotal.bytes += sp->services[(int)ListVerbose].bytes;
		filstotal.messages += sp->services[(int)SendFile].messages;
		filstotal.bytes += sp->services[(int)SendFile].bytes;
		errtotal.messages += sp->services[(int)Errors].messages;
		errtotal.bytes += sp->services[(int)Errors].bytes;
	}

	grandtotal.messages = listtotal.messages + filstotal.messages + listvtotal.messages + errtotal.messages;
	grandtotal.bytes = listtotal.bytes + filstotal.bytes + listvtotal.bytes + errtotal.bytes;

	for ( i = FSCount, epp = list ; i-- ; )
	{
		static char *	format = "\t%ld\t%ld\t%ld\t%ld";

		ep = *epp++;
		sp = ep->type.fils;

		if ( ep->name[0] == '.' )
			Fprintf(fd, "\\&");

		Fprintf(fd, "%s", ep->name);
		Fprintf
		(
			fd,
			format,
			sp->services[(int)List].messages,
			sp->services[(int)List].bytes,
			sp->services[(int)ListVerbose].messages,
			sp->services[(int)ListVerbose].bytes
		);
		Fprintf
		(
			fd,
			format,
			sp->services[(int)SendFile].messages,
			sp->services[(int)SendFile].bytes,
			sp->services[(int)Errors].messages,
			sp->services[(int)Errors].bytes
		);
		Fprintf
		(
			fd,
			"\t%ld\t%ld",
			sp->services[(int)List].messages+sp->services[(int)ListVerbose].messages+sp->services[(int)SendFile].messages+sp->services[(int)Errors].messages,
			sp->services[(int)List].bytes+sp->services[(int)ListVerbose].bytes+sp->services[(int)SendFile].bytes+sp->services[(int)Errors].bytes
		);
		putc('\n', fd);
	}

	Fprintf
	(
		fd,
		"_\n%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\t%ld\n",
		FSCount,
		listtotal.messages,
		listtotal.bytes,
		listvtotal.messages,
		listvtotal.bytes,
		filstotal.messages,
		filstotal.bytes,
		errtotal.messages,
		errtotal.bytes,
		grandtotal.messages,
		grandtotal.bytes
	);

	Fprintf
	(
		fd,
		"_\n%%\t%d\t%d\t%d\t%d",
		PC(listtotal, messages),
		PC(listtotal, bytes),
		PC(listvtotal, messages),
		PC(listvtotal, bytes)
	);

	Fprintf
	(
		fd,
		"\t%d\t%d\t%d\t%d\t100\t100\n.TE\n",
		PC(filstotal, messages),
		PC(filstotal, bytes),
		PC(errtotal, messages),
		PC(errtotal, bytes)
	);

	free((char *)list);
}
