/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "88/07/24 @(#)linkstats.c	1.21";

/*
**	Print out statistics for each link based on contents of state files.
*/

char *	Usage		= "[-V] [-T<level>] [-{c|s}[<sleep>]] [link|node|alias ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"command.h"
#include	"debug.h"
#include	"spool.h"
#include	"state.h"
#undef	Extern
#define	Extern
#include	"Stream.h"

#include	<ndir.h>
#include	<errno.h>

#if	KILL_0 != 1
#include	<signal.h>
#endif	KILL_0 != 1


/*
**	Filenames that must be set up
*/

char *	Slash		= "/";		/* Path name element separator */
char *	Statefile	= STATUSFILE;	/* Filename for daemon status information */
char *	Spooldir	= SPOOLDIR();	/* Spool area directory name */

/*
**	Parameters set from arguments
*/

char *	Name;				/* Program invoked name */
int	SleepTime;			/* Delay for continuous mode scans */
bool	Continuous;			/* Keep scanning status files */
bool	Verbose;			/* Tell everything */
int	Traceflag;			/* Trace level */

/*
**	List for found links.
*/

typedef struct LLel *	LLl_p;

typedef struct LLel
{
	LLl_p	lql_next;
	char *	lql_link;
	ino_t	lql_ino;
}
	LLel;

LLl_p	LinkList;			/* Head of linked list */
LLl_p *	LinkVec;			/* Vector of sorted links */
int	NLinks;				/* No. of links found */

/*
**	Miscellaneous info.
*/

NodeLink LinkD;				/* Info. from FindNode() */

long	StreamSizes[NSTREAMS] =
{
	 SMALL_MESSAGE
	,MEDIUM_MESSAGE
	,LARGE_MESSAGE
};

#define	Fprintf		(void)fprintf
#define	INDENT(A)	Fprintf(stdout,"%*s",A*2,"")
#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))

/*
**	Routines
*/

int	compare();
void	LinkStats(), LongStats(), ShortStats(), ListLink(), SortLinks();
void	ShortHeader(), usage();
bool	FindLinks();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	char		buf[BUFSIZ];

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					continue;

				case 'c':
				case 's':
					Continuous = true;
					if ( (SleepTime = atoi(++*argv)) <= 0 )
						SleepTime = 10;
					break;

				default:
					usage("unrecognised flag '%c'", c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			register int	i;
			register char *	link = *argv;

			for ( i = 0 ; ; i++ )
			{
				char *		linkp = concat(Spooldir, link, Slash, NULLSTR);
				struct stat	statb;

				if
				(
					stat(linkp, &statb) != SYSERROR
					&&
					(statb.st_mode&S_IFMT) == S_IFDIR
				)
				{
#					ifdef	MULTIPATH
					if ( !FindLinks(linkp) )
#					endif	MULTIPATH
						ListLink(Spooldir, link, statb.st_ino);
					break;
				}

				free(linkp);

				if ( i == 0 )
				{
					if ( (link = FindAlias(*argv)) == NULLSTR )
						link = *argv;

					if ( FindNode(link, pt_msg, &LinkD) )
					{
						link = LinkD.nl_name;
						continue;
					}
				}

				Error("unknown link \"%s\"", *argv);
			}
		}
	}

	if ( NLinks == 0 && !FindLinks(Spooldir) )
	{
		Error("no network links found!");
		exit(1);
	}

	SortLinks();

	setbuf(stdout, buf);

	for ( ;; )
	{
		register int	i;

		if ( !Verbose )
			ShortHeader();

		for ( i = 0 ; i < NLinks ; i++ )
			LinkStats(LinkVec[i]->lql_link);

		if ( Continuous )
		{
			if (!Verbose)
				putc('\n', stdout);
			putc('\f', stdout);
			(void)fflush(stdout);
			(void)sleep(SleepTime);
		}
		else
			break;
	}

	(void)fflush(stdout);

	exit(0);
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

#if VARARGS
void
usage(va_alist)
	va_dcl
{
	va_list		ap;
	register char 	*s;

	va_start(ap);
	s = va_arg(ap, char *);
	VMesg("bad arguments", s, ap);
	va_end(ap);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}
#else
/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}
#endif



/*
**	Print stats from state file in directory.
*/

void
LinkStats(link)
	char *		link;
{
	register int	i;
	register int	j;
	static Time_t	lasttime;
	char *		statefile = concat(link, Slash, Statefile, NULLSTR);

	Trace2(1, "link stats from \"%s\"", statefile);

	for ( j = 0 ;; )
	if
	(
		(i = open(statefile, O_READ)) == SYSERROR
		||
		read(i, (char *)&NNstate, sizeof NNstate) != sizeof NNstate
	)
	{
		if ( i != SYSERROR )
			(void)close(i);

		if ( i == SYSERROR || ++j > 3 )
		{
			(void)strclr((char *)&NNstate, sizeof NNstate);
			break;
		}

		(void)sleep(1);
	}
	else
	{
		if ( strcmp(NNstate.version, StreamSCCSID) != STREQUAL )
			if ( NLinks == 1 )
				Warn("\"status\" file version mismatch");
			else
				(void)strclr((char *)&NNstate, sizeof NNstate);

		(void)close(i);
		break;
	}

	free(statefile);

	if ( Continuous && Verbose && NLinks == 1 )
	{
		if ( NNstate.thistime == lasttime )
			return;
		
		lasttime = NNstate.thistime;
	}

	link += strlen(Spooldir);

	if ( Verbose )
		LongStats(link);
	else
		ShortStats(link);
}



void
LongStats(link)
	char *		link;
{
	register Str_p	strp;
	register char *	cp;
	register int	i;
	register int	j;
	register ulong	t;

	if  ( NNstate.starttime == 0 )
	{
		t = 0;
		j = 0;
	}
	else
	{
		if ( (t = NNstate.thistime - NNstate.starttime) == 0 )
			t = 1;
		j = NNstate.thistime - NNstate.lasttime;
	}

	if
	(
		NNstate.procid
		&&
		(
			kill(NNstate.procid, SIG0) != SYSERROR
			||
			errno == EPERM
		)
	)
	{
		register char *	ls = NULLSTR;
		static char *	down = "down";

		switch ( NNstate.procstate )
		{
		default:
		case PROC_IDLE:		cp = "idle";	break;
		case PROC_RUNNING:	cp = "active";	break;
		case PROC_ERROR:	cp = "waiting for error to be fixed";
					ls = down;
					break;
		case PROC_OPENING:	cp = "waiting for open on device";
					ls = down;
					break;
		case PROC_CALLING:	cp = "waiting for connection";
					ls = down;
		}

		if ( ls == NULLSTR )
			switch ( NNstate.linkstate )
			{
			default:
			case LINK_DOWN:		ls = down;	break;
			case LINK_UP:		ls = "up";
			}

		Fprintf
		(
			stdout,
			"%s daemon (%d) %s, link %s. Time: %.8s, up: %s.\n",
			link,
			NNstate.procid,
			cp,
			ls,
			ctime(&NNstate.thistime)+11,
			TimeString(t)
		);
	}
	else
	if ( t != 0 )
	{
		Fprintf
		(
			stdout,
			"%s daemon inactive since %.15s, was up for %s.\n",
			link,
			ctime(&NNstate.thistime)+4,
			TimeString(t)
		);
	}
	else
	{
		if ( NLinks == 1 )
			Fprintf(stdout, "%s daemon inactive.\n", link);
		return;
	}

	{
		register ulong	avg;

		NNstate.allbytes += inByteCount + outByteCount;

		avg = NNstate.activetime ? NNstate.allbytes/NNstate.activetime : (ulong)0;

		Fprintf
		(
			stdout,
			"%s  (bytes/sec.)%7lu%9lu%9lu%9lu%9lu%10lu\n",
			"\n  throughputs:  assumed  maximum  average  overall  receive  transmit\n",
			NNstate.speed,
			NNstate.maxspeed > avg ? NNstate.maxspeed : avg,
			avg,
			t ? NNstate.allbytes/t : (long)0,
			j ? inByteCount/j : (long)0,
			j ? outByteCount/j : (long)0
		);
	}

	Fprintf
	(
		stdout,
		"\n  protocol parameters: buffers %d, data size %3d, receive timeout %2d.\n",
		NNstate.nbufs,
		NNstate.packetsize,
		NNstate.recvtimo
	);

	Fprintf
	(
		stdout,
		"%s  %25ld%13ld%10ld%9ld\n",
		"\n  active totals: packets in  packets out     bytes   messages\n",
		NNstate.inpkts,
		NNstate.outpkts,
		NNstate.allbytes,
		NNstate.allmessages
	);

	Fprintf
	(
		stdout,
		"\n  streams:     range           state    message    size     bytes  messages\n"
	);

	for ( strp = NNstate.streams[0], i = 0 ; i < 2 ; i++ )
	{
		if ( i )
			putc('\n', stdout);

		for ( j = 0 ; j < NSTREAMS ; j++, strp++ )
		{
			Fprintf
			(
				stdout,
				"%10s%7ld..",
				j ? "" : i==INSTREAMS?"receive ":"transmit",
				j?StreamSizes[j-1]:(long)0
			);

			if ( j == (NSTREAMS-1) )
				Fprintf
				(
					stdout,
					"%-6s",
					"."
				);
			else
				Fprintf
				(
					 stdout,
					"%-6ld",
					StreamSizes[j]-1
				);

			switch ( strp->str_state )
			{
			default:
			case STR_ENDED:
			case STR_IDLE:		cp = "idle";		break;
			case STR_WAIT:		cp = "waiting";		break;
			case STR_START:		cp = "starting";	break;
			case STR_ACTIVE:	cp = "active";		break;
			case STR_ENDING:	cp = "terminating";	break;
			case STR_EMPTY:		cp = "empty";		break;
			case STR_ERROR:		cp = "error";		break;
			case STR_INACTIVE:	cp = "inactive";	break;
			}

			switch ( strp->str_state )
			{
			case STR_WAIT:
			case STR_START:
			case STR_ACTIVE:
			case STR_ENDING:
				if ( strp->str_size > 0 )
				{
					Fprintf
					(
						stdout,
						"%11s%8d%%%10ld",
						cp,
						(int)(strp->str_posn * 100 / strp->str_size),
						strp->str_size
					);
					break;
				}

			default:
				Fprintf(stdout, "%11s%19s", cp, "");
			}

			Fprintf
			(
				stdout,
				"%10ld%10ld\n",
				strp->str_bytes,
				strp->str_messages
			);
		}
	}

	if ( NLinks > 1 )
		putc('\n', stdout);
}



void
ShortStats(link)
	char *		link;
{
	register Str_p	strp;
	register char	c;
	register int	i;
	register int	j;
	register char	ls = ' ';

	if
	(
		NNstate.procid
		&&
		(
			kill(NNstate.procid, SIG0) != SYSERROR
			||
			errno == EPERM
		)
	)
	{

		switch ( NNstate.procstate )
		{
		default:
		case PROC_IDLE:		c = 'I';	break;
		case PROC_RUNNING:	c = 'A';	break;
		case PROC_ERROR:	c = 'E';
					ls = 'D';
					break;
		case PROC_OPENING:	c = 'O';
					ls = 'D';
					break;
		case PROC_CALLING:	c = 'C';
					ls = 'D';
					break;
		}

		if ( ls == ' ' )
			switch ( NNstate.linkstate )
			{
			default:
			case LINK_DOWN:		ls = 'D';	break;
			case LINK_UP:		ls = 'U';;
			}

	}
	else
	{
		c = '*';
		ls = '*';
	}

	Fprintf
	(
		stdout,
		"%-*.*s %c%c %10ld %8ld",
		NODE_NAME_SIZE, NODE_NAME_SIZE, link,
		c,
		ls,
		NNstate.allbytes,
		NNstate.allmessages
	);

	for ( strp = NNstate.streams[0], i = 0 ; i < 2 ; i++ )
	{
		putc(' ', stdout);

		for ( j = 0 ; j < NSTREAMS ; j++, strp++ )
		{
			switch ( strp->str_state )
			{
			default:
			case STR_ACTIVE:
			case STR_ENDED:
			case STR_IDLE:		c = ' ';	break;

			case STR_EMPTY:		c = '_';	break;
			case STR_ENDING:	c = '~';	break;
			case STR_ERROR:		c = 'E';	break;
			case STR_INACTIVE:	c = '.';	break;
			case STR_START:		c = '+';	break;
			case STR_WAIT:		c = ',';	break;
			}

			switch ( strp->str_state )
			{
			case STR_WAIT:
			case STR_START:
			case STR_ACTIVE:
			case STR_ENDING:
				if ( strp->str_size > 0 )
				{
					Fprintf
					(
						stdout,
						" %c%3d%%",
						c,
						(int)(strp->str_posn * 100 / strp->str_size)
					);
					break;
				}

			default:
				Fprintf(stdout, "    %c ", c);
			}
		}
	}

	putc('\n', stdout);

	if ( NLinks < 10 )
		putc('\n', stdout);
}



void
ShortHeader()
{
	register int	i;
	register int	j;
	register char	*cp;

	cp = "Receive Channels";
	i = strlen(cp);
	Fprintf(stdout, "%*s%s", NODE_NAME_SIZE+24+((6*NSTREAMS)-i)/2, "", cp);
	cp = "Transmit Channels";
	Fprintf(stdout, "%*s%s\n", ((2*6*NSTREAMS)-i-strlen(cp)+1)/2, "", cp);

	Fprintf
	(
		stdout,
		"%-*s    %10s %8s",
		NODE_NAME_SIZE, "Link",
		"Bytes",
		"Messages"
	);

	for ( i = 0 ; i < 2 ; i++ )
	{
		putc(' ', stdout);

		for ( j = 0 ; j < NSTREAMS ; j++ )
			Fprintf(stdout, "  %3d ", j);
	}

	putc('\n', stdout);
	putc('\n', stdout);
}



/*
**	Queue a network link directory.
*/

void
ListLink(path, link, ino)
	char *		path;
	char *		link;
	ino_t		ino;
{
	register LLl_p	hllp;

	hllp = Talloc(LLel);

	hllp->lql_link = concat(path, link, NULLSTR);
	hllp->lql_ino = ino;

	hllp->lql_next = LinkList;
	LinkList = hllp;

	NLinks++;

	Trace2(1, "list link \"%s\"", hllp->lql_link);
}



/*
**	Search spooldir recursively for link directories
*/

bool
FindLinks(path)
	char *				path;
{
	register DIR *			dirp;
	register struct direct *	direp;
	bool				found = false;

	Trace2(1, "find links in \"%s\"", path);

	if ( (dirp = opendir(path)) == NULL )
	{
		Syserror("cannot read %s", path);
		return found;
	}

	while ( (direp = readdir(dirp)) != NULL )
	if ( direp->d_name[0] != WORKFLAG && !dots(direp->d_name) )
	{
		register LLl_p	hllp;
		struct stat	dstat;
		char *		newpath;

		Trace2(2, "entry \"%s\"", direp->d_name);

		for ( hllp = LinkList ; hllp != (LLl_p)0 ; hllp = hllp->lql_next )
			if ( hllp->lql_ino == direp->d_ino )
				break;

		newpath = concat(path, direp->d_name, Slash, NULLSTR);

		if
		(
			hllp == (LLl_p)0
			&&
			stat(newpath, &dstat) != SYSERROR
			&&
			(dstat.st_mode & S_IFMT) == S_IFDIR
		)
		{
			char *	newname;

			newname = newstr(direp->d_name);	/* direp re-used */

#			ifdef	MULTIPATH
			if ( !FindLinks(newpath) )
#			endif	MULTIPATH
			{
				ListLink(path, newname, direp->d_ino);
				found = true;
			}

			free(newname);
		}

		free(newpath);
	}

	closedir(dirp);

	return found;
}



/*
**	Sort the links into alphabetic order
*/

void
SortLinks()
{
	register LLl_p	lp;
	register LLl_p *lpp;

	LinkVec = lpp = (LLl_p *)Malloc(sizeof(LLl_p) * NLinks);

	for ( lp = LinkList ; lp != (LLl_p)0 ; lp = lp->lql_next )
		*lpp++ = lp;

	DODEBUG(if((lpp-LinkVec)!=NLinks)Fatal1("bad NLinks"));

	Trace2(1, "found %d links", NLinks);

	if ( NLinks > 1 )
		qsort((char *)LinkVec, NLinks, sizeof(LLl_p), compare);
}



/*
**	Alpha compare
*/

int
compare(lpp1, lpp2)
	char *	lpp1;
	char *	lpp2;
{
	return strcmp((*(LLl_p *)lpp1)->lql_link, (*(LLl_p *)lpp2)->lql_link);
}
