/*
**	Dis -- VDU page display program
**
**	"dis [-t<timeout>] [-c<refresh count>] [-u] [-{xX}] [-T<limittime>]"
**
**	Bugs and comments to:	Piers Lauder
**				Dept of Comp Sci
**				Sydney University
**	May '80.
**
**	New options:
**	   -x	The first time through text is displayed in standout.
**		From then on it is displayed in normal video.
**		This allows you to see easily whether stuff has changed.
**	   -X	The same, except changes in standout, first time in normal.
**	   -Tn	Timelimit in minutes.  No or 0 time means no time limit.
**		There is now a default timelimit of 2 hours.
**		This avoids folks forgetting & leaving stuff slowing the
**		machine down.
**					... Brian Coogan 13 Dec 85
*/

/*
#include	<local-system>
*/
#include	<signal.h>
#include	<setjmp.h>
#include	<sgtty.h>
#include	<stdio.h>

/*
**	Parameters
*/

#define		MAXWID		132
#define		MAXLEN		80


/*
**	Screen buffer
*/

short *		Buf;

/*
**	Term Cap
*/

char		*CM, *CL;
char		*SO, *SE;
short		amflag;

short		somovesok;			/* can move in standout */
#define		SO_SET		0x200		/* SO set on/off here */
#define		SO_ON		0x100		/* SO is on here */
#define		so(x)		(x ? (SO_SET|SO_ON) : SO_SET)
#define		is_so(x)	(x & SO_ON)	/* is standout on */
#define	so_bit(x)	(x & 0xFF00)
#define	ch_bit(x)	(x & 0xFF)

#ifdef	hpux
#define ti_extern	/* these are not externs for HP-UX sysV */
#else
#define	ti_extern extern
#endif
ti_extern short	ospeed;
ti_extern char	PC, *BC, *UP;

extern char *	tgoto();
extern char *	tgetstr();

/*
**	Screen macro
*/

#define		putcm(cp,p,c)		if((*cp++&0xff)!=c){\
						if(move((--cp)-p))\
							putc(c,stdout);\
						*cp++ =ch_bit(c)|standon;\
					}

/*
**	Miscellaneous
*/

jmp_buf		alrmbuf;
short		width;
short		Width;		/* width - 1 */
short		length;
short		Length;		/* length - 1 */
char *		name;
unsigned	timeout;
unsigned	toolong = 2 * 3600;	/* Time to end */
short		rcount;
short		standon;		/* True when writing in standout */

void
		tcinit(),
		dis(),
		warn(),
		terror(),
		outc();

int		alrmcatch();
int		intcatch();

/*
**	Externals
*/

int		standout = 0;
int		inverse = 0;

extern char	_sobuf[];
extern char *	tgetstr();
extern char *	malloc();
extern char *	getenv();



main(argc, argv)
	register int		argc;
	register char *		argv[];
{
	register unsigned	size;

	name = *argv++;
	argc--;

	while ( argc )
	{
		switch( argv[0][0] )
		{
		 case '-':	argv[0]++;
				continue;
		 case 'X':	standout++;
				inverse++;
				break;
		 case 'x':	standout++;
				break;
		 case 'T':	toolong = atoi(&argv[0][1])*60;
				break;
		 case 't':	timeout = atoi(&argv[0][1]);
				break;
		 case 'c':	rcount = atoi(&argv[0][1]);
				break;
		 case 'u':	setbuf(stdin, NULL);
				break;
		 default:	fprintf(stderr, "%s: bad arg - %s\n", name, argv[0] );
				return 1;
		}
		argc--;
		argv++;
	}

	setbuf(stdout, _sobuf);
	tcinit();
	size = length * width;
	if ((Buf = (short *)malloc(size * sizeof(*Buf))) == (short *)0)
	{
		fprintf(stderr, "No memory\n");
		exit(2);
	}
	strclr(Buf, size * sizeof(*Buf));

	Length = length - 1;
	Width = width - 1;
	dis(size);

	return 0;
}




void
dis(size)
	unsigned		size;
{
	register short *	ep;
	register short *	p = Buf;
	register int		c;
	register int		line;
	/** on stack to avoid setjmp **/
	short *			cp;
	short *			lastend;
	int			rc;
	extern long		time();
	long			starttime = time((long *)0);

	lastend = p;
	rc = rcount;
	if (standout && is_so((standon = so(! inverse))))
	{
		/*
		 *	If inverse is set, the first time thru we
		 *	paint the screen normally, then inverse from
		 *	then on.
		 */
		
		tputs(SO, 1, outc);
		signal(SIGINT, intcatch);
	}
	if (! timeout && toolong)
	{
		signal(SIGALRM, alrmcatch);
		alarm(toolong);
	}

	do
	{
		line = 0;
		cp = p;
		ep = cp+width;

		if ( /* timeout == 0 || */ setjmp(alrmbuf) == 0 )
		{
			if ( timeout )
			{
				signal(SIGALRM, alrmcatch);
				alarm(timeout);
				standon = so(inverse);
			}
			if (toolong && time(0) - starttime >= toolong)
			{
				fprintf(stderr, "\r\n%s: Time-out\n", name);
				exit(0);
			}
			while ( (c = getchar()) != EOF )
			{
				if ( rcount && !rc )
				{
					tputs(CL, 1, outc);
					strclr(p, size);
					rc = rcount;
					standon = so(! inverse);
				}
				if ( c < ' ' )
				{
					switch ( c )
					{
					 case '\f':
							if ( cp != p )
								break;
							continue;
					 case '\t':
							c = cp - &p[line*width] + 1;
							putcm(cp, p, ' ');
							while ( c++&7 )
								putcm(cp, p, ' ');
							continue;
					 case '\n':
							while ( cp < ep )
								putcm(cp, p, ' ');
							if ( line < Length )
							{
								cp = &p[(++line)*width];
								ep = cp+width;
							}
							continue;
					 default:
							if ( cp < ep )
								putcm(cp, p, '?');
							continue;
					}
				}
				else
				{
					if ( cp < ep )
						putcm(cp, p, c);
					continue;
				}
				break;
			}
			if ( timeout )
				alarm(0);
		}
		if (standout && is_so(standon) && somovesok)
		{
			tputs(SE, 1, outc);
			signal(SIGINT, SIG_DFL);
		}
		standon = so(inverse);
		ep = cp - 1;
		while ( cp < lastend )
			putcm(cp, p, ' ');
		lastend = ep;
		if ( (line = (ep-p)/width) < Length )
			line++;
		(void)move(line*width);
		fflush(stdout);
		if ( rcount )
			--rc;
	}
	while
		( c != EOF );
}




int
move(pos)
	register int	pos;
{
	register int	x = pos%width;
	register int	y = pos/width;
	register int	i;
	static int	oy, ox = -1;

	if ( oy == y )		/* same line */
	{
		if ( (i = x - ox) != 1 )	/* not next char */
		{
			soend(oy*width+ox);
			if ( i <= 3 && i > 0 )	/* only a few chars on */
			{
				i--;
				pos -= i;
				do
					putc(Buf[pos++], stdout);
				while
					( --i > 0 );
			}
			else		/* more than a few chars or back */
				tputs(tgoto(CM, x, y), 1, outc);
			socheck(pos);
		}
		else
			socheck(pos);
	}
	else
	{
		soend(oy*width+ox);
		if ( oy == (y-1) && x == 0 )
		{
			if ( ox != Width || !amflag )
				putc('\n', stdout);
		}
		else
			tputs(tgoto(CM, x, y), oy<y?y-oy:oy-y, outc);
		socheck(pos);
	}

	ox = x; oy = y;
	if ( y==Length && x==Width && amflag )
		return 0;
	return 1;
}




int
alrmcatch()
{
	longjmp(alrmbuf, 1);
}




void
tcinit()
{
	register char *	cp;
	struct sgttyb	gb;
	char		bp[1024];
	static char	buf[100];
	char		*area = buf;

	if ( tgetent(bp, getenv("TERM")) != 1 )
		terror("no \"termcap\" entry");
	if ( (CL = tgetstr("cl", &area)) == (char *)0 )
		terror("not VDU");
	if ( (CM = tgetstr("cm", &area)) == (char *)0 )
		terror("no cursor addressing");
	UP = tgetstr("up", &area);
	BC = tgetstr("bc", &area);
	SO = tgetstr("so", &area);
	SE = tgetstr("se", &area);
	if (SO == (char *)0 || SE == (char *)0)
		standout = standon = 0;
	if ( tgetflag("am") == 1 )
		amflag++;
	somovesok =  ! tgetflag("xs") && tgetflag("mi");
	if ( (cp = getenv("WIDTH")) != (char *)0 )
		width = atoi(cp);
	else
		width = tgetnum("co");
	if ( width > MAXWID )
	{
		width = MAXWID;
		warn("width truncated");
	}
	if ( (length = tgetnum("li")) > MAXLEN )
	{
		length = MAXLEN;
		warn("length truncated");
	}
	if ( (cp = tgetstr("pc", &area)) != (char *)0 )
		PC = *cp;
	gtty(1, &gb);
	ospeed = gb.sg_ospeed;

	tputs(CL, 1, outc);
	fflush(stdout);
}




void
outc(c)
{
	putc(c, stdout);
}




void
warn(s)
	char *	s;
{
	fprintf(stderr, "Warning: %s\n", s);
	sleep(2);
}




void
terror(s)
	char *	s;
{
	fprintf(stderr, "Terminal capability error - %s\n", s);
	exit(1);
}

strclr(p, n)
char	*p;
int	n;
{
	int	i;
	for (i = 0; i < n; i++)
		*p++ = '\0';
}


intcatch()
{
	signal(SIGINT, SIG_IGN);
	tputs(SE, 1, outc);
	exit(0);
}


/*
 *	Check that the standout in the current char position is
 *	equal to standon.  If it's not then change it.
 */
socheck(pos)
int	pos;
{
	register int	val;
	register int	set;
	register short	*cp = &Buf[pos];
	static int	setpos = -1;

	if (somovesok || ! standout)
		return;

	val = so_bit(*cp);
	if (val == standon)
		return;

	/*
	 *	If nothing set at this pos but it was earlier on this
	 *	line and we haven't unset from here since then,
	 *	then we don't need to reset.  This is a clumsy fix.
	 */
	if (! (val & SO_SET) && setpos < pos && setpos/width == pos/width)
		return;

	setpos = pos;

	sofix(standon);

	*cp = ch_bit(*cp) | standon;
}


/*
 *	Send an end standout if we have changed standout
 *	since the last move.
 *	We can tell if we changed standout because the next character will
 *	be different from the current mode which we have already turned on.
 */
soend(pos)
int	pos;
{
	register short	*cp = &Buf[pos+1];

	if (! standout || somovesok || pos < 0)
		return;
	if (standon != so_bit(*cp))
		sofix(so_bit(*cp));
}


sofix(standmode)
int	standmode;
{
	if (is_so(standmode))
		tputs(SO, 1, outc);
	else
		tputs(SE, 1, outc);
}
