/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)linkstats.c	1.19 85/11/08";

/*
**	Print out statistics for each link based on contents of state files.
*/

char *	Usage		= "[-V] [-T<level>] [-{c|s}[<sleep>]] [link|node|alias ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"
#if AUSAS
#define AUSAM
#endif

#include	"command.h"
#include	"debug.h"
#include	"spool.h"
#include	"state.h"
#undef	Extern
#define	Extern
#include	"Stream.h"

#include	<errno.h>

#include	<signal.h>

#ifdef AUSAM
#include <udp.h>
#else
#include <sys/socket.h>
#include <netinet/in.h>
#define	gethtname	gethostbyname
#define	getsvname	getservbyname
#define	gethtaddr	gethostbyaddr
#endif
#include <netdb.h>

/*
**	Filenames that must be set up
*/

char *	Slash		= "/";		/* Path name element separator */

/*
**	Parameters set from arguments
*/

char *	Name;				/* Program invoked name */
int	SleepTime;			/* Delay for continuous mode scans */
bool	Continuous;			/* Keep scanning status files */
bool	Verbose;			/* Tell everything */
int	Traceflag;			/* Trace level */

/*
**	List for found links.
*/
typedef struct LLhl *	LLh_p;

typedef struct LLel *	LLl_p;

typedef struct LLel
{
	LLl_p	lql_next;
	char *	lql_link;
	char *	lql_host;
	LLh_p	lql_ht;
	int	lql_received;
	NN_state lql_state;
}
	LLel;

LLl_p	LinkList;			/* Head of linked list */
LLl_p *	LinkVec;			/* Vector of sorted links */
int	NLinks;				/* No. of links found */
int	NRecv;				/* number of links for which status has been received */
int	NHosts;				/* number of different hosts */

/*
**	List of hosts so we can complain of unresponsive hosts
*/

typedef struct LLhl
{
	LLh_p	hl_next;
	char *	hl_host;
	int	hl_reply;
	char *	hl_links;
#ifdef	AUSAM
	in_addr	hl_addr;
#else
	long	hl_addr;
#endif
}
	LLhl;

LLh_p	HostList;			/* Head of list of hosts */


/*
**	Miscellaneous info.
*/

NodeLink LinkD;				/* Info. from FindNode() */

long	StreamSizes[NSTREAMS] =
{
	 SMALL_MESSAGE
	,MEDIUM_MESSAGE
	,LARGE_MESSAGE
};

int	sock;
#ifdef	AUSAM
udport	peer;
#else
struct sockaddr_in peer;
#endif
int	GotHosts = 0;

#define	Fprintf		(void)fprintf
#define	INDENT(A)	Fprintf(stdout,"%*s",A*2,"")
#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))

/*
**	Routines
*/

int	compare();
int	getpacket();
void	AddHost();
LLh_p	GotHost();
void	LinkStats(), LongStats(), ShortStats(), ListLink(), SortLinks();
void	ShortHeader(), usage();
bool	FindLinks();


time_t now;

int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	char		buf[BUFSIZ];
	struct	servent	*se;
	int st;
	LLh_p hp;


	ignore(SIGALRM);

	se = getsvname("acsstats", "udp");
	if (se == NULL) Error("No service: acsstats");
#ifdef	AUSAM
	{ short port;
	sock = open("/dev/inet/udp", 2);
	port = 0;
	ioctl(sock, NIOC_BIND, &port);
	signal(SIGUSER1, getpacket);
	port = SIGUSER1;
	ioctl(sock, NIOC_ALERT, &port);
	peer.udp_port = se->s_port;
	}
#else
	sock = socket(AF_INET, SOCK_DGRAM, 0);
	bzero(&peer, sizeof(peer));
	peer.sin_family = AF_INET;
	bind(sock, &peer, sizeof(peer));
	signal(SIGIO, getpacket);
	fcntl(sock, F_SETFL, FASYNC|FNDELAY);
	fcntl(sock, F_SETOWN, getpid());
	bzero(&peer, sizeof(peer));
	peer.sin_family = AF_INET;
	peer.sin_port = se->s_port;
#endif



	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					continue;

				case 'c':
				case 's':
					Continuous = true;
					if ( (SleepTime = atoi(++*argv)) <= 0 )
						SleepTime = 10;
					break;

				default:
					usage("unrecognised flag '%c'", c);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			register char *link, *host;
			struct hostent *he;
			char *msg;

			link = *argv;
			host = strchr(link,':');
			if (host == NULLSTR) host = strchr(link,'@');
			if (host != NULLSTR) *host++ = '\0';

			AddHost(host, *link? link: NULLSTR);
		}
	}

	if (HostList == NULL)
		AddHost(NULLSTR,NULLSTR);


	ignore(SIGALRM);
	time(&now);
	do
	{
		alarm(10);
		pause();
		for (hp = HostList ; hp ; hp = hp->hl_next)
			if (hp->hl_reply) break;
	} while( hp != NULL && time(0L) < now+10);
	alarm(0);

	st=0;
	for (hp = HostList ; hp ; hp = hp->hl_next)
		if (hp->hl_reply)
			fprintf(stderr,"No reply from %s\n", hp->hl_host), st++;

	/* no longer interested in HostList or linklist: replies */
	GotHosts = 1;

	if ( NLinks == 0 )
	{
		Error("no network links found!");
		return 1;
	}

	SortLinks();

	setbuf(stdout, buf);

	for ( ;; )
	{
		register int	i;
		LLh_p hp;

		if ( !Verbose )
			ShortHeader();

		NRecv = 0;
		for (hp=HostList ; hp ; hp=hp->hl_next)
		{
			char *msg;

			msg = concat("linkstats", hp->hl_links, NULLSTR);
#ifdef AUSAM
			peer.udp_host = hp->hl_addr;
			ioctl(sock, NIOC_SEND, &peer);
			write(sock, msg, strlen(msg));
#else
			bcopy(&hp->hl_addr, &peer.sin_addr, sizeof(peer.sin_addr));
			sendto(sock, msg, strlen(msg), 0, &peer, sizeof(peer));
#endif
			free(msg);
		}
		ignore(SIGALRM);
		time(&now);
		do {
			alarm(10);
			pause();
		} while (NRecv < NLinks && time(0L) < now+10);
		alarm(0);

		for ( i = 0 ; i < NLinks ; i++ )
			LinkStats(LinkVec[i]);

		if ( Continuous )
		{
			if (!Verbose)
				putc('\n', stdout);
			putc('\f', stdout);
			(void)fflush(stdout);
			(void)sleep(SleepTime);
		}
		else
			break;
	}

	(void)fflush(stdout);

	return 0;
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	Fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}

/*
**	Add an entry to the host list
*/
void
AddHost(host, link)
char *host;
char *link;
{
	int cmp = -1;
	LLh_p *hp;
	char *msg;
	
	if (host == NULLSTR)
	{
		FILE *fp;
		char buf[512];

		fp = fopen(INSLIB(spoolareas), "r");
		if (fp == NULL) AddHost("localhost", link);
		else
		while( fgets(buf, sizeof(buf), fp)!= NULLSTR)
		{
			char * cp;
			cp = strchr(buf, ':');
			if (cp == NULL) continue;
			*cp = '\0';
			AddHost(buf, link);
		}
		fclose(fp);
		return;
	}
	for (hp = &HostList ; *hp ; hp = & (*hp)->hl_next)
		if ((cmp=hostcmp((*hp)->hl_host, host))>=0)
			break;
	if (cmp != 0)
	{
		LLh_p np;
		struct hostent *he;

		he = gethtname(host);
		if (he == NULL) Error("Unknown host %s", host);
		if (hostcmp(host, he->h_name)!= 0)
		{
		    char *cp = concat(he->h_name, NULLSTR);
		    AddHost(cp, link);
		    free(cp);
		    return;
		}
		np = (LLh_p)malloc(sizeof(LLhl));
		np->hl_next = *hp;
		*hp = np;
		np->hl_host = concat(host, NULLSTR);
		np->hl_links = NULLSTR;
		np->hl_reply = 0;
#ifdef AUSAM
		np->hl_addr = he->h_addr;
#else
		bcopy(he->h_addr, &np->hl_addr, he->h_length);
#endif
		NHosts ++;
	}
	(*hp)->hl_reply ++;

	if (link == NULLSTR) msg = concat("linklist", NULLSTR);
	else msg = concat("linklist ", link, NULLSTR);
#ifdef AUSAM
	peer.udp_host = (*hp)->hl_addr;
	ioctl(sock, NIOC_SEND, &peer);
	write(sock, msg, strlen(msg));
#else
	bcopy(&(*hp)->hl_addr, &peer.sin_addr, sizeof(peer.sin_addr));
	sendto(sock, msg, strlen(msg), 0, &peer, sizeof(peer));
#endif
	free(msg);
}

/*
**	Recrease reply count for host
*/
LLh_p
GotHost(host)
char *host;
{
	int cmp = -1;
	LLh_p *hp;
	
	for (hp = &HostList ; *hp ; hp = & (*hp)->hl_next)
		if ((cmp=hostcmp((*hp)->hl_host, host))>=0)
			break;
	if (cmp != 0)
	{
		return NULL;
	}
	(*hp)->hl_reply --;
	return *hp;
}

/*
**	Get a packet from an acsstats server and deal with it
*/
getpacket(sig)
int sig;
{
	int siz;
	char mesg[1400], *host, *link;
	struct hostent *he;

#ifdef	AUSAM
	signal(sig,SIG_IGN);
	peer.udp_port = 0;
	peer.udp_fill = 0;
	while (ioctl(sock, NIOC_RECV, &peer) == 0 && peer.udp_port != 0)
	{
		siz = read(sock, mesg, sizeof(mesg));
		he = gethtaddr(peer.udp_host);
#else
	int peerlen = sizeof(peer);
	signal(sig,SIG_IGN);
	while ((siz = recvfrom(sock, mesg, sizeof(mesg), 0, &peer, &peerlen)) != -1)
	{
		he = gethostbyaddr(&peer.sin_addr, 4, AF_INET);
#endif
		if (he == NULL)
		{
#ifdef AUSAM
			Trace2(1,"Packet from unknown host %lx\n", peer.udp_host);
#else
			Trace2(1,"Packet from unknown host %lx\n", peer.sin_addr);
#endif
			continue;
		}
		host = he->h_name;

		mesg[siz] = '\0';
		Trace3(1,"Message <%s> from %s\n",mesg,host);
		if (!GotHosts)
		{
			if (strncmp(mesg, "linklist:", 9)==0)
			{
				char *lp, *ep;
				LLh_p hp;
				hp = GotHost(host);
				if (hp)
				for (lp = mesg+9 ; *lp ; lp = ep)
				{
					while (*lp == ' ') lp++;
					ep = lp;
					while (*ep && *ep!=' ') ep++;
					if (*ep) *ep++ = '\0';
					if (*lp) ListLink(host, lp, hp);
				}
			}
			if (strncmp(mesg, "badlink:", 8)==0)
				fprintf(stderr, "No such link: %s on %s\n", mesg+8, host);
		}
		else
		if (strncmp(mesg, "linkstats:", 10)==0)
		{
			int d,st;
			char *cp;
			link = mesg+11;
			strclr(&NNstate, sizeof(NNstate));
			if (siz > 15)
			{
			int l= strlen(link) + 11;
			l = (l+4) & ~3;
			cp = mesg + l;


#define l(x)	NNstate.x = ntohl(*((long*)cp)); cp += sizeof(long)
#define	s(x)	NNstate.x = ntohs(*((short*)cp)); cp += sizeof(short)
#define b(x)	NNstate.x = *cp++
	b(linkstate); b(procstate);
	s(procid);
	l(speed); l(maxspeed);
	s(packetsize); s(nbufs); s(recvtimo); cp+= 2;
	l(inbytes); l(outbytes), l(lasttime); l(allbytes); l(allmessages);
	l(starttime); l(thistime); l(activetime); l(inpkts); l(outpkts);
	for (st=0; st<NSTREAMS ; st++)
	for (d=0; d<2 ; d++)
	{
#define ll(x)	l(streams[d][st].str_/**/x)
#define ss(x)	s(streams[d][st].str_/**/x)
#define bb(x)	b(streams[d][st].str_/**/x)
		ll(size); ll(posn); ll(bytes); ll(messages); ll(time);
		ss(fd);
		bb(state); bb(flags);
	}
	for (d=0 ; d<8 ; d++) b(version[d]);
#undef l
#undef s
#undef b
#undef ll
#undef ss
#undef bb

			}
			/* find the link entry */
			for (d=0; d<NLinks ; d++)
			if (strcmp(link, LinkVec[d]->lql_link)== 0
			   && hostcmp(host, LinkVec[d]->lql_host)==0
			   ) break;
			if (d < NLinks)
			{
				LinkVec[d]->lql_received++;
				memcpy(&LinkVec[d]->lql_state, &NNstate, sizeof(NNstate));
				NRecv ++;
				Trace5(2,"Got stats %s@%s %d/%d\n", link, host, NRecv, NLinks);
			}
		}
	}
	signal(sig, getpacket);
}


/*
**	Print stats from state file in directory.
*/

void
LinkStats(link)
	LLl_p	link;
{
	register int	i;
	register int	j;
	static Time_t	lasttime;

	Trace3(1, "link stats from \"%s@%s\"", link->lql_link, link->lql_host);

	memcpy(&NNstate, &link->lql_state, sizeof(NNstate));
	if (link->lql_received == 0) strclr(&NNstate, sizeof(NNstate));
	link->lql_received = 0;
	if ( strcmp(NNstate.version, StreamSCCSID) != STREQUAL )
		if ( NLinks == 1 )
			Warn("\"status\" file version mismatch");
		else
			(void)strclr((char *)&NNstate, sizeof NNstate);


	if ( Continuous && Verbose && NLinks == 1 )
	{
		if ( NNstate.thistime == lasttime )
			return;
		
		lasttime = NNstate.thistime;
	}


	if ( Verbose )
		LongStats(link->lql_link, link->lql_host);
	else
		ShortStats(link->lql_link, link->lql_host);
}



void
LongStats(link, host)
	char *		link;
	char *		host;
{
	register Str_p	strp;
	register char *	cp;
	register int	i;
	register int	j;
	register ulong	t;

	if  ( NNstate.starttime == 0 )
	{
		t = 0;
		j = 0;
	}
	else
	{
		if ( (t = NNstate.thistime - NNstate.starttime) == 0 )
			t = 1;
		j = NNstate.thistime - NNstate.lasttime;
	}

	if
	(
		NNstate.procid
	)
	{
		register char *	ls = NULLSTR;
		static char *	down = "down";

		switch ( NNstate.procstate )
		{
		default:
		case PROC_IDLE:		cp = "idle";	break;
		case PROC_RUNNING:	cp = "active";	break;
		case PROC_ERROR:	cp = "waiting for error to be fixed";
					ls = down;
					break;
		case PROC_OPENING:	cp = "waiting for open on device";
					ls = down;
					break;
		case PROC_CALLING:	cp = "waiting for connection";
					ls = down;
		}

		if ( ls == NULLSTR )
			switch ( NNstate.linkstate )
			{
			default:
			case LINK_DOWN:		ls = down;	break;
			case LINK_UP:		ls = "up";
			}

		Fprintf
		(
			stdout,
			"%s%s%s daemon (%d) %s, link %s. Time: %.8s, up: %s.\n",
			link,
			(NHosts>1)? "@" : "",
			(NHosts>1)? host : "",
			NNstate.procid,
			cp,
			ls,
			ctime(&NNstate.thistime)+11,
			TimeString(t)
		);
	}
	else
	if ( t != 0 )
	{
		Fprintf
		(
			stdout,
			"%s daemon inactive since %.15s, was up for %s.\n",
			link,
			ctime(&NNstate.thistime)+4,
			TimeString(t)
		);
	}
	else
	{
		if ( NLinks == 1 )
			Fprintf(stdout, "%s daemon inactive.\n", link);
		return;
	}

	{
		register ulong	avg;

		NNstate.allbytes += inByteCount + outByteCount;

		avg = NNstate.activetime ? NNstate.allbytes/NNstate.activetime : (ulong)0;

		Fprintf
		(
			stdout,
			"%s  (bytes/sec.)%7lu%9lu%9lu%9lu%9lu%10lu\n",
			"\n  throughputs:  assumed  maximum  average  overall  receive  transmit\n",
			NNstate.speed,
			NNstate.maxspeed > avg ? NNstate.maxspeed : avg,
			avg,
			t ? NNstate.allbytes/t : (long)0,
			j ? inByteCount/j : (long)0,
			j ? outByteCount/j : (long)0
		);
	}

	Fprintf
	(
		stdout,
		"\n  protocol parameters: buffers %d, data size %3d, receive timeout %2d.\n",
		NNstate.nbufs,
		NNstate.packetsize,
		NNstate.recvtimo
	);

	Fprintf
	(
		stdout,
		"%s  %25ld%13ld%10ld%9ld\n",
		"\n  active totals: packets in  packets out     bytes   messages\n",
		NNstate.inpkts,
		NNstate.outpkts,
		NNstate.allbytes,
		NNstate.allmessages
	);

	Fprintf
	(
		stdout,
		"\n  streams:     range           state    message    size     bytes  messages\n"
	);

	for ( strp = NNstate.streams[0], i = 0 ; i < 2 ; i++ )
	{
		if ( i )
			putc('\n', stdout);

		for ( j = 0 ; j < NSTREAMS ; j++, strp++ )
		{
			Fprintf
			(
				stdout,
				"%10s%7ld..",
				j ? "" : i==INSTREAMS?"receive ":"transmit",
				j?StreamSizes[j-1]:(long)0
			);

			if ( j == (NSTREAMS-1) )
				Fprintf
				(
					stdout,
					"%-6s",
					"."
				);
			else
				Fprintf
				(
					 stdout,
					"%-6ld",
					StreamSizes[j]-1
				);

			switch ( strp->str_state )
			{
			default:
			case STR_ENDED:
			case STR_IDLE:		cp = "idle";		break;
			case STR_WAIT:		cp = "waiting";		break;
			case STR_START:		cp = "starting";	break;
			case STR_ACTIVE:	cp = "active";		break;
			case STR_ENDING:	cp = "terminating";	break;
			case STR_EMPTY:		cp = "empty";		break;
			case STR_ERROR:		cp = "error";		break;
			case STR_INACTIVE:	cp = "inactive";	break;
			}

			switch ( strp->str_state )
			{
			case STR_WAIT:
			case STR_START:
			case STR_ACTIVE:
			case STR_ENDING:
				if ( strp->str_size > 0 )
				{
					Fprintf
					(
						stdout,
						"%11s%8d%%%10ld",
						cp,
						(int)(strp->str_posn * 100 / strp->str_size),
						strp->str_size
					);
					break;
				}

			default:
				Fprintf(stdout, "%11s%19s", cp, "");
			}

			Fprintf
			(
				stdout,
				"%10ld%10ld\n",
				strp->str_bytes,
				strp->str_messages
			);
		}
	}

	if ( NLinks > 1 )
		putc('\n', stdout);
}



void
ShortStats(link, host)
	char *		link;
	char *		host;
{
	register Str_p	strp;
	register char	c;
	register int	i;
	register int	j;
	register char	ls = ' ';

	if
	(
		NNstate.procid
	)
	{

		switch ( NNstate.procstate )
		{
		default:
		case PROC_IDLE:		c = 'I';	break;
		case PROC_RUNNING:	c = 'A';	break;
		case PROC_ERROR:	c = 'E';
					ls = 'D';
					break;
		case PROC_OPENING:	c = 'O';
					ls = 'D';
					break;
		case PROC_CALLING:	c = 'C';
					ls = 'D';
					break;
		}

		if ( ls == ' ' )
			switch ( NNstate.linkstate )
			{
			default:
			case LINK_DOWN:		ls = 'D';	break;
			case LINK_UP:		ls = 'U';;
			}

	}
	else
	{
		c = '*';
		ls = '*';
	}

	if (NHosts > 1) link = concat(link,"@",host, NULLSTR);
	Fprintf
	(
		stdout,
		"%-*.*s %c%c %10ld %8ld",
		NODE_NAME_SIZE, NODE_NAME_SIZE, link,
		c,
		ls,
		NNstate.allbytes,
		NNstate.allmessages
	);
	if (NHosts > 1) free(link);

	for ( strp = NNstate.streams[0], i = 0 ; i < 2 ; i++ )
	{
		putc(' ', stdout);

		for ( j = 0 ; j < NSTREAMS ; j++, strp++ )
		{
			switch ( strp->str_state )
			{
			default:
			case STR_ACTIVE:
			case STR_ENDED:
			case STR_IDLE:		c = ' ';	break;

			case STR_EMPTY:		c = '_';	break;
			case STR_ENDING:	c = '~';	break;
			case STR_ERROR:		c = 'E';	break;
			case STR_INACTIVE:	c = '.';	break;
			case STR_START:		c = '+';	break;
			case STR_WAIT:		c = ',';	break;
			}

			switch ( strp->str_state )
			{
			case STR_WAIT:
			case STR_START:
			case STR_ACTIVE:
			case STR_ENDING:
				if ( strp->str_size > 0 )
				{
					Fprintf
					(
						stdout,
						" %c%3d%%",
						c,
						(int)(strp->str_posn * 100 / strp->str_size)
					);
					break;
				}

			default:
				Fprintf(stdout, "    %c ", c);
			}
		}
	}

	putc('\n', stdout);

	if ( NLinks < 10 )
		putc('\n', stdout);
}



void
ShortHeader()
{
	register int	i;
	register int	j;
	register char	*cp;

	cp = "Receive Channels";
	i = strlen(cp);
	Fprintf(stdout, "%*s%s", NODE_NAME_SIZE+24+((6*NSTREAMS)-i)/2, "", cp);
	cp = "Transmit Channels";
	Fprintf(stdout, "%*s%s\n", ((2*6*NSTREAMS)-i-strlen(cp)+1)/2, "", cp);

	Fprintf
	(
		stdout,
		"%-*s    %10s %8s",
		NODE_NAME_SIZE, "Link",
		"Bytes",
		"Messages"
	);

	for ( i = 0 ; i < 2 ; i++ )
	{
		putc(' ', stdout);

		for ( j = 0 ; j < NSTREAMS ; j++ )
			Fprintf(stdout, "  %3d ", j);
	}

	putc('\n', stdout);
	putc('\n', stdout);
}



/*
**	Queue a network link directory.
*/

void
ListLink(host, link, hp)
	char *		host;
	char *		link;
	LLh_p		hp;
{
	register LLl_p	hllp;
	char *cp;

	hllp = Talloc(LLel);

	hllp->lql_link = concat(link, NULLSTR);
	hllp->lql_host = concat(host, NULLSTR);
	hllp->lql_ht = hp;
	hllp->lql_received = 0;
	cp = hp->hl_links;
	hp->hl_links = concat(" ", link, cp, NULLSTR);
	if (cp) free(cp);

	hllp->lql_next = LinkList;
	LinkList = hllp;

	NLinks++;

	Trace2(1, "list link \"%s\"", hllp->lql_link);
}



/*
**	Sort the links into alphabetic order
*/

void
SortLinks()
{
	register LLl_p	lp;
	register LLl_p *lpp;

	LinkVec = lpp = (LLl_p *)Malloc(sizeof(LLl_p) * NLinks);

	for ( lp = LinkList ; lp != (LLl_p)0 ; lp = lp->lql_next )
		*lpp++ = lp;

	DODEBUG(if((lpp-LinkVec)!=NLinks)Fatal1("bad NLinks"));

	Trace2(1, "found %d links", NLinks);

	if ( NLinks > 1 )
		qsort((char *)LinkVec, NLinks, sizeof(LLl_p), compare);
}



/*
**	Alpha compare
*/

int
compare(lpp1, lpp2)
	char *	lpp1;
	char *	lpp2;
{
	int cmp;
	cmp = strcmp((*(LLl_p *)lpp1)->lql_link, (*(LLl_p *)lpp2)->lql_link);
	if (cmp != 0) return cmp;
	return strcmp((*(LLl_p *)lpp1)->lql_host, (*(LLl_p *)lpp2)->lql_host);
}

int
hostcmp(a,b)
char *a,*b;
{
    while (*a && *b && *a!='.' && *b!='.' && *a == *b) a++, b++;
    if (*a=='.') a="";
    if (*b=='.') b="";
    if (*a==*b) return 0;
    if (*a<*b) return -1;
    return 1;
}


/*
**	Catch and ingore signal
*/

ignore(sig)
int sig;
{
	signal(sig, ignore);
	return;
}
