#
/*
 */

/*
 * TU-16 tape driver
 *   Rewrite by S. Tucker Taft
 *      Harvard Science Center
 *
 *
 * This TU-16 driver is copied from the TU-10 driver written at Harvard,
 *  which was itself modeled after the TU-16
 *  driver written at Piscataway.  It has the following features:
 *
 *	- No rewind is done on close
 * for devices with the NOREW bit on in their minor device number.
 * On a close after a write, 2 tape marks are written, and then
 * a single backspace is done.  On a close after a read, a forward
 * seek is done to the next tape mark, leaving the tape positioned after
 * the tape mark.  Hence, to skip a file, simply open the norewind device for
 * reading and then close it.
 *	- With the raw device, it is possible to write a tape mark.
 * A write with zero bytes for a count will cause a tape mark to
 * be written.  Please note that the buffer address must still
 * be a legal address because "physio" checks it.  Writes of less than 4 bytes
 * on the raw device are automatically filled out to 4 bytes.
 *	- On a read, if a tape mark is encountered, an error is
 * returned for the cooked device (errno = EFBIG), and
 * a zero-length record is returned for the raw device.
 *	- Seeks are allowed on the raw device.  Each block is treated
 * as though it were exactly 512 bytes (including tape marks), so that
 * a seek-by-blocks system call will do the desired operation.
 * During a seek on the raw device, tape marks are counted as one block.
 * On the cooked device, an error(EFBIG) will be returned if the seek tries
 * to cross a tape mark.  On the raw device, an error is only returned
 * if the seek tries to cross a double tape mark, leaving
 * the tape positioned after the first of the pair.
 *	- Please note:  If this driver may be used on an 11/70,
 * the symbol "CPU70" must be defined in param.h.
 */

#include "../param.h"
#include "../buf.h"
#include "../conf.h"
#include "../user.h"
#include "../file.h"
#include "../reg.h"

struct {
	int	htcs1;
	int	htwc;
	int	htba;
	int	htfc;
	int	htcs2;
	int	htds;
	int	hter;
	int	htas;
	int	htck;
	int	htdb;
	int	htmr;
	int	htdt;
	int	htsn;
	int	httc;
	int	htbae;	/* 11/70 bus extension */
};

struct	devtab	httab;
struct	buf	rhtbuf,	chtbuf;

/* NUNIT must be power of 2 (NUNIT-1 used as mask) */
#define NUNIT	4
#define	NOREW	040	/* no rewind on close bit (>= NUNIT) */
#define LODEN	0100	/* low density (800 baud) */

struct htinfo {
	char	*h_blkno;	/* block num about to read/write */
	char	*h_rawtm;	/* blkno after last tape mark */
	char	*h_tmblk;	/* block number after last write */
	char	h_openf;	/* 0 = not open, 1 = open, -1 = fatal error */
	char	h_tmwritten;	/* 1 = two eof's already written */
} htinfo[NUNIT];

int	ht_sem	4;	/* tape buffer semaphore */
char	ht_done;	/* i/o done flag */

#define	HTADDR	0172440

#define ETAPEMARK	EFBIG	/* error returned on tape mark */
#define RETRY	90

/* commands (and htcs bits) */
#define	NOCMD	00
#define	GO	01
#define	WEOF	026
#define	SFORW	030
#define	SREV	032
#define	REW	06
#define DCLR	010
#define	IENABLE	0100
#define	CRDY	0200
#define TRE	040000

/* status bits */
#define	HARD	064073	/* hard hter drive errors */
#define CHARD	077400	/* htcs2 hard ctlr errors */
#define EOT	02000
#define	EOF	04
#define P800	01300
#define P1600	02300
#define	MOL	010000	/* medium on line */
#define WRLOCK	04000

#define	SSEEKF	1
#define	SSEEKR	2
#define	SIO	3
#define SCOM	4


htopen(dev, flag)
{
	register ds;
	register struct htinfo *mp;

	mp = &htinfo[dev & (NUNIT-1)];
	if(dev.d_minor & ~((NUNIT-1)|NOREW|LODEN) || mp->h_openf) {
		u.u_error = ENXIO;
		return;
	}
	mp->h_openf++;
	mp->h_blkno = 0;
	/* -1 interpreted as 65535 later */
	mp->h_tmblk = mp->h_rawtm = -1;
	mp->h_tmwritten = 0;
	ds = hcommand(dev, NOCMD);	/* clear controller */
	if ((ds&MOL) == 0 || flag && (ds&WRLOCK)) {
		mp->h_openf = 0;
		prdev("offline or needs ring", dev);
		u.u_error = ENXIO;
		return;
	}
}

htclose(dev, flag)
{
	register int rdev;
	register struct buf *bp;
	register struct htinfo *mp;

	rdev = dev;
	mp = &htinfo[rdev & (NUNIT-1)];
		/* because h_rawtm and h_blkno are updated asyncronously */
		/*  must wait for all i/o to complete */
		hcommand(rdev, NOCMD);

	if(flag && !mp->h_tmwritten) {
		if (mp->h_rawtm != mp->h_blkno) hcommand(rdev, WEOF);
		hcommand(rdev, WEOF);
		hcommand(rdev, rdev&NOREW? SREV: REW);
	} else {
		hcommand(rdev, !(rdev&NOREW)? REW:
		    mp->h_rawtm != mp->h_blkno? SFORW: NOCMD);
	}

	mp->h_openf = 0;
	/* de-associate all blocks */
	for (bp = httab.b_forw; bp != &httab; bp = bp->b_forw)
		if (bp->b_dev == rdev) bp->b_dev.d_minor = -1;
}

hcommand(dev, com)
{
	register struct buf *bp;

	bp = &chtbuf;
	spl5();
	while(bp->b_flags & B_BUSY) {
		bp->b_flags =| B_WANTED;
		sleep(bp, PRIBIO);
	}
	spl0();
	bp->b_dev = dev;
	bp->b_resid = com;
	bp->b_blkno = 0;
	bp->b_flags = B_BUSY | B_READ;
	htstrategy(bp);
	iowait(bp);
	if(bp->b_flags & B_WANTED)
		wakeup(bp);
	bp->b_flags = 0;
	return(bp->b_resid);
}

htstrategy(abp)
struct buf *abp;
{
	register struct buf *bp;
	register struct htinfo *mp;
	register unit;

	bp = abp;
	unit = bp->b_dev&(NUNIT-1);
	mp = &htinfo[unit];

	if (bp == &chtbuf) goto qup;	/* just queue up the spcl coms */
	if (mp->h_openf < 0) {
		/* fatal error */
		bp->b_flags =| B_ERROR;
		iodone(bp);
		return;
	}

	if (mp->h_tmblk <= bp->b_blkno) {
		if (mp->h_tmblk < bp->b_blkno) {
			bp->b_error = ETAPEMARK;
			bp->b_flags =| B_ERROR;
			iodone(bp);
			return;
		}
		if (bp->b_flags&B_READ) {
			/* this may be read-before-write */
			if (bp->b_flags&B_PHYS) bp->b_resid = bp->b_wcount;
			else clrbuf(bp);
			iodone(bp);
			return;
		}
	} else if(mp->h_tmblk!=-1 && (bp->b_flags&B_READ) && !mp->h_tmwritten){
		/* he is about to backspace after writing */
		/* write out two eof's now for him */
		hcommand(unit, WEOF);
		hcommand(unit, WEOF);
		hcommand(unit, SREV);
		hcommand(unit, SREV);
		mp->h_tmwritten++;
	}
	if ((bp->b_flags&B_READ)==0) {
		mp->h_tmblk = bp->b_blkno + 1;
		mp->h_tmwritten = 0;
	}
#ifdef CPU70
	if (bp->b_flags&B_PHYS)
		mapalloc(bp);
#endif CPU70

qup:
	bp->av_forw = 0;
	spl5();
	if (httab.d_actf==0)
		httab.d_actf = bp;
	else
		httab.d_actl->av_forw = bp;
	httab.d_actl = bp;
	if (httab.d_active==0)
		htstart();
	spl0();
	sema_p(&ht_sem, PRIBIO);
	return;
}

htstart()
{
	register struct buf *bp;
	register struct htinfo *mp;
	register unit;

	httab.d_active = 0;
    loop:
	if ((bp = httab.d_actf) == 0)
		return;
	if (ht_done) {
		/* i/o done, chain to next */
		httab.d_actf = bp->av_forw;
		httab.d_errcnt = 0;
		ht_done = 0;
		iodone(bp);
		sema_v(&ht_sem);	/* release semaphore */
		goto loop;
	}
	mp = &htinfo[unit = bp->b_dev & (NUNIT-1)];
	/* select controller and unit */
	HTADDR->htcs2 = (unit>>3)&07;
	HTADDR->httc = unit&07 | (bp->b_dev&LODEN? P800: P1600);
	HTADDR->htcs1 = TRE | DCLR | GO;
	if(bp == &chtbuf) {
		if(bp->b_resid == NOCMD) {
			bp->b_resid = HTADDR->htds;
			ht_done++;
			goto loop;
		}
		httab.d_active = SCOM;
		HTADDR->htfc = 0;	/* i.e. infinity, or until eof */
		HTADDR->htcs1 = bp->b_resid | IENABLE | GO;
		return;
	}
	if (mp->h_openf < 0 || (HTADDR->htcs1 & CRDY)==0) {
		bp->b_flags =| B_ERROR;
		ht_done++;
		goto loop;
	}
	if (mp->h_blkno != bp->b_blkno) {
		if (mp->h_blkno < bp->b_blkno) {
			httab.d_active = SSEEKF;
			HTADDR->htfc = mp->h_blkno - bp->b_blkno;
			mp->h_blkno =- HTADDR->htfc;
			HTADDR->htcs1 = SFORW|IENABLE|GO;
		} else {
			httab.d_active = SSEEKR;
			/* cannot rewind even if b_blkno == 0 */
			/* because may not be in first file of tape */
			HTADDR->htfc = bp->b_blkno - mp->h_blkno;
			/* extra backspace if error retry */
			if (httab.d_errcnt && (httab.d_errcnt&07) == 0 &&
			    bp->b_blkno != 0)
				HTADDR->htfc--;
			mp->h_blkno =+ HTADDR->htfc;
			HTADDR->htcs1 = SREV|IENABLE|GO;
		}
		return;
	}
	httab.d_active = SIO;
	mp->h_blkno++;
	if (!(bp->b_flags&B_READ)) {
		if (bp->b_wcount == -1) {
			HTADDR->htcs1 = IENABLE|WEOF|GO;	/* write eof */
			mp->h_rawtm = mp->h_blkno;
			return;
		}
		mp->h_rawtm = -1;
	}
	rhstart(bp, &HTADDR->htfc, bp->b_wcount<<1, &HTADDR->htbae);
}

htintr()
{
	register struct buf *bp;
	register unit;
	register struct htinfo *mp;

	if ((bp = httab.d_actf)==0)
		return;
	unit = bp->b_dev & (NUNIT-1);
	mp = &htinfo[unit];
	/* ignore errors on SCOM */
	if (httab.d_active == SCOM) {
		ht_done++;
		return(htstart());
	}
	bp->b_resid = HTADDR->htfc>>1;	/* word count wanted */
	if (HTADDR->htcs1&TRE) {

	    /* if error or eof during seek, adjust h_blkno */
	    if (httab.d_active == SSEEKR)
	    	mp->h_blkno =- HTADDR->htfc;
	    else if (httab.d_active == SSEEKF)
	    	mp->h_blkno =+ HTADDR->htfc;
	    if (!(HTADDR->htds&EOF)) {

		/* error */

		if (HTADDR->hter&HARD ||
		    HTADDR->htcs2&CHARD || HTADDR->htds&EOT) {
			mp->h_openf = -1;
			prdev("Hard error", bp->b_dev);
		} else if (++httab.d_errcnt < RETRY) {
			return(htstart());
		}

		deverror(bp, HTADDR->hter, HTADDR->htcs2);
		bp->b_flags =| B_ERROR;
		ht_done++;
		return(htstart());
	    } else {

		/* eof */

		if (bp != &rhtbuf || mp->h_rawtm+1 == mp->h_blkno &&
		    httab.d_active != SSEEKR) {
			/* cooked eof, or two in a row for raw */
			/* must set error or cooked read will not stop */
			bp->b_error = ETAPEMARK;
			bp->b_flags =| B_ERROR;
			/* backspace over eof */
			httab.d_active = SCOM;
			HTADDR->htfc = -1;
			HTADDR->htcs1 = SREV|IENABLE|GO;
			mp->h_blkno--;
			return;
		}
		/* single eof on raw returns as zero length record */
		bp->b_resid = bp->b_wcount;
		mp->h_rawtm = mp->h_blkno;
	    }
	}
	if (httab.d_active == SIO) ht_done++;
	return(htstart());
}

htread(dev)
{
	physio(htstrategy, &rhtbuf, dev, B_READ);
	htadjust();
}

htwrite(dev)
{
	/* PHYSIO complains on count of zero, so */
	/* count of zero is set to 2, indicating write-tape-mark */
	/* count of 2 is set to 4 to avoid confusion and ridiculously */
	/*  small blocks */
	if (u.u_count < 4) u.u_count =+ 2;
	physio(htstrategy, &rhtbuf, dev, B_WRITE);
	htadjust();
}

htadjust()
{
	/* fudge file offset to make all blocks appear to be 512 bytes */
	register struct file *fp;
	struct { long lng; };

	fp = getf(u.u_ar0[R0]);
	fp->f_offset->lng =+ 512 + u.u_count.integ - u.u_arg[1];
}
