#
/*
 */

#include "../param.h"
#include "../user.h"
#include "../proc.h"
#include "../text.h"
#include "../systm.h"
#include "../file.h"
#include "../inode.h"
#include "../buf.h"
#include "../histo.h"
#include "../st_his.h"

/*
 * Give up the processor till a wakeup occurs
 * on chan, at which time the process
 * enters the scheduling queue at priority pri.
 * The most important effect of pri is that when
 * pri<0 a signal cannot disturb the sleep;
 * if pri>=0 signals will be processed.
 * Callers of this routine must be prepared for
 * premature return, and check that the reason for
 * sleeping has gone away.
 */
sleep(chan, pri)
{
	register *rp, s;

	s = PS->integ;
	rp = u.u_procp;
#ifdef HISTON
	histochk(rp,chan);
#endif
	if(pri >= 0) {
		if(issig())
			goto psig;
		spl6();
		rp->p_wchan = chan;
		rp->p_stat = SWAIT;
		rp->p_pri = pri;
		spl0();
		if(runin != 0) {
			runin = 0;
			wakeup(&runin);
		}
		swtch();
		if(issig())
			goto psig;
	} else {
		spl6();
		rp->p_wchan = chan;
		rp->p_stat = SSLEEP;
		rp->p_pri = pri;
		spl0();
		swtch();
	}
	PS->integ = s;
	return;

	/*
	 * If priority was low (>=0) and
	 * there has been a signal,
	 * execute non-local goto to
	 * the qsav location.
	 * (see trap1/trap.c)
	 */
psig:
	aretu(u.u_qsav);
}

/*
 * Wake up all processes sleeping on chan.
 */
wakeup(chan)
{
	register struct proc *p;
	register c, i;
#ifdef	HISTON
	char orflag;
#endif

	c = chan;
	for(p = &proc[0]; p<=maxproc; p++)
		if(p->p_wchan == c) {
			setrun(p);
#ifdef	HISTON
		orflag =| p->p_flag;
#endif
		}
#ifdef	HISTON
	if (orflag & HISTOB) histoent(c);
#endif
}

/*
 * Set the process running;
 * arrange for it to be swapped in if necessary.
 */
setrun(p)
{
	register struct proc *rp;

	rp = p;
	rp->p_wchan = 0;
	rp->p_stat = SRUN;
	if(rp->p_pri < curpri)
		runrun++;
	if(runout != 0 && (rp->p_flag&SLOAD) == 0) {
		runout = 0;
		wakeup(&runout);
	}
}

/*
 * Set user priority.
 * The rescheduling flag (runrun)
 * is set if the priority is higher
 * than the currently running process.
 */
setpri(up)
{
	register *pp, p;

	pp = up;
	p = (pp->p_cpu & 0377)/16;
	p =+ PUSER + pp->p_nice;
	if(p > 127)
		p = 127;
	if(p > curpri)	/* shouldn't this be p < curpri???? stt-2/6/76 */
		runrun++;
	pp->p_pri = p;
}

/*
 * The main loop of the scheduling (swapping)
 * process.
 * The basic idea is:
 *  see if anyone wants to be swapped in;
 *  swap out processes until there is room;
 *  swap him in;
 *  repeat.
 * The runout flag is set whenever someone is swaped out.
 * Sched sleeps on it awaiting work.
 *
 * Sched sleeps on runin whenever it cannot find enough
 * core (by swapping out or otherwise) to fit the
 * selected swapped process. It is awakened when the
 * core situation changes and in any event once per second.
 */
sched()
{
	register struct proc *rp, *p;
	register int outage, inage;
 	int maxsize;

	/*
	 * find user to swap in
	 * of users ready, select one out longest
	 */

loop:
	spl6();
	outage = -20000;
	for(rp = &proc[0]; rp <= maxproc; rp++)
	if(rp->p_stat==SRUN && (rp->p_flag&SLOAD)==0 &&
	   rp->p_time-rp->p_nice*8 > outage) {
		p = rp;
		outage = rp->p_time-rp->p_nice*8;
	}
	if(outage == -20000) {
		runout++;
		sleep(&runout, PSWP);
		goto loop;
	}
	spl0();

	/*
	 * see if there is core for that process
	 * if so, swap it in.
	 */

	if (swapin(p))
		goto loop;

	/*
	 * none found.
	 * look around for core.
	 * Select the largest of those sleeping
	 * at bad priority; if none, select the oldest.
	 */

	spl6();
	p = NULL;
	maxsize = -1;
	inage = -1;
	for (rp = &proc[0]; rp <= maxproc; rp++) {
		if ((rp->p_flag&(SSYS|SLOCK|SLOAD))!=SLOAD)
			continue;
		if (rp->p_stat==SWAIT || rp->p_stat==SSTOP) {
			if (maxsize < rp->p_size) {
				p = rp;
				maxsize = rp->p_size;
			}
		} else if (maxsize<0 && (rp->p_stat==SRUN||rp->p_stat==SSLEEP)) {
			if (rp->p_time+rp->p_nice > inage) {
				p = rp;
				inage = rp->p_time+rp->p_nice;
			}
		}
	}
	spl0();
	/*
	 * Swap found user out if sleeping at bad pri,
	 * or if he has spent at least 2 seconds in core and
	 * the swapped-out process has spent at least 3 seconds out.
	 * Otherwise wait a bit and try again.
	 */
	if (maxsize >= 0 || (outage >= 3 && inage >= 2)) {
		p->p_flag =& ~SLOAD;
		xswap(p, 1, 0);
		goto loop;
	}
	spl6();
	runin++;
	sleep(&runin, PSWP);
	goto loop;
}

/*
 * Swap a process in.
 * Allocate data and possible text seperately.
 * It would be better to do the largest first.
 */
swapin(pp)
struct proc *pp;
{
	register struct proc *p;
	register struct text *xp;
	register int a;
	int x;

	p = pp;
	if ((a = malloc(coremap, p->p_size)) == NULL)
		return(0);
	if (xp = p->p_textp) {
		if(xp->x_ccount == 0) {
			if ((x = malloc(coremap, xp->x_size)) == NULL) {
				mfree(coremap, p->p_size, a);
				return(0);
			}
			if (swap(xp->x_daddr, x, xp->x_size, B_READ)) {
				/* swap error */
				mfree(coremap, xp->x_size, x);
				x = xp->x_daddr;
			swaperr:
				mfree (coremap, p->p_size, a);
				log(LOG_LOG, "swaperror bn%o,pid%d\n",x, p->p_pid);
				p->p_time = 0;
				/* ##stt 3/11/77 force into wait */
				p->p_stat = SWAIT;
				return(1);
			}
			xp->x_caddr = x;
		}
		xp->x_ccount++;
	}
	if (swap(p->p_addr, a, p->p_size, B_READ)) {
		xccdec(xp);
		x = p->p_addr;
		goto swaperr;
	}
	mfree(swapmap, (p->p_size+7)>>3, p->p_addr);
	p->p_addr = a;
	p->p_flag =| SLOAD;
	p->p_time = 0;
	return(1);
}


/*
 * This routine is called to reschedule the CPU.
 * if the calling process is not in RUN state,
 * arrangements for it to restart must have
 * been made elsewhere, usually by calling via sleep.
 */
#ifdef	HISTON
int	cputime;	/* holds time between process switches */
#endif
swtch()
{
	static struct proc *p;
	register  n;
	register struct proc *rp,*rp1;

	if(p == NULL)
		p = &proc[0];
#ifdef	HISTON
	else if (p->p_flag&HISTOB) {
		if ((cputime =+ lbolt)<=0) cputime =+ HZ;
		histoent(CPUCHAN);	/* ##stt-2/6/76 histo cpu times */
	}
#endif
	/*
	 * Remember stack of caller
	 */
	savu(u.u_rsav);
	/*
	 * Switch to scheduler's stack
	 */
	retu(proc[0].p_addr);

loop:
	runrun = 0;
	rp = p;
	rp1 = rp;
	p = NULL;
	n = 128;
	/*
	 * Search for highest-priority runnable process
	 */
	do {
		rp++;
		if(rp > maxproc)
			rp = &proc[0];
		if(rp->p_stat==SRUN && (rp->p_flag&SLOAD)!=0) {
			if(rp->p_pri < n) {
				p = rp;
				n = rp->p_pri;
			}
		}
	} while(rp != rp1);
	/*
	 * If no process is runnable, idle.
	 */
	if(p == NULL) {
		p = rp;
		st_word =| ST_IDLE;	/* indicate in idle state */
		idle();
		st_word =& ~ST_IDLE;	/* clear indicator */
		goto loop;
	}
	rp = p;
	curpri = n;
	/*
	 * Switch to stack of the new process and set up
	 * his segmentation registers.
	 */
	retu(rp->p_addr);
	sureg();
	/*
	 * If the new process paused because it was
	 * swapped out, set the stack level to the last call
	 * to savu(u_ssav).  This means that the return
	 * which is executed immediately after the call to aretu
	 * actually returns from the last routine which did
	 * the savu.
	 *
	 * You are not expected to understand this.
	 */
	if(rp->p_flag&SSWAP) {
		rp->p_flag =& ~SSWAP;
		aretu(u.u_ssav);
	}
#ifdef	HISTON
	histochk(rp, CPUCHAN);	/* ##stt-2/6/76 check if to be histo'd */
	cputime = -lbolt;
#endif
	/*
	 * The value returned here has many subtle implications.
	 * See the newproc comments.
	 */
	return(1);
}

/*
 * Create a new process-- the internal version of
 * sys fork.
 * It returns 1 in the new process.
 * How this happens is rather hard to understand.
 * The essential fact is that the new process is created
 * in such a way that appears to have started executing
 * in the same call to newproc as the parent;
 * but in fact the code that runs is that of swtch.
 * The subtle implication of the returned value of swtch
 * (see above) is that this is the value that newproc's
 * caller in the new process sees.
 */
newproc()
{
	int a1, a2;
	struct proc *p, *up;
	register struct proc *rpp;
	register *rip, n;
	struct { struct proc *pptr; };	/* used to make rip into proc ptr */

	p = NULL;
	/*
	 * First, just locate a slot for a process
	 * and copy the useful info from this process into it.
	 * The panic "cannot happen" because fork has already
	 * checked for the existence of a slot.
	 */
retry:
	mpid++;
	if(mpid < 0) {
		mpid = 0;
		goto retry;
	}
	for(rpp = &proc[0]; rpp < &proc[NPROC]; rpp++) {
		if(rpp->p_stat == NULL && p==NULL)
			p = rpp;
		if (rpp->p_pid==mpid)
			goto retry;
	}
	if ((rpp = p)==NULL)
		panic("no procs");
	n = PS->integ;
	spl7();
	if(rpp > maxproc){
		maxproc = rpp;
	}
	PS->integ = n;

	/*
	 * make proc entry for new proc
	 */

	rip = u.u_procp;
	up = rip;
	rpp->p_stat = SRUN;
	rpp->p_flag = SLOAD;
	rpp->p_uid = rip->p_uid;
	rpp->p_nice  = rip->p_nice;
	rpp->p_textp = rip->p_textp;
	rpp->p_pid = mpid;
	rpp->p_ppid = rip->p_pid;
	rpp->p_time = 0;
	/* copy pgrp pointer */
	/* ##stt 3/2/77 */
	rpp->p_grp = rip->p_grp;
	/* clear out alarm in child */
	rpp->p_alarm = 0;

	/*
	 * make duplicate entries
	 * where needed
	 */

/* protocol change... ofile->pfile below ##bsb 3-27-76 */
	for(rip = &u.u_ofile[0]; rip < &u.u_pfile[NOFILE];)
		if((rpp = *rip++) != NULL)
			rpp->f_count++;
	if((rpp=up->p_textp) != NULL) {
		rpp->x_count++;
		rpp->x_ccount++;
	}
	u.u_cdir->i_count++;
	/*
	 * Partially simulate the environment
	 * of the new process so that when it is actually
	 * created (by copying) it will look right.
	 */
	savu(u.u_rsav);
	rpp = p;
	u.u_procp = rpp;
	rip = up;
	n = rip->p_size;
	a1 = rip->p_addr;
	rpp->p_size = n;
	a2 = malloc(coremap, n);
	/*
	 * If there is not enough core for the
	 * new process, swap out the current process to generate the
	 * copy.
	 */
	if(a2 == NULL) {
		rip->p_stat = SIDL;
		rpp->p_addr = a1;
		savu(u.u_ssav);
		xswap(rpp, 0, 0);
		rpp->p_flag =| SSWAP;
		rip->p_stat = SRUN;
	} else {
	/*
	 * There is core, so just copy.
	 */
		rpp->p_addr = a2;
		while(n--)
			copyseg(a1++, a2++);
	}
	u.u_procp = rip;
	return(0);
}

/*
 * Change the size of the data+stack regions of the process.
 * If the size is shrinking, it's easy-- just release the extra core.
 * If it's growing, and there is core, just allocate it
 * and copy the image, taking care to reset registers to account
 * for the fact that the system's stack has moved.
 * If there is no core, arrange for the process to be swapped
 * out after adjusting the size requirement-- when it comes
 * in, enough core will be allocated.
 * Because of the ssave and SSWAP flags, control will
 * resume after the swap in swtch, which executes the return
 * from this stack level.
 *
 * After the expansion, the caller will take care of copying
 * the user's stack towards or away from the data area.
 */
expand(newsize)
{
	int i, n;
	register *p, a1, a2;

	p = u.u_procp;
	n = p->p_size;
	p->p_size = newsize;
	a1 = p->p_addr;
	if(n >= newsize) {
		mfree(coremap, n-newsize, a1+newsize);
		return;
	}
	savu(u.u_rsav);
	a2 = malloc(coremap, newsize);
	if(a2 == NULL) {
		savu(u.u_ssav);
		xswap(p, 1, n);
		p->p_flag =| SSWAP;
		swtch();
		/* no return */
	}
	p->p_addr = a2;
	for(i=0; i<n; i++)
		copyseg(a1+i, a2++);
	mfree(coremap, n, a1);
	retu(p->p_addr);
	sureg();
}
#ifdef	HISTON
/*	histogramming routines ##stt-2/6/76 */
histochk(ap,achan)
{
	register *p, *chan;
	register struct histr *hp;

	p=ap;
	chan=achan;
	for (hp = &histo.grams[0]; hp->h_lim[0]; hp++) {
		if (chan < hp->h_lim[0] || chan > hp->h_lim[1]) continue;
		if (hp->h_inum == 0 ||
		    (p->p_textp != 0 &&
    		    p->p_textp->x_iptr->i_number == hp->h_inum)) {
			p->p_flag =| HISTOB;
			return;
		}
	}
	p->p_flag =& ~HISTOB;
}

histoent(achan)
{
	register *chan;
	register a;
	register struct histr *hp;

	chan = achan;
	for (hp = &histo.grams[0]; hp->h_lim[0]; ) {
		if ((a = *hp->h_valp) < 0 ||
		    chan < hp->h_lim[0] || chan > hp->h_lim[1]) {
			hp++;
			continue;
		}
		a = a/hp->h_scal + &hp->h_data[0];
		hp->h_tot++;
		*hp->h_valp = -1;	/* indicate histo'd */
		if (&a->integ < ++hp)
			++(a->integ);
	}
}

#endif
