/*
 *	Scan error record file looking for RM03 error reports
 *	& interpret them at several levels of verbosity
 *
 *		Robert Elz,
 *		Melb Uni,
 *		Comp Sci,
 *		March 80
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/errlog.h>
#include <sys/hperrs.h>

#define	B_READ	0001		/* from buf.h - too difficult to include it */

char	*erfile = "/etc/loggederrors";	/* default error file */

int	longbits = 0;		/* long or short bit descriptions */
int	form  = 0;		/* to use form feeds or not */
int	thelot = 0;		/* != 0 ==> print regs after recovery */
int	verbose = 0;		/* != 0 ==> print reg contents */
int	guess = 0;		/* != 0 ==> have a guess at cause of error */
int	seq1, seq2;		/* only errors between these seq numbers */

int	option = 0;		/* output options */
#	define	RECOV		0x0001	/* recovered errors */
#	define	HARD		0x0002	/* hard (non recovered) errors */
#	define	VOLINVAL	0x0004	/* volume invalid status (disc change) */
#	define	DISCOFF		0x0008	/* operation when disc off line */
#	define	TOTALS		0x0010	/* error type totals */

struct hplog hplog;
struct hplog savlog;
int	saved;
#define	SIZE	(sizeof(struct hplog)-sizeof(struct errhead))

struct	{
	long	tim;
	short	len;
} hd;

int	cyl, head, sector;			/* disk posn info */

int cyloff[8] = {		/* start cyl for logical discs,
				   should agree with table in sys/sys/hp.c */
	0,
	100,
	0,
	442,
	0,
	380,
	310,
	0,
};

int	hard, recov, inval, mol, seq, count;	/* statistics */

struct	bitinfo	{
	char	*meaning;		/* meaning if bit is (is not) set */
	char type;			/* indication of error cause */
					/* nb: these are very probably not
					   correct, they are just my guesses
								... kre   */
	char	bit;			/* bit number (as in RM03 manual) */
	char	on;			/* msg is signifigant if bit same as this */
};
#define	THE_END		{ (char *)0, 0, 0, 0 }

	/* error causes : NB arder is signifigant to 'guessat()' */
#define	INFO	0			/* information bit */
#define	MBA_F	1			/* indicates MBA fault */
#define	SBI_F	2			/* indicates SBI fault */
#define	DV_F	3			/* indicates drive fault */
#define	MEM_F	4			/* indicates memory fault */
#define	OP_F	5			/* indicates operator error */
#define	DAT_F	6			/* indicates data error (bad pack) */
#define	PG_F	7			/* indicates program (software) err */


struct bitinfo mbacsr[] = {
	{ "SBI Parity Error",		SBI_F,	31,	1	},
	{ "Write data Seq error",	SBI_F,	30,	1	},
	{ "Unexpected Read Data",	SBI_F,	29,	1	},
	{ "Multiple Transmitter",	SBI_F,	27,	1	},
	{ "XMTFLT",			INFO,	26,	1	},
	{ "adaptor Power Down",		MBA_F,	23,	1	},
	{ "adaptor Power Up",		INFO,	22,	1	},
	{ "Over Temparature !?!",	MBA_F,	21,	1	},
	THE_END
};

struct bitinfo mbasr[] = {
	{ "Data Transfer BUSY",		INFO,	31,	1	},
	{ "No Response CONFirmation",	SBI_F,	30,	1	},
	{ "Corrected Read Data",	INFO,	29,	1	},
	{ "ProGram Error",		PG_F,	19,	1	},
	{ "Non-Existing Drive",		PG_F,	18,	1	},
	{ "Massbus Control Parity Error", MBA_F, 17,	1	},
	{ "ATTentioN",			INFO,	16,	1	},
	{ "Data Transfer CoMPleted",	INFO,	13,	1	},
	{ "Data Transfer ABorTed",	INFO,	12,	1	},
	{ "Data LaTe",			MBA_F,	11,	1	},
	{ "Write ChecK UPper ERRor",	DAT_F,	10,	1	},
	{ "Write ChecK LoWeR ERRor",	DAT_F,	9,	1	},
	{ "Miss transfer error|XF",	MBA_F,	8,	1	},
	{ "MassBus EXCeption",		MBA_F,	7,	1	},
	{ "Massbus Data Parity Error",	MBA_F,	6,	1	},
	{ "page frame MAP Parity Error",MBA_F,	5,	1	},
	{ "INValid MAP",		PG_F,	4,	1	},
	{ "ERRor CONFirmation",		MEM_F,	3,	1	},
	{ "Read Data Substitute",	MEM_F,	2,	1	},
	{ "Interface Sequence TIMEOUT",	MEM_F,	1,	1	},
	{ "Read Data TIMEOUT",		MEM_F,	0,	1	},
	THE_END
};

struct bitinfo hpcs1 [] = {
	{ "Special Condition",		INFO,	15,	1	},
	{ "TRansfer Error",		INFO,	14,	1	},
	{ "Massbus Control Parity Error", MBA_F, 13,	1	},
	{ "Drive aVAilable",		INFO,	11,	1	},
	{ "Port SELect",		INFO,	10,	1	},
	{ "ReaDY",			INFO,	7,	1	},
	{ "Interrupt Enable",		INFO,	6,	1	},
	THE_END
};

struct bitinfo hpds [] = {
	{ "ATtention Active",		INFO,	15,	1	},
	{ "ERRor",			INFO,	14,	1	},
	{ "Positioning In Progress",	INFO,	13,	1	},
	{ "Medium On Line",		INFO,	12,	1	},
	{ "Medium On Line",		OP_F,	12,	0	},
	{ "WRite Lock",			INFO,	11,	1	},
	{ "Last Block Transferred",	INFO,	10,	1	},
	{ "ProGraMmable",		INFO,	9,	1	},
	{ "Drive PResent",		INFO,	8,	1	},
	{ "Drive ReadY",		INFO,	7,	1	},
	{ "Volume Valid",		INFO,	6,	1	},
	{ "Volume Valid",		PG_F,	6,	0	},
	{ "Offset Mode",		INFO,	0,	1	},
	THE_END
};

struct bitinfo hper1 [] = {
	{ "Data ChecK error",		DAT_F,	15,	1	},
	{ "UNSafe",			DV_F,	14,	1	},
	{ "OPeration Incomplete",	DV_F,	13,	1	},
	{ "Drive Timing Error",		DV_F,	12,	1	},
	{ "Write Lock Error",		OP_F,	11,	1	},
	{ "Invalid Address Error",	PG_F,	10,	1	},
	{ "Address OvErflow",		PG_F,	9,	1	},
	{ "Header CRC error",		DAT_F,	8,	1	},
	{ "Header Compare Error",	DAT_F,	7,	1	},
	{ "ECc Hard error",		DAT_F,	6,	1	},
	{ "Write Clock Fail",		MBA_F,	5,	1	},
	{ "Format ERror",		PG_F,	4,	1	},
	{ "bus PARity error",		DV_F,	3,	1	},
	{ "Register Modification Refused",PG_F,2,	1	},
	{ "ILlegal Register",		PG_F,	1,	1	},
	{ "ILlegal Function",		PG_F,	0,	1	},
	THE_END
};

struct bitinfo hper2 [] = {
	{ "Bad Sector Error",		DAT_F,	15,	1	},
	{ "SeeK Incomplete",		DV_F,	14,	1	},
	{ "Operator Plug Error",	OP_F,	13,	1	},
	{ "InValid Command",		PG_F,	12,	1	},
	{ "Loss of System Clock",	DV_F,	11,	1	},
	{ "Loss of Bit Clock",		DV_F,	10,	1	},
	{ "DeVice Check",		DV_F,	7,	1	},
	{ "Data Parity Error",		DV_F,	3,	1	},
	THE_END
};

struct bitinfo hpof [] = {
	{ "FMT 16",			INFO,	12,	1	},
	{ "FMT 16",			INFO,	12,	0	},
	{ "Error Correction Inhibit",	INFO,	11,	1	},
	{ "Header Compare Inhibit",	INFO,	10,	1	},
	{ "OFfset Direction",		INFO,	7,	1	},
	THE_END
};

struct bitinfo nobits [] = {
	THE_END
};

char	*errlevel[] = {		/* indexed by error cause */
	"",			/* never needed for INFO */
	"MBA fault",
	"SBI fault",
	"Drive fault",
	"Memory error",
	"Operator error",
	"Data (pack) error",
	"Software error"	/* this never happens !!! */
};

#define	RP	022
#define	RM	024
#define	DRQ	04000
#define	MOH	020000

#define	NHP	2
#define	BLKHP	0
#define	RAWHP	4

#define	NRMSECT	32
#define	NRMTRAC	5

char *cvflags();
char *getnum();
int	mbacsrfn(), mbabcrfn(), hpcmdfn(), nillfn(), eccposfn(), eccpatfn();

main(argc, argv)
char **argv;
{
	register l, efd;
	register char *p;

	while (--argc > 0 && *(p = *++argv) == '-')
		switch (*++p) {
		case 'g':	guess++;		break;
		case 'f':	form++;			break;
		case 'V':	thelot++;		break;
		case 'v':	verbose++;
				if (p[1] == 'v')
					longbits++;	break;
		case 'h':	option |= HARD;		break;
		case 'r':	option |= RECOV;	break;
		case 'm':	option |= DISCOFF;	break;
		case 'p':	option |= VOLINVAL;	break;
		case 't':	option |= TOTALS;	break;

		case 's':	if (*++p == '\0')
					if (--argc <= 0) {
						fprintf(stderr, "No sequence number[s]\n");
						exit(1);
					} else
						p = *++argv;
				p = getnum(p, &seq1);
				if (*p == ',')
					getnum(++p, &seq2);
				else
					seq2 = seq1;
				if (seq1)
					seq1--;		/** nasty kludge **/
							break;

		case 'a':
		case 'b':
		case 'd':
		case 'x':
				fprintf(stderr, "'%c' option unsupported\n", *p);
				break;

		default:
				fprintf(stderr, "Unknown option: %s\n", p);
				break;
		}
	if (guess && option == 0)
		option = RECOV|HARD;
	if (option == 0)
		option = TOTALS;

	if (argc > 0)
		erfile = *argv;

	if ((efd = open(erfile, 0)) < 0) {
		perror(erfile);
		exit(1);
	}

	count = 0;
	for (;;) {
		if ((l = read(efd, &hd, sizeof hd)) != sizeof hd) {
			if (l < 0)
				perror(erfile);
			else if (l != 0)
				fprintf(stderr, "malformed error file: %s\n",
							erfile);
			if (option & TOTALS) {
				printf("Hard errors	: %4d\n", hard);
				printf("Recovered errors: %4d\n", recov);
				printf("Pack changes	: %4d\n", inval/2);
				printf("Op's off line	: %4d\n", mol);
			}
			exit(l != 0);
		}
		count++;
		if (seq2 && count > seq2)
			exit(0);
/**	to come
		if (hd.tim > last)
			exit(0);
**/
		if (hd.len != SIZE || (seq1 && count < seq1)
/**	to come
					|| hd.tim < first
**/
								) {
			lseek(efd, (long)hd.len, 1);
			continue;
		}

		if ((l = read(efd, &hplog.er_dev, SIZE)) != SIZE) {
			if (l < 0)
				perror(erfile);
			else
				fprintf(stderr, "malformed error file: %s\n",
							erfile);
			exit(1);
		}
		if (major(hplog.er_dev) != RAWHP && major(hplog.er_dev) != BLKHP
				|| minor(hplog.er_dev) >= (NHP<<3))
			continue;

		if (seq && hplog.er_ecnt != seq+1)
			if (seq < hplog.er_ecnt)
				fprintf(stderr, "Lost errs near %s",
							ctime(&hd.tim));
			else if (hplog.er_ecnt != 1)
				fprintf(stderr, "Sequence failure\n");
		seq = hplog.er_ecnt;
/**	to come
		if (dselect && minor(hpolg.er_dev)>>3 != selected)
			continue;
**/
		switch (hplog.er_why) {

		case HPER_VV:
			if (option & VOLINVAL)
				format(&hplog, "Volume invalid");
			saved = 0;
			inval++;
			break;

		case HPER_MOL:
			if (option & DISCOFF)
				format(&hplog, "Disk off line");
			saved = 0;
			mol++;
			break;

		case HPER_ERR:
	/*
	 * This is not good enough in general, it is quite possible for
	 * there to be an error on one drive occurring between the initial
	 * error report (HPERR_ERR) and the final indication of what happened
	 * (HPERR_OK or HPERR_BAD). This code assumes nothing of that sort
	 * will occur (& so far we have been lucky)
	 */
			if (option & (RECOV|HARD)) {
				savlog = hplog;
				saved = 1;
			}
			break;

		case HPER_OK:
			if (option & RECOV) {
				if (!saved || hplog.er_ecnt != savlog.er_ecnt+1)
					format(&hplog, "Stray recovery");
				else {
					if (!thelot)
						savlog.er_errcnt = hplog.er_errcnt;
					format(&savlog, "Recovered error");
					if (thelot)
						format(&hplog, "After recovery");
				}
			}
			saved = 0;
			recov++;
			break;

		case HPER_BAD:
			if (option & HARD) {
				if (!saved || hplog.er_ecnt != savlog.er_ecnt+1)
					format(&hplog, "Stray hard error");
				else {
					if (!thelot)
						savlog.er_errcnt = hplog.er_errcnt;
					format(&savlog, "Hard error");
					if (thelot)
						format(&hplog, "After abandoned");
				}
			}
			saved = 0;
			hard++;
			break;

		default:
/*
			fprintf(stderr, "Unknown error type %d", hplog.er_why);
*/
			format(&hplog, "Unknown error");
			continue;
		}
	}
}

format(log, s)
register struct hplog *log;
char *s;
{

	if (form)
		putchar('\f');
	else if (guess || verbose)
		putchar('\n');

	printf("%-20s Dev %2u/%-2u\t(%d)\tSeq %-5d\t%s"
		, s
		, major(log->er_dev)
		, minor(log->er_dev)
		, log->er_errcnt
		, count
		, ctime(&hd.tim)
	);
	if (!verbose && !guess)
		return;
	if (log->er_hpdt & DRQ)
		printf("Dual port ");
	if ((log->er_hpdt & MOH) == 0)
		printf("Fixed head (!?!) ");
	switch (log->er_hpdt & 0777) {
		case RP:
			printf("RP06");
			break;
		case RM:
			printf("RM03");
			break;
		default:
			printf("Drive type %03o ?", log->er_hpdt&0777);
			break;
	}
	printf(" #%x\n", log->er_hpsn);
	if ((log->er_hpdt & 0777) != RM) {
		printf("No interpretation available for this device\n");
		return;
	}
	if (guess)
		guessat(log);
	if (!verbose)
		return;
	blktoad(log);
	printf
	(	"System: %s of block %d (Cyl %d Track %d Sect %d) [ %ld bytes ]\n"
		, cvflags(log->er_bflags)
		, log->er_blk
		, cyl
		, head
		, sector
		, log->er_bcount
	);
	printf
	(	"Drive: Cyl %d, Track %d, Sect %d\n"
		, log->er_hpdc
		, log->er_hpda>>8
		, log->er_hpda&0377
	);

	bitreg("MBA CSR", log->er_mbacsr, mbacsr, mbacsrfn);
	bitreg("MBA SR ", log->er_mbasr, mbasr, nillfn);
	bitreg("MBA BCR", log->er_mbabcr, nobits, mbabcrfn);
	bitreg("Cmd/Status", log->er_hpcs1, hpcs1, hpcmdfn);
	bitreg("Drive Status", log->er_hpds, hpds, nillfn);
	bitreg("err reg 1", log->er_hper1, hper1, nillfn);
	bitreg("err reg 2", log->er_hper2, hper2, nillfn);
	bitreg("offset reg", log->er_hpof, hpof, nillfn);
	bitreg("ECC position", log->er_hpec1, nobits, eccposfn);
	bitreg("ECC pattern", log->er_hpec2, nobits, eccpatfn);
}

blktoad(log)
register struct hplog *log;
{
	register unsigned n;

	cyl = log->er_blk / (NRMSECT * NRMTRAC);
	cyl += cyloff[minor(log->er_dev) & 07];
	n = log->er_blk % (NRMSECT * NRMTRAC);
	head = n / NRMSECT;
	sector = n % NRMSECT;
}

char *
cvflags(f)
{
	static char buf[80];

	if (f & B_READ)
		strcpy(buf, "Read");
	else
		strcpy(buf, "Write");

	/* add lots more here someday, .. phys I/O, ... */

	return(buf);
}

bitreg(s, v, rd, fn)
char *s;
register long v;
register struct bitinfo *rd;
int (*fn)();
{
	register i = 0;
	char *fullmsg(), *abbrev();

	printf("%-20s	%08lx", s, v);
	if (longbits && rd->meaning)
		putchar('\n');
	while (rd->meaning) {
		if ( ((v >> rd->bit) & 1) == rd->on ) {
			if (longbits)
				printf("\t\t\t%08lx  %s%s\n"
					, 1L << rd->bit
					, rd->on ? "" : "--- Not "
					, fullmsg(rd->meaning)
				);
			else if (rd->on)
				printf("%c%s"
					, i ? ',' : ' '
					, abbrev(rd->meaning)
				);
			i++;
		}
		rd++;
	}
	(*fn)(v, i);
	putchar('\n');
}

char *
fullmsg(s)
register char *s;
{
	static char outbuf[80];
	register char *p = outbuf;
	register c;

	while ( (*p = c = *s++) && c != '|') {
		if (isupper(c))
			*p = tolower(c);
		p++;
	}
	*p = '\0';
	if ((c = outbuf[0]) && islower(c))
		outbuf[0] = toupper(c);
	return(outbuf);
}

char *
abbrev(s)
register char *s;
{
	static char outbuf[20];
	register char *p = outbuf;
	register c;

	while (c = *s++)
		if (isupper(c))
			*p++ = c;
	*p = '\0';
	return(outbuf);
}

mbacsrfn(v, i)
long v;
int i;
{
	register code = v & 0xff;

	if (longbits) {
		printf("\t\t\t%08x  Adaptor code for ", code);
		switch (code) {
		case 0x20:
			printf("Massbus");
			break;
		default:
			printf("Unknown Adaptor !!");
		}
		putchar('\n');
	} else
		printf("%cCode=%x", i ? ',' : ' ', code);
}

mbabcrfn(v, i)
long v;
int i;
{
	register mba = (v >> 16) & 0xffff;
	register sbi = v & 0xffff;

	if (longbits) {
		if (mba)
			printf("\n\t\t\t%04x0000  Mba byte count %d"
				, mba
				, 0x10000 - mba
			);
		if (sbi)
			printf("\n\t\t\t%08x  SBI byte count %d"
				, sbi
				, 0x10000 - sbi
			);
		putchar('\n');
	} else {
		if (mba || sbi)
			putchar(' ');
		if (mba)
			printf("%d", 0x10000-mba);
		if (mba && sbi)
			putchar(',');
		if (sbi)
			printf("%d", 0x10000-sbi);
	}
}

hpcmdfn(v, i)
long v;
int i;
{
	register fn = v & 0x3e;

	if (longbits)
		printf("\t\t\t%08x  Last command: ", fn);
	else
		printf("%cFn=", i ? ',' : ' ');

	switch (fn) {
	case 000:	printf("No-op");		break;
	case 004:	printf("Seek");			break;
	case 006:	printf("Recalibrate");		break;
	case 010:	printf("Drive Clear");		break;
	case 012:	printf("Release");		break;
	case 014:	printf("Offset");		break;
	case 016:	printf("Rtn to centre");	break;
	case 020:	printf("Preset");		break;
	case 022:	printf("Pack acknowledge");	break;
	case 030:	printf("Search");		break;
	case 050:	printf("Write check data");	break;
	case 052:	printf("Wrt chk hdr & data");	break;
	case 060:	printf("Write");		break;
	case 062:	printf("Write hdr & data");	break;
	case 070:	printf("Read");			break;
	case 072:	printf("Read hdr & data");	break;
	default :	printf("? %03.1o ?", fn);	break;
	}

	if (v & 01)
		printf(" + GO");
	if (longbits)
		putchar('\n');
}

eccposfn(v, i)
long v;
int i;
{
	if (v)
		printf("  %6ld (10)", v);
	if (longbits)
		putchar('\n');
}

eccpatfn(v, i)
long v;
int i;
{
	if (v)
		printf("  %06lo (8)", v);
	if (longbits)
		putchar('\n');
}

nillfn()
{
}

guessat(log)
register struct hplog *log;
{
	register i;
	register char f;

	switch (log->er_why) {

	case HPER_MOL:
		printf("User error: drive off line\n");
		return;

	case HPER_VV:
		printf("Informative: pack changed\n");
		return;
	}

	for ( f = INFO+1; f <= PG_F; f++) {
		i = 0;
		i = faults(f, i, (long)log->er_mbacsr, mbacsr);
		i = faults(f, i, (long)log->er_mbasr, mbasr);
		i = faults(f, i, (long)log->er_hpcs1, hpcs1);
		i = faults(f, i, (long)log->er_hpds, hpds);
		i = faults(f, i, (long)log->er_hper1, hper1);
		i = faults(f, i, (long)log->er_hper2, hper2);
		if (i)
			break;
	}
	if (!i)
		printf("Cause undeterminable\n");
}

faults(f, i, v, rd)
register char f;
register i;
register long v;
register struct bitinfo *rd;
{
	while (rd->meaning) {
		if (rd->type == f && ((v >> rd->bit) & 1) == rd->on) {
			if (i == 0)
				printf("%s indicated\n", errlevel[f]);
			printf("\t%s\n", fullmsg(rd->meaning));
			i++;
		}
		rd++;
	}
	return(i);
}

char *
getnum(p, ip)
register char *p;
int *ip;
{
	register n = 0;

	while (isdigit(*p)) {
		n *= 10;
		n += *p++ - '0';
	}
	*ip = n;
	return(p);
}
