/*
 *	rxsh.c
 *	Program to allow a subset of shell commands to users
 *	of rx disc
 *	Deals with mounting and unmounting the discs
 *	Peter Collinson
 *	May 1977
 */

char mtab[]	"/etc/mtab";
char rxdev[]	"/dev/rx0";
char rxdir[]	"/usr/rx0";

/* Allowable commands are:- */

char *commands[]
{	"ar",
	"cat",
	"chdir",
	"chmod",
	"copy",
	"cmp",
	"cp",
	"cd",
	"date",
	"dcheck",
	"df",
	"dsw",
	"du",
	"echo",
	"find",
	"icheck",
	"icp",
	"kill",
	"ln",
	"ls",
	"mkdir",
	"mv",
	"ncheck",
	"pcp",
	"prot",
	"ps",
	"pwd",
	"quit",			/* Internal command to produce icheck on exit */
	"rm",
	"rmdir",
	"time",
	"unprot",
	"who",
	"write",
	0
};

/* Buffers */
char buf[512];
char sbuf[512];
char pwbuf[80];			/* Stores defualt directory */
int nohome 0;		/* =1 when home dir obtained */
char *chdirvec[25];		/* Table of pointers to arguments for chdir - must be done internaly */
int chdtype[25];		/* Contains 2 if cd and 5 if chdir - used to find start of command */
int chdirct	0;		/* Count of entries */
int prompt	1;		/* Non-zero if prompt characters to be printed */
char *ifile 	0;		/* Points to input file specified on command line */
char *ofile	0;		/* Points to output file (for logging) specified on the command line */
int ifd;
int ofd;

int quitfound	0;		/* = 1 if quit typed */

/* Mtab file - internal structure */

struct
{	char	mtab_p[32];
	char	mtab_dev[32];
};

char *comstart;		/* Points to start of command string for checking */
int chkcom;			/* = 0 if command still need to be checked */
int pid;			/* Current process number */
int status;			/* exit status */

main(argc, av)
int argc; char **av;
{	int fd;
	register char **argv;
	argv = av;

	sigsoff();

	while(argc-- > 1)
	{	argv++;
		if(**argv == '0' ||  **argv == '1')
		{	rxdev[7] = **argv;
			rxdir[7] = **argv;
		}
		else
		if(**argv == '-')
		{	if(*++*argv == 'i')
				ifile = *++argv;
			else
			if(**argv == 'o')
				ofile = *++argv;
			else goto illearg;
			argc--;
			prompt = 0;
		}
		else
		{	illearg:
			printf("Argument not recognised: %s\n", *argv);
			exit();
		}
	}

	/* Check for device already mounted */

	if((fd = open(mtab, 0)) < 0)
	{	printf("Cannot open mtab\n");
		exit();
	}

	/* Read through mtab looking for device */

	while(read(fd, buf, 64) > 0)
	{	if(buf->mtab_dev[0] == 'r' &&
		   buf->mtab_dev[1] == 'x' &&
		   buf->mtab_dev[2] == rxdev[7])
		{
			printf("%s in use\n", rxdev);
			exit();
		}
	}
	close(fd);

	/* open the input and output file */
	if(ifile)
	{	if((ifd = open(ifile, 0)) < 0)
		{	printf("Cannot open:%s\n", ifile);
			exit();
		}
	}
	if(ofile)
	{	if((ofd = creat(ofile, 0666)) < 0)
		{	printf("Cannot creat: %s\n", ofile);
			exit();
		}
	}


	/* Mount the device into appropriate directory */

	if(newproc()) wait(&status);
	else
	{	execl("/etc/mount",  "mount", rxdev, rxdir, 0);
		printf("Cannot execute mount\n");
		exit();
	}

	/* See if it worked */

	if((status>>8)&01)
	{	printf("Mount failed\n");
		exit();
	}

	/* Juggle with input and output files */
	if(ifile)
	{	close(0); dup(ifd); close(ifd);	}
	if(ofile)
	{	close(1); dup(ofd); close(ofd);	}
	printf("rx shell for %s\n", rxdev);

	process_commands();

	if(prompt) printf("\n");

	/* End of commands - umount the rx */


	chdir("/");

	if(newproc()) wait(&status);
	else
	{	execl("/etc/umount", "umount", rxdev, 0);
		printf("Cannot execute umount\n");
		exit();
	}

	/* Execute icheck if wanted */
	if(quitfound)
	{
		sigtrap();
		printf("icheck on- ");
		if(newproc()) wait(&status);
		else
		{	execl("/bin/icheck", "icheck", rxdev, 0);
			printf("Cannot execute icheck\n");
		}
	}
	exit();
}

int newproc()
{

	if((pid = fork()) <0)
	{	printf("Cannot create new processes\n");
		exit();
	}
	return(pid);
}

process_commands()
{	register char *source, *dest, c;
	int quote;

	sigtrap();
	while(1)
	{	if(prompt) {	printf("rx%c%% ", rxdev[7]); flush();	}
		/* Read in a command - this method is not very elegant - but it is easy */
		dest = buf;
		while((c = getchar()) != '\n')
		{	if(c == '\0') goto leave;
			*dest++ = c;
		}
		*dest = '\0';
		/* Interpret commands */

		dest = sbuf;
		source = buf;
		if(*source == '\0') goto out;
		comstart = sbuf;
		chkcom = 1;
		quote = 0;
		chdirct = 0;

		while(*dest++ = *source++)
		{	rescan:
			switch(*source)
			{
			case ';':
			case '|':
				if(quote) break;
				if(chkcom && illegalcom(dest)) goto out;
				comstart = dest + 1;
				chkcom = 1;
				break;

			case ' ':
				if(chkcom)
				{	if(illegalcom(dest)) goto out;
					if(compstr(comstart, "quit"))
					{	quitfound++;
						goto leave;
					}
				}
				break;


			case '"':
				quote = ~quote;
				break;

			case 'x':
				if(quote == 0 &&
				   dest[-1] == 'r' &&
				   dest[-2] == ' ' &&
				   special(source[1]))
				{	dest = copystr(rxdir, --dest);
					source++;	/* Ignore the 'x' */
					goto rescan;
				}
			default:;
			}
		}


		if(chkcom && illegalcom(dest)) goto out;
		/* Look for quit */
		if(compstr(comstart, "quit")) { quitfound++; goto leave; 	}


		if(chdirct)
		{	if(execchdir())  goto leave;	}

		/* The line checked out as legal - invoke shell */

		if(sbuf[0] == '\0') goto out;	/* Do nothing if line empty */

		if(callshell(sbuf)) goto leave;
	out:;
	}
	leave:
	sigsoff();
	return;
}

int illegalcom(cend)
char *cend;
{	register char **com;

	*cend = '\0';
	if(*comstart == '\0') return(0);

	chkcom--;
	com = commands;

	do
	{	if(compstr(comstart, *com))
		{
			if(compstr("cd", comstart) || compstr("chdir", comstart))
			{	chdtype[chdirct] = (comstart[1] == 'd' ? 2:5);
				chdirvec[chdirct++] = cend;
			}
			return(0);
		}
	}	 while(*++com);

	printf("%s is not allowed in rxsh\n", comstart);
	return(1);
}


char *copystr(s1, s2)
char *s1, *s2;
{	register char *s, *d;
	s = s1;
	d = s2;

	while(*d++ = *s++);
	return(--d);
}

int compstr(str1, str2)
char *str1, *str2;
{	register char *a, *b, c;

	a = str1; b = str2;
	while((c = *a++))
		if(c != *b++) return(0);
	return(*b == '\0');
}

/* Routines dealing with signals */

onintr()
{	int statret;
	sigsoff();
	if(pid)
	{	kill(pid, 2);
		wait(&statret);
	}
	pid = 0;
	sigtrap();
}

sigsoff()
{	signal(2, 1);
	signal(3, 1);
}

sigtrap()
{	signal(2, onintr);
	signal(3, onintr);
}

int special(c)
char c;
{	register int ch;
	ch = c;

	return(ch == '\0' || ch == ' ' || ch == ';' || ch == '>' || ch == '<' ||
	       ch == '&' || ch == '|' || ch == '\t' || ch == '/');
}

int callshell(arg)
char *arg;
{	int retstat;

	if(newproc()) wait(&retstat);
	else
	{	execl("/bin/sh", "sh", "-c", arg, 0);
		printf("Cannot execute shell\n");
		return(1);
	}
	pid = 0;
	return(0);
}


execchdir()
{	register char *s, *comst;
	register int i;
	char *wkpt;

	ex_sigtrap();
	comst = chdirvec[0] - chdtype[0];

	if(comst != sbuf)  	/* Commands before chdir */
	{	comst[-1] = '\0';
		if(callshell(sbuf)) return(1);
	}

	for(i=0; chdirct--; i++)
	{	s = chdirvec[i];
		while(*s == ' ') s++;
		comst = s;
		while(*s != '\0' && *s != ';' && *s != ' ') s++;
		if(*s == '\0') s[1] = '\0';
		*s++ = '\0';
		if(*comst != '\0')
		{	if(chdir(comst) < 0)
			{	printf("chdir - bad argument: %s\n", comst);
				sbuf[0] = '\0';
				return(0);
			}
		}
		else
		{
			if(getback() == 0) return(0);
		}

		if(*s != '\0')
		{	if(chdirct)
			{	wkpt = chdirvec[i+1] - chdtype[i+1] -1;
				*wkpt++ = '\0';
			}
			if(s != wkpt)
			{
				if(callshell(s)) return(1);
			}
		}
	}
	sbuf[0] = '\0';
	sigtrap();
	return(0);
}


ex_sigs()
{	register char *p;
	sigsoff();
	for(p = sbuf; p < sbuf+512; p++) *p = '\0';
	chdirct = 0;
	onintr();
	ex_sigtrap();
}

ex_sigtrap()
{	signal(2, ex_sigs);
	signal(3, ex_sigs);
}

gethomename()
{	register int i;
	register char *s, *t;
	nohome++;
	if(getpw(getuid()&0377, pwbuf))
	{	pwbuf[0] = 0;
		return;
	}
	s = pwbuf;
	t = pwbuf;
	i = 5;
	/* Ignore 5 ':'s */
	do
	{	while(*t++ != ':');
	} while(--i);
	while(*t != ':')
		*s++ = *t++;
	*s = '\0';
}

int getback()
{	if(nohome == 0) gethomename();
	if(chdir(pwbuf) < 0)
	{	printf("Chdir - cannot get home directory name\n");
		sbuf[0] = '\0';
		return(0);
	}
	return(1);
}
