/*
 * symbol.c - TIG-pack library software character generator
 *
 *	last edit: 02-Dec-1980  D A Gwyn
 *
 * This routine plots a character string.
 *
 * Calling sequence:
 * symbol( string , x , y , height , theta );
 *	where
 * char string[]	= null-terminated string of ASCII chars
 * unsigned x,y		= position (1000ths of inch) of LLC of string
 * int height		= char height (= spacing) (1000ths of inch)
 * double theta		= rotation angle (degrees CCW).
 *
 * Compile:
 *	# cc -c -O symbol.c
 *	# ar r /lib/libg.a symbol.o
 *
 * Usage:
 *	% cc application_program.c -lg
 */

#include	<math.h>

static char	*sstroke[128] = { 0 };	/* char starting-stroke locs */

/* bits in stroke vectors */
#define	SHIFTY	0200
#define	VISIBLE	0100
#define	XSTROKE	0070
#define	YSTROKE	0007

#define	XSHIFT	3

#define	STDHT	6.0			/* height from stroke tables */
#define	CHSPAC	6			/* char-to-char spacing */

#define	DEGRAD	57.29577951308232087679815481410517033240547246656424

#define	ASCMASK	0177			/* 7-bit mask */

#define	DOWN	0
#define	UP	1
static int	pen ;			/* pen state */

struct	fcoords
	{
	double		fx ;
	double		fy ;
	};

static struct fcoords	llc ;		/* floating-point copy of xy */
static double	symcos , symsin ;	/* transformation coeffs */

struct	coords
	{
	unsigned	x ;
	unsigned	y ;
	};


symbol( string , xy , height , theta )
	char		string[];
	struct coords	xy ;
	int		height ;
	double		theta ;
	{
	extern char	stroke[];
	register char	*p ;
	register int	c , cornerx , cpos ;
	double		angle , ht ;

	/* once-only code to initialize starting stroke pointers */
	if ( sstroke[0] == 0 )
		{
		p = stroke ;
		for ( c = 0 ; c < 128 ; ++ c )
			{
			sstroke[c] = p ;	/* starts here */
			while ( *p++ )
				;	/* 0 terminates this char */
			}
		}

	pen = DOWN ;			/* force initial lift */
	llc.fx = (double)xy.x ;
	llc.fy = (double)xy.y ;
	ht = height / STDHT ;		/* normalize */
	angle = theta / DEGRAD ;	/* convert to radians */
	symcos = cos( angle ) * ht ;
	symsin = sin( angle ) * ht ;

	for ( cpos = 0 ; c = *string++ ; ++ cpos )	/* next char */
		{
		cornerx = cpos * CHSPAC ;
		move( cornerx , 0 , 0 ) ;	/* char LLC */
		/* no move optimization done here */
		p = sstroke[c & ASCMASK];	/* -> starting stroke */
		while ( c = *p++ )	/* get stroke from table */
			move( cornerx + ((c & XSTROKE) >> XSHIFT) ,
				(c & YSTROKE) - ((c & SHIFTY) ? 2 : 0) ,
				c & VISIBLE );	/* move or draw */
		}
	penup();
	}


static
move( dx , dy , vis )			/* put adjusted stroke */
	int	dx , dy ;		/* relative to string LLC */
	int	vis ;			/* nonzero => visible */
	{
	double		xf , yf , temp ;
	struct coords	pos ;

	if ( vis && pen == UP )
		{
		pendown();
		pen = DOWN ;
		}
	else if ( ! vis && pen == DOWN )
		{
		penup();
		pen = UP ;
		}
	/* that's all the optimization we do here */

	/* in the following, we limit to the valid TIG "UPA" range */
	xf = (double)dx ;
	yf = (double)dy ;
	temp = llc.fx + xf * symcos - yf * symsin ;
	if ( temp < 0.0 )
		temp = 0.0 ;
	else if ( temp > 65535.0 )
		temp = 65535.0 ;
	pos.x = (unsigned)(temp + 0.5);
	temp = llc.fy + yf * symcos + xf * symsin ;
	if ( temp < 0.0 )
		temp = 0.0 ;
	else if ( temp > 65535.0 )
		temp = 65535.0 ;
	pos.y = (unsigned)(temp + 0.5);

	movepen( pos );
	}


/*
 * stroke tables -- each character stroke sequence is terminated by 0
 *
 * These prototypes are contained in a 4 x 6 unit area, except for
 * occasional descenders up to 2 units lower.  All visible strokes are
 * "basic" vectors, for best results with small characters.  The first
 * 16 characters are non-standard extra symbols, the next 16 are Calcomp
 * "centered plotting symbols" (not centered here!), and the final 96
 * characters are the complete ASCII graphic set (DEL plots as a grid).
 *
 * A stroke is given as SVXXXYYY bits, with:
 *	S	= 1 if YYY needs to have 2 subtracted
 *		  0 if YYY is correct as is
 *	V	= 1 if stroke is visible (draw)
 *		  0 if stroke is invisible (move)
 *	XXX	= final X coord of stroke (0..4)
 *	YYY	= final Y coord of stroke (0..6)
 */

static char	stroke[] =
	{
	0003,0105,0123,0143,0141,0121,0125,0,
	0006,0115,0112,0142,0022,0121,0141,0140,0120,0013,0133,0034,
						0114,0015,0126,0,
	0021,0125,0105,0103,0123,0141,0143,0,
	0012,0114,0034,0104,0106,0126,0124,0033,0113,0021,0141,0042,
							0122,0120,0,
	0005,0125,0134,0145,0143,0023,0125,0015,0113,0,
	0011,0131,0142,0144,0135,0115,0104,0102,0111,0012,0114,0134,
						0133,0113,0023,0132,0,
	0011,0131,0142,0144,0135,0115,0104,0102,0111,0034,0114,0112,
							0132,0,
	0021,0122,0142,0133,0134,0124,0125,0024,0114,0113,0102,0122,0,
	0012,0103,0114,0003,0143,0,
	0003,0143,0034,0143,0132,0,
	0012,0121,0132,0021,0125,0,
	0021,0125,0014,0125,0134,0,
	0012,0121,0132,0021,0125,0014,0125,0134,0,
	0012,0103,0114,0003,0143,0034,0143,0132,0,
	0004,0124,0126,0106,0104,0014,0112,0142,0034,0130,0,
	0021,0123,0013,0115,0025,0105,0003,0123,0141,0143,0,
	0023,0125,0145,0141,0101,0105,0125,0,
	0023,0125,0135,0144,0142,0131,0111,0102,0104,0115,0125,0,
	0023,0124,0142,0102,0124,0,
	0021,0125,0003,0143,0,
	0001,0145,0005,0141,0,
	0023,0125,0143,0121,0103,0125,0,
	0021,0125,0143,0103,0125,0,
	0001,0145,0105,0141,0,
	0005,0145,0101,0141,0,
	0023,0121,0005,0123,0145,0,
	0023,0145,0034,0132,0141,0032,0112,0101,0012,0114,0105,0014,
							0134,0,
	0001,0145,0025,0121,0041,0105,0003,0143,0,
	0001,0141,0105,0145,0101,0,
	0021,0125,0,
	0023,0125,0024,0142,0102,0124,0021,0122,0144,0104,0122,0,
	0023,0143,0,
	0,
	0020,0121,0022,0126,0,
	0014,0116,0036,0134,0,
	0010,0116,0036,0130,0042,0102,0004,0144,0,
	0002,0111,0131,0142,0133,0113,0104,0115,0135,0144,0026,0120,0,
	0001,0145,0025,0114,0105,0116,0125,0032,0141,0130,0121,0132,0,
	0040,0104,0105,0116,0125,0124,0102,0101,0110,0120,0142,0,
	0014,0136,0,
	0030,0112,0114,0136,0,
	0010,0132,0134,0116,0,
	0001,0145,0025,0121,0041,0105,0,
	0021,0125,0003,0143,0,
	0211,0120,0121,0,
	0003,0143,0,
	0020,0121,0,
	0001,0145,0,
	0001,0145,0136,0116,0105,0101,0110,0130,0141,0145,0,
	0010,0130,0020,0126,0115,0,
	0005,0116,0136,0145,0144,0100,0140,0,
	0001,0110,0130,0141,0142,0133,0144,0145,0136,0116,0105,0023,
								0133,0,
	0030,0136,0024,0102,0142,0,
	0001,0110,0130,0141,0143,0134,0114,0103,0106,0146,0,
	0002,0113,0133,0142,0141,0130,0110,0101,0105,0116,0136,0145,0,
	0006,0146,0145,0112,0110,0,
	0013,0102,0101,0110,0130,0141,0142,0133,0113,0104,0105,0116,
						0136,0145,0144,0133,0,
	0001,0110,0130,0141,0145,0136,0116,0105,0104,0113,0133,0144,0,
	0020,0121,0023,0124,0,
	0211,0120,0121,0023,0124,0,
	0030,0103,0136,0,
	0002,0142,0044,0104,0,
	0010,0143,0116,0,
	0005,0116,0136,0145,0144,0122,0021,0120,0,
	0031,0133,0124,0113,0112,0121,0131,0142,0144,0135,0115,0104,
						0101,0110,0130,0,
	0104,0126,0144,0140,0042,0102,0,
	0130,0141,0142,0133,0144,0145,0136,0106,0100,0003,0133,0,
	0045,0136,0116,0105,0101,0110,0130,0141,0,
	0130,0141,0145,0136,0106,0100,0,
	0003,0133,0046,0106,0100,0140,0,
	0106,0146,0033,0103,0,
	0022,0142,0141,0130,0110,0101,0105,0116,0136,0145,0,
	0106,0046,0140,0043,0103,0,
	0010,0130,0020,0126,0016,0136,0,
	0001,0110,0130,0141,0146,0,
	0106,0046,0102,0013,0140,0,
	0006,0100,0140,0,
	0106,0124,0146,0140,0,
	0106,0005,0141,0040,0146,0,
	0010,0130,0141,0145,0136,0116,0105,0101,0110,0,
	0106,0136,0145,0144,0133,0103,0,
	0010,0130,0141,0145,0136,0116,0105,0101,0110,0022,0140,0,
	0106,0136,0145,0144,0133,0103,0013,0140,0,
	0001,0110,0130,0141,0142,0133,0113,0104,0105,0116,0136,0145,0,
	0020,0126,0006,0146,0,
	0006,0101,0110,0130,0141,0146,0,
	0006,0102,0120,0142,0146,0,
	0006,0100,0122,0140,0146,0,
	0101,0145,0146,0006,0105,0141,0140,0,
	0020,0123,0105,0106,0046,0145,0123,0,
	0040,0100,0101,0145,0146,0106,0013,0133,0,
	0030,0110,0116,0136,0,
	0005,0141,0,
	0010,0130,0136,0116,0,
	0004,0126,0144,0,
	0201,0341,0,
	0016,0134,0,
	0003,0114,0134,0143,0140,0042,0112,0101,0110,0130,0141,0,
	0106,0001,0110,0130,0141,0143,0134,0114,0103,0,
	0043,0134,0114,0103,0101,0110,0130,0141,0,
	0043,0134,0114,0103,0101,0110,0130,0141,0040,0146,0,
	0002,0142,0143,0134,0114,0103,0101,0110,0130,0141,0,
	0010,0115,0126,0136,0145,0034,0104,0,
	0201,0310,0330,0341,0144,0041,0130,0110,0101,0103,0114,0134,
								0143,0,
	0106,0003,0114,0134,0143,0140,0,
	0020,0124,0114,0025,0126,0,
	0201,0310,0330,0341,0144,0045,0146,0,
	0106,0044,0100,0022,0140,0,
	0020,0126,0116,0,
	0104,0003,0114,0123,0120,0040,0143,0134,0123,0,
	0104,0003,0114,0134,0143,0140,0,
	0010,0130,0141,0143,0134,0114,0103,0101,0110,0,
	0001,0110,0130,0141,0143,0134,0114,0103,0004,0300,0,
	0041,0130,0110,0101,0103,0114,0134,0143,0044,0340,0,
	0104,0003,0114,0134,0143,0,
	0001,0110,0130,0141,0132,0112,0103,0114,0134,0143,0,
	0004,0134,0015,0111,0120,0130,0141,0,
	0004,0101,0110,0130,0141,0040,0144,0,
	0004,0102,0120,0142,0144,0,
	0004,0101,0110,0121,0022,0121,0130,0141,0144,0,
	0144,0004,0140,0,
	0201,0310,0330,0341,0144,0004,0101,0110,0130,0141,0,
	0004,0144,0100,0140,0,
	0030,0121,0122,0113,0124,0125,0136,0,
	0020,0126,0,
	0010,0121,0122,0133,0124,0125,0116,0,
	0005,0116,0134,0145,0,
	0140,0146,0106,0100,0010,0116,0026,0120,0030,0136,0,
	};
