/* FILE PERT_Xstubs.c ****************************************
 *
 * xplan - project planning tool
 * Copyright (C) 1992 Brian Gaubert, Mark M. Lacey, Richard Malingkas,
 * and Mike Marlow.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License (distributed with this program in the file
 * COPYING) for more details.
 * 
 * If you did not received a copy of the GNU General Public License
 * along with this program, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * Since this was a project for a one semester software engineering
 * course, the authors will not be offering support for the product
 * after its release.
 *
 * DESCRIPTION OF CONTENTS
 *
 * Notify and event callback functions for the PERT chart window.
 * Originally generated by guide and then removed from the guide
 * system due to guide's little support of scrollbars.
 *
 */

#include <stdio.h>
#include <sys/param.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/textsw.h>
#include <xview/xv_xrect.h>
#include "db.h"
#include "julian.h"
#include "xplan.h"
#include "PERT_Xui.h"

/* FUNCTION PERT_Popup_resize_callback ***********************************

   PURPOSE

   Resize callback for function `canvas'.

   INPUTS

   canvas --- the canvas that was resized

   width --- the new width of the canvas

   height --- the new height of the canvas

   AUTHOR/AUDITOR/TESTER
   
   Author.... Mark M. Lacey, 7 Dec 1992
   Tester.... Mark M. Lacey, 7 Dec 1992
*/
void
PERT_Popup_resize_callback(Canvas canvas, int width, int height)
{
	/* gxv_start_connections DO NOT EDIT THIS SECTION */

	/* gxv_end_connections */

}

/* FUNCTION PERT_Popup_repaint_callback ***********************************

   PURPOSE

   Repaint callback for function `canvas'.

   INPUTS

   canvas --- the canvas that needs repainting

   paint_window --- the window to be repainted

   display --- a pointer to the diplay

   xid --- the X identifier for this window

   rects --- a list of objects to be drawn - not used

   AUTHOR/AUDITOR/TESTER
   
   Author.... Mark M. Lacey, 7 Nov 1992
   Tester.... Mark M. Lacey, 7 Dec 1992
*/
void
PERT_Popup_repaint_callback(Canvas canvas, Xv_window paint_window, Display *display, Window xid, Xv_xrectlist *rects)
{
   draw_pert_boxes(display, xid);
}

/* FUNCTION draw_pert_boxes *****************************************

   PURPOSE

   Serves as a driver function function to draw the PERT boxes and
   lines.

   INPUTS

   display --- a pointer to the diplay

   xid --- the X identifier for this window

   AUTHOR/AUDITOR/TESTER
   
   Author.... Mark M. Lacey, 7 Nov 1992
   Tester.... Mark M. Lacey, 7 Dec 1992
*/
void draw_pert_boxes(Display *display, Window xid)
{
   GC gc;
   struct task_node *temp;
   int tasknum=1;

   /* get the default graphics context */
   gc = DefaultGC(display, DefaultScreen(display));

   /* get the head of the task list */
   temp = get_main_task_list()->head;

   /* go through the list and draw each box and then each line */
   while (temp) {
      /* as long as it either has a dependency or is a dependency of */
      /* something else, draw it --- otherwise don't, since its */
      /* not part of the task network */
      if (temp->data->dependencies->head != NULL ||
	  temp->data->dependents->head != NULL) {

	 /* draw the box */
	 draw_pert_box(display, xid, gc, temp->data, tasknum);

	 /* draw all the dependency lines */
	 draw_dependency_lines(display, xid, gc, temp->data,
			       temp->data->dependencies);
      }
      temp = temp->next;
      ++tasknum;
   }
}

/* FUNCTION draw_pert_box *****************************************

   PURPOSE

   Draws a single PERT box to the window.

   INPUTS

   display --- a pointer to the diplay

   xid --- the X identifier for this window

   gc --- the graphics context to use

   data --- the task that we are going to draw the box for

   tasknum --- the task number of the task

   AUTHOR/AUDITOR/TESTER
   
   Author.... Mark M. Lacey, 7 Nov 1992
   Tester.... Mark M. Lacey, 7 Dec 1992
*/
void draw_pert_box(Display *display, Window xid, GC gc, struct
		   task_info *data, int tasknum)
{
   int x, y;
   char str[30];
   char *tmp;

   /* find the upper-left coordinate */
   x = data->x_pert*(PERT_BOX_WIDTH+PERT_BOX_HORIZ_SPACE);
   y = (data->y_pert+1)*(PERT_BOX_HEIGHT+PERT_BOX_VERT_SPACE);

   /* Draw the box itself */
   XDrawRectangle(display, xid, gc, x, y, PERT_BOX_WIDTH,
		  PERT_BOX_HEIGHT);

   /* copy the name into a field 25 long */
   sprintf(str, "%.25s", data->name);

   /* Display the name of the task */
   XDrawString(display, xid, gc, 
	       x+STRING_X_OFFSET, y+PERT_BOX_HEIGHT/3+STRING_Y_OFFSET,
	       str, strlen(str));

   /* get a string representation for the task number */
   sprintf(str, "%d", tasknum);

   /* Display the task number */
   XDrawString(display, xid, gc, 
	       x+STRING_X_OFFSET, y+STRING_Y_OFFSET,
	       str, strlen(str));

   /* Draw horizontal line 2/3rds down from top */
   XDrawLine(display, xid, gc, x, y+2*PERT_BOX_HEIGHT/3,
	     x+PERT_BOX_WIDTH, y+2*PERT_BOX_HEIGHT/3);

   /* Draw vertical line in the middle, starting 2/3rds down from top */
   XDrawLine(display, xid, gc, 
	     x+PERT_BOX_WIDTH/2, y+2*PERT_BOX_HEIGHT/3,
	     x+PERT_BOX_WIDTH/2, y+PERT_BOX_HEIGHT);

   /* convert the date to a string representation, and display it */
   tmp = julian_to_str_date(data->planned_start_date);
   strcpy(str, tmp);
   
   /* put in the planned start date */
   XDrawString(display, xid, gc, 
	       x+STRING_X_OFFSET,
	       y+2*PERT_BOX_HEIGHT/3+STRING_Y_OFFSET,
	       str, strlen(str));
   
   /* convert the date to a string representation, and display it */
   tmp = julian_to_str_date(data->planned_end_date);
   strcpy(str, tmp);
   
   /* put in the planned end date */
   XDrawString(display, xid, gc, 
	       x+PERT_BOX_WIDTH/2+STRING_X_OFFSET,
	       y+2*PERT_BOX_HEIGHT/3+STRING_Y_OFFSET,
	       str, strlen(str));
}

/* FUNCTION draw_dependency_lines **************************************

   PURPOSE

   Draws a single PERT box to the window.

   INPUTS

   display --- a pointer to the diplay

   xid --- the X identifier for this window

   gc --- the graphics context to use

   data --- the task that we are going to draw the box for

   dependencies --- the list of dependencies for this task

   AUTHOR/AUDITOR/TESTER
   
   Author.... Mark M. Lacey, 7 Nov 1992
   Tester.... Mark M. Lacey, 7 Dec 1992
*/
void draw_dependency_lines(Display *display, Window xid, GC gc, struct
			   task_info *data, struct task_list
			   *dependencies)
{
   struct task_node *temp;
   int x_start, x_end, y_start, y_end;
   XGCValues gcv;

   /* get the head of the dependency list */
   temp = dependencies->head;

   /* find out where all these lines will be originating from */
   x_start =
     data->x_pert*(PERT_BOX_WIDTH+PERT_BOX_HORIZ_SPACE);
   y_start =
     (data->y_pert+1)*(PERT_BOX_HEIGHT+PERT_BOX_VERT_SPACE)+
       PERT_BOX_HEIGHT/2;
   
   /* for each dependency, draw out the line */
   while (temp) {

      /* if it is part of the critical path, make the lines thicker */
      if (data->critical_path && temp->data->critical_path) {
	 gcv.line_width = 3;
	 XChangeGC(display, gc, GCLineWidth, &gcv);
      } else {
	 gcv.line_width = 0;
	 XChangeGC(display, gc, GCLineWidth, &gcv);
      }
      
      /* find end of the line */
      x_end =
	temp->data->x_pert*(PERT_BOX_WIDTH+PERT_BOX_HORIZ_SPACE)+
	  PERT_BOX_WIDTH;
      y_end =
	(temp->data->y_pert+1)*(PERT_BOX_HEIGHT+PERT_BOX_VERT_SPACE)+
	  PERT_BOX_HEIGHT/2;
      
      /* actually draw the line */
      XDrawLine(display, xid, gc, x_start, y_start, x_end, y_end);

      /* move on to the next dependency */
      temp = temp->next;
   }

   /* change the GC back */
   gcv.line_width = 0;
   XChangeGC(display, gc, GCLineWidth, &gcv);
}



