/*
**	Patterns and functions to call using the Netcomm ``Trailblazer'' modem
**	(mostly Hayes compatible.)
**
**
**	Here are suggested Sun III setup parameters for use with Trailblazer modems.
**	These parameters give an average real data throughput of about 680 bytes/sec.
**	to/from somewhat overloaded VAX 11/780s, and allow connection at the slower
**	speeds as well by sending <BREAK> to trigger the getty speed selection cycle.
**
**	The calling Trailblazer should be configured with the following command:-
**
**	at &f s45=0 s50=0 s51=4 s52=2 s54=3 s58=3 s66=0 s68=3 s110=1 q6 &w
**
**	and the ``getty'' Trailblazer with this command:-
**
**	at &f s45=0 s50=0 s51=4 s52=2 s53=3 s54=3 s55=3 s58=3 s66=0 s68=3 s110=1 e0 q3 &w
**
**	The call program should invoke ``PNdaemon'' with the following parameters
**	(and any others) in the ``params'' file (at both ends of the link):-
**		-CXNH -b8 -f2 -s500 -w1 -z512
**
**	These configure ``cooked'' protocol with XON/XOFF flow control,
**	and half-duplex message transfer on one channel.
**
**	I've tried ``-z1024'' and it seems sometimes to achieve around 850 bytes/sec.
**	but in general gets more errors. This may be due to overloaded vaxen, and
**	so you might try the larger buffers on faster CPUs (with DMA tty i/o.)
**
**	Those of you with a version of ``NNdaemon/recvControl.c'' less than 1.20
**	should ``fetchfile'' from basser the later versions of the daemon source.
**	(Try ``fetchfile -LVdbasser.cs.su.oz ACSnet/NNdaemon''.)
*/

#include	"global.h"
#include	"caller.h"
#include	<signal.h>

#define TRACE

#define	MAXPHONES	8

extern	int	Traceflag;

char
	*callargs	= CALLARGS,
	*daemon1	= NNDAEMON,	/* standard daemon */
	*daemon2	= NN2DAEMON,	/* alternate protocol */
	*daemon3	= PNDAEMON,	/* high-speed protocol */
	*hostflg	= "-BF",	/* dialup, don't fork, cooked */
	*host	 	= NULLSTR,	/* connecting from where? */
	*target  	= NULLSTR,	/* connecting to where? */
	*passwd  	= NULLSTR,
	*dialpw  	= NULLSTR,
	*afterst	= "ATE0H0M1V1X3Q6S0=2",
	*ph_num[MAXPHONES + 1],		/* array of phone number pointers */
	**phoneno	= ph_num,	/* Current phone number to call */
	*device  	= NULLSTR,	/* Which call unit? (cul0) */
	*speed   	= "EXTA",
	*opentime	= "60",
	*logintime	= "35";		/* MUST be greater then 15 seconds
						(modem carrier detect time) */


int
	rclcount = 0,	/* count of resets calls */
	rcount	 = 0,	/* count of resets */
	tcount	 = 0,	/* count of timeouts */
	ecount	 = 0,	/* count of unexpected eofs */
	lcount	 = 0,	/* count of Unix login attempts */
	dcount	 = 0,	/* count of dial attempts */
	flush	 = 0,	/* flush input on connect flag */
	ccitt	 = 1;	/* 1200 baud using V22 standard */

int
	callbusy(),
	callefail(),
	calltfail(),
	eof(),
	gotlogin(),
	gotpasswd(),
	gotdialpw(),
	ignoreline(),
	nocarrier(),
	resettimo(),
	resetunit(),
	start2(),
	started(),
	startpn(),
	timeout(),
	tocallunit(),
	tounix1200(),
	tounix19200(),
	tounix2400(),
	tounix4800(),
	tounix();

struct patlist atreset[] =
{
	{ "OK",    tocallunit },
	{ TIMEOUT, resettimo },
	{ 0, 0 }
};


struct patlist atcallunit[] =
{
	{ "CONNECT",	  tounix19200 },
	{ "BUSY",	  callbusy },
	{ "NO CARRIER",   nocarrier },
	{ "ERROR",	  callefail },
	{ TIMEOUT,	  calltfail },
	{ 0, 0 }
};

struct patlist atunix[] =
{
	{ "[Ii]llegal", ignoreline },
	{ "[Ii]ncorrect", ignoreline },
	{ "[Ll]ogin", gotlogin },
	{ "[Dd]ial[iu][np] [Pp]assword", gotdialpw },
	{ "[Ww]rong [Pp]assword", ignoreline },
	{ "[Pp]assword", gotpasswd },
	{ PNSTARTS, startpn },	/* high-speed protocol */
	{ START2MSG, start2 },	/* alternate protocol */
	{ STARTMSG, started},
	{ "NO CARRIER", nocarrier },
	{ TIMEOUT, timeout },
	{ EOFSTR, eof },
	{ 0, 0 }
};

args(argc, argv)
	register int	argc;
	register char *	argv[];
{
	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'b':
					ccitt = 0;	/* Bell 212 1200 */
					break;
				case 'c':
					ccitt = 1;	/* CCITT V22 1200 */
					break;
				case 'F':
					flush = 1;	/* Flush input on connect */
					break;
				case 'd':
					device = ++*argv;
					goto break2;
				case 'l':
					host = ++*argv;
					goto break2;
				case 's':
					speed = ++*argv;
					goto break2;
				case 'p':
					if (phoneno != &ph_num[MAXPHONES])
					{
					    *phoneno++ = ++*argv;
					    goto break2;
					}
					Command("fail too many phone numbers", NULLSTR);
					exit(1);
				case 'P':
					passwd = ++*argv;
					goto break2;
				case 'D':
					dialpw = ++*argv;
					goto break2;
				case 'a':
					afterst = ++*argv;
					goto break2;
				case '1':
					daemon1 = ++*argv;
					goto break2;
				case '2':
					daemon2 = ++*argv;
					goto break2;
				case 'T':
#ifdef TRACE
					Traceflag = atoi(++*argv);
#endif
					goto break2;
				default:
					Command("fail unexpected flag \"-", *argv, "\" in ", CALLARGS, NULLSTR);
					exit(1);
				}
			}
break2:			;
		}
		else
			target = *argv;
	}
}

setafter()
{
	Command("after sleep 2", NULLSTR);
	Command("after write +++", NULLSTR);
	Command("after sleep 2", NULLSTR);
	Command("after speed ", speed, NULLSTR);
	Command("after write \r", NULLSTR);
	Command("after write ", afterst, "\r", NULLSTR);
	Command("after close", NULLSTR);
}

sigabort()
{
#ifdef	TRACE
	if (Traceflag)
		Command("trace sigterm received - exiting", NULLSTR);	/**/
#endif	TRACE
	Command("sleep 2", NULLSTR);
	flushinput();
	Command("write +++", NULLSTR);
	Command("sleep 2", NULLSTR);
	flushinput();
	fail("sigterm received");
}

void
init(argc, argv)
	int	argc;
	char	*argv[];
{

	args(argc, argv);
	if (target == NULLSTR)
	{
	    Command("fail no target", NULLSTR);
	    exit(1);
	}

	callargs = concat(SPOOLDIR(), target, "/", callargs, NULLSTR);
	(void)readargs(callargs, args);
	phoneno = ph_num;
	if (*phoneno == NULLSTR)
	{
	    Command("fail no phone numbers", NULLSTR);
	    exit(1);
	}
	signal(SIGTERM, sigabort);
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace opening line......", NULLSTR);
#endif	TRACE
	Command("timeout ", opentime, NULLSTR);
	Command("opendial ", device, NULLSTR);
	Command("speed ", speed, NULLSTR);
	Command("write \r", NULLSTR);
	Command("timeout ", logintime, NULLSTR);
	setafter();
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace starting reader ......", NULLSTR);
#endif	TRACE
	Command("read", NULLSTR);
	resetunit();
}

resetunit()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace resetting modem ......", NULLSTR);
#endif	TRACE
	if (++rclcount > 9)
		fail("call fail - modem reset called 10 times.");
	state(atreset);
	Command("sleep 2", NULLSTR);
	flushinput();
	Command("write +++", NULLSTR);
	Command("sleep 2", NULLSTR);
	flushinput();
	Command("speed ", speed, NULLSTR);
	Command("write \r", NULLSTR);
	Command("write ATH0", NULLSTR);
	Command("write Q6E0", NULLSTR);
	Command("write X3V1", NULLSTR);
#ifdef	TRACE
	if (Traceflag)
		Command("write M1", NULLSTR);
	else
#endif	TRACE
	Command("write M0", NULLSTR);
	Command("write S0=0\r", NULLSTR);
	reset();
}

resettimo()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace issuing another reset comand ......", NULLSTR);
#endif	TRACE
	if (++rcount > 9)
		fail("call fail - 10 modem resets failed.");
	state(atreset);
	Command("sleep 2", NULLSTR);
	flushinput();
	Command("write +++", NULLSTR);
	Command("sleep 2", NULLSTR);
	flushinput();
	Command("speed ", speed, NULLSTR);
	Command("write \r", NULLSTR);
	Command("write ATH0", NULLSTR);
	Command("write Q6E0", NULLSTR);
	Command("write X3V1", NULLSTR);
#ifdef	TRACE
	if (Traceflag)
		Command("write M1", NULLSTR);
	else
#endif	TRACE
	Command("write M0", NULLSTR);
	Command("write S0=0\r", NULLSTR);
	reset();
}

tocallunit()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace dialing......", NULLSTR);
#endif	TRACE
	state(atcallunit);
	Command("write ATD", *phoneno++, "\r", NULLSTR);
	if (*phoneno == NULLSTR)
	    phoneno = ph_num;
	reset();
}

callbusy()
{
	if (++dcount > 9)
		fail("was busy 10 times");
	else
	{
#ifdef	TRACE
		if (Traceflag)
			Command("trace reset modem - call busy", NULLSTR);
#endif	TRACE
		resetunit();
	}
}

calltfail()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace waiting......", NULLSTR);
#endif	TRACE
	if (++dcount > 9)
		fail("timeout 9 times");
	else
		reset();
}

callefail()
{
	if (++dcount > 9)
		fail("error");
	else
	{
#ifdef	TRACE
		if (Traceflag)
			Command("trace reset modem - error", NULLSTR);
#endif	TRACE
		flushinput();
		resetunit();
	}
}

ignoreline()
{
	reset();
}

nocarrier()
{
	if (++dcount > 9)
		fail("no carrier");
	else
	{
#ifdef	TRACE
		if (Traceflag)
			Command("trace reset modem - no carrier", NULLSTR);
#endif	TRACE
		resetunit();
	}
}

tounix1200()
{
	tounix("1200");
}

tounix19200()
{
	tounix("EXTA");
}

tounix2400()
{
	tounix("2400");
}

tounix4800()
{
	tounix("4800");
}

tounix(sp)
char *sp;
{
	if (strcmp(speed, sp) != 0)
	{
		Command("speed ", sp, NULLSTR);
	}
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace waiting for login......", NULLSTR);
#endif	TRACE
	if (flush)
	{
		Command("sleep 5", NULLSTR);
		flushinput();	/* toss away junk */
		Command("write ", host, "\r", NULLSTR);
	}
	state(atunix);
	reset();
}

gotlogin()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace got login ......", NULLSTR);
#endif	TRACE
	if ( ++lcount >= 9 )
		fail("too many login attempts");
	Command("write ", host, "\r", NULLSTR);
	reset();
}


gotpasswd()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace got passwd ......", NULLSTR);
#endif	TRACE
	Command("write ", passwd, "\r", NULLSTR);
	reset();
}


gotdialpw()
{
#ifdef	TRACE
	if (Traceflag >= 2 )
		Command("trace got dialin passwd ......", NULLSTR);
#endif	TRACE
	Command("write ", dialpw, "\r", NULLSTR);
	reset();
}


startpn()
{
#ifdef	TRACE
	if (Traceflag)
		Command("trace call succeeded, PNdaemon starting", NULLSTR);	/**/
#endif	TRACE
	Command("daemon ", daemon3, NULLSTR);
	Command("succeed ", hostflg, " ", target, NULLSTR);
	exit(0);
}


started()
{
#ifdef	TRACE
	if (Traceflag)
		Command("trace call succeeded, daemon starting", NULLSTR);	/**/
#endif	TRACE
	Command("succeed ", target, NULLSTR);
	exit(0);
}


start2()
{
#ifdef	TRACE
	if (Traceflag)
		Command("trace call succeeded, daemon 2 starting", NULLSTR);	/**/
#endif	TRACE
	Command("daemon ", daemon2, NULLSTR);
	Command("succeed ", hostflg, " ", target, NULLSTR);
	exit(0);
}


timeout()
{
	if (++tcount >= 9)
	{
		Command("sleep 2", NULLSTR);
		Command("write +++", NULLSTR);
		Command("sleep 2", NULLSTR);
		fail("9 timeouts in login");
	}
	if (tcount & 1)
		Command("write \r", NULLSTR);
	reset();
}

eof()
{
	Command("close", NULLSTR);
	Command("timeout ", opentime, NULLSTR);
	Command("opendial ", device, NULLSTR);
	Command("speed ", speed, NULLSTR);
	Command("write \r", NULLSTR);
	Command("timeout ", logintime, NULLSTR);
	if (++ecount > 5)
		fail("unexpected eof");
#ifdef	TRACE
	if (Traceflag)
		Command("trace unexpected eof", NULLSTR);	/**/
#endif	TRACE
	Command("read", NULLSTR);
	resetunit();
}

fail(s)
register char *s;
{
	Command("write \r", NULLSTR);
	Command("write ", afterst, "\r", NULLSTR);
	Command("fail ", s, NULLSTR);
	exit(1);
}
