/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)printer.c	1.9 87/10/26";

/*
**	Process print jobs from message received from network.
*/

#define	FILE_CONTROL
#define	STAT_CALL
#define	RECOVER
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"debug.h"
#include	"ftheader.h"
#include	"header.h"
#include	"spool.h"
#include	"sub_proto.h"

#include	<errno.h>

/*
**	Parameters set from arguments.
*/

bool	Broadcast;		/* Message has been everywhere */
char *	CommandsFile;		/* Commands of files to be pre-pended to Message */
char *	HomeNode;		/* Name of this node */
char *	LinkNode;		/* Message arrived from this node */
char *	Message;		/* Message */
char *	Name;			/* Program invoked name */
char *	SourceNode;		/* Message originated at this node */
Time_t	StartTime;		/* Time message started at source */
int	Traceflag;		/* Global tracing control */

/*
**	Miscellaneous
*/

bool	Ack;			/* Acknowledgement of delivery received */
char *	DataName;		/* Used by ExpandArgs() */
FthFD_p*FilesList;		/* Sorted vector of file descriptors */
int	Pid;			/* Used by UniqueName() in Exec...() */
char	ProtoType[2];		/* Header protocol type */
bool	Returned;		/* True if invoked for returned message */
int	RetVal;			/* Value returned by program */
char *	SenderName;		/* Used by ExpandArgs() */
Time_t	Time;			/* Used by UniqueName() in Exec...() */
char *	UserName;		/* Used by ExpandArgs() */

char *	AckMessage	= WORKDIR(ack....message);

#define	Fprintf		(void)fprintf
#define	free(A)

void	finish(), printfiles(), returnmsg(), recvack(), sendack(), setuser();
void	sortfiles();
int	byname();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	register int	fd;
	FthReason	reason;
	char *		cp;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'B':
					Broadcast = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'c':
					CommandsFile = ++*argv;
					goto break2;

				case 'd':
					DataLength = atol(++*argv);
					break;

				case 'e':
					HdrEnv = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 's':
					SourceNode = ++*argv;
					goto break2;

				case 't':
					StartTime = Time - atol(++*argv);
					goto break2;

				default:
					Trace2(1, "unrecognised flag '%c'", c);
					goto break2;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		if ( Message != NULLSTR )
			Error("only one message allowed");
		else
			Message = *argv;
	}

	if ( (cp = GetEnv(ENV_RETURNED)) != NULLSTR )
	{
		free(cp);
		Returned = true;
	}

	if ( (cp = GetEnv(ENV_ACK)) != NULLSTR )
	{
		free(cp);
		Ack = true;
	}

	while ( (fd = open(Message, O_READ)) == SYSERROR )
		Syserror("Can't open \"%s\"", Message);

	if
	(
		(reason = ReadFtHeader(fd, DataLength, !Returned && !Ack)) != fth_ok
		||
		(reason = GetFthFiles()) != fth_ok
	)
		Error("File transfer header \"%s\" error", FTHREASON(reason));

	if ( Ack || Returned )
	{
		sortfiles();
		recvack();
	}
	else
#	ifdef	PRINTER
#	ifdef	PRINTORIGINS
	{
		if ( InList(SourceNode, PRINTORIGINS, NULLSTR) )
			printfiles(fd);
		else
			sendack(concat(SourceNode, " is not a legal origin for print jobs", NULLSTR));
	}
#	else	PRINTORIGINS
		sendack(concat(SourceNode, " is not a legal origin for print jobs", NULLSTR));
#	endif	PRINTORIGINS
#	else	PRINTER
		sendack(concat(HomeNode, " has no printer", NULLSTR));
#	endif	PRINTER

	exit(RetVal);
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int	error;
{
	(void)unlink(AckMessage);
	(void)exit(error);
}



#ifdef	PRINTER

/*
**	Read the file transfer header from the message containing the file(s),
**	check and acknowledge the data if requested,
**	identify any users at this host, and pass the print jobs
**	to the local print spooler with any optional flags.
*/

void
printfiles(ifd)
	int		ifd;
{
	register char *	cp;
	register int	n;
	register int	r;
	register FILE *	fd;
	char *		errs;
	VarArgs		va;
	bool		piperr;
	char		buf[BUFSIZ];

	Recover(ert_return);

	SenderName = FthFrom;
	UserName = FthTo;
	DataName = FthFiles->f_name;

	FIRSTARG(&va) = PRINTER;

#	ifdef	PRINTERARGS
	ExpandArgs(&va, PRINTERARGS, NULLSTR);
#	endif	PRINTERARGS

	if ( (cp = GetEnv(ENV_HANDLER_FLAGS)) != NULLSTR )
	{
		SplitArg(&va, cp);
		free(cp);
	}

	fd = ExecPipe(&va);

	if ( CommandsFile != NULLSTR )
		CopyFromComFile(CommandsFile, fileno(fd), "pipe", NULLSTR);

	(void)lseek(ifd, (long)0, 0);

	piperr = false;

	while
	(
		(r = FtDataLength > sizeof buf ? sizeof buf : FtDataLength) > 0
		&&
		(n = read(ifd, buf, r)) > 0
	)
	{
		FtDataLength -= n;
		cp = buf;

		while ( (r = write(fileno(fd), cp, n)) != n )
		{
			if ( r == SYSERROR )
			{
				if ( errno == EPIPE )
				{
					piperr = true;
					goto break2;
				}

				Syserror("Can't write to pipe!");
			}
			else
			{
				cp += r;
				n -= r;
			}
		}
	}

	if ( n == SYSERROR )
		Syserror("Can't read message \"%s\"", Message);

break2:
	if
	(
		(errs = ExPipeClose(fd)) != NULLSTR
		||
		(piperr && (errs = newstr("broken pipe")))
	)
	{
		sendack(errs);			/* send NAK */
	}
	else
		if ( FthType[0] & FTH_ACK )
			sendack(NULLSTR);	/* send ack */

	Recover(ert_finish);
}

#endif	PRINTER



/*
**	Sort files in message
*/

void
sortfiles()
{
	register FthFD_p	fp;
	register FthFD_p *	fpp;

	FilesList = fpp = (FthFD_p*)Malloc(sizeof(FthFD_p)*NFthFiles);

	for ( fp = FthFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
		*fpp++ = fp;
	
	if ( NFthFiles > 1 )
		qsort((char *)FilesList, NFthFiles, sizeof(FthFD_p), byname);
}


/*
**	Alphabetically
*/

int
byname(fpp1, fpp2)
	char *	fpp1;
	char *	fpp2;
{
	return strcmp((*(FthFD_p *)fpp1)->f_name, (*(FthFD_p *)fpp2)->f_name);
}



/*
**	Make up message for returned files.
*/

void
returnmsg(fd)
	FILE *		fd;
{
	register char *	cp;

	Fprintf
	(
		fd, 
		"Subject: Print spooler failed at %s\n\n",
		SourceNode
	);

	Fprintf
	(
		fd,
		"\nPRINT JOB \"%s\" NOT SPOOLED AT \"%s\"\n\n",
		FthFiles->f_name,
		SourceNode
	);

	if ( (cp = GetEnv(ENV_ERR1)) != NULLSTR )
	{
		Fprintf(fd, "Failure explanation follows :-\n");
		Fprintf(fd, "%s\n\n", cp);
		free(cp);
	}
}



/*
**	Send an acknowledgement message to source.
*/

void
sendack(err_mesg)
	char *			err_mesg;
{
	register FthFD_p	fp;
	register long		size;
	register char *		errs;
	int			fd;
	VarArgs			va;

	if ( err_mesg == NULLSTR )
		HdrEnv = MakeEnv(ENV_ACK, NULLSTR, ENV_NORET, NULLSTR, NULLSTR);
	else
	{
		HdrEnv = MakeEnv
			 (
				ENV_RETURNED, NULLSTR,
				ENV_ERR1, err_mesg,
				NULLSTR
			 );

		free(err_mesg);
	}

	HdrDest = SourceNode;
	HdrSource = HomeNode;
	HdrHandler = Name;

	Time = time((long *)0);

	for ( fp = FthFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
		fp->f_time = Time;

	SetFthFiles();

	while ( (fd = creat(UniqueName(AckMessage, (long)0, Time), 0600)) == SYSERROR )
		Syserror("Can't creat \"%s\"", AckMessage);

	while ( (size = WriteFtHeader(fd, (long)0, false, false)) == SYSERROR )
		Syserror("Can't write \"%s\"", AckMessage);

	ProtoType[0] = FTP;
	HdrSubpt = ProtoType;

	while ( WriteHeader(fd, size, 0) == SYSERROR )
		Syserror("Can't write \"%s\"", AckMessage);
	
	(void)close(fd);

	FIRSTARG(&va) = RECEIVER;
	NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);
	NEXTARG(&va) = AckMessage;

	if ( (errs = Execute(&va)) != NULLSTR )
	{
		Error("Acknowledgement failed;\n%s\n", errs);
		free(errs);
		(void)unlink(AckMessage);
		RetVal = 1;
	}
}



/*
**	Receive an acknowledgement generated by 'sendack()'.
**
**	Notify sender of acknowledgement.
*/

void
recvack()
{
	register int		i;
	register FILE *		fd;
	register FthFD_p *	fpp;
	char *			errs;
	VarArgs			va;

	SenderName = FthTo;	/* Ack from receiver of files */

	FIRSTARG(&va) = BINMAIL;

#	ifdef	BINMAILARGS
	ExpandArgs(&va, BINMAILARGS, NULLSTR);
#	endif	BINMAILARGS

	NEXTARG(&va) = FthFrom;	/* Ack to original sender */

	fd = ExecPipe(&va);
	
	Fprintf(fd, "From postmaster %s", ctime(&Time));

	if ( Returned )
		returnmsg(fd);
	else
	{
		Fprintf
		(
			fd,
			"Subject: Acknowledgement received for %s printed at %s\n\n",
			NFthFiles>1?"jobs":FthFiles->f_name,
			SourceNode
		);

		Fprintf
		(
			fd,
			"The following print %s spooled on %.12s at %s:\n      Size   Spool time    Name\n",
			NFthFiles>1?"jobs were":"job was",
			ctime(&StartTime)+4,
			SourceNode
		);

		for ( fpp = FilesList, i = NFthFiles ; --i >= 0 ; fpp++ )
		{
			Fprintf
			(
				fd,
				"%10ld %.15s %s\n",
				(*fpp)->f_length,
				ctime(&(*fpp)->f_time)+4,
				(*fpp)->f_name
			);
		}
	}

	if ( (errs = ExPipeClose(fd)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}
}
