/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)recvControl.c	1.11 85/07/21
*/

/*
**	Deal with control message from link.
**	All messages have an "id",
**	some may also have a size,
**	and a SOM has a wait time.
*/

#include	"global.h"
#include	"command.h"
#include	"debug.h"

#include	"Stream.h"
#include	"driver.h"
#include	"AQ.h"
#include	"CQ.h"
#include	"proto.h"



void
recvControl(chan, data, size)
	int		chan;
	char *		data;
	int		size;
{
	register int	i;
	register char *	cp;
	register Str_p	strp;
	register int	items;
	char *		item[3];

	Trace4
	(
		 1
		,"recvControl for channel %d inState %d outState %d"
		,chan
		,inStreams[chan].str_state
		,outStreams[chan].str_state
	);

	if ( data[0] != MQ_EMPTY )
	{
		/*
		**	Data in control message are null terminated strings.
		*/

		size--;

		cp = &data[1];
		item[0] = cp;

		for ( items = 0, i = 0 ; i < size ; i++, cp++ )
			if ( (*cp & 0177) == '\0' )
			{
				*cp = '\0';
				if ( ++items >= 3 )
					break;
				item[items] = cp+1;
			}

		if
		(
			items == 0
			||
			(item[1] - item[0]) > (STREAMIDZ+1)
			||
			i < (size-1)
		)
		{
			Fatal4("bad control message from channel %d: \"%.*s\"", chan, size, &data[1]);
			return;
		}

		Trace3(1, "recvControl \"%d\" id=\"%s\"", data[0], item[0]);
	}
	else
		Trace1(1, "recvControl MQ_EMPTY");

	switch ( data[0] )
	{
	case SOM:
		/*
		**	Start-Of-Message.
		**	"data" contains a message "id", its size, and its wait time.
		*/
		
		if ( items != 3 )
		{
			Fatal4
			(
				 "bad SOM format from channel %d: \"%.*s\""
				,chan
				,size
				,&data[1]
			);
			return;
		}

		Trace3(1, "SOM size %s wait %s", item[1], item[2]);

		strp = &inStreams[chan];

		switch ( strp->str_state )
		{
		case STR_ENDING:
			Report4
			(
				"SOM received for id \"%s\" while stream %d ending id \"%s\"",
				item[0],
				chan,
				strp->str_id
			);
			return;	/* EOMA will sort things out ... */

		case STR_AGAIN:
			if
			(
				strcmp(strp->str_id, item[0]) == STREQUAL
				&&
				strp->str_size == atol(item[1])
			)
				strp->str_flags |= STR_DUP;
			else
		case STR_ENDED:
			if
			(
				strcmp(strp->str_id, item[0]) == STREQUAL
				&&
				strp->str_size == atol(item[1])
			)
			{
				SndEOMA(chan);
				return;
			}
		case STR_IDLE:
		case STR_START:
		case STR_EMPTY:
			(void)strcpy(strp->str_id, item[0]);
			strp->str_size = atol(item[1]);
			strp->str_time = LastTime - atol(item[2]);

			if ( strp->str_state == STR_START )
				return;
			
			strp->str_state = STR_START;

			qAction(MakeTemp, (AQarg)chan);

			return;

		case STR_ACTIVE:
			if
			(
				strcmp(strp->str_id, item[0]) == STREQUAL
				&&
				strp->str_size == atol(item[1])
				&&
				strp->str_posn <= strp->str_size
			)
			{
				Report4
				(
					 "Input message \"%s\" restarted at %ld, size %ld"
					,strp->str_id
					,strp->str_posn
					,strp->str_size
				);

				break;
			}

			Report4
			(
				 "Input message \"%s\" aborted at %ld, size %ld"
				,strp->str_id
				,strp->str_posn
				,strp->str_size
			);

			(void)strcpy(strp->str_id, item[0]);
			strp->str_size = atol(item[1]);
			strp->str_time = LastTime - atol(item[2]);

			if ( strp->str_posn > strp->str_size )
			{
				(void)unlink(strp->str_fname);
				(void)close(strp->str_fd);
				strp->str_fd = 0;
				strp->str_state = STR_START;
				qAction(MakeTemp, (AQarg)chan);
				return;
			}

			(void)lseek(strp->str_fd, (long)0, 0);

			strp->str_posn = 0;
			break;

		default:
			Fatal2("bad stream state %d", strp->str_state);
		}

		SndSOMA(chan);
		return;

	case SOM_ACCEPT:
		/*
		**	SOM Accept.
		**	"data" contains a message id and its restart address.
		*/

		if ( items != 2 )
		{
			Fatal2("bad SOM_ACCEPT format for stream %d", chan);
			return;
		}

		Trace2(1, "SOM_ACCEPT posn %s", item[1]);

		strp = &outStreams[chan];

		if
		(
			strp->str_state != STR_START
			||
			strcmp(strp->str_id, item[0]) != STREQUAL
		)
		{
			Report3("unexpected SOM_ACCEPT on stream %d, state %d", chan, strp->str_state);

			if ( strp->str_state != STR_ENDED )
			{
				strp->str_state = STR_IDLE;
				qAction(NewMessage, (AQarg)chan);
			}

			return;
		}

		Transmitting++;

		SetSpeed();

		if ( (strp->str_posn = atol(item[1])) > 0 )
		{
			if ( strp->str_posn > strp->str_size )
				Fatal4("message \"%s\" bad restart postion %ld on size %ld", strp->str_id, strp->str_posn, strp->str_size);
			Report3("Output message \"%s\" restarted at %ld", strp->str_id, strp->str_posn);
		}

		qAction(PosnStream, (AQarg)chan);
		return;

	case EOM:
		/*
		**	End-Of-Message.
		*/

		if ( items != 1 )
		{
			Fatal2("bad EOM format for stream %d", chan);
			return;
		}

		Trace1(1, "EOM");

		strp = &inStreams[chan];

		if
		(
			strcmp(strp->str_id, item[0]) != STREQUAL
			||
			strp->str_state != STR_ACTIVE
			||
			strp->str_size != strp->str_posn
		)
		{
			Report2("unexpected EOM on stream %d", chan);
			Preset(chan);
			return;
		}

		(void)close(strp->str_fd);
		strp->str_fd = 0;

		if ( --Receiving < 0 )
			Receiving = 0;

		strp->str_state = STR_ENDING;

		qAction(EndMessage, (AQarg)chan);
		return;

	case EOM_ACCEPT:
		/*
		**	EOM Accept.
		**	May get this if restarting
		**	and message has already been accepted.
		*/

		if ( items != 1 )
		{
			Fatal2("bad EOM_ACCEPT format for stream %d", chan);
			return;
		}

		Trace1(1, "EOM_ACCEPT");

		strp = &outStreams[chan];

		if
		(
			(
				strp->str_state != STR_START
				&&
				strp->str_state != STR_ENDING
			)
			||
			strcmp(strp->str_id, item[0]) != STREQUAL
		)
		{
			Report3("unexpected EOM_ACCEPT on stream %d, state %d", chan, strp->str_state);

			if ( strp->str_state != STR_ERROR )
				strp->str_state = STR_IDLE;
			return;	/* Wait for a timeout to set things right */
		}

		strp->str_messages++;
		NNstate.allmessages++;
		strp->str_bytes += strp->str_posn;

		strp->str_state = STR_ENDED;

		qAction(NewMessage, (AQarg)chan);
		return;

	case MQ_EMPTY:
		/*
		**	Remote message queue for this channel is empty.
		*/

		strp = &inStreams[chan];

		switch ( strp->str_state )
		{
		case STR_ACTIVE:
			Report3
			(
				"unexpected MQ_EMPTY on stream %d, input message \"%s\" aborted",
				chan,
				strp->str_id
			);
			(void)unlink(strp->str_fname);
			(void)close(strp->str_fd);
			strp->str_fd = 0;
		case STR_IDLE:
		case STR_ENDED:
		case STR_AGAIN:
			strp->str_state = STR_EMPTY;
		}

		return;

	default:
		Fatal3("unrecognised control for stream %d = %d", chan, data[0]);
	}
}



/*
**	Timeout from remote on idle transmit channel
*/

void
rTimeout(chan)
	int	chan;
{
	Trace4(1, "rTimeout on channel %d, instate %d, outstate %d", chan, inStreams[chan].str_state, outStreams[chan].str_state);

	(void)flushCq(chan, false);

	switch ( outStreams[chan].str_state )
	{
	case STR_START:
		break;					/* SOM is on its way ... */

	case STR_IDLE:
	case STR_EMPTY:
	case STR_ERROR:
		qAction(NewMessage, (AQarg)chan);	/* Scan for a new message */
		break;
	
	case STR_ACTIVE:
		while ( Psend(chan) > 0 );		/* Maybe protocol needs prodding */
		break;

	case STR_ENDED:					/* Patience -- we are being slow */
	case STR_ENDING:				/* Patience -- other end is being slow */
		break;
	}
}



/*
**	Reset on receive channel from remote: reset receiver.
*/

void
rReset(chan)
	int		chan;
{
	register Str_p	strp = &inStreams[chan];

	Trace3(1, "rReset on channel %d, instate %d", chan, strp->str_state);

	(void)flushCq(chan, false);

	if ( strp->str_state == STR_ACTIVE )
		strp->str_posn = lseek(strp->str_fd, (long)0, 2);
}



/*
**	Reset on transmit channel from remote: reset transmitter.
*/

void
xReset(chan)
	int	chan;
{
	Trace3(1, "xReset on channel %d, outstate %d", chan, outStreams[chan].str_state);

	(void)flushCq(chan, true);

	switch ( outStreams[chan].str_state )
	{
	case STR_EMPTY:
		outStreams[chan].str_state = STR_IDLE;
	case STR_IDLE:
	case STR_ERROR:
		qAction(NewMessage, (AQarg)chan);
	case STR_ENDED:
		break;

	default:
		SendSOM(chan);
	}
}
