/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)DoRoute.c	1.14 85/08/19
*/

/*
**	Applies routing algorithm to each element of address in 'HdrDest',
**	and for each outgoing link found builds a new destination address
**	in 'HdrDest' and calls '*funcp' with a new source address as arg.
**
**	If any address error is encountered,
**	'*errfuncp' is called with the offending node name as argument,
**	and an error reason in nlp->nl_name.
**
**	Returns true if '*funcp' was called, otherwise false.
*/

#include	"global.h"

#include	"address.h"
#include	"debug.h"
#include	"header.h"
#include	"state.h"

#include	"route.h"


/*
**	Declare structures for making multicast disjoint sets.
*/

typedef struct LinkEl *	LinkEl_p;

typedef struct LinkEl
{
	LinkEl_p	le_next;
	char *		le_addr;
}
		LinkEl;

typedef struct
{
	LinkEl_p	ls_next;
	NodeLink	ls_link;
	short		ls_count;
}
		LinkSet;

static LinkSet *LinkSets;		/* One for each possible link */

/*
**	Legal addresses
*/

#define	EXPL_ADDR	1
#define	MULT_ADDR	2
#define	BROAD_ADDR	4

/*
**	Miscellaneous
*/

static char *	Dest2;			/* Tail of address if explicit */
static void	(*Errfuncp)();
static char *	Link;
static NodeLink *Nlp;
static char *	Source;

static char *	parse();



bool
DoRoute(source, link, nlp, funcp, errfuncp)
	char *		source;		/* The source address */
	char *		link;		/* The incoming link address */
	NodeLink *	nlp;		/* Place for details of outgoing link */
	void		(*funcp)();	/* Function to send message on link */
	void		(*errfuncp)();	/* Address error function */
{
	register LinkSet *lsp;
	register LinkEl *lep;
	register int	i;
	register char *	cp;
	bool		val;
	NodeLink	snl;
	Address *	ap;

	Trace4(1, "DoRoute link \"%s\", source \"%s\", home \"%s\"", link, source, HomeNode);

	if ( RouteBase == NULLSTR && !ReadRoute() )
	{
		/*
		**	No route file!
		*/

		nlp->nl_name = "no routing table";

		(*errfuncp)(source, true);

		return false;
	}

	if
	(
		link != NULLSTR
		&&
		!FindNode(link, pt_msg, &snl)
	)
		Fatal2("Unknown link \"%s\" in DoRoute", link);

	/*
	**	Need to make disjoint sets of nodes with common
	**	shortest paths from here.
	*/

	LinkSets = lsp = (LinkSet *)Malloc(LinkCount * sizeof(LinkSet));

	for ( i = 0 ; i < LinkCount ; i++, lsp++ )
	{
		lsp->ls_next = (LinkEl_p)0;
		lsp->ls_count = 0;
	}

	/*
	**	Parse the address into link sets
	*/

	Dest2 = NULLSTR;
	Nlp = nlp;
	Errfuncp = errfuncp;
	Source = source;
	Link = link;

	if ( (nlp->nl_name = parse(HdrDest, EXPL_ADDR|MULT_ADDR|BROAD_ADDR)) != NULLSTR )
		(*Errfuncp)(HdrDest, false);

	/*
	**	Make an address set in HdrDest for each outgoing link.
	*/

	val = false;
	ap = SplitAddress(source);

	for ( lsp = LinkSets, i = 0 ; i < LinkCount ; i++, lsp++ )
	{
		if ( lsp->ls_count == 0 )
			continue;

		cp = HdrDest ;		/* NB: length of sub-set <= original */

		if ( Dest2 != NULLSTR )
			*cp++ = ATYP_EXPLICIT;

		if ( lsp->ls_count == 1 )
		{
			lep = lsp->ls_next;
			cp = strcpyend(cp, lep->le_addr);
			free(lep->le_addr);
			free((char *)lep);
		}
		else
		for
		(
			;
			(lep = lsp->ls_next) != (LinkEl_p)0 ;
			lsp->ls_next = lep->le_next,
			free((char *)lep)
		)
		{
			*cp++ = ATYP_MULTICAST;
			cp = strcpyend(cp, lep->le_addr);
			free(lep->le_addr);
		}

		if ( Dest2 != NULLSTR )
		{
			*cp++ = ATYP_EXPLICIT;
			(void)strcpy(cp, Dest2);
		}

		*Nlp = lsp->ls_link;

		Trace3
		(
			2,
			"Multi-cast subset \"%s\" on link \"%s\"",
			HdrDest,
			Nlp->nl_name
		);

		if
		(
			link != NULLSTR
			&&
			(cp = AdjustSource(source, ap, snl.nl_index, RT_LINK(Nlp->nl_link)->le_index)) != NULLSTR
		)
		{
			(*funcp)(cp);
			free(cp);
		}
		else
			(*funcp)(Source);

		val = true;
	}

	FreeAddress(ap);

	if ( Dest2 != NULLSTR )
		free(Dest2);

	free((char *)LinkSets);

	return val;
}


/*
**	Parse the address.
**	Add each element into appropriate linkset.
*/

static
char *
parse(dest, legal)
	char *		dest;
	int		legal;
{
	register int	i;
	register int	j;
	register char *	cp;
	register LinkSet *lsp;
	register LinkEl	*lep;
	bool		freecp;
	int		primdom;	/* Target domain index */
	char *		domain;

	Trace2(2, "parse dest \"%s\"", dest);

	switch ( dest[0] )
	{
	case ATYP_EXPLICIT:
		if
		(
			!(legal & EXPL_ADDR)
			||
			(cp = strchr(&HdrDest[1], ATYP_EXPLICIT)) == NULLSTR
		)
			return "Bad explicit address";
		
		*cp++ = '\0';
		Dest2 = newstr(cp);

		return parse(++dest, legal);

	case ATYP_MULTICAST:
		if ( !(legal & MULT_ADDR) )
			return "Bad multicast address";

		legal &= ~(MULT_ADDR|EXPL_ADDR);

		do
		{
			if ( (cp = strchr(++dest, ATYP_MULTICAST)) != NULLSTR )
				*cp = '\0';

			if ( (Nlp->nl_name = parse(dest, legal)) != NULLSTR )
				(*Errfuncp)(dest, false);
		}
			while ( (dest = cp) != NULLSTR );
		
		return NULLSTR;

	case ATYP_BROADCAST:
		/*
		**	Extended Reverse Path Forwarding Algorithm:
		**
		**	If this message has come via the link on the
		**	shortest path back to the source, then it should
		**	be propagated on all those links for which a bit
		**	is set in the broadcast table, and which is a
		**	member of the target domain.
		*/

		if ( dest[1] == DOMAIN_SEP )
		{
			domain = &dest[2];

			if ( domain[0] == ATYP_BROADCAST )
				primdom = LINK_N_A;
			else
			{
				NodeLink	dom_link;

				if ( !FindDomain(domain, &dom_link) )
				{
					/*
					**	Unknown domain!
					*/

					return dom_link.nl_name;	/* Return fail reason */
				}

				primdom = dom_link.nl_domind;
			}
		}
		else
		{
			if ( dest[1] != '\0' )
				return "Bad broadcast address";

			domain = NULLSTR;
		}

do_broadcast:
		if ( strcmp(Source, HomeNode) == STREQUAL )
		{
			/*
			**	Start of broadcast.
			**	Set boolean for each link
			**	on its own shortest path
			**	which is also in the target domain.
			*/

			if ( domain == NULLSTR )
				primdom = RT_NODE(NodeCount-1)->ne_primary;

			cp = Malloc(1+LinkCount/8);
			j = 0;	/* Offset */
			freecp = true;

			for ( i = 0 ; i < LinkCount ; i++ )
				if ( RT_NODE(RT_LINK(i)->le_index)->ne_shortest[(int)pt_msg] == i )
					cp[i/8] |= (1<<(i%8));
				else
					cp[i/8] &= ~(1<<(i%8));
		}
		else
		{
			cp = newstr(Source);	/* Might get modified by "FindAddress()" */

			if ( !FindAddress(cp, Nlp) )
			{
				/*
				**	Unknown source!
				*/

				(*Errfuncp)(Source, true);

				free(cp);
				return NULLSTR;
			}

			Trace3
			(
				2,
				"Broadcast source \"%s\" (address \"%s\")",
				Nlp->nl_name,
				cp
			);

			if ( Link != NULLSTR && strcmp(Nlp->nl_name, Link) != STREQUAL )
			{
				/*
				**	Didn't come the right way, so drop it.
				*/

				free(cp);
				return NULLSTR;
			}

			if ( domain == NULLSTR )
			{
				if ( strchr(cp, DOMAIN_SEP) )
				{
					/*
					**	This one has escaped from its primary domain, so drop it!
					*/

					free(cp);
					return NULLSTR;
				}

				primdom = RT_NODE(Nlp->nl_index)->ne_primary;
			}

			free(cp);
			freecp = false;

			j = Nlp->nl_index * LinkCount;
			cp = &ForwTable[j/8];
			j %= 8;	/* Offset */
		}

		for ( i = 0 ; i < LinkCount ; i++, j++ )
			if ( cp[j/8] & (1<<(j%8)) )
			{
				(void)GetLink(i, Nlp);

				if
				(
					primdom != LINK_N_A
					&&
					(
						domain != NULLSTR
						||
						Nlp->nl_domind != LINK_N_A
					)
				)
				{
					register int	k;

					/*
					**	Test link lies on route to target domain.
					*/

					k = primdom * LinkCount + i;

					if ( !(DomForwTable[k/8] & (1<<(k%8))) )
						continue;
				}

				Trace2
				(
					2,
					"Broadcast link \"%s\"",
					Nlp->nl_name
				);

				lep = Talloc(LinkEl);

				lep->le_addr = newstr(dest);

				if ( (lep->le_next = (lsp = &LinkSets[Nlp->nl_link])->ls_next) == (LinkEl *)0 )
					lsp->ls_link = *Nlp;

				lsp->ls_next = lep;
				lsp->ls_count++;
			}

		if ( freecp )
			free(cp);

		return NULLSTR;
	}

	for ( i = 2 ; --i >= 0 ; )
	{
		if ( FindAddress(dest, Nlp) )
			break;

		if ( i == 0 || (dest = FindAlias(dest)) == NULLSTR )
		{
			/*
			**	Unknown destination!
			*/

			return Nlp->nl_name;
		}
	}

#	if	DOM_NON_HIER
	if
	(
		(Nlp->nl_domflags & DOM_NON_HIER)
		&&
		(cp = strrchr(dest, DOMAIN_SEP)) != NULLSTR
	)
	{
		/*
		**	This is a distaster for hierarchical domains!
		**	REMOVED -- until such time as there is
		**	a "non-hierarchic" flag for domains.
		*/

		domain = cp;
		primdom = Nlp->nl_domind;
		goto do_broadcast;
	}
#	endif	DOM_NON_HIER

	lep = Talloc(LinkEl);

	lep->le_addr = newstr(dest);

	if ( (lep->le_next = (lsp = &LinkSets[Nlp->nl_link])->ls_next) == (LinkEl *)0 )
		lsp->ls_link = *Nlp;

	lsp->ls_next = lep;
	lsp->ls_count++;

	return NULLSTR;
}
