/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "%W% %E%";

/*
**	Process request from remote site for a file from our system
**
**	SETUID -> root
*/

#define	MAXFILES	28		/* depends on MAXVARARGS */

#define	FILE_CONTROL
#define	RECOVER
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"address.h"
#include	"debug.h"
#include	"header.h"
#include	"spool.h"
#include	"ndir.h"

#include	<errno.h>

#if	AUSAS == 0
#include	<grp.h>
#endif

/*
**	Parameters set from arguments.
*/

bool	Broadcast;		/* Message has been everywhere */
char *	CommandsFile;		/* Commands of files to be pre-pended to Message */
char *	HomeNode;		/* Name of this node */
char *	LinkNode;		/* Message arrived from this node */
char *	Message;		/* Message */
char *	Name;			/* Program invoked name */
char *	SourceNode;		/* Message originated at this node */
int	Traceflag;		/* Global tracing control */

/*
**	Parameters read from header of file
*/

char *	Options;		/* Service related options */
char *	RemUser;		/* Name of user requesting service */
char *	Service;		/* Service requested */

/*
**	Miscellaneous
*/

char *	DataName;		/* Used by ExpandArgs() */
int	Pid;			/* Used by UniqueName() in Exec...() */
char *	PublicDir;		/* Place where files are stored */
int	RetVal;			/* Value to return to receiver */
char *	SenderName;		/* Used by ExpandArgs() */
Time_t	Time;			/* Used by UniqueName() in Exec...() */
char *	UserName;		/* Used by ExpandArgs() */

#ifdef	PUBLICFILES
char *	PublicFiles = PUBLICFILES;	/* where we find about public files */
#endif

#define	Fprintf		(void)fprintf
#define	free(A)

void	finish(), DoRequest(), MailBack(), Return(), GetFSHdr();
void	flist(), vlist(), blist(), sendfiles(), sendlist();
char *	filename(), * FileStr(), * CheckPerm();
struct	group *getgrnam();
struct	passwd *getpwnam();


int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	register FILE *	fd;
	char *		cp;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);
	DataName = SenderName = UserName = "fileserver";

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'B':
					Broadcast = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'c':
					CommandsFile = ++*argv;
					goto break2;

				case 'd':
					DataLength = atol(++*argv);
					break;

				case 'e':
					HdrEnv = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 's':
					SourceNode = ++*argv;
					goto break2;

				default:
					Trace2(1, "unrecognised flag '%c'", c);
					goto break2;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		if ( Message != NULLSTR )
			Error("only one message allowed");
		else
			Message = *argv;
	}

	if ( (cp = GetEnv(ENV_ACK)) != NULLSTR )
	{
		free(cp);
		Error("file request acknowledged from \"%s\"", SourceNode);
	}

	while ( (fd = fopen(Message, "r")) == NULL )
		Syserror("Can't open \"%s\"", Message);

	Recover(ert_finish);

	if ( (cp = GetEnv(ENV_RETURNED)) != NULLSTR )
	{
		free(cp);
		MailBack(GetEnv(ENV_ERR1), fd);
	}
	else
		DoRequest(fd);

	return RetVal;
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}




/*
**	Check validity of file request, and send files back
*/

void
DoRequest(ifd)
	FILE *		ifd;
{
	register char *	cp;
	register char * dp;
	register char **fp;
	FILE *		fd;
	struct stat	statb;
	char *		FileList[MAXFILES];

#ifndef	PUBLICFILES

	Error("There are no remotely accessible files at %s.  Sorry.",
	    HomeNode);

#else
	GetFSHdr(ifd);

	if ( stat(PublicFiles, &statb) == SYSERROR )
	{
		Error("No public files are currently available.");
	}

	switch (statb.st_mode & S_IFMT)
	{
	default:
		Error("%s is a wierd file.  Nothing available", PublicFiles);

	case S_IFDIR:
		PublicDir = PublicFiles;
		break;

	case S_IFREG:
		/*
		** This can't fail - file exists, and we are root ..
		*/
		if ( (fd = fopen(PublicFiles, "r")) == NULL )
		{
			Error("Public files are private!");
		}

		PublicDir = CheckPerm(fd);

		fclose(fd);
		break;
	}

#if	AUSAS == 0
#ifdef	SERVERGROUP
	{
		register struct group *gp;

		gp = getgrnam(SERVERGROUP);
		if (gp == NULL)
			Syserror("Group %s does not exist!", SERVERGROUP);

		setgid(gp->gr_gid);
	}
#endif
#endif

#ifdef	SERVERUSER
	{
		Passwd	pw;

		if ( !GetUid(&pw, SERVERUSER) )
			Error("User %s does not exist!", SERVERUSER);

#		if	AUSAS == 1
		setgid(pw.P_gid);
#		endif

		setuid(pw.P_uid);
	}
#endif

	if ( chdir(PublicDir) == SYSERROR )
		Error("Public files are inaccessible");

	Recover(ert_return);

	fp = FileList;

	while (fp < &FileList[MAXFILES-1])
	{
		if  ((cp = FileStr(ifd)) == NULLSTR)
			break;

		if (*cp == '/')
		{
			Error("%s of %s refused: absolute path illegal",
			    Service, cp);
			RetVal = 1;
			continue;
		}

		if (strncmp(cp, "../", 3) == STREQUAL ||
		    strcmp(cp, "..") == STREQUAL)
		{
			Error("Values of B will give rise to Dom <%s>", cp);
			RetVal = 1;
			continue;
		}

		for (dp = cp + 1; (dp = strchr(dp, '.')) != NULLSTR; dp++)
		{
			if (strncmp(dp-1, "/../", 4) == STREQUAL)
				break;
			if (strcmp(dp-1, "/..") == STREQUAL)
				break;
		}

		if (dp != NULLSTR)
		{
			Error("Illegal \"..\" in filename <%s>", cp);
			RetVal = 1;
			continue;
		}

		if (access(cp, 4) == SYSERROR || stat(cp, &statb) == SYSERROR)
		{
			Error("%s of %s refused: unavailable", Service, cp);
			RetVal = 1;
			continue;
		}

		if ((statb.st_mode & S_IFMT) == S_IFDIR &&
		    strcmp(Service, "SendFile") == STREQUAL)
		{
			Error("%s is a directory: not returned", cp);
			RetVal = 1;
			continue;
		}

		*fp++ = cp;
	}

	if ( fp == FileList )
	{
		if ( RetVal == 0 )
			Error("No files requested.");
		RetVal = 1;
		return;
	}

	*fp++ = NULLSTR;

	if ( cp != NULLSTR )
	{
		Warn("Too many files requested, excess requests ignored");
		RetVal = 1;
	}

	if ( strcmp(Service, "SendFile") == STREQUAL )
		sendfiles(FileList);
	else if ( strcmp(Service, "List") == STREQUAL )
		sendlist(FileList, false);
	else if ( strcmp(Service, "ListVerbose") == STREQUAL )
		sendlist(FileList, true);
	else
	{
		Error("Service \"%s\" is not implemented at %s",
		    Service, HomeNode);
		RetVal = 1;
	}
#endif
}

#ifdef	PUBLICFILES
void
sendfiles(list)
	register char **list;
{
	register char * errs;
	VarArgs		va;

	FIRSTARG(&va) = SEND;
	NEXTARG(&va) = concat("-q", RemUser, "@", SourceNode, NULLSTR);
	if (Options != NULLSTR)
		NEXTARG(&va) = Options;
	if (Traceflag != 0)
		NEXTARG(&va) = NumericArg('T', (long)Traceflag);

	while (*list != NULLSTR)
	{
		NEXTARG(&va) = *list;
		list++;
	}

	if ((errs = Execute(&va)) != NULLSTR)
	{
		Error("Sendfile said '%s'", errs);
		RetVal = 1;
	}
}

void
sendlist(list, verbose)
	register char **list;
	bool		verbose;
{
	FILE	      * sfd;
	char	      * errs;
	VarArgs		va;

	FIRSTARG(&va) = SEND;
	NEXTARG(&va) = concat("-q", RemUser, "@", SourceNode, NULLSTR);
	if (Options != NULLSTR)
		NEXTARG(&va) = Options;
	if (Traceflag != 0)
		NEXTARG(&va) = NumericArg('T', (long)Traceflag);
	NEXTARG(&va) = "-nPublicFileList";

	sfd = ExecPipe(&va);

	Fprintf(sfd, "Results of %slist request at %s on %s\n",
	    (verbose ? "verbose " : ""), HomeNode, ctime(&Time));

	if (verbose)
		Fprintf(sfd, "%-9s %-16s %10s  %-24s\n\n",
		    "Type", "Name", "Size", "Last Modified");
	while (*list != NULLSTR)
	{
		if (verbose)
			vlist(*list, sfd);
		else
			blist(*list, sfd);
		list++;
	}

	if (!verbose)
		flist(NULLSTR, sfd);

	if ((errs = ExPipeClose(sfd)) != NULLSTR)
	{
		Error("SendFile said '%s'", errs);
		RetVal = 1;
	}
}

void
vlist(file, fd)
	register char *file;
	register FILE *fd;
{
	struct stat    statb;
	register DIR  *dd;
	register struct direct *dp;

	if (stat(file, &statb) == SYSERROR)
	{
		Fprintf(fd, "%s: nonexistant\n", filename(file));
		return;
	}

	/*
	** Note: we expressly do not tell owner, link count, permissions ..
	*/
	switch (statb.st_mode & S_IFMT)
	{
	case S_IFREG:
		Fprintf(fd, "%-9s", "file");
		break;

	case S_IFDIR:
		Fprintf(fd, "%-9s", "directory");
		break;

	default:
		Fprintf(fd, "%-9s", "special");
		break;
	}

	Fprintf(fd, " %-16s %10ld  %s", filename(file),
	    statb.st_size, ctime(&statb.st_mtime));

	if ((statb.st_mode & S_IFMT) != S_IFDIR)
		return;

	if ((dd = opendir(file)) == NULL)
	{
		Fprintf(fd, "\nContents unavailable\n\n");
		return;
	}

	Fprintf(fd, "\nContents:\n");

	while ((dp = readdir(dd)) != NULL)
	{
		if (dp->d_name[0] == '.')
			continue;

		flist(dp->d_name, fd);
	}
	flist(NULLSTR, fd);
	Fprintf(fd, "\n");
	closedir(dd);
}

void
blist(file, fd)
	register char *file;
	register FILE *fd;
{
	flist(filename(file), fd);
}

void
flist(file, fd)
	register char *file;
	register FILE *fd;
{
	static int linelen = 0;
	register   len;

	if (file == NULLSTR)
	{
		if (linelen > 0)
			Fprintf(fd, "\n");
		linelen = 0;
		return;
	}

	len = strlen(file);
	if (len == 0)
		return;

	if (linelen + len > 70)
	{
		linelen = 0;
		Fprintf(fd, "\n");
	}
	else
	if (linelen > 0)
	{
		Fprintf(fd, "\t");
		linelen |= 7;
		linelen++;
	}
	
	Fprintf(fd, "%s", file);

	linelen += len;
}

char *
filename(file)
	register char *file;
{
	register n = strlen(PublicDir);

	if (strncmp(file, PublicDir, n) == STREQUAL)
		return (file + n + 1);

	return (file);
}

char *
CheckPerm(fd)
	register FILE *fd;
{
	register char *cp;
	register char *hp;
	char buf[BUFSIZ];

	while (fgets(buf, sizeof buf, fd) != NULLSTR)
	{
		if ((cp = strchr(buf, '\n')) != NULLSTR)
			*cp = '\0';

		if (cp = strpbrk(buf, " \t"))
		{
			*cp++ = '\0';
			cp += strspn(cp, " \t");
		}

		if (hp = strchr(buf, '@'))
			*hp++ = '\0';
		else
			hp = buf;

		if ( *hp != '*' && strccmp(hp, SourceNode) != STREQUAL )
			continue;

		if (hp != buf && buf[0] != '*' &&
		    strcmp(buf, RemUser) != STREQUAL)
			continue;

		if (cp == NULLSTR)
			break;

		return newstr(cp);
	}

	Error("You have no access permissions at %s", HomeNode);

	/*NOTREACHED*/
}
#endif	PUBLICFILES

void
GetFSHdr(ifd)
	FILE	* ifd;
{
	RemUser = FileStr(ifd);
	Service = FileStr(ifd);
	Options = FileStr(ifd);

	if ( Service == NULLSTR || RemUser == NULLSTR ||
	     getc(ifd) != '\0' )
		Error("File request header bad");
}

char *
FileStr(fd)
	register FILE * fd;
{
	register	c;
	register char * cp;
	char		string[128];

	for (cp = string; cp < &string[sizeof string]; cp++)
	{
		if ((c = getc(fd)) == EOF)
			return NULLSTR;

		*cp = c;

		if (c == '\0')
		{
			if (cp == string)
				return NULLSTR;

			return newstr(string);
		}
	}

	return NULLSTR;
}

void
MailBack(mesg, fd)
	char *	mesg;
	FILE *	fd;
{
	VarArgs	va;
	char *	cp;

#ifdef	SERVERUSER
	SenderName = SERVERUSER;
#else
	SenderName = "FileServer";
#endif
	GetFSHdr(fd);

	FIRSTARG(&va) = BINMAIL;

#ifdef	BINMAILARGS
	if ( HomeAddress(SourceNode, (char **)0, EXCL_HIER) )
		ExpandArgs(&va, BINMAILARGS, NULLSTR);
#endif

	NEXTARG(&va) = RemUser;

	fd = ExecPipe(&va);

	Fprintf(fd, "From %s %sFrom: %s@%s\nTo: %s@%s\n",
	    SenderName, ctime(&Time), SenderName, SourceNode,
	    RemUser, HomeNode);

	Fprintf(fd, "Subject: Your request for files from %s\n\n", SourceNode);

	Fprintf(fd, "Some or all of the files you requested from %s\n",
		SourceNode);

	Fprintf(fd, "are not available.  ");

	if ( mesg != NULLSTR )
	{
		if ( (cp = StripErrString(mesg)) != NULLSTR )
		{
			Fprintf(fd, "The reason given was:\n\n\t%s\n", mesg);
			free(cp);
		}
		free(mesg);
	}
	else
		Fprintf(fd, "No reason was given.\n");
	
	(void)ExPipeClose(fd);
}
