/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)request.c	1.16 85/08/16";

/*
**	Request remote state files.
**
**	Send request message to statehandlers.
*/

#define	FILE_CONTROL
#define	LOCKING
#define	STDIO

#include	"global.h"

#include	"address.h"
#include	"debug.h"
#include	"handlers.h"
#include	"header.h"
#include	"state.h"
#include	"sub_proto.h"


/*
**	Parameters set from arguments.
*/

bool	All;			/* Send all local state info. */
bool	AllLinks;		/* Include all links in broadcast */
bool	Broadcast;		/* Broadcast request */
bool	Global;			/* Global broadcast request */
char *	HomeNode;		/* Name of this node */
char *	Name;			/* Program invoked name */
bool	NoEnq;			/* Don't ask for remote reply */
int	Traceflag;		/* Global tracing control */

/*
**	Define structure to remember addresses
*/

typedef struct Addr *	Addr_p;

typedef struct Addr
{
	Addr_p	a_next;
	char *	a_name;
}
	Addr;

Addr *	AddrList;
int	AddrCount;
int	AddrLength;

/*
**	Miscellaneous
*/

NodeLink LinkD;			/* Details of link */
int	MesgFd;			/* File descriptor of Message */
int	Pid;			/* Used by UniqueName() in Exec...() */
char	ProtoType[2];		/* Header protocol type */
Time_t	Time;			/* Used by UniqueName() in Exec...() */
char	Ttd[TIME_SIZE+1];	/* "Time-to-die" for message */

char *	Message	= WORKDIR(requestmessage);

#define	WEEK	(1*7*24*60*60)
#define	free(A)

void	finish(), listaddr(), send();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	char *		addr;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);

	SetUser(ACSNETUID, ACSNETGID);

	while ( (MesgFd = creat(UniqueName(Message, (long)0, Time), 0600)) == SYSERROR )
		Syserror("Can't creat \"%s\"", Message);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					All = true;
					continue;

				case 'G':
					Global = true;
				case 'B':
					Broadcast = true;
					continue;

				case 'L':
					AllLinks = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'N':
					NoEnq = true;
					continue;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				default:
					Error("unrecognised flag '%c'", c);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			if ( (addr = FindAlias(*argv)) == NULLSTR )
				addr = *argv;

			listaddr(addr);
		}
	}

	if ( HomeNode == NULLSTR )
		HomeNode = NodeName();

	if ( AddrCount == 0 )
	{
		if ( !Broadcast )
		{
			for ( argc = 0 ; GetLink(argc, &LinkD) ; argc++ )
				if
				(
					(LinkD.nl_flags & S_MSG)
					&&
					!(LinkD.nl_flags & S_FOREIGN)
					&&
					LinkD.nl_spooler == NULLSTR
				)
					listaddr(LinkD.nl_name);

			if ( AddrCount == 0 )
			{
				Warn("Can't find any links");
				finish(0);
			}
		}
#		ifndef	OLD_ADDR_FORMAT
		else
		if ( !Global )
		{
			register char *	cp;
			int		domind = PrimDom();
			
			if ( AllLinks )
			{
				for ( argc = 0 ; GetLink(argc, &LinkD) ; argc++ )
					if
					(
						(LinkD.nl_flags & S_MSG)
						&&
						!(LinkD.nl_flags & S_FOREIGN)
						&&
						LinkD.nl_spooler == NULLSTR
						&&
						DomInPrim(LinkD.nl_domind) == NULLSTR
					)
						listaddr(DomainName(LinkD.nl_domind));
				if ( AddrCount == 0 )
				{
					Warn("Can't find any links");
					finish(0);
				}
			}

			if ( (cp = DomainName(domind)) != NULLSTR )
			do
				listaddr(cp);
			while
				( (cp = DomInHier(&domind)) != NULLSTR );
		}
#		endif	OLD_ADDR_FORMAT
	}
	else
	if ( Broadcast )
		Error("Can't specify addresses with broadcast request");

	if
	(
		All
		&&
		(
			Broadcast
			||
			AddrCount > 1
			||
			(
				AddrCount == 1 
				&&
				AddrList->a_name[0] == ATYP_BROADCAST
			)
		)
	)
		Error("Can only send/request all info to/from specific node");

	send();

	return 0;
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int		error;
{
	register int	i;

	(void)unlink(Message);

	for ( i = 0 ; i < 20 ; i++ )
		UnLock(i);

	(void)exit(error);
}



/*
**	Send Message.
**
**	Write Message, and call receiver to process it.
**
**	NB: this will invoke (via receiver) statehandler for this node.
*/

void
send()
{
	char *	errs;
	FILE *	fd;
	VarArgs	va;
	char	addr[4];

	if ( Broadcast )
	{
		addr[0] = ATYP_BROADCAST;
		addr[1] = DOMAIN_SEP;
		
		if ( AddrCount == 0 )
		{
			if ( Global )
			{
				addr[2] = ATYP_BROADCAST;
				addr[3] = '\0';
			}
			else
				addr[1] = '\0';

			listaddr(addr);
			Broadcast = false;
		}
		else
		{
			addr[2] = '\0';
			AddrLength += AddrCount * (strlen(addr)+1);
		}
	}

	if ( AddrCount )
	{
		register char *	cp;
		register Addr *	ap;

		HdrDest = cp = Malloc(AddrLength+1);

		for ( ap = AddrList ; ap != (Addr *)0 ; ap = ap->a_next )
		{
			*cp++ = ATYP_MULTICAST;

			if ( Broadcast )
				cp = strcpyend(cp, addr);

			if ( ap->a_name == NULLSTR )
				*--cp = '\0';
			else
				cp = strcpyend(cp, ap->a_name);
		}

		if ( AddrCount == 1 )
			HdrDest++;
	}
	else
		return;

	HdrSource = HomeNode;

	/*
	**	Invoke STATE to make state message.
	*/

	FIRSTARG(&va) = STATE;
	NEXTARG(&va) = All?"-Q":"-O";		/* Output local state */
	if ( All && !NoEnq )
		NEXTARG(&va) = "-CS";		/* Update statefile */
	NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);
#	ifdef	DEBUG
	if ( Traceflag )
		NEXTARG(&va) = Traceflag==1?"-T1":"-T2";
#	endif	DEBUG

	fd = ExPipeO(&va, MesgFd);

	if ( All && ! NoEnq )
		(void)fprintf(fd, "clear %s", HdrDest);	/* Mark statefile so we will accept all info back */

	if ( (errs = ExOPipeClose(fd)) != NULLSTR )
	{
		Error(errs);
		free(errs);
	}

	/*
	**	Send message.
	*/

	HdrEnv = MakeEnv
		 (
			ENV_NORET, NULLSTR,
			ENV_STATE, NULLSTR,
			ENV_NO_AUTOCALL, NULLSTR,
			NoEnq?NULLSTR:(All?ENV_ENQALL:ENV_ENQ), NULLSTR,
			NULLSTR
		 );

	(void)sprintf(Ttd, "%lu", (ulong)WEEK);
	HdrTtd = Ttd;

	HdrHandler = STATEHANDLER;

	DataLength = lseek(MesgFd, (long)0, 2);

	ProtoType[0] = STATE_PROTO;
	HdrSubpt = ProtoType;

	while ( WriteHeader(MesgFd, DataLength, 0) == SYSERROR )
		Syserror("Can't write \"%s\"", Message);

	(void)close(MesgFd);

	FIRSTARG(&va) = RECEIVER;
#	ifdef	DEBUG
	if ( Traceflag )
		NEXTARG(&va) = Traceflag==1?"-T1":"-T2";
#	endif	DEBUG
	NEXTARG(&va) = "-L";
	NEXTARG(&va) = concat("-h", HomeNode, NULLSTR);
	NEXTARG(&va) = Message;
	NEXTARG(&va) = NULLSTR;

	(void)execve(ARG(&va, 0), &ARG(&va, 0), StripEnv());
	Syserror("Can't execve \"%s\"", ARG(&va, 0));
}



/*
**	Add an address to list.
*/

void
listaddr(addr)
	char *		addr;
{
	register Addr *	ap;

	Trace2(1, "listaddr \"%s\"", addr);

	for ( ap = AddrList ; ap != (Addr_p)0 ; ap = ap->a_next )
		if
		(
			addr == ap->a_name
			||
			(
				addr != NULLSTR
				&&
				ap->a_name != NULLSTR
				&&
				strcmp(addr, ap->a_name) == STREQUAL
			)
		)
			return;

	ap = Talloc(Addr);

	ap->a_next = AddrList;
	AddrList = ap;

	ap->a_name = addr;
	AddrLength += strlen(addr)+1;

	AddrCount++;
}
