// fe.c - Front-end 8080 communication routines for KS10 processor
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS-10 Emulator.
// See README for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "pdp10/defs.h"
#include "pdp10/fe.h"
#include "pdp10/proto.h"
#include "emu/proto.h"
#include "emu/socket.h"

#ifdef SVR4SIGNALS
#include <sys/file.h>
#include <unistd.h>
#include <stropts.h>
#include <signal.h>
#endif

#if defined(linux)
#include <endian.h>
#endif

#define FE_CTYPORT   5000 // Default CTY port
#define FE_KLUPORT   5001 // Default KLU port
#define FE_ESCAPE    0x1C // ASCII Ctrl-\  
#define FE_COUNTDOWN 100  // Default countdown
#define FE_PASSWORD  "savior7" // KLINIK Password
#define FE_PASSLEN   80   // Maximum Password Length
#define FE_PASSTRY   3    // Number of tries allowed

// CTY/KLU Flag <20:27>
#define FE_NOACT   0  // No active
#define FE_PENDING 1  // Pending character
#define FE_INITED  2  // Initialized/New call
#define FE_HANGUP  2  // Hang up/Terminated
#define FE_NOCAR   3  // Carrier Loss

#if BYTE_ORDER == LITTLE_ENDIAN
// For little endian machines
#define FE_FLAG 1
#define FE_CHAR 0
#else
// For big endian machines
#define FE_FLAG 6
#define FE_CHAR 7
#endif BYTE_ORDER

// CTY Buffer
static uchar inBuffer[4096];
static char  outBuffer[4096];
static int   idxInQueue, idxOutQueue;
static int   idxOutBuffer;
static char  lastSeen;

// KLINIK Buffer
static int   klu_Mode;
static int   klu_tryPassword;
static int   klu_idxPassword;
static char  klu_inPassword[FE_PASSLEN+1];
static char  *klu_Password;

static int   klu_idxInQueue, klu_idxOutQueue;
static int   klu_idxOutBuffer;
static uchar klu_inBuffer[4096];
static char  klu_outBuffer[4096];
static char  klu_lastSeen;

// Direct access to KS10 main memory
// for better optimization.

// Console TTY - Input Word
static int36 *ctyInWord;
static uchar *ctyInFlag;
static uchar *ctyInChar;

// Console TTY - Output Word
static int36 *ctyOutWord;
static uchar *ctyOutFlag;
static uchar *ctyOutChar;

// KLINIK - Input Word
static int36 *kluInWord;
static uchar *kluInFlag;
static uchar *kluInChar;

// KLINIK - Output Word
static int36 *kluOutWord;
static uchar *kluOutFlag;
static uchar *kluOutChar;

static SOCKET *ctyServer = NULL; // CTY Listening Socket
static SOCKET *ctySocket = NULL; // CTY Socket
static SOCKET *kluServer = NULL; // KLINIK Listening Socket
static SOCKET *kluSocket = NULL; // KLINIK Socket

// Send initialization codes to make sure that telnet behave correctly.
static int  n_telnetInit = 15;
static char telnetInit[] =
{
	255, 251,  34, // IAC WILL LINEMODE
	255, 251,   3, // IAC WILL SGA
	255, 251,   1, // IAC WILL ECHO
	255, 251,   0, // IAC WILL BINARY
	255, 253,   0, // IAC DO BINARY
};

int p10_ctyCountdown = -1;

void p10_ctyAccept(SOCKET *);
void p10_ctyEof(SOCKET *, int, int);
void p10_ctyInput(SOCKET *, char *, int);

void p10_kluAccept(SOCKET *);
void p10_kluEof(SOCKET *, int, int);
void p10_kluPassword(SOCKET *, char *, int);
void p10_kluInput(SOCKET *, char *, int);

int p10_ctyInitialize(void)
{
	// Set up the listing socket for CTY device.
	ctyServer = sock_Open(FE_CTYPORT, NET_SERVER);
	if (ctyServer != NULL) {
		ctyServer->maxConns = 1;
		ctyServer->nConns   = 0;
		ctyServer->Accept   = p10_ctyAccept;
		ctyServer->Eof      = NULL;
		ctyServer->Process  = NULL;

		// Now accept incoming connections;
		sock_Listen(ctyServer, 5);
	}

	// Set up the listening socket for KLINIK device.
	kluServer = sock_Open(FE_KLUPORT, NET_SERVER);
	if (kluServer != NULL) {
		kluServer->maxConns = 1;
		kluServer->nConns   = 0;
		kluServer->Accept   = p10_kluAccept;
		kluServer->Eof      = NULL;
		kluServer->Process  = NULL;

		klu_Password = FE_PASSWORD;
		klu_Mode     = 0; // Status: Not Connected

		// Now accept incoming connections;
		sock_Listen(kluServer, 5);
	}

	// Set up pointers from KS10 main memory
	// for optimization.

	// Set up CTYIWD/CTYOWD pointers
	ctyInWord  = p10_pAccess(FE_CTYIWD);
	ctyInFlag  = &((uchar *)ctyInWord)[FE_FLAG];
	ctyInChar  = &((uchar *)ctyInWord)[FE_CHAR];

	ctyOutWord = p10_pAccess(FE_CTYOWD);
	ctyOutFlag = &((uchar *)ctyOutWord)[FE_FLAG];
	ctyOutChar = &((uchar *)ctyOutWord)[FE_CHAR];

	// Set up KLUIWD/KLUOWD pointers
	kluInWord  = p10_pAccess(FE_KLUIWD);
	kluInFlag  = &((uchar *)kluInWord)[FE_FLAG];
	kluInChar  = &((uchar *)kluInWord)[FE_CHAR];

	kluOutWord = p10_pAccess(FE_KLUOWD);
	kluOutFlag = &((uchar *)kluOutWord)[FE_FLAG];
	kluOutChar = &((uchar *)kluOutWord)[FE_CHAR];

	return EMU_OK;
}

int p10_ctyCleanup(void)
{
	sock_Close(ctySocket);
	sock_Close(ctyServer);
	sock_Close(kluSocket);
	sock_Close(kluServer);

	ctySocket = NULL;
	ctyServer = NULL;
	kluSocket = NULL;
	kluServer = NULL;

	return EMU_OK;
}

void p10_ctySendDone(void)
{
	// Send a done interrupt to the KS10 Processor.
	p10_aprInterrupt(APRSR_F_CON_INT);
}

void p10_ctyAccept(SOCKET *srvSocket)
{
	SOCKET *newSocket;

	if (newSocket = sock_Accept(srvSocket)) {
		// First, check if CTY connection already was taken.
		// If so, tell operator that.
		if (ctySocket != NULL) {
			int idSocket = newSocket->idSocket;

			sock_Send(idSocket, "Console (CTY) connection already was taken.\r\n", 0);
			sock_Send(idSocket, "Check other terminal which has that connection.\r\n", 0);
			sock_Send(idSocket, "\r\nTerminated.\r\n", 0);
			sock_Close(newSocket);

			return;
		}
 
		// Set up the CTY socket connection.
		ctySocket = newSocket;
		ctySocket->Accept  = NULL;
		ctySocket->Eof     = p10_ctyEof;
		ctySocket->Process = p10_ctyInput;

		// Send initialization codes and welcome messages
		sock_Send(ctySocket->idSocket, telnetInit, n_telnetInit);
		sock_Send(ctySocket->idSocket, "Welcome to KS10 Emulator\r\n\r\n", 0);

		// Reset all buffer for this new connection.
		idxInQueue   = 0;
		idxOutQueue  = 0;
		idxOutBuffer = 0;
	}
}

void p10_ctyEof(SOCKET *Socket, int rc, int nError)
{
	sock_Close(ctySocket);

	ctySocket = NULL;
}

void p10_ctyInput(SOCKET *Socket, char *keyBuffer, int len)
{
	boolean okSend = FALSE;
	int     idx;

	// Process telnet codes and filter them out of data stream.
	if (len > 1) {
		if ((len = sock_ProcessTelnet(keyBuffer, len)) == 0)
			return;
	}

	for (idx = 0; idx < len; idx++) {
		uchar ch = keyBuffer[idx];

		// Press ^\ twice to disconnect.
		if (ch == FE_ESCAPE) {
			if (lastSeen == FE_ESCAPE) {
				p10_ctyEof(ctySocket, 0, 0);
				return;
			}
			lastSeen = ch;
			continue;
		}

		// Convert CR NL to CR line.
		if ((ch == 012) && (lastSeen == 015))
			continue;
		lastSeen = ch;

		okSend = TRUE;
		inBuffer[idxInQueue] = ch;
		if (++idxInQueue == 4096)
			idxInQueue = 0;
		if (idxInQueue == idxOutQueue) {
#ifdef DEBUG
			dbg_Printf("CTY: Error - Overrun!!\n");
#endif DEBUG
			break;
		}
	}

	if (okSend && (*ctyInFlag == 0)) {
		*ctyInChar = inBuffer[idxOutQueue];
		*ctyInFlag = 1;
		if (++idxOutQueue == 4096)
			idxOutQueue = 0;
		p10_aprInterrupt(APRSR_F_CON_INT);
	}
}

void p10_kluAccept(SOCKET *srvSocket)
{
	SOCKET *newSocket;

	if (newSocket = sock_Accept(srvSocket)) {
		// First, check if KLU connection already was taken.
		// If so, tell operator that.
		if ((kluSocket != NULL) || (klu_Mode > 0)) {
			int idSocket = newSocket->idSocket;

			sock_Send(idSocket, "Console (KLU) connection already was taken.\r\n", 0);
			sock_Send(idSocket, "Check other terminal which has that connection.\r\n", 0);
			sock_Send(idSocket, "\r\nTerminated.\r\n", 0);
			sock_Close(newSocket);

			return;
		}
		klu_Mode = 1; // State: Password Prompt
 
		// Set up the CTY socket connection.
		newSocket->Accept  = NULL;
		newSocket->Eof     = p10_kluEof;
		newSocket->Process = p10_kluPassword;

		// Send initialization codes and welcome messages
		sock_Send(newSocket->idSocket, telnetInit, n_telnetInit);
		sock_Send(newSocket->idSocket, "Welcome to KS10 Emulator\r\n\r\n", 0);

		if (klu_Password == NULL) {
			sock_Send(newSocket->idSocket, "No Access. Terminated.\r\n", 0);
			p10_kluEof(newSocket, 0, 0);
			return;
		}

		sock_Send(newSocket->idSocket, "Password: ", 0);

		klu_idxPassword = 0;
		klu_tryPassword = 0;

#ifdef DEBUG
		if (dbg_Check(DBG_CONSOLE)) {
			dbg_Printf("KLU: Opened on <date/time>\n");
		}
#endif DEBUG
	}
}

void p10_kluEof(SOCKET *Socket, int rc, int nError)
{
	// Send a CARRIER LOSS signal to KS10 Processor.
	// Let system know a carrier loss signal on KLINIK
	// connection.

	if (klu_Mode >= 2) {
		*kluInFlag = FE_NOCAR;
		*kluInChar = 0;
		p10_aprInterrupt(APRSR_F_CON_INT);
	}

	sock_Close(Socket);

	klu_Mode  = 0;
	kluSocket = NULL;

#ifdef DEBUG
	if (dbg_Check(DBG_CONSOLE)) {
		if (rc < 0) {
			dbg_Printf("KLU: Socket Error: %s\n", strerror(nError));
			dbg_Printf("KLU: *** Carrier Loss ***\n");
		}
		dbg_Printf("KLU: Closed on <date/time>\n");
	}
#endif DEBUG
}

void p10_kluPassword(SOCKET *Socket, char *keyBuffer, int len)
{
	int idx;

	// Process telnet codes and filter them out of data stream.
	if (len > 1) {
		if ((len = sock_ProcessTelnet(keyBuffer, len)) == 0)
			return;
	}

	for (idx = 0; idx < len; idx++) {
		uchar ch = keyBuffer[idx];

		// Press ^\ twice to disconnect.
		if (ch == FE_ESCAPE) {
			if (klu_lastSeen == FE_ESCAPE) {
				p10_kluEof(Socket, 0, 0);
				return;
			}
			klu_lastSeen = ch;
			continue;
		}

		// Convert CR NL to CR line.
		if ((ch == 012) && (klu_lastSeen == 015))
			continue;
		klu_lastSeen = ch;

		// Now check a password.
		if (ch == 015) {
			klu_inPassword[klu_idxPassword] = '\0';
			if (strcmp(klu_inPassword, klu_Password)) {
				// Sorry, wrong password.
				if (++klu_tryPassword == FE_PASSTRY) {
#ifdef DEBUG
					if (dbg_Check(DBG_CONSOLE))
						dbg_Printf("KLU: *** Wrong Password! Access Denied. ***\n");
#endif DEBUG
					sock_Send(Socket->idSocket, "Invalid Password.\r\n", 0);
					p10_kluEof(Socket, 0, 0);
					return;
				}
#ifdef DEBUG
				if (dbg_Check(DBG_CONSOLE))
					dbg_Printf("KLU: *** Wrong Password! Try Again. ***\n");
#endif DEBUG
				sock_Send(Socket->idSocket, "Invalid Password.\r\n", 0);
				sock_Send(Socket->idSocket, "Password: ", 0);
				klu_idxPassword = 0;
				return;
			}
			sock_Send(Socket->idSocket, "Password accepted.\r\n", 0);
			break;
		}

		if (klu_idxPassword < FE_PASSLEN)
			klu_inPassword[klu_idxPassword++] = ch;
		return;
	}

	// Reset all buffer for this new connection.
	klu_idxInQueue   = 0;
	klu_idxOutQueue  = 0;
	klu_idxOutBuffer = 0;
	klu_Mode         = 2; // State: Connected.

	kluSocket = Socket;
	kluSocket->Process = p10_kluInput;

	// Let system know that KLINIK is inited.
	*kluInFlag = FE_INITED;
	*kluInChar = 0;
	p10_aprInterrupt(APRSR_F_CON_INT);
}

void p10_kluInput(SOCKET *Socket, char *keyBuffer, int len)
{
	boolean okSend = FALSE;
	int     idx;

	// Process telnet codes and filter them out of data stream.
	if (len > 1) {
		if ((len = sock_ProcessTelnet(keyBuffer, len)) == 0)
			return;
	}

	for (idx = 0; idx < len; idx++) {
		uchar ch = keyBuffer[idx];

		// Press ^\ twice to disconnect.
		if (ch == FE_ESCAPE) {
			if (klu_lastSeen == FE_ESCAPE) {
				p10_kluEof(kluSocket, 0, 0);
				return;
			}
			klu_lastSeen = ch;
			continue;
		}

		// Convert CR NL to CR line.
		if ((ch == 012) && (klu_lastSeen == 015))
			continue;
		klu_lastSeen = ch;

		okSend = TRUE;
		klu_inBuffer[klu_idxInQueue] = ch;
		if (++klu_idxInQueue == 4096)
			klu_idxInQueue = 0;
		if (klu_idxInQueue == klu_idxOutQueue) {
#ifdef DEBUG
			dbg_Printf("CTY: Error - Overrun!!\n");
#endif DEBUG
			break;
		}
	}

	if (okSend && ((*kluInFlag == FE_NOACT) || (*kluInFlag == FE_INITED))) {
		*kluInChar = klu_inBuffer[klu_idxOutQueue];
		*kluInFlag = 1;
		if (++klu_idxOutQueue == 4096)
			klu_idxOutQueue = 0;
		p10_aprInterrupt(APRSR_F_CON_INT);
	}
}

void p10_ctyCheckQueue(void)
{
	static int count = 0;
	int   pi;

	if (count++ == 1000) {
		count = 0;

		// Update CTYIWD queue.
		if (idxOutQueue != idxInQueue) {
			if (*ctyInFlag == FE_NOACT) {
				*ctyInChar = inBuffer[idxOutQueue];
				*ctyInFlag = FE_PENDING;
				if (++idxOutQueue == 4096)
					idxOutQueue = 0;
				p10_aprInterrupt(APRSR_F_CON_INT);
			}
			if (ctySocket)
				p10_ctyOutput();
		}

		// Update KLUIWD queue.
		if (klu_idxOutQueue != klu_idxInQueue) {
			if (*kluInFlag == FE_NOACT) {
				*kluInChar = klu_inBuffer[klu_idxOutQueue];
				*kluInFlag = FE_PENDING;
				if (++klu_idxOutQueue == 4096)
					klu_idxOutQueue = 0;
				p10_aprInterrupt(APRSR_F_CON_INT);
			}
			if (kluSocket)
				p10_ctyOutput();
		}
	}
}

void p10_ctyOutput(void)
{
	int36 cty;
	uchar flag, ch;
	int pi;
	int status;

	// Write a CTY character to the terminal
	if (kluSocket && *kluOutWord) {
		switch (*kluOutFlag) {
			case FE_PENDING:
				// Print a character on terminal
				ch = *kluOutChar & 0177;
				sock_Send(kluSocket->idSocket, &ch, 1);

				// Log a character into a log file.
				if (ch == '\n') {
					klu_outBuffer[klu_idxOutBuffer++] = ch;
					klu_outBuffer[klu_idxOutBuffer++] = '\0';
#ifdef DEBUG
					if (dbg_Check(DBG_CONSOLE))
						dbg_Printf("KLU: %s", klu_outBuffer);
#endif DEBUG
					if (emu_logFile >= 0)
						write(emu_logFile, klu_outBuffer, klu_idxOutBuffer);
					klu_idxOutBuffer = 0;
				} else {
					if (ch == '\b' || ch == 127) {
						if (klu_idxOutBuffer > 0)
							klu_idxOutBuffer--;
					} else if (ch != '\r' && ch != '\0') {
						if (klu_idxOutBuffer < 4095)
							klu_outBuffer[klu_idxOutBuffer++] = ch;
					}
				}
				break;

			case FE_HANGUP:
				// Send a carrier loss to KS10 Processor
#ifdef DEBUG
				if (dbg_Check(DBG_CONSOLE))
					dbg_Printf("KLU: *** Hangup Request ***\n");
#endif DEBUG
				// Disconnected now.
				p10_kluEof(kluSocket, 0, 0);
				break;

			default:
#ifdef DEBUG
				if (dbg_Check(DBG_CONSOLE))
					dbg_Printf("KLU: Unknown flag = %03o\n", *kluOutFlag);
#endif DEBUG
		}

		p10_ctyCountdown = FE_COUNTDOWN;
		*kluOutWord = 0;
//		p10_aprInterrupt(APRSR_F_CON_INT);
	}

	if (ctySocket && *ctyOutWord) {
		if (*ctyOutFlag == FE_PENDING) {
			ch = *ctyOutChar & 0177;
			sock_Send(ctySocket->idSocket, &ch, 1);

			// Log a character into a log file.
			if (ch == '\n') {
				outBuffer[idxOutBuffer++] = ch;
				outBuffer[idxOutBuffer++] = '\0';
#ifdef DEBUG
				if (dbg_Check(DBG_CONSOLE))
					dbg_Printf("CTY: %s", outBuffer);
#endif DEBUG
				if (emu_logFile >= 0)
					write(emu_logFile, outBuffer, idxOutBuffer);
				idxOutBuffer = 0;
			} else {
				if (ch == '\b' || ch == 127) {
					if (idxOutBuffer > 0)
						idxOutBuffer--;
				} else if (ch != '\r' && ch != '\0') {
					if (idxOutBuffer < 4095)
						outBuffer[idxOutBuffer++] = ch;
				}
			}

			p10_ctyCountdown = FE_COUNTDOWN;
			*ctyOutWord = 0;
//			p10_aprInterrupt(APRSR_F_CON_INT);
		}
	}
}
