// cpu_kxio.c - All Processors: I/O access routines.
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "pdp10/defs.h"

// Function Code Definitions
#define IOF_BLKI   0 // (00) Block In
#define IOF_DATAI  1 // (04) Data In
#define IOF_BLKO   2 // (10) Block Out
#define IOF_DATAO  3 // (14) Data Out
#define IOF_CONO   4 // (20) Conditions Out
#define IOF_CONI   5 // (24) Conditions In
#define IOF_CONSZ  6 // (30) Conditions Skip Zero
#define IOF_CONSO  7 // (34) Conditiion Skip Ones

typedef struct {
	char   *devName;   // Device Name
	DEVICE *Device; 
	UNIT   *Unit;
	void   (*Function[8])(UNIT *);
} P10DEVICE;

// Up to 128 devices per CPU.
P10DEVICE *p10_kxDevices[0200];

#ifdef DEBUG
char *p10_funcNames[] = {
	"BLKI",  // (00) Block In
	"DATAI", // (04) Data In
	"BLKO",  // (10) Block Out
	"DATAO", // (14) Data Out
	"CONO",  // (20) Condition Out
	"CONI",  // (24) Condition In
	"CONSZ", // (30) Condition Skip Zero
	"CONSO", // (34) Condition Skip One
};
#endif DEBUG

void p10_kxOpcode_IO(void)
{
	int  opDev  = INST_GETDEV(HR);
	int  opFunc = INST_GETFUNC(HR);
	void (*DoFunction)(void);

	if (p10_kxDevices[opDev]) {
#ifdef DEBUG
		if (dbg_Check(DBG_IOREGS)) {
			char *devName = p10_kxDevices[opDev]->devName;

			dbg_Printf("KXIO: Device: %s (%03o) Function: %s (%02o)\n",
				devName ? devName : "(Unknown)", (opDev << 1),
				p10_funcNames[opFunc], (opFunc << 2));
		}
#endif DEBUG
		if (DoFunction = p10_kxDevices[opDev]->Function[opFunc]) {
			DoFunction(p10_kxDevices[opDev]->Unit);
			return;
		}
	}

	// Default function for undefined devices.
	switch (opFunc) {

	}
}
