// tcu150.c - TCU routines.
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <time.h>
#include "emu/defs.h"
#include "dev/proto.h"

// Article posting from alt.sys.pdp10 by Bob Supnik.
//
// From KSSER.MAC,
//
// Base address = 760770 on Unibus adapter 3.
//
// Register 0
//   15:9   years since 1900 (0-127)
//    8:5   month (1-12)
//    4:0   day (1-31)
//
// Register 2
//   12:8   hour (0-24, with midnight represented as 24:00,
//          but the KS10 will accept 00:00)
//    5:0   minute (0-59)
//
// Register 4
//    5:0   second (0-59)
//
// I have no idea what happens if you write, I can't find any references
// to that.  Implementation is easy, a tm struct has all the fields one
// needs.
//
// Of course, this gives the KS10 a year 2028 problem...

// TCU150 Registers on 760770.
#define TCU_DATE  0 // (R) Date Entry (Year/Month/Day)
#define TCU_TIME1 1 // (R) Time Entry (Hour/Minute)
#define TCU_TIME2 2 // (R) Time Entry (Second)
#define TCU_SR    3 // (R) Status Register

// 760770 - Date Register
#define TCU_DATE_YEAR    0177000 // (R) Year   (0-127)
#define TCU_DATE_MONTH   0000740 // (R) Month  (1-12)
#define TCU_DATE_DAY     0000037 // (R) Day    (1-31)

// 760772 - Time 1 Register
#define TCU_TIME1_HOUR   0017400 // (R) Hour   (0-24)
#define TCU_TIME1_MINUTE 0000077 // (R) Minute (0-59)

// 760774 - Time 2 Register
#define TCU_TIME2_SECOND 0000077 // (R) Second (0-59)

// 760776 - Status Register
#define TCU_SR_READY     0000200 // (R) Ready

DTYPE tcu_dTypes[] =
{
	{
		"TCU100",      // Controller Name
		"TCU-100",
		UNIT_DISABLE,  // Controller is disable
		0, 0, 0, 0,    // Not used
		000000,        // Drive Type ID - Not Used
		NULL,          // Slave Drive Types - Not Used
	},

	{
		"TCU150",      // Controller Name
		"TCU-150",
		UNIT_DISABLE,  // Controller is disable
		0, 0, 0, 0,    // Not used
		000000,        // Drive Type ID - Not Used
		NULL,          // Slave Drive Types - Not Used
	},

	{ NULL }
};

DEVICE tcu_Device =
{
	"TCU",           // Device Name
	"Timing Control Unit",
	"v0.1 (Alpha)",  // Version
	tcu_dTypes,      // Drive Type
	NULL,            // RH11 Units - up to 8 RH controllers
	NULL,            // Listing of devices
	NULL,            // List of Commands
	NULL,            // List of Set Commands
	NULL,            // List of Show Commands

	0,               // Number of Devices
	0,               // Number of Units

	tcu_Initialize,  // Initialize Routine
	tcu_Reset,       // Reset Routine
	tcu_Create,      // Create Routine
	tcu_Delete,      // Delete Routine
	NULL,            // Configure Routine
	NULL,            // Enable Routine
	NULL,            // Disable Routine
	NULL,            // Attach/Mount Routine
	NULL,            // Detach/Unmount Routine
	NULL,            // Format Routine
	tcu_ReadIO,      // Read I/O Routine
	tcu_WriteIO,     // Write I/O Routine
	NULL,            // Process Routine
	NULL,            // Boot Routine
	NULL             // Execute Routine
};

#ifdef DEBUG
char *tcu_regNames[] = {
	"TCU_DATE",
	"TCU_TIME1",
	"TCU_TIME2",
	"TCU_SR",
};
#endif DEBUG

void tcu_Initialize(UNIT *tcuUnit)
{
}

void tcu_Reset(UNIT *tcuUnit)
{
}

// Create TCU device
int tcu_Create(UNIT *pUnit, char *devName, int argc, char **argv)
{
	UNIT *uptr;
	int  unit;
	int  len, st;
	int  idx1, idx2;

	// Get a unit pointer by device name.
	len  = strlen(tcu_Device.Name);
	unit = toupper(devName[len++]) - 'A';
	uptr = &pUnit->sUnits[unit];

	if (!(uptr->Flags & UNIT_PRESENT)) {
		char  *pAddress, *epAddress;
		char  *pMask,    *epMask;
		int32 nAddress, nMask;
		int idx;

		for (idx1 = 0; tcu_dTypes[idx1].Name; idx1++) {
			if (!strcasecmp(argv[0], tcu_dTypes[idx1].Name))
				break;
		}

		if (!tcu_dTypes[idx1].Name)
			return EMU_ARG;

		pAddress = argv[1];
		pMask    = argv[2];

		nAddress = util_ToInteger(pAddress, &epAddress, 8);
		nMask    = util_ToInteger(pMask, &epMask, 8);

		if ((pAddress == epAddress) || (pMask == epMask))
			return EMU_ARG;
				
		if (st = pUnit->Device->Configure(pUnit, NULL, nAddress, nMask))
			return st;

/*
		*strchr(argv[5], ':') = '\0';
		if (st = unit_mapCreateDevice(argv[5], uptr)) {
			free(rhUnits);
			return st;
		}
*/

		// This unit is a time clock unit
		uptr->tFlags = UT_CLOCK;
		uptr->dType  = &tcu_dTypes[idx1];
		uptr->Device = &tcu_Device;
		uptr->Flags  = tcu_dTypes[idx1].Flags | UNIT_PRESENT;
		uptr->pUnit  = pUnit;

		pUnit->Device->Configure(pUnit, uptr, nAddress, nMask);
	} else
		return EMU_ARG;
	return EMU_OK;
}

// Disable the desired RH11 controller
int tcu_Delete(UNIT *tcuUnit)
{
	if (tcuUnit->Flags & UNIT_DISABLED)
		return EMU_ARG;

	tcuUnit->Flags |= UNIT_DISABLED;

	return EMU_OK;
}

void tcu_WriteIO(UNIT *tcuUnit, int32 addr, int32 data)
{
	int32 reg = (addr & 07);

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: (W) Illegal Register: %02o (%06o)\n",
			tcuUnit->dType->Name, reg, addr);
#endif DEBUG
}

int32 tcu_ReadIO(UNIT *tcuUnit, int32 addr)
{
	int32  reg = (addr & 07);
	struct tm *tm;
	time_t now;
	int32  data;

	// Get time clock data
	now = time(NULL);
	tm  = localtime(&now);

	switch (reg >> 1) {
		case TCU_DATE:
			// Return current year/month/day.
			data = ((tm->tm_year & 0177) << 9) |
			       (((tm->tm_mon + 1) & 017)  << 5) |
			       (tm->tm_mday & 037);
			break;

		case TCU_TIME1:
			// Return current hours and minutes
			data = ((tm->tm_hour & 037) << 8) |
			       (tm->tm_min & 077);
			break;

		case TCU_TIME2:
			// Return current seconds.
			data = tm->tm_sec & 077;
			break;

		case TCU_SR:
			// Always tell it is ready.
			data = TCU_SR_READY;
			break;

#ifdef DEBUG
		default:
			if (dbg_Check(DBG_IOREGS))
				dbg_Printf("%s: (R) Illegal Register: %02o (%06o)\n",
					tcuUnit->dType->Name, reg, addr);
			return 0;
#endif DEBUG
	}

#ifdef DEBUG
	if (dbg_Check(DBG_IOREGS))
		dbg_Printf("%s: (R) %s (%02o) => %06o\n", tcuUnit->dType->Name,
			tcu_regNames[reg >> 1], reg, data);
#endif DEBUG

	return data;
}
