/* VPort-50 Monitor
   Copyright (c) 2004, Hans Rosenfeld

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
*/

#include <build.h>
#include <common/command.h>
#include <common/break.h>
#include <common/stdio.h>
#include <common/string.h>
#include <common/mem.h>
#include <common/intel.h>
#include <mach/addr.h>
#include <mach/mach.h>
#include <mach/setjmp.h>
#include <mach/status.h>
#include <mach/asm.h>


/* Befehle */
const struct command c_comm[] = {
        { "dump",        CMD_DUMP   },
        { "set",         CMD_SET    },
        { "show",        CMD_SHOW   },
        { "reset",       CMD_RESET  },
        { "input",       CMD_INPUT  },
        { "output",      CMD_OUTPUT },
        { "move",        CMD_MOVE   },
        { "disassemble", CMD_DISASM },
        { "start",       CMD_START  },
        { "continue",    CMD_CONT   },
        { "help",        CMD_HELP   },
        { "clear",       CMD_CLR    },
        { 0, 0 }
};

/* Default-Qualifier */
const struct command c_def[] = {
        { "address",     DEF_ADDR },
        { "port",        DEF_PORT },
        { 0, 0 }
};

/* Parameter fr clear */
const struct command c_del[] = {
        { "breakpoint",  CLR_BRK },
        { "current",     CLR_CUR },
        { "registers",   CLR_REG },
        { 0, 0 }
};

/* Parameter fr set */
const struct command c_set[] = {
        { "memory",      SET_MEM },
        { "breakpoint",  SET_BRK },
        { "registers",   SET_REG },
        { "trace",       SET_TRC },
        { "radix",       SET_RAD },
        { 0, 0 }
};

/* Qualifier fr set memory */
const struct command c_setmem[] = {
        { "address",     SET_MEM_A },
        { "intel",       SET_MEM_I },
        { "hex",         SET_MEM_I },
        { 0, 0 }
};

/* Qualifier fr set memory/intel */
const struct command c_intel[] = {
        { "start",       SET_MEM_I_AUTO },
        { 0, 0 }
};

/* Argumente fr set trace */
const struct command c_trace[] = {
        { "on",          TRC_ON  },
        { "off",         TRC_OFF },
        { "toggle",      TRC_TOG },
        { 0, 0 }
};

/* Parameter fr show */
const struct command c_show[] = {
        { "current",     SHO_CUR },
        { "breakpoint",  SHO_BRK },
        { "registers",   SHO_REG },
        { "machine",     SHO_PRM },
        { 0, 0 }
};

/* Register - werden in mach.c definiert */
extern const struct command c_reg[];

/* Hauptprogramm, wird von _main in main.s aufgerufen,
   sobald der Stack und die Interrupt-Vektoren eingerichtet sind
   diese Routine kehrt nie zurck
*/
void monitor()
{
        char input[80];
        char *ptr;
        unsigned int cmd=0, tmp=0, val=0, cnt=0, num=0, bcnt=0;
        unsigned int set=0, setmem=0, sho=0, iport=0, oport=0, reg=0, clr=0;
        addr_t src, dst, start;

        init();
        brkcnt = 0;
        brkcur = NBRK;
        trace = 0;
        clr_addr(&src);
        clr_addr(&dst);
        clr_addr(&start);
        radix = RADIX;

        printf("\n\n%C Monitor Build %d, %C\n", NAME, BUILD, DATE);
        setjmp();

restart:
        printf("$ ");
        ptr=gets(input, 79);
        ptr=skip(ptr, " \t\n");
        if(*ptr != '\0') {
                ptr=find_command(c_comm, ptr, &cmd);
        } else if(cmd == 0) goto restart;

        /* das ist zwar nicht besonders schn, so alles in einem,
           aber vorher war es _noch_ hsslicher
        */
        switch(cmd) {
        case CMD_DUMP:
                /* dump[/address=][addr]
                   Gibt einen Speicherabzug von 16 * 8 Bytes ab addr hexadezimal
                   und in ASCII aus.
                */
                if(*ptr == '/') {
                        ptr=find_command(c_def, ptr+1, &tmp);
                        if(tmp != DEF_ADDR)
                                goto qualifier;
                }
                if(*ptr) {
                        ptr=scan_addr(ptr, &src);
                        bcnt=0;
                }
                add_addr(&src, bcnt);
                odd_addr_inc(&src);
                bcnt=dump(&src);
                break;

        case CMD_SET: /* set */
                if(*ptr)
                        ptr=find_command(c_set, ptr, &set);
                switch(set) {
                case SET_MEM:
                        /* set memory[/address=addr] [{ string | [val [cnt]] }]]
                           Schreibt das Argument in den Speicher an addr.
                           Wird kein Argument angegeben, wird der aktuelle Speicherinhalt
                           angezeigt und die Eingabe eines String- oder Byte-Arguments
                           erwartet.
                           set memory/intel
                           Liest Daten im Intel-Hex-Format in den Speicher.
                        */
                        if(*ptr == '/') {
                                ptr=find_command(c_setmem, ptr+1, &setmem);
                                switch(setmem) {
                                case SET_MEM_I:
                                        if(*ptr == '/')
                                                ptr=find_command(c_intel, ptr+1, &setmem);
                                        cmd = 0;
                                        intel(&start);
                                        printf("> ");
                                        if(setmem == SET_MEM_I_AUTO) {
                                                odd_addr_err(&start);
                                                initreg(&start);
                                                exec();
                                        }
                                        goto restart;
                                        break;
                                case SET_MEM_A:
                                        ptr=scan_addr(ptr, &src);
                                        ptr=skip(ptr, " \t\n");
                                        odd_addr_inc(&src);
                                        break;
                                default:
                                        goto qualifier;
                                }
                        }
                        if(*ptr)
                                edit(ptr, &src, &val, &cnt);
                        else for(;;) {
                                odd_addr_inc(&src);
                                printf("%A ", &src);
                                printf(MEMFMT, read_mem(&src));
                                ptr=gets(input, 79);
                                ptr=skip(ptr, " \t\n");
                                if(*ptr == 'x' || *ptr == '.')
                                        break;
                                if(*ptr) 
                                        edit(ptr, &src, &val, &cnt);
                                else add_addr(&src, sizeof(mem_t));
                        }
                        bcnt=0;
                        break;
                case SET_BRK:
                        /* set breakpoint[/address=][addr]
                           Trgt addr als Breakpoint ein.
                        */
                        if(brkcnt == NBRK) {
                                printf("no more than %d breakpoints possible\n", NBRK);
                                goto restart;
                        }
                        if(*ptr == '/') {
                                ptr=find_command(c_def, ptr+1, &tmp);
                                if(tmp != DEF_ADDR)
                                        goto qualifier;
                        }
                        if(*ptr) {
                                ptr=scan_addr(ptr, &src);
                        }
                        odd_addr_err(&src);
                        setbreak(&src);
                        break;
                case SET_REG:
                        /* set registers
                           Setzt die angegebenen Register auf die angegebenen Werte.
                           Die eigentliche Zuweisung funktioniert nur, weil (bzw. wenn)
                           die fr c_reg in mach.h definierten Konstanten fr die
                           Register Werte haben, die ihrer Reihenfolge in struct jmpbuf
                           in setjmp.h entsprechen.
                        */
                        do {
                                if(*ptr) {
                                        ptr=find_register(c_reg, ptr, &reg);
                                        ptr=skip(ptr, " \t\n=");
                                }
                                if(reg) {
                                        if(*ptr) {
                                                ptr=scann(ptr, &val, 0);
                                                ptr=skip(ptr, " \t\n");
                                        }
                                        ((unsigned int*) &status)[reg-1] = val;
                                } else
                                        goto parameter;
                                trace = istrace();
                        } while(*ptr);
                        break;
                case SET_TRC:
                        /* set trace [{on|off|toggle}]
                           Setzt das Trace-Flag im Flag-Register und in der Variable
                           trace. 
                           Falls wir gerade auf einem Breakpoint stehen, setzen wir in
                           jedem Fall das Trace-Flag, weil der Breakpoint-Handler es
                           verwendet.
                        */
                        if(*ptr) {
                                ptr=find_command(c_trace, ptr, &tmp);
                                switch(tmp) {
                                case TRC_ON:
                                        settrace();
                                        break;
                                case TRC_OFF:
                                        untrace();
                                        break;
                                case TRC_TOG:
                                        togtrace();
                                        break;
                                default:
                                        goto parameter;
                                }
                        } else
                                togtrace();
                        printf("trace is %C\n", (trace = istrace()) ? "on" : "off");
                        if(brkcur != NBRK)
                                settrace();
                        break;
                case SET_RAD:
                        /* set radix
                           undokumentiert, setzt den Default-Radix des Monitors
                        */
                        ptr=scann(ptr, &tmp, 10);
                        if(tmp!=16 && tmp!=10 && tmp!=8)
                                printf("illegal radix - %d\n", tmp);
                        else radix=tmp;
                        break;
                default:
                        goto parameter;
                }
                break;
        case CMD_SHOW:
                /* show { current | state | breakpoints | registers | machine } */
                if(*ptr)
                        ptr=find_command(c_show, ptr, &sho);
                switch(sho) {
                case SHO_CUR:
                        printf("\
src = %A, dst = %A, start = %A,\n\
val = %r, cnt = %r, bcnt = %r, reg = %d, num = %d,\n\
iport = %r, oport = %r\n",
                               &src, &dst, &start, val, cnt, bcnt, reg, num, iport, oport);
                        printf("trace is %C\n", trace ? "on" : "off");
                        break;
                case SHO_BRK:
                        if(brkcnt == 0)
                                printf("no breakpoints set\n");
                        else for(tmp=0; tmp!=brkcnt; tmp++)
                                printf("breakpoint %d: %A\n", tmp, &brktbl[tmp].addr);
                        break;
                case SHO_REG:
                        showreg();
                        break;
                case SHO_PRM:
                        printf(
                                "%C %d \"%C\" \"%C\" \"%C\" \"%C\" \"%C\" %d %d %d %d %d\n",
                                NAME, BUILD, ADRFMT, MEMFMT, IOFMT, DMPFMT_A, DMPFMT_S,
                                sizeof(addr_t), sizeof(mem_t),  sizeof(break_t),
                                RADIX, radix);
                        break;
                default:
                        goto parameter;
                }
                break;

        case CMD_RESET:
                /* reset
                   reset() springt den Reset-Vektor an
                */
                reset();
                break;

        case CMD_INPUT:
                /* input[/port=][port]
                   Liest vom IO-Port port in val
                */
                if(*ptr == '/') {
                        ptr=find_command(c_def, ptr+1, &tmp);
                        if(tmp != DEF_PORT)
                                goto qualifier;
                }
                if(*ptr) {
                        ptr=skip(ptr, " \t");
                        ptr=scann(ptr, &iport, 0);
                }
                odd_port(iport);
                val=in(iport);
                printf(IOFMT,  val);
                break;

        case CMD_OUTPUT:
                /* output [/port=port] [val]
                   Schreibt val auf den IO-Port port
                */
                if(*ptr) {
                        if(*ptr == '/') {
                                ptr=find_command(c_def, ptr+1, &tmp);
                                if(tmp != DEF_PORT)
                                        goto qualifier;
                                ptr=skip(ptr, " \t");
                                ptr=scann(ptr, &oport, 0);
                                ptr=skip(ptr, " \t\n");
                                if(*ptr) {
                                        ptr=skip(ptr, " \t");
                                        ptr=scann(ptr, &val, 0);
                                }
                        } else {
                                ptr=skip(ptr, " \t");
                                ptr=scann(ptr, &tmp, 0);
                                ptr=skip(ptr, " \t\n");
                                if(*ptr) {
                                        oport=tmp;
                                        ptr=skip(ptr, " \t");
                                        ptr=scann(ptr, &val, 0);
                                } else val=tmp;
                        }
                }
                odd_port(oport);
                out(oport, val);
                break;

        case CMD_MOVE:
                /* move [addr [daddr [count]]]
                   Kopiert einen Speicherbereich der Lnge count von addr an daddr
                */
                if(*ptr) {
                        ptr=skip(ptr, " \t");
                        ptr=scan_addr(ptr, &src);
                }
                if(*ptr) {
                        ptr=skip(ptr, " \t");
                        ptr=scan_addr(ptr, &dst);
                }
                if(*ptr) {
                        ptr=skip(ptr, " \t");
                        ptr=scann(ptr, &cnt, 0);
                }
                move(&src, &dst, cnt);
                bcnt=0;
                break;

        case CMD_DISASM:
                /* disasm[/addr=][addr]
                   disassembliert die nchsten 8 Befehle ab addr
                */
                if(*ptr == '/') {
                        ptr=find_command(c_def, ptr+1, &tmp);
                        if(tmp != DEF_ADDR)
                                goto qualifier;
                }
                if(*ptr)
                        ptr=scan_addr(ptr, &src);
                odd_addr_err(&src);
                for(tmp=0; tmp!=8; tmp++)
                        disasm(&src);	  
                break;

        case CMD_START:
                /* start[/addr=][addr]
                   startet das Programm ab addr
                   setzt ausserdem einen Stack fr das Programm auf
                */
                if(*ptr == '/') {
                        ptr=find_command(c_def, ptr+1, &tmp);
                        if(tmp != DEF_ADDR)
                                goto qualifier;
                }
                if(*ptr) {
                        ptr=skip(ptr, " \t");
                        ptr=scan_addr(ptr, &start);
                }
                odd_addr_err(&start);
                initreg(&start);
                brkcur=NBRK;
                trace ? settrace() : untrace();
                cmd = CMD_CONT;
        case CMD_CONT:
                /* continue
                   Breakpoints einrichten und Programm starten
                */
                ptr=skip(ptr, " \t\n");
                if(*ptr)
                        goto parameter;
                dobreak();
                exec();
                break;

        case CMD_CLR: /* clear {reg | current | break } */
                if(*ptr)
                        ptr=find_command(c_del, ptr, &clr);
                switch(clr) {
                case CLR_REG: /* setzt alle Register auf 0 */
                        clrreg();
                        break;
                case CLR_CUR: /* setzt die Current Values auf 0 */
                        cmd=tmp=val=cnt=set=setmem=sho=iport=oport=reg=clr=bcnt = 0;
                        clr_addr(&src);
                        clr_addr(&dst);
                        break;
                case CLR_BRK: /* lscht Breakpoints */
                        cnt=0;
                        while(*ptr) {
                                ptr  = scann(ptr, &num, 10);
                                num -= cnt;
                                cnt++;
                                if(num < brkcnt) {
                                        for(tmp = num; tmp != (brkcnt-1); tmp++) {
                                                brktbl[tmp].inst = brktbl[tmp+1].inst;
                                                cpy_addr(&brktbl[tmp+1].addr, &brktbl[tmp].addr);
                                        }
                                        brkcnt--;
                                } else
                                        printf("no such breakpoint - %d\n", num+cnt-1);
                                ptr=skip(ptr, " \t\n");
                        }
                        break;
                default:
                        goto parameter;
                }
                break;

        case CMD_HELP:
                printf("\n\n%C Monitor Build %d, %C\n\n", NAME, BUILD, DATE);
                printf("\
dump memory contents\t\t- dump[/address=][addr]\n\
disassemble program\t\t- disassemble[/address=][addr]\n\
set memory contents\t\t- set memory[/address=addr] [{string|val[,cnt]}]\n\
move memory block\t\t- move [addr [addr [cnt]]]\n\
read Intel-Hex data\t\t- set memory/intel\n\
show register contents\t\t- show registers\n\
set register contents\t\t- set registers [reg[=][val]]\n\
zero registers\t\t\t- clear registers\n\
read I/O port\t\t\t- input[/port=][iport]\n\
write I/O port\t\t\t- output[[[/port=]oport] val]\n\
set breakpoint\t\t\t- set breakpoint[/address=][addr]\n\
delete breakpoint\t\t- clear breakpoint [num]\n\
list breakpoints\t\t- show breakpoints\n\
trace mode on/off/toggle\t- set trace [{on | off | toggle}]\n\
start program execution\t\t- start[/address=][addr]\n\
continue program execution\t- continue\n\
show current values\t\t- show current\n\
clear current values\t\t- clear current\n\
reset Monitor\t\t\t- reset\n\
\n\
All commands and parameters can be abbreviated to a minimum of 2 characters.\n");
                break;
        default:
                printf("unknown command\n");
        }
        goto restart;

parameter:
        printf("unknown parameter\n");
        goto restart;
qualifier:
        printf("unknown qualifier\n");
        goto restart;
}
