/* VPort-50 Monitor
   Copyright (c) 2005, Hans Rosenfeld

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

#include <arm/lpc2xxx.h>
#include <arm/asm.h>
#include <arm/uart.h>
#include <arm/mach.h>
#include <arm/setjmp.h>
#include <common/stdio.h>

/* Zeiger auf Register-Strukturen */
struct pin * const pin = (struct pin *) PIN_BASE;
struct mem * const mem = (struct mem *) MEM_BASE;
struct mam * const mam = (struct mam *) MAM_BASE;
struct pll * const pll = (struct pll *) PLL_BASE;

/* unbenutzt:
   struct pwr * const pwr = (struct pwr *) PWR_BASE;
   struct vpb * const vpb = (struct vpb *) VPB_BASE;
   struct ext * const ext = (struct ext *) EXT_BASE;
   struct emc * const emc = (struct emc *) EMC_BASE;
*/

/* Initialisierung:
   - vom Monitor verwendete Hardware initialisieren (PLL, MAM und den UART)
   - UART-Pins aktivieren
   - Interrupt-Vektoren aus Flash in RAM kopieren
   - Memory-Mapping auf RAM umschalten
   - Interrupts aktivieren
*/
void init(void)
{
        int i;
        int *fptr = (int *) FLASH_START;
        int *sptr = (int *) SRAM_START;

        pll_init();
        mam_init();
        uart_init();

        pin->sel0 = 0x00000005;

        for(i = 0; i != 64; i++)
                *sptr++ = *fptr++;

        mem->map = MEM_SRAM;

        untrace();
        enable_interrupts();
}

/* PLL-Initialisierung
   P = 2, M = 5
*/
void pll_init(void)
{
        pll->cfg = PLL_PSEL(2) | PLL_MSEL(5);

        pll->con = PLL_ENBL;
        pll->feed = PLL_FEED1;
        pll->feed = PLL_FEED2;
     
        while((pll->stat & PLL_LOCK) == 0);

        pll->con = PLL_ENBL | PLL_CONN;
        pll->feed = PLL_FEED1;
        pll->feed = PLL_FEED2;
}
 
/* MAM-Initialisierung
   MAM Fully Enabled, 4 cycles/fetch
*/    
void mam_init(void)
{
        mam->tim = MAM_CYCLE(4);
        mam->cr = MAM_FENB;
}

/* Zeiger auf IAP-Routine im Bootcode */
void (* const iap)(struct iap_param *, struct iap_param *) = (void (*)(struct iap_param*, struct iap_param *)) IAP_LOCATION;

void iap_wrapper(struct iap_param *param, struct iap_param *result)
{
        disable_interrupts();
        iap(param, result);
        enable_interrupts();
}

int has_flash(void)
{
        struct iap_param param, result;

        param.code = IAP_PARTID;
        iap_wrapper(&param, &result);

        if(result.code == IAP_CMD_SUCCESS)
                return(PID_FLASH(result.p[0]));
        else {
                printf("%c%creading part id failed, result code = %d\n", 7, 7, result.code);
                longjmp();
        }
}

void write_flash(addr_t addr, char *mem)
{
        struct iap_param param, result;
        int secnum;

        param.code = IAP_PARTID;
        iap_wrapper(&param, &result);

        if(result.code != IAP_CMD_SUCCESS) {
                printf("%c%creading part id failed, result code = %d\n", 7, 7, result.code);
                longjmp();
        }

        switch(PID_FLASH(result.p[0])) {
        case 0:
                printf("%c%cno flash memory present\n", 7, 7);
                longjmp();
        case 1:
                secnum = (addr >> 13) & 0xf;
                break;
        case 2:
                secnum = (addr >> 13) & 0x1f;
                if(secnum >= 8 && secnum < 16)
                        secnum = 8;
                else if(secnum >= 16 && secnum < 24)
                        secnum = 9;
                else if(secnum >= 24)
                        secnum -= 14;
                break;
        default:
                printf("%c%cunknown flash memory system\n", 7, 7);
                longjmp();
        }

        param.code = IAP_PREPARE;
        param.p[0] = secnum;
        param.p[1] = secnum;
        iap_wrapper(&param, &result);

        if(result.code != IAP_CMD_SUCCESS) {
                printf("%c%cpreparing flash sector %d failed, result code = %d\n", 7, 7, secnum, result.code);
                longjmp();
        }

        param.code = IAP_COPY;
        param.p[0] = addr;
        param.p[1] = (addr_t) mem;
        param.p[2] = 512;
        param.p[3] = 60000;
        iap_wrapper(&param, &result);

        if(result.code != IAP_CMD_SUCCESS) {
                printf("%c%ccopying 0x%r to 0x%r (sector %d) failed, result code = %d\n", 7, 7,
                       mem, addr, secnum, result.code);
                longjmp();
        }

        param.code = IAP_COMPARE;
        param.p[0] = addr;
        param.p[1] = (addr_t) mem;
        param.p[2] = 512;
        iap_wrapper(&param, &result);

        if(result.code != IAP_CMD_SUCCESS) {
                printf("%c%ccompare error at 0x%r, result code = %d\n", 7, 7, result.p[0], result.code);
                longjmp();
        }
}
