/* VPort-50 Monitor
   Copyright (c) 2005 Hans Rosenfeld

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
*/

.section .text

. = 0x00000000
main:  
reset_vector:
        ldr     pc, reset_addr
undef_vector:
        ldr     pc, undef_addr
swi_vector:
        ldr     pc, swi_addr
pabort_vector:
        ldr     pc, pabort_addr
dabort_vector:
        ldr     pc, dabort_addr
valid_user_program_key:
        nop
irq_vector:
        ldr     pc, irq_addr
fiq_vector:
        ldr     pc, fiq_addr

reset_addr:
        .word   start
undef_addr:
        .word   undef_handler
swi_addr:
        .word   swi_handler
pabort_addr:
        .word   pabort_handler
dabort_addr:
        .word   dabort_handler
irq_addr:
        .word   irq_handler
fiq_addr:
        .word   fiq_handler

        
/* Eintrittspunkt in den Monitor
   Stack einrichten und Monitor aufrufen
   die Interrupt-Vektoren werden in den RAM-Bereich kopiert,
   das Memory-Mapping wird jedoch nicht in User-RAM-Mode umgeschaltet
*/
.globl  main
. = 0x00000040
start:
        msr     cpsr_c, #0xd3           /* Supervisor Mode setzen */
        mrs     r0, cpsr                /* CPSR lesen             */
        and     r0, r0, #0x1f           /* Mode Bits              */
        cmp     r0, #0x13               /* Supervisor Mode?       */
        bne     start                   /* wenn nicht, dann nicht */
        
        mov     r0, #0x40000000         /* RAM-Bereich   */
        add     sp, r0, #0x800          /* 2k Stack      */
        bl      monitor                 /* Monitor aufrufen, kehrt nicht zurck */
        b       reset                   /* und wenn doch... */
        
/* enable_interrupts(), disable_interrupts() */
.globl enable_interrupts, disable_interrupts
enable_interrupts:
        mrs     r0, cpsr
        bic     r0, r0, #0xc0
        msr     cpsr_c, r0
        mov     pc, lr

disable_interrupts:
        mrs     r0, cpsr
        orr     r0, r0, #0xc0
        msr     cpsr_c, r0
        mov     pc, lr
        
/* Exception-Handler
   - der Stackpointer der Exception-Modi Abort und Undefined ist identisch
     mit dem in jmpbuf.r13 gespeicherten SP fr den Supervisor-Mode
   - wie beim SWI werden die User-Mode Register auf den Supervisor-Stack
     geschoben
   - umschalten in Supervisor-Mode, Korrektur des Stackpointers
   - die syscall()-Routine wird mit einem Funktionscode gerufen, der
     auerhalb des Wertebereiches fr SWI-Codes liegt
   - bei Exceptions innerhalb des Monitors wird der CPU-Zustand nicht korrekt
     gespeichert und z.T. zerstrt
*/
     
undef_handler:
        str     lr, [sp, #-4]!          /* Rcksprungadresse  */
        mov     lr, sp                  /* SP sichern         */
        sub     sp, sp, #60             /* Platz fr Register */
        stmfd   lr, {r0-r14}^           /* R0-R14 User Mode   */
        mrs     r1, spsr                /* CPSR User Mode     */
        str     r1, [sp, #-4]!
        mov     r0, sp                  /* Stackpointer fr syscall()    */
        add     sp, sp, #68             /* Stackpointer zurckdrehen     */
        msr     cpsr_c, #0xd3           /* Umschalten in Supervisor-Mode */
        mov     sp, r0                  /* Stackpointer fr syscall()    */
        mov     r0, #0x01000000         /* "Undefined Instruction"       */
        bl      syscall                 /* syscall(0x01000000, d1, d2, d3, cpsr, r0, r1, r2, r3, r4, r5, r6, r7, r8, r9, r10, r11, r12, r13, r14, r15) */
        /* kehrt nicht zurck? */
        b       reset
                        
pabort_handler: 
        str     lr, [sp, #-4]!          /* Rcksprungadresse  */
        mov     lr, sp                  /* SP sichern         */
        sub     sp, sp, #60             /* Platz fr Register */
        stmfd   lr, {r0-r14}^           /* R0-R14 User Mode   */
        mrs     r1, spsr                /* CPSR User Mode     */
        str     r1, [sp, #-4]!
        mov     r0, sp                  /* Stackpointer fr syscall()    */
        add     sp, sp, #68             /* Stackpointer zurckdrehen     */
        msr     cpsr_c, #0xd3           /* Umschalten in Supervisor-Mode */
        mov     sp, r0                  /* Stackpointer fr syscall()    */
        mov     r0, #0x02000000         /* "Prefetch Abort"   */
        bl      syscall                 /* syscall(0x02000000, d1, d2, d3, cpsr, r0, r1, r2, r3, r4, r5, r6, r7, r8, r9, r10, r11, r12, r13, r14, r15) */
        /* kehrt nicht zurck? */
        b       reset

dabort_handler: 
        str     lr, [sp, #-4]!          /* Rcksprungadresse  */
        mov     lr, sp                  /* SP sichern         */
        sub     sp, sp, #60             /* Platz fr Register */
        stmfd   lr, {r0-r14}^           /* R0-R14 User Mode   */
        mrs     r1, spsr                /* CPSR User Mode     */
        str     r1, [sp, #-4]!
        mov     r0, sp                  /* Stackpointer fr syscall()    */
        add     sp, sp, #68             /* Stackpointer zurckdrehen     */
        msr     cpsr_c, #0xd3           /* Umschalten in Supervisor-Mode */
        mov     sp, r0                  /* Stackpointer fr syscall()    */
        mov     r0, #0x03000000         /* "Data Abort"       */
        bl      syscall                 /* syscall(0x03000000, d1, d2, d3, cpsr, r0, r1, r2, r3, r4, r5, r6, r7, r8, r9, r10, r11, r12, r13, r14, r15) */
        /* kehrt nicht zurck? */
        b       reset

irq_handler:    
        subs    pc, lr, #4
        
fiq_handler:
        subs    pc, lr, #4   

/* SWI-Handler */
.globl  syscall
swi_handler:
        str     lr, [sp, #-4]!          /* Rcksprungadresse */
        mov     lr, sp                  /* SP sichern        */
        sub     sp, sp, #60             /* Platz fr Register */
        stmfd   lr, {r0-r14}^           /* R0-R14 User Mode  */
        mrs     r1, spsr                /* CPSR User Mode    */
        str     r1, [sp, #-4]!
        ldr     lr, [lr]                /* Rcksprungadresse wiederherstellen */
        ldr     r0, [lr, #-4]           /* SWI Befehl        */
        bics    r0, r0, #0xff000000     /* Opcode entfernen  */
        beq     reset                   /* SWI 0 ist Reset   */
        bl      syscall                 /* syscall(func, d1, d2, d3, cpsr, r0, r1, r2, r3, r4, r5, r6, r7, r8, r9, r10, r11, r12, r13, r14, r15) */
        ldr     r1, [sp], #8            /* CPSR User Mode in R1, SP auf gesichertes R1 */
        msr     spsr_fsxc, r1           /* CPSR User Mode    */
        ldmfd   sp, {r1-r14}^           /* R0-R14 User Mode  */
        nop                             /* Zugriff auf Register vermeiden */
        ldr     lr, [sp, #56]!          /* Rcksprungadresse */
        add     sp, sp, #4              /* Stack aufrumen   */
        movs    pc, lr                  /* Rcksprung, R0 enthlt Rckgabe von syscall()
        
/* setjmp(), longjmp() fr C
   Achtung: im Gegensatz zu den blichen Versionen bentigen diese Funkionen
   kein Argument, es steht genau ein Speicherplatz zur Verfgung
*/
.globl  setjmp, longjmp, jmpbuf, status
setjmp:
        ldr     r0, =jmpbuf     /* Adresse von jmpbuf in R0       */
        mrs     r1, cpsr
        str     r1, [r0], #4    /* CPSR in jmpbuf.cpsr            */
        stmia   r0, {r0-r14}    /* R0-R14 in jmpbuf.r0-r14        */

        /* an dieser Stelle bietet es sich an, die Stackpointer fr
           Abort- und Undefined-Mode zu setzen
        */
        mov     r0, sp          /* Stackpointer (== jumpbuf.r13)  */
        mrs     r1, cpsr        /* CPSR sichern                   */
        msr     cpsr_c, #0xd7   /* Abort Mode                     */
        mov     sp, r0          /* Abort Mode SP setzen           */
        msr     cpsr_c, #0xdb   /* Undefined Mode                 */
        mov     sp, r0          /* Undefined Mode SP setzen       */
        msr     cpsr_c, r1      /* CPSR wiederherstellen          */
        
        ldr     r0, =jmpbuf     /* Adresse von jmpbuf in R0       */
        mov     r1, pc          /* &(mov pc, lr) in R0            */
        str     r1, [r0, #64]   /* &(mov pc, lr) in jmpbuf.r15    */
        mov     pc, lr          /* Rcksprung (auch bei longjmp)  */
        
longjmp:
        ldr     r0, =jmpbuf     /* Adresse von jmpbuf in R0       */
        ldr     r1, [r0], #8    /* jmpbuf.cpsr in R1              */
        msr     cpsr_fsxc, r1   /* jmpbuf.cpsr in CPSR            */
        ldmia   r0!, {r1-r14}   /* jmpbuf.r1-r14 in R1-R14        */
        ldr     pc, [r0]        /* Rcksprung zum Ende von setjmp */

/* Programm starten
   vor dem eigentlichen Start wird der Stackpointer des Monitors
   auf den in jmpbuf gespeicherten Wert zurckgedreht
   Rckkehr aus exec ist nicht mglich
*/
.globl  trace_emulation
.globl  exec
exec:
        bl      trace_emulation
        ldr     r14, =jmpbuf    /* Adresse von jmpbuf in R14      */
        ldr     sp, [r14, #56]  /* jmpbuf.r13 in R13              */
        mov     r0, sp          /* Stackpointer sichern           */
        msr     cpsr_c, #0xd7   /* Abort Mode                     */
        mov     sp, r0          /* Abort Mode SP setzen           */
        msr     cpsr_c, #0xdb   /* Undefined Mode                 */
        mov     sp, r0          /* Undefined Mode SP setzen       */
        
        ldr     r14, =status    /* Adresse von status in R14      */
        ldr     r1, [r14], #4   /* status.cpsr in R1              */
        msr     spsr_fsxc, r1   /* status.cpsr in SPSR            */
        ldmia   r14, {r0-r14}^  /* User-Mode Register laden       */
        nop                     /* Zugriff auf Register vermeiden */
        ldr     r14, [r14, #60] /* status.r15 in R14              */
        movs    pc, lr          /* Rcksprung ins Programm        */

/* Software-Reset
   wenn der Prozessor nicht im Supervisor Mode luft, Reset-SWI aufrufen
   andernfalls Sprung zum Reset-Handler
*/
.globl  reset
reset:
        mrs     r0, cpsr        /* CPSR in R0                     */
        and     r0, r0, #0x1f   /* alles auer Mode-Bits lschen  */
        cmp     r0, #0x13       /* Supervisor Mode?               */
        swine   #0x0            /* Reset-SWI falls nicht          */
        mov     pc, #0          /* reset-handler aufrufen         */

/* unsigned int ror(r0, r1)
   r0 wird um r1 Stellen bitweise nach rechts rotiert
*/
.globl ror
ror:    mov     r0, r0, ror r1
        mov     pc, lr

