;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	@(#)format.s	1.2	07/12/29
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	.area CODE (ABS)

	.include "memcfg.s"
	
	.org 0x0100

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Available ROM routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

cold_start		= rom+3*0
warm_start		= rom+3*1
con_rxstat		= rom+3*2
con_txstat		= rom+3*3
getchar			= rom+3*4
putchar			= rom+3*5
gets			= rom+3*6
put			= rom+3*7
puts			= rom+3*8
crlf			= rom+3*9
prhex			= rom+3*10
_spio13_rxstat		= rom+3*11
_spio13_txstat		= rom+3*12
_spio13_sctrl		= rom+3*13
_spio13_sread		= rom+3*14
_spio13_swrite		= rom+3*15
_ide_address		= rom+3*16
_ide_hard_reset		= rom+3*17
_ide_drive_id		= rom+3*18
_ide_read_a_sector	= rom+3*19
_ide_write_a_sector	= rom+3*20


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Disk definitions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
TYPE_EXT	= 0x05		; extended partition type
TYPE_CPM	= 0x52		; CP/M partition
LBA		= 0b01000000	; LBA flag for the drive/head byte
BURN		= 63		; burn the first 63 sectors of the disk
SECSIZE		= 512		; sector size
DRM		= 1023		; DRM value from the BIOS
DIRBLK		= (DRM+1)*32/SECSIZE	; number of directory blocks to format

drive		= 0		; bits 0-6 always 0, bit 7 set if bootable
head		= 1		; partition starting point (CHS)
sector		= 2		;
cylinder	= 3		;
type		= 4		; partition type
end_head	= 5		; partition ending point (CHS)
end_sector	= 6		;
end_cylinder	= 7		;
start_lba	= 8		; starting LBA of the partition
len_lba		= 12		; length of partition in blocks


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Start of program
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
start:
	xor a,a			;
	ld (drivec),a		; clear the drive count
	ld hl,#msg2		; get heading message
	call puts		; print it
	call _ide_address	; get the controller address
	ld b,#1			; get CHS 1
	ld e,#0			;
	ld d,#0			;
	ld a,#0			;
	ld hl,#buffer		; buffer address
	call _ide_read_a_sector	; read the MBR into memory
	call scanpart		; scan the partition entries
	call getdrive		; get the drive to format in A
	ret nc			; leave if bad response

	ld (driven),a		; otherwise, save the drive number
	ld hl,#msg6		; print formatting message
	call put		;
	ld a,(driven)		; otherwise, save the drive number
	add a,#'A		; print drive letter we're formatting
	call putchar		;
	ld a,#0x20		; print space
	call putchar		;
	ld a,#'@		; print @
	call putchar		;
	xor a,a			; clear the drive count
	ld (drivec),a		;

	call finddrive		; get the LBA for the drive we're formatting
	ld a,(fptype)		; get partition type
	cp a,#1			; extended?
	;jr z,1$			; yes, don't add offset
	ld de,#BURN		; reserve these sectors for the OS
	ld hl,(flba)		; get the LBA
	add hl,de		; add the reserve offset
	ld (flba),hl		;
	ld de,#0		;
	ld hl,(flba+2)	;
	adc hl,de		;
	ld (flba+2),hl	;

1$:	ld a,(flba+3)		; print the LBA
	call prhex		;
	ld a,(flba+2)		;
	call prhex		;
	ld a,(flba+1)		;
	call prhex		;
	ld a,(flba)		;
	call prhex		;
	call crlf		;

	call format		; create the directory entries
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Scan the disk for CP/M partitions and
;	report to the user.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
scanpart:
	ld a,#TYPE_CPM		; get the CP/M partition type
	ld ix,#part1		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,1$		; no, continue
	call reportpart		; yes, report it
1$:	ld ix,#part2		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,2$		; no, continue
	call reportpart		; yes, report it
2$:	ld ix,#part3		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,3$		; no, continue
	call reportpart		; yes, report it
3$:	ld ix,#part4		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,4$		; no, continue
	call reportpart		; yes, report it

4$:	ld a,#TYPE_EXT		; get the CP/M partition type
	ld ix,#part1		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,5$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
5$:	ld ix,#part2		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,6$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
6$:	ld ix,#part3		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,7$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
7$:	ld ix,#part4		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,8$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
8$:	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Scan extended partition for CP/M
;	partitions and report to the user.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
scanext:
	ld a,#TYPE_CPM		; get the CP/M partition type
	ld ix,#epart1		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,1$		; no, continue
	call copyrunlba		; yes, add LBA
	call reportpart		; report it
1$:	ld ix,#epart2		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,2$		; no, continue
	call copyrunlba		; yes, add LBA
	call reportpart		; report it
2$:	ld ix,#epart3		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,3$		; no, continue
	call copyrunlba		; yes, add LBA
	call reportpart		; report it
3$:	ld ix,#epart4		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,4$		; no, continue
	call copyrunlba		; yes, add LBA
	call reportpart		; report it

4$:	ld a,#TYPE_EXT		; get the CP/M partition type
	ld ix,#epart1		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,5$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
5$:	ld ix,#epart2		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,6$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
6$:	ld ix,#epart3		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,7$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
7$:	ld ix,#epart4		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,8$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext		; get partition table of ext partition
8$:	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Report the partition.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
reportpart:
	bit 7,drive(ix)		; partition bootable?
	jr z,2$			; no, print a space
	ld a,#'*		; yes, get an asterisk
	jr 3$			; continue
2$:	ld a,#0x20		; get a space
3$:	call putchar		; print it
	ld a,(drivec)		; get the drive count
	add a,#'A		; convert to a letter
	call putchar		; print it
	ld a,#':		; get colon
	call putchar		; print it
	ld a,#0x20		; get space
	call putchar		; print it
	ld a,(drivec)		; get drive count
	inc a			; increment it
	ld (drivec),a		; save it
	ld a,start_lba+3(ix)	; print the LBA
	call prhex		;
	ld a,start_lba+2(ix)	;
	call prhex		;
	ld a,start_lba+1(ix)	;
	call prhex		;
	ld a,start_lba(ix)	;
	call prhex		;
	ld hl,#msg3		; print size message
	call put		;
	ld a,len_lba+3(ix)	; print length
	call prhex		;
	ld a,len_lba+2(ix)	; 
	call prhex		;
	ld a,len_lba+1(ix)	;
	call prhex		;
	ld a,len_lba(ix)	;
	call prhex		;
	call crlf		; EOL
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load an extended partition table.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
getext:
	call getaddr		; get the sector address
	or a,#LBA		; set the LBA flag
	call _ide_read_a_sector	; load the sector
	call scanext		; scan it (recursively)
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the LBA address and the memory
;	address
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
getaddr:
	xor a,a			; clear the block count
	ld (blocks),a		; 
	call _ide_address	; get the controller address
	ld b,start_lba(ix)	; get the LBA
	ld e,start_lba+1(ix)	;
	ld d,start_lba+2(ix)	;
	ld a,start_lba+3(ix)	;
	ld hl,#buf2		; get the target address
	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Get the drive to format.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
getdrive:
	ld hl,#msg4		; ask which drive to format.
	call put		;
	ld hl,#kbuf		; get the keyboard buffer address
	call gets		; get the drive letter
1$:	ld a,(hl)		; get the next character in the buffer
	cp #0x20		; space?
	jr nz,2$		; no, leave
	inc hl			; yes, skip to the next buffer position
	jr 1$			; check the character
2$:	or a,a			; EOL?
	jp nz,3$		; no, continue
	or a,a			; yes clear carry flag
	ret			; leave
3$:	and a,#0xdf		; convert drive letter to uppercase
	sub a,#'A		; calculate the drive number
	ld hl,#drivec		; get the drive counter
	cp a,(hl)		; out of range?
	ret nc			; if so, leave
	push af			; otherwise, save drive
	ld hl,#msg5		; ask if sure
	call put		;
	ld hl,#kbuf		; get the keyboard buffer address
	call gets		; get response
4$:	ld a,(hl)		; get the next character in the buffer
	cp #0x20		; space?
	jr nz,5$		; no, leave
	inc hl			; yes, skip to the next buffer position
	jr 4$			; check the character
5$:	or a,a			; EOL?
	jp nz,6$		; no, continue
	pop af			; yes, clean up the stack
	or a,a			; clear carry flag
	ret			; leave
6$:	and a,#0xdf		; convert to uppercase
	sub a,#'Y		; sure?
	jr z,7$			; yes,
	pop af			; no, leave
	or a,a			; clear carry
	ret			; leave
7$:	pop af			; clean up
	scf			; set carry
	ret			; done


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Find the LBA of the a particular drive.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
finddrive:
	xor a,a			; clear partition type
	ld (fptype),a		;

	ld a,#TYPE_CPM		; get the CP/M partition type
	ld ix,#part1		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,1$		; no, continue
	call getpart		; yes, look for a match
	ret c			; return if match found
1$:	ld ix,#part2		; scan partition 2
	cp a,type(ix)		; CP/M?
	jr nz,2$		; no, continue
	call getpart		; yes, look for a match
	ret c			; return if match found
2$:	ld ix,#part3		; scan partition 3
	cp a,type(ix)		; CP/M?
	jr nz,3$		; no, continue
	call getpart		; yes, look for a match
	ret c			; return if match found
3$:	ld ix,#part4		; scan partition 4
	cp a,type(ix)		; CP/M?
	jr nz,4$		; no, continue
	call getpart		; yes, look for a match
	ret c			; return if match found

4$:	ld a,#TYPE_EXT		; get the CP/M partition type
	ld ix,#part1		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,5$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
5$:	ld ix,#part2		; scan partition 2
	cp a,type(ix)		; Extended?
	jr nz,6$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
6$:	ld ix,#part3		; scan partition 3
	cp a,type(ix)		; Extended?
	jr nz,7$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
7$:	ld ix,#part4		; scan partition 4
	cp a,type(ix)		; Extended?
	jr nz,8$		; no, continue
	call savelba		; yes, save the LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
8$:	ret


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Load an extended partition table.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
getext2:
	call getaddr		; get the sector address
	or a,#LBA		; set the LBA flag
	call _ide_read_a_sector	; load the sector
	call findext		; scan it (recursively)
	ret			; done.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Search an extended partition for CP/M
;	partitions and report to the user.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
findext:
	ld a,#1			; set partition type
	ld (fptype),a		;

	ld a,#TYPE_CPM		; get the CP/M partition type
	ld ix,#epart1		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,1$		; no, continue
	call copyrunlba		; yes, add LBA
	call getpart		; look for a match
	ret c			; return if match found
1$:	ld ix,#epart2		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,2$		; no, continue
	call copyrunlba		; yes, add LBA
	call getpart		; look for a match
	ret c			; return if match found
2$:	ld ix,#epart3		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,3$		; no, continue
	call copyrunlba		; yes, add LBA
	call getpart		; look for a match
	ret c			; return if match found
3$:	ld ix,#epart4		; scan partition 1
	cp a,type(ix)		; CP/M?
	jr nz,4$		; no, continue
	call copyrunlba		; yes, add LBA
	call getpart		; look for a match
	ret c			; return if match found

4$:	ld a,#TYPE_EXT		; get the CP/M partition type
	ld ix,#epart1		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,5$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
5$:	ld ix,#epart2		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,6$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
6$:	ld ix,#epart3		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,7$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
7$:	ld ix,#epart4		; scan partition 1
	cp a,type(ix)		; Extended?
	jr nz,8$		; no, continue
	call addlba		; yes, add LBA
	call saverunlba		; save running LBA
	call getext2		; get partition table of ext partition
	ret c			; return if match found
8$:	ret


getpart:
	ld a,(drivec)		; get the drive count
	ld b,a			; save in B
	ld a,(driven)		; get the drive number
	cp a,b			; same?
	jr z,1$			; yes, save the LBA & leave
	ld a,(drivec)		; no, get drive count
	inc a			; increment it
	ld (drivec),a		; save it
	or a,a			; clear carry
	ret			; leave
1$:	call saveflba		; save the drive LBA
	;pop hl			; throw away the return address
	scf			; set carry
	ret			; done



savelba:
	ld a,start_lba(ix)
	ld (extlba),a
	ld a,start_lba+1(ix)
	ld (extlba+1),a
	ld a,start_lba+2(ix)
	ld (extlba+2),a
	ld a,start_lba+3(ix)
	ld (extlba+3),a
	ret

saveflba:
	ld a,start_lba(ix)
	ld (flba),a
	ld a,start_lba+1(ix)
	ld (flba+1),a
	ld a,start_lba+2(ix)
	ld (flba+2),a
	ld a,start_lba+3(ix)
	ld (flba+3),a
	ret

saverunlba:
	ld a,start_lba(ix)
	ld (runlba),a
	ld a,start_lba+1(ix)
	ld (runlba+1),a
	ld a,start_lba+2(ix)
	ld (runlba+2),a
	ld a,start_lba+3(ix)
	ld (runlba+3),a
	ret

addlba:
	ld l,start_lba(ix)
	ld h,start_lba+1(ix)
	ld a,(extlba)
	ld e,a
	ld a,(extlba+1)
	ld d,a
	add hl,de
	ld start_lba(ix),l
	ld start_lba+1(ix),h
	ld l,start_lba+2(ix)
	ld h,start_lba+3(ix)
	ld a,(extlba+2)
	ld e,a
	ld a,(extlba+3)
	ld d,a
	adc hl,de
	ld start_lba+2(ix),l
	ld start_lba+3(ix),h
	ret

addrunlba:
	ld l,start_lba(ix)
	ld h,start_lba+1(ix)
	ld a,(runlba)
	ld e,a
	ld a,(runlba+1)
	ld d,a
	add hl,de
	ld start_lba(ix),l
	ld start_lba+1(ix),h
	ld l,start_lba+2(ix)
	ld h,start_lba+3(ix)
	ld a,(runlba+2)
	ld e,a
	ld a,(runlba+3)
	ld d,a
	adc hl,de
	ld start_lba+2(ix),l
	ld start_lba+3(ix),h
	ret

copyrunlba:
	ld a,(runlba)
	ld start_lba(ix),a
	ld a,(runlba+1)
	ld start_lba+1(ix),a
	ld a,(runlba+2)
	ld start_lba+2(ix),a
	ld a,(runlba+3)
	ld start_lba+3(ix),a
	ret

format:
	ld hl,#buf2		; zero out buffer 2
	ld b,#0			;
1$:	ld (hl),#0		;
	inc hl			;
	djnz 1$			;
2$:	ld (hl),#0		;
	inc hl			;
	djnz 2$			;

	ld de,#20
	ld hl,#buf2		; start of buffer
4$:	ld (hl),#0xe5		; unsed directory record value
	inc hl			; next position
	ld b,#11		; write 11 spaces (filename.ext)
3$:	ld (hl),#0x20		;
	inc hl			;
	djnz 3$			;
	add hl,de		; skip to the next record position
	ld a,h			; end of sector?
	cp a,#>(buf2+SECSIZE)	;
	jr c,4$			; no, continue
	ld a,l			;
	cp a,#<(buf2+SECSIZE)	;
	jr c,4$			; no, continue

	ld a,#DIRBLK		; number of directory blocks to write
	ld (blocks),a		; save it

	ld a,(flba)		; load the LBA
	ld b,a			;
	ld a,(flba+1)		;
	ld e,a			;
	ld a,(flba+2)		;
	ld d,a			;
	ld a,(flba+3)		;
	or a,#LBA		; set the LBA flag
	push af			; save A
6$:	pop af
	push af
	ld hl,#msg7		; print writing block message
	call put		;
	pop af
	push af
	call prhex		; print the LBA
	ld a,d			;
	call prhex		;
	ld a,e			;
	call prhex		;
	ld a,b			;
	call prhex		;
	call crlf		;
	pop af
	push af
	ld hl,#buf2		; get the sector buffer
	call _ide_write_a_sector; write the sector
	pop af			; restore A
	inc b			; increment the LBA
	jr nz,5$		;
	inc e			;
	jr nz,5$		;
	inc d			;
	jr nz,5$		;
	inc a			;
5$:	push af			; save A
	ld a,(blocks)		; decrement the block count
	dec a			;
	ld (blocks),a		;
	jr nz,6$		; continue if not done.
	pop af			; clean up the stack
	ret			; leave

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Initialized data.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
msg1:	.asciz "NO BOOT DISK FOUND"
msg2:	.asciz "CP/M PARTITIONS FOUND:"
msg3:	.asciz "  SIZE: "
msg4:	.asciz "FORMAT WHICH DRIVE? "
msg5:	.asciz "ARE YOU SURE? "
msg6:	.asciz "FORMATTING DRIVE "
msg7:	.asciz "WRITING BLOCK "

drivec:	.db	0		; drive counter
driven:	.db	0		; drive 
blocks:	.db	0		; number of blocks loaded from disk


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Uninitialized data.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
extlba:	.ds	4
runlba:	.ds	4
flba:	.ds	4
fptype:	.ds	1
kbuf:	.ds	128


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	Sector buffers.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
buffer:	.ds	446		; not used
part1:	.blkb	16		; partition 1
part2:	.blkb	16		; partition 2
part3:	.blkb	16		; partition 3
part4:	.blkb	16		; partition 4
	.blkb	2		; end of boot sector

buf2:	.ds	446		; not used
epart1:	.blkb	16		; partition 1
epart2:	.blkb	16		; partition 2
epart3:	.blkb	16		; partition 3
epart4:	.blkb	16		; partition 4
	.blkb	2		; end of sector

