/*************************************************************************

	KB_CU.C      -- Keyboard (KB) Subroutines
			for Simple Software.

			Implementation for use with CURSES systems.

			The use of Simple Software is governed by the
			"Simple Software User Agreement" (see file
			SS.DOC).

			Copyright (c) 1988, Ted A. Campbell

			Bywater Software
			Box 4023
			Duke Station,
			Durham, NC  27706

**************************************************************************/

#include "curses.h"
#include "kb.h"

#ifndef CR
#define	CR	0x0d
#define LF	0x0a
#define	ESC	0x1b
#endif

#define	DELBACK	0x08
#define TAB	0x09

/*************************************************************************

	FUNCTION:       kb_init()

	DESCRIPTION:    This function should perform any initialization
			necessary for the keyboard system.

	INPUT:          none.

	RETURNS:        none.

**************************************************************************/

kb_init()
	{
	raw();
	noecho();
	}

/*************************************************************************

	FUNCTION:       kb_deinit()

	DESCRIPTION:    This function should perform any necessary
			deinitialization, that is, return the keyboard
			to its default state when a Simple Software
			program is to be exited.

	INPUT:          none.

	RETURNS:        none.

**************************************************************************/

kb_deinit()
	{
	noraw();
	echo();
	}

/*************************************************************************

	FUNCTION:       kb_cis()

	DESCRIPTION:    This function determines whether a character is
			ready from the console.  The function is used
			especially in telecommunications programs,
			where it is necessary to poll the keyboard
			without locking up the program waiting for a
			response.

	INPUT:          none.

	RETURNS:        The function returns 0 if no character is
			available and 1 if a character is available.

	NOTE:		Not implemented.

**************************************************************************/

kb_cis()
	{
	return 1;
	}

/*************************************************************************

	FUNCTION:       kb_ci()

	DESCRIPTION:    This function returns a single character from
			the keyboard.  If a character is not available
			it waits.  The function should be able to
			recognize any special keys, and return the
			appropriate Simple Software KB conventions
			designated for them.

	INPUT:          none.

	RETURNS:        The function returns the ASCII code for the
			key pressed, or the Simple Software KB convention
			(see kb.h) for a function or other special key.

**************************************************************************/

kb_ci()
	{
	int c;
	c = getch();
	if ( ( c >= 0x20 ) && ( c < 0x7f ) )
		{
		return c;
		}
	else
		{
		switch ( c )
			{
			case 0x02:              /* Control-B    */
				return KB_LEFT;
			case 0x04:              /* Control-D    */
				return KB_D_CHAR;
			case 0x06:              /* Control-F    */
				return KB_RIGHT;
			case 0x0e:              /* Control-N    */
				return KB_DOWN;
			case 0x10:              /* Control-P    */
				return KB_UP;
			case 0x1b:
				return cu_esc();
			case LF:
			case CR:
			case TAB:
			case DELBACK:
				return c;
			default:
				return 0;
				break;
			}
		}
	}

/*************************************************************************

	FUNCTION:       cu_esc()

	DESCRIPTION:    This function handles escape sequences from the
			keyboard.  It calls the kb_emenu() function
			(defined by the application program), accepts
			a new command letter, then replace the menu
			with kb_dmenu (the default menu defined by
			the application program).

	INPUT:          none.

	RETURNS:        The function returns the Simple Software
			convention for a special key sequence (see
			kb.h), or 0.

**************************************************************************/

cu_esc()
	{
	register int c;
	kb_emenu();
	c = getch();
	kb_dmenu();
	switch ( toupper(c) )
		{
		case 'B':
			return KB_W_LEFT;
		case 'F':
			return KB_W_RIGHT;
		case 'P':
			return KB_P_UP;
		case 'N':
			return KB_P_DOWN;
		case 'H':
			return KB_HOME;
		case 'E':
			return KB_END;
		case 'D':
			return KB_D_WORD;
		case 'I':
			return KB_INSERT;
		case 'M':
			return KB_FIND;
		case 'R':
			return KB_REPLACE;
		case 'S':
			return KB_SAVE;
		case 'Q':
			return KB_ABANDON;
		case 'X':
			return KB_EXIT;
		default:
			return 0;
			break;
		}
	}

