/* xcomm.c		External (and XMODEM) Communications.
 *			By Eric E. Coe
 *
 * This program is in the public domain, and anyone can use it for personal
 * or non-profit use. Feel free to study the code and borrow from it, (I
 * borrowed from other programs to create it).  I am not supporting this
 * program, but if you have something interesting to say, my address is:
 *
 * Eric E. Coe c/o CARP
 * 481 8th Ave.  Rm 640
 * New York, NY 10001
 *
 * 
 * Source files  Function
 * ------------  --------
 * xcomm.c     - Initialization, command line parsing, main command
 *               loop, put and take, sub-shell, misc parameter fiddling.
 * xcomm.h     - Global defines and variables.
 * xcterm.c    - Terminal mode, capture buffer, autodial, text dribble.
 * xcxmdm.c    - XMODEM protocol file transfer routines.
 * xcsubs.c    - Generally used subroutines.
 * xcport.c    - Modem port configuration and i/o.
 *
 * To compile:
 *
 *	CC -O xcomm.c xcterm.c xcxmdm.c xcsubs.c xcport.c -o xcomm -s
 *
 * Interactive communications program with XMODEM protocol data transfer.
 * Designed to replace the cu utility.  Binary and text file transfer with
 * both CP/M systems using XMODEM and UNIX systems using umodem, rb, sb, uc,
 * or others; is handled using the XMODEM routines, (those with names starting
 * with wc...).  Text can be captured while in terminal mode, is temporarily
 * stored in a memory buffer, and is saved to disk at a time chosen by the
 * operator (when toggling the buffer closed or when leaving terminal mode).
 * Text can be transmitted from a local file to the remote computer in terminal
 * mode; a "dribble" option is available that delays after the the end of each
 * line of text (2sec default), this is for programs that expected input at
 * typing speed, (like my favorite RCPM's message program).  The shell is
 * availble for commands on the local system.  A hook has been left for
 * other data transfer protocols.
 *
 * An optional is feature is a cu - compatible put and take facility. This
 * allows easier file transfer with UNIX type systems that do not have any
 * XMODEM compatible transfer utilitys; but do have stty and cat. Unfortuantely
 * I have not been able to exclude on the take option the next command (shell)
 * prompt at the end of the file.  To disable this feature comment out the
 * PUT_TAKE define in xcomm.h
 *
 * Since this program is designed to fufill just about every need as far as
 * local communications programs go, it has suffered a bad case of "creeping
 * feature-itis".  I put in anything that would help me in my late-nite
 * excursions on the RCP/M's and Bulletin Boards.
 * 
 * Code parts and concepts culled from rb, sb, umodem, cu, etc., in their
 * various PD versions.  Written on/for a TRS-80 Mod 16 w/ XENIX-68000 V2.3A
 * system.  (This is fairly close to UNIX V7).
 *
 * If someone changes the progeam, please change the version number and
 * describe who, when, and what. (It stacks up, as you can see.)
 */

#define  VERSION   "1.1"

/* Version  Who            When      What
 * -------  ---            ----      ----
 * 1.1      Eric E. Coe    7/21/85   Autodial for Hayes - compatible modem.
 *                                   General re-arranging of the code.
 * 1.0      Eric E. Coe    4/12/85   Program created
 */

#include <stdio.h>
#include <signal.h>
#include <ctype.h>
#include <sgtty.h>
#include "xcomm.h"

/* globals */
jmp_buf erret;				/* non-local error return	*/
int	verbose		= 1,		/* verbosity level		*/
	mungmode	= FALSE;	/* ok to overwrite files?	*/

/* locals */
static char vblist[3][8] = { "quiet", "normal", "verbose" };
static struct sgttyb
    newmode,	/* tty struct for raw console mode */
    oldmode;	/* tty struct for standard (cooked) console mode */

/* command mode help messages (must be edited if any defaults are changed)
 */
char *help[] = {
"\fHelp for XCOMM command mode.  These commands are available:",
"",
"t           <*- Enter terminal mode. These keys are interpreted by the",
"                local program and not transmitted:",
"  shift-F1    - return to comand mode",
"  shift-F2    - toggle capture buffer open/closed",
"  shift-F3    - transmit a text file (w/ \"dribble\" option)",
"  shift-F4    - select a phone number to dial",
"",
"rb [file]   <-- Receive file(s) from remote XMODEM program; no file name",
"                given means the sender is expected to provide the name(s),",
"                and many files can be transfered",
"rt [file]   <-- Like above but with CP/M-to-UNIX text conversions",
"sb file ... <-- Send binary file(s) to remote XMODEM program",
"st file ... <-- Send file(s) with UNIX-to-CP/M text conversions",
#ifdef PUT_TAKE
"%p file     <-- Put file (transmit a file to a UNIX system)",
"%t file     <-- Take file (receive a file from a UNIX system)",
#endif
"\n",
"b num       <*- Change modem port baud rate to 'num' (1200 baud default)",
"c num       <*- Change capture buffer size to 'num' kilobytes (32k default)",
"n file      <*- Change name of capture buffer save file ('commsave' default)",
"p file      <*- Change name of phone number file ('.phonelist' default)",
"7           <*- Set a 7-bit data mask",
"8           <*- Set a 8-bit data mask (default)",
"k           <*- Toggle the CRC check feature (default is ON)",
"v           <*- Make program more verbose (more status messages)",
"q           <*- Make program more quiet (less status messages)",
"m           <*- Allow overwriting files (toggle mungmode)",
"z           <*- Toggle bad telephone line mode. (more purging for r & b cmds)",
"",
"! cmd       <-- Execute shell command string on the local system",
"!           <-- Execute a local interactive shell",
"!!          <-- Re-execute the last command string",
"$ cmd       <-- Execute a shell command string with standard input and output",
"                re-directed to the modem port. (Use this for other binary",
"                transfer protocols like kermit, TERM II, etc.)",
"",
"x           <-- Exit XCOMM program",
"?           <*- Print this help message\n",
"Note: those commands with a * can also be accessed from the command line,",
"Along with the 'l' command, that designates an alternate modem port from",
"the one designated in the environment variable MODEM. The command line is",
"given in a format similar to tar(1), where the first argument is key formed",
"from the command letters and the other arguments (if any) correspond to those",
"command letters that require arguments, (see above). For example:",
"",
"xcomm qbn7lt 300 saveit /dev/tty02",
"",
"in order; sets the quiet reporting mode, a port baud rate of 300, a capture",
"file name of 'saveit' and the modem port to '/dev/tty02', and immediately",
"enter terminal mode.\n",
NULL
};

main(argc, argv)
int argc;
char *argv[];
{
    int c, i, j, catch();
    char inbuf[WBSIZE], oldshell[WBSIZE], *rest, *p, *index();

    ioctl(0, TIOCGETP, &oldmode);	/* get current console tty mode */
    newmode = oldmode;			/* copy (structure) to newmode	*/
    newmode.sg_flags |= RAW;		/* prepare to set raw interface	*/
    newmode.sg_flags &= ~ECHO;		/* prepare to set echoing off	*/
    oldshell[0] = '\0';			/* set last command to blank	*/
    j = FALSE;				/* no direct jump into terminal mode */
    if(setjmp(erret))			/* set error handler to exit	*/
	exit(0);			/* while parsing command line	*/
    signal(SIGINT, catch);		/* catch break & quit signals/keys */
    signal(SIGQUIT, catch);

    printf("\fXCOMM: External Communications - Ver %s  By EEC\n", VERSION);
    if(argc > 1){			/* any arguments ?		*/
	rest = argv[1];			/* set command key in first arg */
	i = 2;				/* other args start from 2	*/
	while(c = mklow(*rest++)){	/* get each command letter in lc */
	    if(index("lbcnp", c) != NULL){ /* argument needed?		*/
		if(i >= argc){		/* got enough left?		*/
		    printf("Not enough arguments.\n");
		    prhelp();
		    exit(-1);
		}
		p = argv[i++];		/* save pointer and incr. counter */
	    }
	    switch(c){
	    case 'l':			/* set modem port name		*/
		mport(p);
		break;
	    case 'b':			/* set modem port baud rate	*/
		if(mbaud(p) < 0)
		    printf("Invalid speed argument %s\n", p);
		break;
	    case 'c':			/* set capture buffer size	*/
		if(atoi(p) > 0)
		    cbsiz = atoi(p) * K;
		else
		    printf("Invalid capture buffer size %s\n", p);
		break;
	    case 'n':			/* set capture file name	*/
		strcpy(captfile, p);
		break;
	    case 'p':			/* set phone number file name	*/
		strcpy(phonefile, p);
		break;
	    case 't': j = TRUE;        break;	/* jump into terminal mode */
	    case '7': bitmask = 0x7F;  break;	/* 7-bit interface	*/
	    case '8': bitmask = 0xFF;  break;	/* 8-bit interface	*/
	    case 'k': crcheck = FALSE; break;	/* set CRC mode off 	*/
	    case 'v': verbose = 2;     break;	/* more messages	*/
	    case 'q': verbose = 0;     break;	/* less messages	*/
	    case 'm': mungmode = TRUE; break;	/* can overwrite files	*/
	    case 'z': badline = TRUE;  break;	/* bad phone line mode	*/
	    case '?': prhelp();        break;	/* print help message	*/
	    default:  prhelp();       exit(-1);	/* bad command		*/
	    }
	}
    }
    if(mopen() < 0){	/* open modem port and configure */
	printf("XCOMM: Error in modem initialization routine.\n");
	printf("XCOMM: Check environment variable MODEM for modem port.\n");
	exit(-1);
    }
    if(verbose)
	status();	/* display all parameters */
    if(j)
	goto term;	/* jump directly to terminal mode */
    while(TRUE){
	while(setjmp(erret))		/* set error restart to command loop */
	    printf("...  re-start.\n");
	if(verbose)	/* allowable commands list	*/
#ifdef PUT_TAKE
	    printf("Type t,x,b,c,n,7,8,v,q,rb,rt,sb,st,%%p,%%t,!,$, or ? for help\n");
#else
	    printf("Type t,x,b,c,n,7,8,v,q,rb,rt,sb,st,!,$, or ? for help\n");
#endif
	printf("XCOMM> ");		/* command prompt		*/
	gets(rest = inbuf);		/* get command			*/
	skipspace(&rest);		/* skip leading spaces		*/
	if(!(c = mklow(*rest++)))	/* get first command char	*/
	    continue;			/* if blank line prompt again	*/
					/* check for modifer for r, s, and % */
	if(*rest && !isspace(*rest) && (c == 'r' || c == 's'
#ifdef PUT_TAKE
							    || c == '%'
#endif
									))
	    i = *rest++;		/* save modifier char	*/
	skipspace(&rest);		/* move up to args (if any) */
	switch(c) {
	case '7': /* set 7-bit i/o interface */
	    bitmask = 0x7F;
	    if(verbose)
		printf("Seven-bit interface enabled.\n");
	    break;
	case '8': /* set 8-bit i/o interface */
	    bitmask = 0xFF;
	    if(verbose)
		printf("Eight-bit interface enabled.\n");
	    break;
	case 'k': /* toggle CRC check mode */
	    crcheck = !crcheck;
	    if(verbose)
		printf("%s error checking mode is enabled.\n",
					crcheck ? "CRC check" : "Checksum");
	    break;
	case 'v': /* increment verbose mode */
	    if(verbose < 2)
		verbose++;
	    goto vbmp;
	case 'q': /* decrement verbose mode */
	    if(verbose > 0)
		verbose--;
vbmp:
	    printf("Messages will now be in %s mode.\n", vblist[verbose]);
	    break;
	case 'm': /* toggle file overwriting */
	    mungmode = !mungmode;
	    if(verbose)
		printf("File overwriting is %sabled.\n", mungmode?"en":"dis");
	    break;
	case 'z': /* toggle bad phone line mode */
	    badline = !badline;
	    if(verbose)
		printf("Bad phone line mode is %sabled.\n", badline?"en":"dis");
	    break;
	case 'b': /* change the port's baud rate */
	    if(verbose)
		printf("Modem port's speed was %d baud.\n",mbaud((char *)NULL));
	    if(mbaud(rest) < 0){
		printf("Incorrect baud rate.\n");
		break;
	    }
	    if(verbose)
		printf("Changed to %d baud.\n", mbaud((char *)NULL));
	    break;
	case 'c': /* set size of the capture buffer */
	    if(verbose)
		printf("Capture buffer size was set to %dk bytes.\n",cbsiz / K);
	    cbsiz = atoi(rest) * K;
	    if(verbose)
		printf("Changed to %dk bytes.\n", cbsiz / K);
	    break;
	case 'n': /* set capture file name */
	    if(verbose)
		printf("Capture buffer save file was '%s'.\n", captfile);
	    if(copyitem(captfile, rest))
		printf("Must have filename.\n");
	    if(verbose)
		printf("Name changed to '%s'.\n", captfile);
	    break;
	case 'p': /* set phone number file name */
	    if(verbose)
		printf("Phone number file was '%s'.\n", phonefile);
	    if(copyitem(phonefile, rest))
		printf("Must have filename.\n");
	    if(verbose)
		printf("Name changed to '%s'.\n", phonefile);
	    break;
	case 't': /* enter terminal mode */
term:
	    ioctl(0, TIOCSETP, &newmode);
	    terminal();
	    ioctl(0, TIOCSETP, &oldmode);
	    break;
	case 'r': /* receive XMODEM transmissions */
	    xreceive(i, rest);
	    break;
	case 's': /* send XMODEM transmissions */
	    xsend(i, rest);
	    break;
#ifdef PUT_TAKE
	case '%': /* do cu - style put and take */
	    puttake(i, rest);
	    break;
#endif
	case '!': /* exec a shell or shell command */
	    if(*rest == '!')
		rest = oldshell;
	    else
		strcpy(oldshell, rest);
	    if(verbose > 1)
		printf("Local xenix shell (/bin/sh).\n");
	    if(forkem() == 0){
		/* the child process: execute the shell */
		if(strlen(rest) > 0)
		    execl("/bin/sh", "sh", "-c", rest, 0); /* read from rest */
		else
		    execl("/bin/sh", "sh", "-i", 0);	   /* interactive */
		/* if we got here, it's an error */
		printf("XCOMM: exec failed\n");
		exit(-1);
		/*NOTREACHED*/
	    }
	    /* parent process: wait for any children. */
	    while(wait((int *) 0) >= 0)
		;
	    break;
	case '$': /* exec a data transfer program */
	    if(verbose)
		printf("Invoking data transfer program.\n");
	    if(strlen(rest) > 0){
		if(forkem() == 0){
		    /* the child process: attach the standard input and */
		    /* output to the modem port and execute the program */
		    mattach();
		    /* use the shell to parse the command string */
		    execl("/bin/sh", "sh", "-c", rest, 0);
		    /* if we got here, it's an error */
		    printf("XCOMM: exec failed.\n");
		    exit(-1);
		    /*NOTREACHED*/
		}
		/* parent process: wait for any children. */
		while(wait((int *) 0) >= 0)
		    ;
	    }
	    else
		printf("XCOMM: must have arguments for data transfer.\n");
	    break;
	case 'x': /* exit program */
	    if(verbose)
		printf("Exiting the XCOMM program.\n\n");
	    exit(0);
	    /*NOTREACHED*/
	case '?': /* print help and status messages */
	    prhelp();
	    status();
	    break;
	default: /* some other letter */
	    if(verbose)
		printf("Garbled command.\n");
	    break;
	}
    }
}

#ifdef PUT_TAKE
/* Put and take a file to/from a UNIX-type system. Unfortunately the stty
 * command is one of those commands that always gets changed with diffrent
 * UNIX systems; so you will get (at least) a file full of ^M on the take
 * command, for systems later than V7, or work-alikes. Also the take takes
 * a little too much.
 */
puttake(c, p)
register int c;
char *p;
{
    FILE *fp;
    char wrkbuff[WBSIZE];
    register int i;

    if(!*p){
	printf("Must give a filename with the put and take options.\n");
	return;
    }
    switch(mklow(c)){
    case 'p':
	if((fp = fopen(p, "r")) == NULL){
	    printf("Can't open file '%s' for read.\n", p);
	    break;
	}
	if(verbose)
	    printf("Putting file '%s' to UNIX system.\n", p);
	sprintf(wrkbuff, "stty -echo;cat >%s;stty echo\n", p);
	sendstr(wrkbuff);	/* send command string to remote shell */
	i = 0;
	while((c = getc(fp)) != EOF){
	    if(++i > 64){	/* this prevents an overload on the */
		i = 0;		/* receiver's input buffer (64=kludge) */
		sleep(DRIBBLE);
	    }
	    sendbyte(c);	/* send characters to cat command */
	}
	fclose(fp);
	sendbyte('D' - '@');	/* send a ^D to cat */
	purge();		/* get rid of whatever was sent back */
	break;
    case 't':
	if(!mungmode && !access(p, 0)){
	    printf("Can't over write existing file '%s'.\n", p);
	    break;
	}
	if((fp = fopen(p, "w")) == NULL){
	    printf("Can't open file '%s' for write.\n", p);
	    break;
	}
	if(verbose)
	    printf("Taking file '%s' from UNIX system.\n", p);
	sprintf(wrkbuff, "stty nl tabs; cat %s;stty -nl -tabs", p);
	sendstr(wrkbuff);	/* send command string to remote shell */
	purge();		/* get rid of the echo */
	sendbyte('\n');		/* start the command */
	while((c = readbyte(3)) != -1)	/* while more chars are being sent */
	    putc(c, fp);
	fclose(fp);
	break;
    default:
	printf("Type %%p or %%t only.\n");
    }
}
#endif PUT_TAKE

/* catch a signal (break key press) and jump to main
 */
catch()
{
    printf("XCOMM: Interrupt");
    signal(SIGINT, catch);
    signal(SIGQUIT, catch);
    longjmp(erret, 1);
}

/* print help message, with paging
 */
prhelp()
{
    register int i, c;

    for(i = 0; help[i] != NULL; i++){
	printf("%s\n", help[i]);
	if(*(help[i] + strlen(help[i]) - 1) == '\n'){
	    ioctl(0, TIOCSETP, &newmode);
	    printf("Press ENTER or RETURN to continue.> ");
	    while((c = getchar()) != '\n' && c != '\r')
		;
	    printf("\r                                    \r");
	    ioctl(0, TIOCSETP, &oldmode);
	}
    }
}

/* print all kinds of info about the state of the program
 */
status()
{
    putchar('\n');
    printf("Modem port is '%s'.\n", mport((char *)NULL));
    printf("Speed is %d baud.\n", mbaud((char *)NULL));
    printf("%s-bit communication mask enabled.\n",
					(bitmask == 0xFF) ? "Eight" : "Seven");
    printf("XMODEM %s protocol enabled.\n", crcheck ? "CRC check" : "checksum");
    if(badline)
	printf("Extra bad telephone line purging enabled.\n");
    printf("Capture buffer size is %d kilobytes.\n", cbsiz / K);
    printf("Capture save file is '%s'.\n", captfile);
    printf("Phone number file is '%s'.\n", phonefile);
    printf("Message status is %s.\n", vblist[verbose]);
    putchar('\n');
}

