/*
This program converts a ProComm directory to an ascii file and
vice-versa.  Invoke it: PRODIR [dirfile ascfile /x]
	where:	dirfile is the ProComm directory
		ascfile is the ASCII file
		x is either A for convert directory to ASCII or
			    D for convert ASCII to directory
*/

/*

Written January 10, 1987 by Scott Romanowski,
	using Computer Innovations C86 (tm) Compiler

    Donated to the public domain January 13, 1987

*/

/*********************************************************************/

#include <stdio.h>

#define isspace(x) ( (x)==' ' || (x)=='\t' || (x)=='\n' )


FILE *dir, *asc;
char dirname[50], ascname[50], action[10];

/*********************************************************************/
	
main( argc, argv ) int argc; char *argv[];
{

    switch( argc ){
	case 4:
	    if( strlen(argv[1]) > 50 )
		abort( "Directory file name too long (%s)\n", argv[1] );
	    if( strlen(argv[2]) > 50 )
		abort( "ASCII file name too long (%s)\n", argv[2] );
	    if( strlen(argv[3]) > 1 )
		abort( "Action (%s) too long\n", argv[3] );
	    strcpy( dirname, argv[1] ); upper( dirname );
	    strcpy( ascname, argv[2] ); upper( ascname );
	    strcpy( action, argv[3] ); upper( action );
	    if( action[0] != 'A' && action[0] != 'D' )
		abort( "Invalid action %c\n", action[0] );
	    break;
	case 1:
	    printf( "ProComm directory file: " );
	    gets( dirname, 50 ); upper( dirname );
	    printf( "ASCII file: " );
	    gets( ascname, 50 ); upper( ascname );
	    do {
		printf( "Make directory (D) or ASCII (A): " );
		gets( action, 10 ); upper( action );
		if( action[0] != 'A' && action[0] != 'D' )
		    printf( "Invalid action (%c)\n", action[0] );
	    } while( action[0] != 'A' && action[0] != 'D' );
	    break;
	default:
	    puts( "PRODIR converts ProComm directory files to ASCII files" );
	    puts( "and vice-versa.  Run PRODIR with the following syntax:" );
	    puts( "PRODIR [dirfile ascfile x]" );
	    puts( "Where:\t\tdirfile is the ProComm directory file" );
	    puts( "\t\tascfile is the ASCII file" );
	    puts( "\t    and x is either:" );
	    puts( "\t\t    D to make the directory file from the ASCII" );
	    puts( "\t\t    A to make the ASCII file from the directory" );
	    exit();
	    break;
    }

    if( action[0] == 'D' ){
	asc = fopen( ascname, "r" );
	if( asc == NULL )
	    abort( "Error opening ASCII file '%s'\n", ascname );
	dir = fopen( dirname, "wb" );
	if( dir == NULL )
	    abort( "Error opening directory file '%s'\n", dirname );
    }
     else
     {
	dir = fopen( dirname, "rb" );
	if( dir == NULL )
	    abort( "Error opening directory file '%s'\n", dirname );
	asc = fopen( ascname, "w" );
	if( asc == NULL )
	    abort( "Error opening ASCII file '%s'\n", ascname );
     }
    
    if( action[0] == 'D' ) asc_to_dir();
     else dir_to_asc();

    fclose( dir ); fclose( asc );
}

/*********************************************************************/

/* format of a ProComm directory file is:
    16 bytes for dialing command
    100 bytes for long distance codes
	25 bytes for each long distance code, in the order: -,+,@,#
    34 bytes of zeroes
    6000 bytes for 100 directory entries, in the format:
	name (25 bytes), zero-terminated
	number (15 bytes), zero-terminated
	baud (5 bytes), zero-terminated
	20h (ASCII space)
	parity (1 byte)
	data bits (1 byte)
	stop bits (1 byte)
	echo (1 byte)
	command file (10 bytes), zero-terminated
	TOTAL: 60 bytes
	
    all entries are in ASCII
*/
    
dir_to_asc()
{
    char temp[60];
    int i;

    printf( "Converting directory %s and writing %s\n", dirname, ascname );
    
    /* read modem command */
    if( fread( temp, 1, 16, dir ) != 16 )
	abort( "Error while reading ProComm directory!\n" );
    fprintf( asc, "Modem Command: %s\n", temp );
    
    /* read long distance codes */
    if( fread( temp, 1, 25, dir ) != 25 )
	abort( "Error while reading ProComm directory!\n" );
    fprintf( asc, "Long distance code -: %s\n", temp );

    if( fread( temp, 1, 25, dir ) != 25 )
	abort( "Error while reading ProComm directory!\n" );
    fprintf( asc, "Long distance code +: %s\n", temp );

    if( fread( temp, 1, 25, dir ) != 25 )
	abort( "Error while reading ProComm directory!\n" );
    fprintf( asc, "Long distance code @: %s\n", temp );

    if( fread( temp, 1, 25, dir ) != 25 )
	abort( "Error while reading ProComm directory!\n" );
    fprintf( asc, "Long distance code #: %s\n\n", temp );
	
    /* read the zeroes */
    if( fread( temp, 1, 34, dir ) != 34 )
	abort( "Error while reading ProComm directory!\n" );
    
    /* lable the columns of the directory entries */
    fprintf( asc, "%-25s %-15s %-4s %c %c %c %c %s\n",
	"     Name", "    Number", "Baud", 'P','D','S', 'E',
	" CMD File" );
    for( i=0; i<66; ++i ) fputc( '-', asc );
    fputc( '\n', asc );
    
    /* read the directory entries */
    for( i=0; i<100; ++i ){
	if( fread( temp, 1, 60, dir ) != 60 )
	    abort( "Error while reading ProComm directory!\n" );

	/* name number baud parity-data-stop echo command */
	fprintf( asc, "%-25s %15s %-4s %c %c %c %c  %s\n",
	    temp, temp+25, temp+40, temp[46], temp[47], temp[48],
	    temp[49], temp+50 );
    }
}

/*********************************************************************/

/* converts ASCII files make by dir_to_asc() to ProComm directory format */

#define CLR_TEMP for( k=0; k<100; ++k ) temp[k] = EOS;
#define CLR_BUFFER for( k=0; k<25; ++k ) buffer[k] = EOS;

asc_to_dir()
{
    char *temp, *c, *number, *baud, parity, data, stop, echo,
    	*command, *buffer;
    int i, j, k;
    
    printf( "Converting %s and writing directory %s\n", ascname, dirname );
    temp = calloc( 100, 1 );
    number = calloc( 16, 1 );
    baud = calloc( 5, 1 );
    buffer = calloc( 25, 1 );
    
    /* read the modem command */
    fgets( temp, 100, asc );
    c = strchr( temp, ':' );	/* find the : */
    if( c == NULL )
	abort( "Error while reading the modem command\n\t%s\n",temp );
    copy_to_end( buffer, c+1 );
    fwrite( buffer, 1, 16, dir ); CLR_TEMP; CLR_BUFFER;
    
    /* read the long distance codes */
    fgets( temp, 100, asc );
    c = strchr( temp, ':' );	/* find the : */
    if( c == NULL )
	abort( "Error while reading the long distance code\n\t%s\n",temp );
    copy_to_end( buffer, c+1 );
    fwrite( buffer, 1, 25, dir ); CLR_TEMP; CLR_BUFFER;

    fgets( temp, 100, asc );
    c = strchr( temp, ':' );	/* find the : */
    if( c == NULL )
	abort( "Error while reading the long distance code\n\t%s\n",temp );
    copy_to_end( buffer, c+1 );
    fwrite( buffer, 1, 25, dir ); CLR_TEMP; CLR_BUFFER;

    fgets( temp, 100, asc );
    c = strchr( temp, ':' );	/* find the : */
    if( c == NULL )
	abort( "Error while reading the long distance code\n\t%s\n",temp );
    copy_to_end( buffer, c+1 );
    fwrite( buffer, 1, 25, dir ); CLR_TEMP; CLR_BUFFER;

    fgets( temp, 100, asc );
    c = strchr( temp, ':' );	/* find the : */
    if( c == NULL )
	abort( "Error while reading the long distance code\n\t%s\n",temp );
    copy_to_end( buffer, c+1 );
    fwrite( buffer, 1, 25, dir ); CLR_TEMP; CLR_BUFFER;
    
    /* write the zeroes */
    for( i=0; i<34; ++i ) fputc( 0, dir );
    
    /* read up to the underline */
    fscanf( asc, "Name" );
    fgets( temp, 100, asc );

    /* read the underline */
    fgets( temp, 100, asc );
    
    /* read the directory entries */
    for( i=0; i<100; ++i ){
	CLR_TEMP;
	if( feof(asc) )
	    strcpy( temp,
	         "........................    . ... ...-.... 1200 N 8 1 N  " );
	 else fgets( temp, 100, asc );

	/* delete trailing white space */
	for( j = strlen( temp ) - 1; isspace( temp[j] ) && j >= 0; --j )
	    temp[ j ] = EOS;
	
	/* was it all spaces? */
	if( strlen( temp ) == 0 )
	    strcpy( temp,
	         "........................    . ... ...-.... 1200 N 8 1 N  " );
	
	/* find the start of the command file */
	command = temp + strlen( temp );
	for( ; !isspace( *command ) && command != temp; --command );

	if( command == temp )
	    abort( "Can't parse directory entry\n\t%s\n", temp );
	
	/* find echo */
	j = command - temp;	/* offset */
	while( isspace( temp[j] ) && j > 0 ) --j;
	if( j == 0 ) abort( "Can't parse directory entry\n\t%s\n", temp );

	/* is it a valid echo? */
	if( toupper( temp[j] ) != 'Y' && toupper( temp[j] ) != 'N' ){
	    if( temp[j] == '1' || temp[j]=='2' ){  /* no, it's stop bits */
		j++; sscanf( command, "%1s", &echo );
		command = NULL;
	    }
	    else abort( "Can't parse directory entry\n\t%s\n", temp );
	}
	else echo = temp[j];

	/* stop, data, & parity */
	for( --j; isspace( temp[j] ) && j>0; --j );
	if( j==0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	stop = temp[j];
	for( --j; isspace( temp[j] ) && j>0; --j );
	if( j==0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	data = temp[j];
	for( --j; isspace( temp[j] ) && j>0; --j );
	if( j==0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	parity = temp[j];

	/* baud */
	for( --j; isspace( temp[j] ) && j>0; --j );
	if( j==0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	for( --j; !isspace( temp[j] ) && j>0; --j );   /* go to start of baud */
	if( j == 0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	sscanf( temp+j, "%s", baud );
	
	/*force leading blank in baud */
	if( strlen( baud ) == 3 ){
	    for( k=5; k>0; --k ) baud[k] = baud[k-1];
	    baud[0] = ' ';
	}
	
	/* number */
	for( k=0; k<16; ++k ) number[k] = EOS;
	for( ; isspace( temp[j] ) && j>0; --j );
	if( j==0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	
	/* is entry assigned? */
	if( temp[j] == '.' ){
	    j -= 13; strncpy( number, temp+j, 14 );
	    number[15] = EOS; --j;
	}
	 else {
	    for(; !isspace( temp[j] ) && j>0; --j );	/* go to start of number */
	    if( j==0 ) abort( "Can't parse directory entry\n\t%s\n", temp );
	    sscanf( temp+j, "%s", number );
	 }
	
	/* name */
	/* remove trailing blanks */
	for( ; isspace( temp[j] ) && j>0; --j ) temp[j] = EOS;
	
	/* output to dir file */
	fwrite( temp, 1, 25, dir );	/* name */
	fwrite( number, 1, 15, dir );	/* number */
	fwrite( baud, 1, 5, dir );	/* baud */
	fputc( ' ', dir );
	fputc( toupper( parity ), dir );		/* parity */
	fputc( toupper( data ), dir );		/* data bits */
	fputc( stop, dir );		/* stop bits */
	fputc( toupper( echo ), dir );		/* echo on/off */
	copy_to_end( buffer, command );
	upper( buffer );
	fwrite( buffer, 1, 10, dir );	/* CMD file */
    }
}

/*********************************************************************/

/* copy characters up to EOS or \n.  Skip leading blanks */
copy_to_end( to, from ) char to[], *from;
{
    int j;
    
    /* skip over leading spaces */
    while( isspace( *from ) && *from ) ++from;
    
    /* copy */
    for( j=0; *from && *from != '\n'; ++j, ++from ) to[j] = *from;
    
    to[j] = EOS;
}
