/*
 * @(#)sggpio.h	1.1	 sggpio.h
 * Include file for GPIB based devices
 * This file includes:
 *	get/set device def structure gdef
 *	get/set gpib status structure gstat
 *	get/set gpib parameter structure gparam
 *
 * Note: the NOEXTRA flag turns off normally unused defines.
 */

#ifndef NOEXTRA

/*
 * structure for getting or setting the device definition
 */
 struct gdef {
	char		df_dev_num;	/* minor device number */
	char		df_gid;		/* gpib device driver  */
	char		df_priadrs;	/* primary gpib address */
	char		df_secadrs;	/* secondary gpib address */
	char		df_dev_stat;	/* device status */
};


/*
 * structure for getting system status
 */
struct systat {
	char		gpibadrs;	/* gpib address setting */
	char		state;		/* gpib interface state */
	char		busdev;		/* dev 'assigned' the gpib */
};

#endif

/*
 * device status bit definitions
 * these are also in devtab.h - definitions must be the same 
 */
#define	D_OPEN		0x01	/* device open */
#define	D_EXCL		0x02	/* exclusive use */
#define D_ASGN		0x04	/* private gpib use assigned */
#define	D_BUSY		0x20	/* device busy */
#define	D_WAIT		0x40	/* device waiting */
#define	D_DONE		0x80	/* done */



/*
 * structure for getting GPIB status
 */
 struct gstat {
	char		gs_srq_stat;	/* serial poll status */
	char		gs_gpib_stat;	/* gpib status flags */
};

/*
 * gpib status bit definitions
 * these are also in devtab.h - definitions must be the same 
 */
#define	G_NORET		0x20	/* no status returned */
#define	G_SRQNEW	0x40	/* new srq status */
#define G_SRQON		0x80	/* SRQ enable */



/*
 * structure for get and set gpib parameter calls
 */
struct gparam {
	int		hand_time;	/* handshake timeout value */
	int		poll_time;	/* poll timeout value */
	int		term_mode;	/* termination mode */
};
/*
 * Termination Modes
 */
#define	TERM_EOI	0
#define	TERM_LF		1
#define	TERM_CRLF	2
#define	TERM_NONE	3


/*
 * GPIB interface state
 * (values of GpibState)
 */

#define	GS_NCINC	0	/* not controller in charge	*/
#define GS_CINC		1	/* controller in charge		*/


/*
 * GPIO ioctl commands
 */
#define CMD_NOP		 0 	/* no operation GIOCNOP */
#define CMD_DEFG	 1 	/* get device defination GIOCDEFG */
#define CMD_DEFS	 2 	/* set device defination GIOCDEFS */

#define CMD_GETG	 3	/* get GPIB Parameters GIOCGETG */
#define CMD_SETG	 4	/* set GPIB Parameters GIOCSETG */

#define CMD_ASGN	 5	/* set Assigned GPIB define GIOCASGN */
#define CMD_NASGN	 6	/* clear Assigned GPIB GIOCNASGN */

#define CMD_GETP	 7	/* get Device parameters GIOCGETP */
#define CMD_SETP	 8	/* set Device parameters GIOCSETP */
#define CMD_SETN	 9	/* set Device param (now) GIOCSETN */
#define CMD_GETD	 10	/* get dev par defaults GIOCGETD */
#define CMD_SETD	 11	/* set dev par defaults GIOCSETD */

#define CMD_EXCL	 12	/* set exclusive open GIOCEXCL */
#define CMD_NXCL	 13	/* clear exclusive open GIOCNXCL */

#define CMD_DOP		 14	/* do device operation GIOCDOP */
#define CMD_DSTAT	 15	/* Get device Status GIOCDSTAT */
#define CMD_FLUSH	 16	/* Flush device Buffers GIOCFLUSH */


/*
 * Normal Command Codes
 */

#define	CMD_OPEN	0x41
#define	CMD_CLOSE	0x42
#define	CMD_READ	0x43
#define	CMD_WRITE	0x44
#define CMD_RESTART	0x45		/* restart command */
#define CMD_SETQ	0x46		/* set control queues */
#define CMD_ABORT	0x47		/* abort i/o command */


#ifndef NOEXTRA

/*
 * Boot/Diagnostic ROM Command Codes
 */

#define BR_MIN		32		/* Lowest num. command	*/
#define BR_NOP		BR_MIN+0	/* NOP			*/
#define BR_BOOT		BR_MIN+1	/* Boot			*/
#define	BR_READ		BR_MIN+2	/* Diag Read		*/
#define BR_WRITE	BR_MIN+3	/* Diag Write		*/
#define BR_WRRD		BR_MIN+4	/* Diag Write/Read	*/
#define BR_WIGGLE	BR_MIN+5	/* Diag GPIB		*/
#define BR_ERRINT	BR_MIN+6	/* Diag ERR/INT Test	*/
#define BR_OUTBYTE	BR_MIN+7	/* Diag Force out byte	*/
#define BR_DMABYTE	BR_MIN+8	/* Diag DMA out byte	*/
#define BR_SELFTEST	BR_MIN+9	/* Selftest		*/
#define BR_MAX		BR_SELFTEST	/* Highest num. command	*/

/*
 * 'device' byte bit definitions for Diagnostic Commands
 */

#define DEVBIT_WORD	0x01		/* 1 => Word operation	*/
#define DEVBIT_XCEIV	0x02		/* Enable GPIB Xceivers	*/


/*
 * GPIO ioctl commands
 */
#define GIOCNOP		(('g'<<8) |CMD_NOP) 	/* no operation */
#define GIOCDEFG	(('g'<<8) |CMD_DEFG) 	/* get device def */
#define GIOCDEFS	(('g'<<8) |CMD_DEFS) 	/* set device def */

#define GIOCGETG	(('g'<<8) |CMD_GETG)	/* get GPIB Param */
#define GIOCSETG	(('g'<<8) |CMD_SETG)	/* set GPIB Param */

#define GIOCASGN	(('g'<<8) |CMD_ASGN)	/* set Assigned GPIB */
#define GIOCNASGN	(('g'<<8) |CMD_NASGN)	/* clear Asgn GPIB */

#define GIOCGETP	(('g'<<8) |CMD_GETP)	/* get Device param */
#define GIOCSETP	(('g'<<8) |CMD_SETP)	/* set Device param */
#define GIOCSETN	(('g'<<8) |CMD_SETN)	/* set param (now) */
#define GIOCGETD	(('g'<<8) |CMD_GETD)	/* get dev defaults */
#define GIOCSETD	(('g'<<8) |CMD_SETD)	/* set dev defaults */

#define GIOCEXCL	(('g'<<8) |CMD_EXCL)	/* set exclusive open */
#define GIOCNXCL	(('g'<<8) |CMD_NXCL)	/* clear excl open */

#define GIOCDOP		(('g'<<8) |CMD_DOP)	/* do dev operation */
#define GIOCDSTAT	(('g'<<8) |CMD_DSTAT)	/* Get device status */
#define GIOCFLUSH	(('g'<<8) |CMD_FLUSH)	/* Flush dev Buffers */

#define GIOCLOAD	(('g'<<8) |BR_BOOT)	/* Load gpio software */
#define GIOCRESET	(('g'<<8) | CMD_RESTART) /* restart command */
#define GIOCSETQ	(('g'<<8) | CMD_SETQ)	/* set control queues */
#define GIOCABORT	(('g'<<8) | CMD_ABORT)	/* abort i/o command */



/*
 *	initialization constants
 *
 */

#define DEF_HSTIME	50		/* 5 second */
#define DEF_POLLTIME	50		/* 5 second */
#define DEF_TMODE	TERM_EOI

#define DEF_DSTAT	D_EXCL		/* default device status */
#define DEF_GSTAT	0		/* default gpib status */

#endif

#define FXFERTIME	100		/* fast loop xfer count */
