/***********************************************************************
*
* sim911.c - 911 VDT processing for the TI 990 Simulator.
*
* Changes:
*   11/06/03   DGP   Original.
*   11/17/03   DGP   Added escape sequence processing.
*
***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>

#if defined (_WIN32)
#define __TTYROUTINES 0
#include <conio.h>
#include <windows.h>
#include <signal.h>
#include <process.h>
#include <winsock.h>
#define IAC 255
#define WILL 251
#define TELOPT_SGA 3
#define TELOPT_ECHO 1
#endif

#if defined(UNIX)
#include <unistd.h>
#include <sys/time.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <arpa/telnet.h>
#include <pthread.h>
#endif

#include "simdef.h"

#define DEPTH 1
#define ESCAPE  0x1B

extern uint16 pcreg;	/* The program PC */
extern uint16 statreg;	/* The program status register */
extern uint16 wpreg;	/* The program Workspace Pointer */
extern uint16 lights;	/* The panel lights */

extern int run;
extern int idle;
extern int fault;
extern int devcnt;
extern int model;
extern int mapenabled;
extern int pancount;
extern char view[MAXVIEW][81];

extern Device devices[MAXDEVICES];

static struct sockaddr_in sinme;
static int ungetchar = -1;

/***********************************************************************
* Nwrite - Network write.
***********************************************************************/

static int
Nwrite (int fd,
	uint8 *buf,
	int count)
{
   int cnt;

   cnt = send (fd, buf, count, 0);
#ifdef DEBUG911TASK
   if (cnt < 0) PERROR ("Nwrite");
#endif
   return (cnt);
}

/***********************************************************************
* Nread - Network read.
***********************************************************************/

static int
Nread (int fd,
       uint8 *buf,
       int count)
{
   int cnt;

   if (ungetchar > 0)
   {
      cnt = ungetchar;
      ungetchar = -1;
   }
   else
   {
      cnt = recv (fd, buf, count, 0);
#ifdef DEBUG911TASK
      if (cnt < 0) PERROR ("Nread");
#endif
   }
   return (cnt);
}

/***********************************************************************
* v911doiac - VDT 911 IAC processor. We just eat them.
***********************************************************************/

static void
v911doiac (int fd)
{
   uint8 buf[2];

#ifdef DEBUG911TASK
   fprintf (stderr, "v911doiac: entered\n");
#endif
   if (Nread (fd, buf, 1) == 1)
   {
#ifdef DEBUG911TASK
      fprintf (stderr, "   c1 = %02X\n", buf[0]);
#endif
      if (Nread (fd, buf, 1) == 1)
      {
#ifdef DEBUG911TASK
	 fprintf (stderr, "   c2 = %02X\n", buf[0]);
#endif
      }
   }
}

/***********************************************************************
* v911escape - VDT 911 escape processor.
***********************************************************************/

static uint8
v911escape (int fd)
{
   uint8 retchar = 0;
   uint8 buf[2];

#ifdef DEBUG911ESCAPE
   fprintf (stderr, "v911escape: entered\n");
#endif
   if (Nread (fd, buf, 1) == 1)
   {
#ifdef DEBUG911ESCAPE
      fprintf (stderr, "   c1 = %02X(%c)\n", buf[0], buf[0]);
#endif
      switch (buf[0])
      {
      case 'O':
	 if (Nread (fd, buf, 1) == 1)
	 {
#ifdef DEBUG911ESCAPE
	    fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
	    switch (buf[0])
	    {
	    case 'P': /* F1 */
	       retchar = 0x92;
	       break;
	    case 'Q': /* F2 */
	       retchar = 0x93;
	       break;
	    case 'R': /* F3 */
	       retchar = 0x94;
	       break;
	    case 'S': /* F4 */
	       retchar = 0x95;
	       break;
	    default: ;
	    }
	 }
         break;
      case '[':
	 if (Nread (fd, buf, 1) == 1)
	 {
#ifdef DEBUG911ESCAPE
	    fprintf (stderr, "   c2 = %02X(%c)\n", buf[0], buf[0]);
#endif
	    switch (buf[0])
	    {
	    case 'A': /* Up arrow */
	       retchar = 0x89;
	       break;
	    case 'B': /* Down arrow */
	       retchar = 0x8B;
	       break;
	    case 'C': /* Right arrow */
	       retchar = 0x8A;
	       break;
	    case 'D': /* Left arrow */
	       retchar = 0x88;
	       break;
	    case '1':
	       if (Nread (fd, buf, 1) == 1)
	       {
#ifdef DEBUG911ESCAPE
		  fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	          switch (buf[0])
		  {
		  case '5': /* F5 */
		     retchar = 0x96;
		     break;
		  case '7': /* F6 */
		     retchar = 0x97;
		     break;
		  case '8': /* F7 */
		     retchar = 0x98;
		     break;
		  case '9': /* F8 */
		     retchar = 0x99;
		     break;
		  case '~': /* HOME */
		     retchar = 0x82;
		     break;
		  default: ;
		  }
		  while (buf[0] != '~')
		  {
		     Nread (fd, buf, 1); /* Suck up tilde */
#ifdef DEBUG911ESCAPE
		     fprintf (stderr, "   cx = %02X(%c)\n", buf[0], buf[0]);
#endif
		  }
	       }
	       break;
	    case '2':
	       if (Nread (fd, buf, 1) == 1)
	       {
#ifdef DEBUG911ESCAPE
		  fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	          switch (buf[0])
		  {
		  case '0': /* F9  - CMD */
		     retchar = 0x9B;
		     break;
		  case '1': /* F10 - GOLD */
		     retchar = 0x9C;
		     break;
		  case '3': /* F11 - ERASE FIELD */
		     retchar = 0x80;
		     break;
		  case '4': /* F12 - ERASE INPUT */
		     retchar = 0x81;
		     break;
		  case '~': /* INSERT */
		     retchar = 0x86;
		     break;
		  default: ;
		  }
		  while (buf[0] != '~')
		  {
		     Nread (fd, buf, 1); /* Suck up tilde */
#ifdef DEBUG911ESCAPE
		     fprintf (stderr, "   cx = %02X(%c)\n", buf[0], buf[0]);
#endif
		  }
	       }
	       break;
	    case '3':
	       if (Nread (fd, buf, 1) == 1)
	       {
#ifdef DEBUG911ESCAPE
		  fprintf (stderr, "   c3 = %02X(%c)\n", buf[0], buf[0]);
#endif
	          switch (buf[0])
		  {
		  case '~': /* DELETE */
		     retchar = 0x84;
		     break;
		  default: ;
		  }
		  while (buf[0] != '~')
		  {
		     Nread (fd, buf, 1); /* Suck up tilde */
#ifdef DEBUG911ESCAPE
		     fprintf (stderr, "   cx = %02X(%c)\n", buf[0], buf[0]);
#endif
		  }
	       }
	       break;
	    default: 
	       while (buf[0] != '~')
	       {
		  Nread (fd, buf, 1); /* Suck up tilde */
#ifdef DEBUG911ESCAPE
		  fprintf (stderr, "   cx = %02X(%c)\n", buf[0], buf[0]);
#endif
	       }
	    }
	 }
	 break;
      default: 
         ungetchar = buf[0];
	 return (ESCAPE);
      }
   }

#ifdef DEBUG911ESCAPE
   fprintf (stderr, "   retchar = %02X\n", retchar);
#endif
   return (retchar);
}

/***********************************************************************
* v911task - VDT 911 connection task
***********************************************************************/

static void
v911task (void *vdev)
{
   Device *dev;
   int sfd;
   int fd;
   int maxfd;
   int i;
   int fromlen;
   int row, col;
   struct timeval tv;
   struct sockaddr_in peer;
   struct sockaddr_in frominet;
   fd_set readfds;
   char msg[82];
   uint8 buf[2];
   uint8 will[] = {IAC, WILL, '%', 'c', 0};

   dev = (Device *)vdev;

   sfd = (int)dev->fd;

SERVER_LOOP:

#ifdef DEBUG911TASK
   fprintf (stderr, "v911task: starting: dev = %s, sfd = %d\n", dev->dev, sfd);
#endif

   sprintf (dev->file, "%d", dev->unit);

   /*
   ** Accept user connection
   */

   fromlen = sizeof(struct sockaddr_in);
   if ((fd = accept (sfd,
		     (struct sockaddr *)&frominet,
		     (void *)&fromlen)) < 0)
   {
      if (ERRNO == EAGAIN || ERRNO == EINTR)
      {
	 goto SERVER_LOOP;
      }
      PERROR ("v911task: accept");
      return;
   }
#ifdef DEBUG911TASK
   fprintf (stderr, "v911task: accepting: fd = %d\n", fd);
#endif

   i = sizeof(peer);

   if (getpeername (fd,
		    (struct sockaddr *)&peer,
		    (void *)&i) < 0)
   {
      PERROR ("v911task: getpeername");
      CLOSE (fd);
      goto SERVER_LOOP;
   }
   sprintf (dev->file, "%s:%d", inet_ntoa (peer.sin_addr), dev->unit);

#ifdef DEBUG911TASK
   fprintf (stderr, "v911task: from %s\n", dev->file);
#endif
   
   /*
   ** Tell the telnet client we're doing the work
   */

   sprintf (msg, will, TELOPT_SGA);
   Nwrite (fd, msg, 3);
   sprintf (msg, will, TELOPT_ECHO);
   Nwrite (fd, msg, 3);

   /*
   ** Display the introduction banner
   */

   sprintf (msg, "%c[2J", ESCAPE);
   Nwrite (fd, msg, strlen(msg));
   sprintf (msg, "%c[%d;%dH", ESCAPE, 1, 1);
   Nwrite (fd, msg, strlen(msg));
   sprintf (msg, "sim990 VDT911 %s on %s", dev->dev, dev->file);
   Nwrite (fd, msg, strlen(msg));
   dev->info.vdtinfo.cursorpos = 1840;


   /*
   ** Process connection
   */

   while (TRUE)
   {
      /*
      ** Display screen, if updated
      */

      if (dev->cbreak)
      {
	 dev->cbreak = FALSE;
#ifdef DEBUG911TASKaa
         fprintf (stderr, "Display screen\n");
#endif
	 sprintf (msg, "%c[2J", ESCAPE);
	 if (Nwrite (fd, msg, strlen(msg)) < 0) break;
	 sprintf (msg, "%c[%d;%dH", ESCAPE, 1, 1);
	 if (Nwrite (fd, msg, strlen(msg)) < 0) break;
	 for (i = 0; i < 24; i++)
	 {
	    if (Nwrite (fd, &dev->info.vdtinfo.screen[i*80], 80) < 0) break;
	 }
      }

      /*
      ** Position cursor on screen
      */

      row = (dev->info.vdtinfo.cursorpos / 80) + 1;
      col = (dev->info.vdtinfo.cursorpos % 80) + 1;
      sprintf (msg, "%c[%d;%dH", ESCAPE, row, col);
      if (Nwrite (fd, msg, strlen(msg)) < 0) break;

      /*
      ** Await user input or timeout
      */

      FD_ZERO (&readfds);
      FD_SET (fd, &readfds);

      tv.tv_sec = 0;
      tv.tv_usec = 1000;
      if ((maxfd = select (fd+1,
			   &readfds,
			   NULL,
			   NULL,
			   &tv)) < 0)
      {
	 if (ERRNO != EINTR && ERRNO != EAGAIN)
	 {
	    PERROR ("v911task: select failed");
	    break;
	 }
      }
#ifdef DEBUG911TASKaa
      fprintf (stderr, "Select: fd = %d, maxfd = %d\n", fd, maxfd);
#endif
      /*
      ** If input data, process it
      */

      if (maxfd > 0)
      {
         if (Nread (fd, buf, 1) == 1)
	 {
	    if (buf[0] == IAC)
	    {
	       v911doiac (fd);
	    }
	    else if (buf[0] == ESCAPE)
	    {
	       uint8 c;

	       if ((c = v911escape (fd)) != 0)
	       {
		  dev->inchar = c;
		  dev->info.vdtinfo.kbchar = TRUE;
		  if (dev->intenabled)
		     dev->info.vdtinfo.kbint = TRUE;
#ifdef DEBUG911TASK
		  fprintf (stderr, "Complete select: eschar = %02X\n", c);
#endif
	       }
	    }
	    else if (buf[0] > 0 && buf[0] < 0x7F)
	    {
	       dev->info.vdtinfo.kbchar = TRUE;
	       dev->inchar = buf[0];
#ifdef DEBUG911TASK
	       fprintf (stderr, "Complete select: inchar = %02X(%c)\n",
			buf[0] & 0xFF, isprint(buf[0]) ? buf[0] : '.');
#endif
	       if (dev->intenabled)
		  dev->info.vdtinfo.kbint = TRUE;
	    }
	 }
	 else break;
      }
   }

   CLOSE (fd);
   dev->cbreak = TRUE;
   goto SERVER_LOOP;

}

/***********************************************************************
* v911start - Start connection task
***********************************************************************/

void
v911start (Device *dev)
{
#if defined(UNIX)
   pthread_t thread_handle;
#endif

   memset (dev->info.vdtinfo.screen, ' ', V911SCREENSIZE);
   dev->info.vdtinfo.kbchar = FALSE;

#if defined(UNIX)
   if (pthread_create (&thread_handle,
		       NULL,
		       (void *(*)(void *)) v911task,
		       (void *)dev) < 0)
#endif
#if defined(_WIN32)
   if (_beginthread (v911task,
		     NULL,
		     (void *)dev) < 0)
#endif
   {
      sprintf (view[0], "v911start: thread create failed: %s",
	       strerror (ERRNO));
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
#endif
   }

}

/***********************************************************************
* v911open - Open a connection for the 911
***********************************************************************/

FILE *
v911open (char *bp, char *mode)
{
   int sfd;
   short port;
#if defined(_WIN32)
   WSADATA nt_data;
#endif /* _WIN32 */

#ifdef DEBUG911TASK
   fprintf (stderr, "v911open: bp = %s\n", bp);
#endif

   memset ((char *)&sinme,
           '\0',
	   sizeof(sinme));

#if defined(_WIN32)
   if (WSAStartup (MAKEWORD (1, 1), &nt_data) != 0)
   {
      sprintf (view[0], "v911open: WSAStartup failed: %s",
	       strerror (ERRNO));
      sprintf (view[1], "socket: %s", bp);
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
      fprintf (stderr, "%s\n", view[1]);
#endif
      return (NULL);
   }
#endif /* _WIN32 */

   port = atoi(bp);
   sinme.sin_port = htons (port);
   sinme.sin_family = AF_INET;

   if ((sfd = socket (AF_INET,
		     SOCK_STREAM,
		     0)) < 0)
   {
      sprintf (view[0], "v911open: socket failed: %s",
	       strerror (ERRNO));
      sprintf (view[1], "socket: %s", bp);
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
      fprintf (stderr, "%s\n", view[1]);
#endif
      return (NULL);
   }

   if (bind (sfd,
	     (struct sockaddr *)&sinme,
	     sizeof(sinme)) < 0)
   {
      sprintf (view[0], "v911open: bind failed: %s",
	       strerror (ERRNO));
      sprintf (view[1], "socket: %s", bp);
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
      fprintf (stderr, "%s\n", view[1]);
#endif
      return (NULL);
   }

   if (listen (sfd,
	       DEPTH) < 0)
   {
      sprintf (view[0], "v911open: listen failed: %s",
	       strerror (ERRNO));
      sprintf (view[1], "socket: %s", bp);
#ifdef DEBUG911TASK
      fprintf (stderr, "%s\n", view[0]);
      fprintf (stderr, "%s\n", view[1]);
#endif
      return (NULL);
   }

   return ((FILE *)sfd);
}

/***********************************************************************
* v911seto - Set CRU bit to one.
***********************************************************************/

void
v911seto (Device *dev, uint8 disp)
{
#ifdef DEBUG911CRU
   fprintf (stderr, "CRU %04X SBO %d\n", R12, disp);
#endif

   if (dev->cdevaddr == R12)
   {
      disp += 8;
   }

   if (dev->info.vdtinfo.wrdsel == 0)
   {
      switch (disp)
      {
      case 7: /* Hi/Lo intensity select */
         break;

      case 8: /* Write data strobe */
         dev->info.vdtinfo.screen[dev->info.vdtinfo.cursorpos] = dev->outchar;
	 dev->cbreak = TRUE;
         break;

      case 9: /* Test mode */
         break;

      case 10: /* Cursor Move */
         dev->info.vdtinfo.cursorpos--;
	 if (dev->info.vdtinfo.cursorpos < 0)
	    dev->info.vdtinfo.cursorpos = V911SCREENSIZE;
	 break;

      case 11: /* Blinking cursor enable */
         break;

      case 12: /* Keyboard interrupt enable */
	 dev->intenabled = TRUE;
	 if (dev->info.vdtinfo.kbchar)
	    geninterrupt (dev->intlvl);
	 break;

      case 13: /* Hi/Lo intensity enable */
	 break;

      case 14: /* Display Enable */
	 break;

      case 15: /* Word select */
	 dev->info.vdtinfo.oldsel = dev->info.vdtinfo.wrdsel;
	 dev->info.vdtinfo.wrdsel = 1;
	 break;

      default: ;
      }
   }
   else
   {
      switch (disp)
      {
      case 12: /* Display cursor */
         break;

      case 13: /* Keyboard ack */
         break;

      case 14: /* Beep enable strobe */
         break;

      case 15: /* Word select */
	 dev->info.vdtinfo.oldsel = dev->info.vdtinfo.wrdsel;
	 dev->info.vdtinfo.wrdsel = 1;
	 break;

      default: ;
      }
   }
}

/***********************************************************************
* v911setz - Set CRU bit to zero.
***********************************************************************/

void
v911setz (Device *dev, uint8 disp)
{

#ifdef DEBUG911CRU
   fprintf (stderr, "CRU %04X SBZ %d\n", R12, disp);
#endif

   if (dev->cdevaddr == R12)
   {
      disp += 8;
   }

   if (dev->info.vdtinfo.wrdsel == 0)
   {
      switch (disp)
      {
      case 7: /* Hi/Lo intensity select */
         break;

      case 8: /* Write data strobe */
         dev->info.vdtinfo.screen[dev->info.vdtinfo.cursorpos] = dev->outchar;
	 dev->cbreak = TRUE;
	 break;

      case 9: /* Test mode */
         break;

      case 10: /* Cursor Move */
         dev->info.vdtinfo.cursorpos++;
	 if (dev->info.vdtinfo.cursorpos > V911SCREENSIZE)
	    dev->info.vdtinfo.cursorpos = 0;
	 break;

      case 11: /* Blinking cursor enable */
         break;

      case 12: /* Keyboard interrupt enable */
	 dev->intenabled = FALSE;
	 break;

      case 13: /* Hi/Lo intensity enable */
	 break;

      case 14: /* Display Enable */
	 break;

      case 15: /* Word select */
	 dev->info.vdtinfo.oldsel = dev->info.vdtinfo.wrdsel;
	 dev->info.vdtinfo.wrdsel = 0;
	 break;

      default: ;
      }
   }
   else
   {
      switch (disp)
      {
      case 12: /* Display cursor */
         break;

      case 13: /* Keyboard ack */
         break;

      case 14: /* Beep enable strobe */
         break;

      case 15: /* Word select  */
	 dev->info.vdtinfo.oldsel = dev->info.vdtinfo.wrdsel;
	 dev->info.vdtinfo.wrdsel = 0;
	 break;

      default: ;
      }
   }
}

/***********************************************************************
* v911tb - Test CRU bit.
***********************************************************************/

void
v911tb (Device *dev, uint8 disp)
{

#ifdef DEBUG911CRU
   fprintf (stderr, "CRU %04X TB %d", R12, disp);
#endif

   CLR_EQ;
   if (dev->cdevaddr == R12)
   {
      disp += 8;
   }

   if (dev->info.vdtinfo.wrdsel == 0)
   {
      switch (disp)
      {
      case 7: /* Low intensity */
         break;

      case 15: /* Keyboard ready */
	 if (dev->info.vdtinfo.kbchar)
	 {
	    dev->info.vdtinfo.kbchar = FALSE;
	    SET_EQ;
#ifdef DEBUG911CRU
	    fprintf (stderr, " TRUE");
#endif
	    if (dev->intenabled) geninterrupt (dev->intlvl);
	 }
	 break;

      default: ;
      }
   }
   else
   {
      switch (disp)
      {
      case 10: /* Cursor MSB */
         if (dev->info.vdtinfo.cursorpos & 0x400) SET_EQ;
	 break;

      case 11: /* Keyboard MSB */
         if (dev->inchar & 0x80)
	 {
	    SET_EQ;
#ifdef DEBUG911CRU
	    fprintf (stderr, " TRUE");
#endif
	 }
	 break;

      case 12: /* Terminal ready */
	 break;

      case 13: /* Previous state */
	 if (dev->info.vdtinfo.oldsel == 1)
	 {
	    SET_EQ;
#ifdef DEBUG911CRU
	    fprintf (stderr, " TRUE");
#endif
	 }
         break;

      case 14: /* Keyboard parity error */
	 break;

      case 15: /* Keyboard ready */
	 if (dev->info.vdtinfo.kbchar)
	 {
	    dev->info.vdtinfo.kbchar = FALSE;
	    SET_EQ;
#ifdef DEBUG911CRU
	    fprintf (stderr, " TRUE");
#endif
	 }
	 break;

      default: ;
      }
   }
#ifdef DEBUG911CRU
   fprintf (stderr, "\n");
#endif
}

/***********************************************************************
* v911ldcr - Load CRU.
***********************************************************************/

void
v911ldcr (Device *dev, uint16 cnt, uint16 dat)
{

#ifdef DEBUG911CRU
   fprintf (stderr, "CRU %04X LDCR CNT %d DAT %02X", R12, cnt, dat);
#endif

   if (dev->info.vdtinfo.wrdsel == 0)
   {
      dat &= 0x7F;
      if (dat < ' ' || dat > 0x7E) dat = ' ';
#ifdef DEBUG911CRU
      if (isprint (dat))
	 fprintf (stderr, "(%c)", dat);
#endif
      dev->outchar = dat;
   }
   else
   {
      if (dat >= V911SCREENSIZE) dat = V911SCREENSIZE - 1;
      dev->info.vdtinfo.cursorpos = dat;
   }
#ifdef DEBUG911CRU
   fprintf (stderr, "\n");
#endif
}

/***********************************************************************
* v911stcr - Store CRU.
***********************************************************************/

uint16
v911stcr (Device *dev, uint16 cnt)
{
   uint16 dat = 0;

   if (dev->info.vdtinfo.wrdsel == 0)
   {
      int mask;

      if (cnt == 7) mask = 0x7F;
      else mask = 0xFF;

      if (R12 == dev->devaddr)
      {
	 if (cnt > 8)
	    dat = (dev->inchar << 8)
		  | dev->info.vdtinfo.screen[dev->info.vdtinfo.cursorpos];
	 else
	    dat = dev->info.vdtinfo.screen[dev->info.vdtinfo.cursorpos];
      }
      else
	 dat = dev->inchar & mask;

#ifdef DEBUG911CRU
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %02X", R12, cnt, dat);
      if (isprint (dat))
	 fprintf (stderr, "(%c)", dat);
      fprintf (stderr, "\n");
#endif
   }
   else
   {
      dat = dev->info.vdtinfo.cursorpos;
#ifdef DEBUG911CRU
      fprintf (stderr, "CRU %04X STCR CNT %d DAT %02X\n", R12, cnt, dat);
#endif
   }

   return (dat);
}
