/************************************************************************
*
* lnkpunch - Punchs out memory.
*
* Changes:
*   06/05/03   DGP   Original.
*   06/23/03   DGP   Put date/time and creator on EOF record.
*   06/25/03   DGP   Added punchsymbols for partial link.
*   04/08/04   DGP   Added "Cassette" mode.
*   04/09/04   DGP   Added object checksum.
*   05/25/04   DGP   Added long ref/def support.
*
************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

#include "lnkdef.h"

extern FILE *lstfd;
extern int listmode;
extern int pc;
extern int errcount;
extern int absentry;
extern int cassette;
extern int relentry;
extern int symbolcount;
extern int cassette;
extern uint8 memory[MEMSIZE];
extern Memory memctl[MEMSIZE];
extern char idtbuf[IDTSIZE+2];
extern SymNode *symbols[MAXSYMBOLS];
extern struct tm *timeblk;

static int objcnt = 0;
static int objrecnum = 0;
static char objbuf[80];
static char objrec[82];

/***********************************************************************
* punchfinish - Punch a record with sequence numbers.
***********************************************************************/

static void
punchfinish (FILE *outfd)
{
   if (objcnt)
   {
      int i;
      short int cksum;
      char temp[10];

      cksum = 0;
      for (i = 0; i < objcnt; i++) cksum += objrec[i];
      cksum += CKSUM_TAG;
      cksum = -cksum;
      sprintf (temp, "%c%04X%c", CKSUM_TAG, cksum & 0xFFFF, EOR_TAG);
      strncpy (&objrec[objcnt], temp, 6);
      objcnt += 6;

      if (cassette)
      {
	 strcpy (&objrec[objcnt], "\n");
      }
      else
      {
	 sprintf (&objrec[SEQUENCENUM], SEQFORMAT, ++objrecnum);
      }
      fputs (objrec, outfd);
      memset (objrec, ' ', sizeof(objrec));
      objcnt = 0;
   }
}

/***********************************************************************
* punchrecord - Punch an object value into record.
***********************************************************************/

static void 
punchrecord (FILE *outfd)
{
   int len = strlen (objbuf);

   if (objcnt+len >= (cassette ? CHARSPERCASREC : CHARSPERREC))
   {
      punchfinish (outfd);
   }
   strncpy (&objrec[objcnt], objbuf, len);
   objcnt += len;
   objbuf[0] = '\0';
}

/***********************************************************************
* puncheof - Punch EOF mark.
***********************************************************************/

static void
puncheof (FILE *outfd)
{
   char temp[80];

   punchfinish (outfd);
   objrec[0] = EOFSYM;
   if (cassette)
   {
      strcpy (&objrec[1], "\n");
   }
   else
   {
      sprintf (temp, "%-8.8s  %02d/%02d/%02d  %02d:%02d:%02d    LNK990 %s",
	       idtbuf,
	       timeblk->tm_mon+1, timeblk->tm_mday, timeblk->tm_year - 100,
	       timeblk->tm_hour, timeblk->tm_min, timeblk->tm_sec,
	       VERSION);
      strncpy (&objrec[7], temp, strlen(temp));
      sprintf (&objrec[SEQUENCENUM], SEQFORMAT, ++objrecnum);
   }
   fputs (objrec, outfd);
}

/***********************************************************************
* punchsymbols - Punch REF and DEF symbols.
***********************************************************************/

static void
punchsymbols (FILE *outfd)
{
   int i;

   for (i = 0; i < symbolcount; i++)
   {
      if (symbols[i]->external)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LREFFORMAT,
		  symbols[i]->relocatable ? LRELEXTRN_TAG : LABSEXTRN_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	    sprintf (objbuf, REFFORMAT,
		  symbols[i]->relocatable ? RELEXTRN_TAG : ABSEXTRN_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
         
         punchrecord (outfd);
      }
      else if (symbols[i]->global)
      {
	 if (symbols[i]->longsym)
	    sprintf (objbuf, LDEFFORMAT,
		  symbols[i]->relocatable ? LRELGLOBAL_TAG : LABSGLOBAL_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
	 else
	    sprintf (objbuf, DEFFORMAT,
		  symbols[i]->relocatable ? RELGLOBAL_TAG : ABSGLOBAL_TAG,
		  symbols[i]->value & 0xFFFF, symbols[i]->symbol);
         punchrecord (outfd);
      }
   }
}

/***********************************************************************
* punchmemory - Punch out memory.
***********************************************************************/

int
lnkpunch (FILE *outfd, int partiallink)
{
   int i;
   char temp[80];

   memset (objrec, ' ', sizeof(objrec));
   sprintf (objbuf, IDTFORMAT, IDT_TAG, pc, idtbuf);
   punchrecord (outfd);
   sprintf (objbuf, OBJFORMAT, RELORG_TAG, 0);
   punchrecord (outfd);

   for (i = 0; i < pc; )
   {
      if (memctl[i].tag)
      {
         sprintf (objbuf, OBJFORMAT, memctl[i].tag, GETMEM(i));
         punchrecord (outfd);
	 if (memctl[i].tag == RELORG_TAG) i = GETMEM(i);
	 else i += 2;
      }
      else i++;
   }

   if (relentry >= 0)
   {
      sprintf (objbuf, OBJFORMAT, RELENTRY_TAG, relentry);
      punchrecord (outfd);
   }

   if (partiallink) punchsymbols (outfd);

   puncheof (outfd);

   return (0);
}

