/***********************************************************************
*
* fpydir - Directory listing for the TI 990 FD800 disks.
*
* Changes:
*   06/21/03   DGP   Original.
*   07/16/03   DGP   Sorted the dir entries.
*	
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#define FALSE 0
#define TRUE  1

/*
** Disk geometry stuff
*/

#define MAXAUS 333	/* Max Allocation units (AU) on disk */
#define SECPERAU 6	/* Sectors per AU */

#define SECPERTRK 26	/* Sectors per track */
#define SECLEN 128	/* Sector length */

/*
** Location of things
*/

#define DISKLABELAU 4	/* Disk label AU */
#define DISKDIRAU 5	/* Disk directory AU start */

#define DISKLABEL 0	/* Offset to disk label */
#define DISKVOLNAM 38	/* Offset to disk volume name */

#define DIRENTSIZE 16	/* Size of a directory entry */
#define DIRENTPSEC 8	/* Directory entries per sector */
#define DIRFILE 0	/* Offset to a file name */
#define DIRFILELEN 7	/* Length of the file name */
#define DIREXT 7	/* Offset to a file name extension */
#define DIREXTLEN 3	/* Length of the file name extension */
#define DIRFCB 10	/* Offset to the FCB location */
#define DIRFILEPROT 12	/* Offset to file protection */

#define FCBEXTENT 20	/* Offset to FCB file extents */

static int longlist = FALSE;
static int extentlist = FALSE;

typedef struct
{
   char filename[DIRFILELEN+DIREXTLEN+2];
   char fileprot;
   int  fcbaddr;
} dirent_t;

/*
** Skew (interleaving) table
*/

static int skew[26] = {
    4,10,16,22, 2, 8,
   14,20, 0, 6,12,18,
   24, 5,11,17,23, 3,
    9,15,21, 1, 7,13,
   19,25
};

static unsigned char disklabel[SECLEN];
static unsigned char dirblock[SECLEN];
static unsigned char fcbblock[SECLEN];

static dirent_t *dirent[SECPERAU*DIRENTPSEC];

#define HEXDUMP(file,ptr,size,offset)	\
{\
   int jjj;\
   int iii;\
   char *tp;\
   char *cp;\
   for (iii = 0, tp = (char *)(ptr), cp = (char *)(ptr); iii < (size); )\
   {\
      fprintf ((file), "%04X  ", iii+offset);\
      for (jjj = 0; jjj < 8; jjj++)\
      {\
	 if (cp < ((char *)(ptr)+(size)))\
	 {\
	    fprintf ((file), "%02.2X", *cp++ & 0xFF);\
	    if (cp < ((char *)(ptr)+(size)))\
	    {\
	       fprintf ((file), "%02.2X ", *cp++ & 0xFF);\
	    }\
	    else\
	    {\
	       fprintf ((file), "   ");\
	    }\
	 }\
	 else\
	 {\
	    fprintf ((file), "     ");\
	 }\
	 iii += 2;\
      }\
      fprintf ((file), "   ");\
      for (jjj = 0; jjj < 8; jjj++)\
      {\
	 if (tp < ((char *)(ptr)+(size)))\
	 {\
	    if (isprint(*tp))\
	       fprintf ((file), "%c", *tp);\
	    else\
	       fprintf ((file), ".");\
	    tp++;\
	    if (tp < ((char *)(ptr)+(size)))\
	    {\
	       if (isprint(*tp))\
		  fprintf ((file), "%c ", *tp);\
	       else\
		  fprintf ((file), ". ");\
	       tp++;\
	    }\
	    else\
	    {\
	       fprintf ((file), "  ");\
	    }\
	 }\
	 else\
	 {\
	    fprintf ((file), "   ");\
	 }\
      }\
      fprintf ((file), "\n");\
   }\
}

/***********************************************************************
* cvtau - Convert AU address into Track/sector
***********************************************************************/

static int
cvtau (int au, int aus, int *ntrk, int *nsec)
{
   int trk;
   int sec;

   if (au < MAXAUS && aus < SECPERAU)
   {
      trk = (au * SECPERAU + aus) / SECPERTRK;
      sec = (au * SECPERAU + aus) % SECPERTRK;
      sec = skew[sec] + 1;
      *ntrk = trk;
      *nsec = sec;
#ifdef DEBUG
      printf ("trk = %d, sec = %d\n", trk, sec);
#endif
      return (0);
   }
   return (-1);
}

/***********************************************************************
* readdisk - Read the disk
***********************************************************************/

static int
readdisk (FILE *diskfd, int au, int aus, char *buffer)
{
   int diskloc;
   int trk, sec;

#ifdef DEBUG
   printf ("readdisk: au = %d, aus = %d\n", au, aus);
#endif
   cvtau (au, aus, &trk, &sec);
   diskloc = ((SECLEN * SECPERTRK) * trk) + (SECLEN * (sec-1));
   if (fseek (diskfd, diskloc, SEEK_SET) < 0)
   {
      perror("Disk seek error");
      return (-1);
   }

   if (SECLEN != fread (buffer, 1, SECLEN, diskfd))
   {
      perror ("Disk read error");
      return (-1);
   }
#ifdef DEBUG
   printf ("Buffer:\n");
   HEXDUMP (stdout, buffer, SECLEN, 0);
#endif

   return (SECLEN);

}

/***********************************************************************
* Main procedure
***********************************************************************/

int
main (int argc, char **argv)
{
   FILE *diskfd;
   char *disk = NULL;
   char *bp;
   int dircnt = 0;
   int maxlen = 0;
   int i, j, k, l;
   char temp[256];

   /*
   ** Process args
   */

   for (i = 1; i < argc; i++)
   {
      bp = argv[i];

      if (*bp == '-')
      {
	 for (bp++; *bp; bp++) switch (*bp)
	 {
	 case 'e':
	    longlist = TRUE;
	    extentlist = TRUE;
	    break;

	 case 'l':
	    longlist = TRUE;
	    break;

	 default:
      USAGE:
	    printf ("usage: fpydir [-l][-e] ti990.fpy\n");
	    exit (1);
	 }
      }
      else
      {
         if (disk != NULL) goto USAGE;
	 disk = argv[i];
      }
   }
   if (disk == NULL) goto USAGE;
   
   /*
   ** Open the disk image.
   */

   if ((diskfd = fopen (disk, "r")) == NULL)
   {
      sprintf (temp, "fpydir: Can't open disk image: %s", disk);
      perror (temp);
      return (1);
   }

   /*
   ** Read the disk label.
   */

   readdisk (diskfd, DISKLABELAU, 0, disklabel);
   if (longlist)
      printf ("DISK ID: %-32.32s     VOLNAME: %-4.4s\n\n",
	       &disklabel[DISKLABEL], &disklabel[DISKVOLNAM]);

   /*
   ** Read the directory, the size of the directory is 1 AU.
   */

   for (i = 0; i < SECPERAU; i++)
   {
      /*
      ** Read a directory sector.
      */

      readdisk (diskfd, DISKDIRAU, i, dirblock);
      for (j = 0; j < DIRENTPSEC; j++)
      {
	 /*
	 ** A ".." in the filename is an empty slot.
	 */

	 if (strncmp (&dirblock[j*DIRENTSIZE+DIRFILE], "..", 2))
	 {
	    dirent_t *dir;
	    char file[DIRFILELEN+1];
	    char ext[DIREXTLEN+1];

	    /*
	    ** Get filename component
	    */

	    strncpy (file, &dirblock[j*DIRENTSIZE+DIRFILE], DIRFILELEN);
	    file[DIRFILELEN] = '\0';
	    for (l = 0; l < DIRFILELEN; l++) if (file[l] == ' ') file[l] = '\0';

	    /*
	    ** Get extension component
	    */

	    strncpy (ext, &dirblock[j*DIRENTSIZE+DIREXT], DIREXTLEN);
	    ext[DIREXTLEN] = '\0';
	    for (l = 0; l < DIREXTLEN; l++) if (ext[l] == ' ') ext[l] = '\0';

	    if ((dir = (dirent_t *)malloc (sizeof (dirent_t))) == NULL)
	    {
	       fprintf (stderr, "Unable to allocate memeory\n");
	       exit (1);
	    }

	    sprintf (dir->filename, "%s.%s", file, ext);

	    /*
	    ** Get the address of the FCB
	    */

	    dir->fcbaddr = (dirblock[j*DIRENTSIZE+DIRFCB] << 8) |
			    dirblock[j*DIRENTSIZE+DIRFCB+1];
	    /*
	    ** Get the file protection
	    */

	    dir->fileprot = dirblock[j*DIRENTSIZE+DIRFILEPROT];

	    /*
	    ** Insert pointer in sort order.
	    */

	    if (dircnt == 0)
	    {
	       dirent[dircnt] = dir;
	       dircnt++;
	    }
	    else
	    { 
	       for (k = 0; k < dircnt; k++)
	       {
		  if (strcmp (dirent[k]->filename, dir->filename) > 0)
		  {
		     for (l = dircnt + 1; l > k; l--)
		     {
			dirent[l] = dirent[l-1];
		     }
		     dirent[k] = dir;
		     dircnt++;
		     break;
		  }
	       }
	       if (k == dircnt)
	       {
		  dirent[dircnt] = dir;
		  dircnt++;
	       }
	    }
         }
      }
   }

   /*
   ** Print the entries
   */

   k = 0;
   for (j = 0; j < dircnt; j++)
   {
      dirent_t *dir;

      dir = dirent[j];

      printf ("%-11.11s  ", dir->filename);
      if (longlist)
      {
	 int fcbext;
	 int totalaus;
	 int start, numaus;

	 /*
	 ** Print the file protection
	 */

	 printf ("  %c  ", dir->fileprot);

#ifdef DEBUG
	 printf (" read fcb: fcbaddr = %d\n", dir->fcbaddr);
#endif
	 /*
	 ** Read the FCB
	 */

	 readdisk (diskfd, dir->fcbaddr, 0, fcbblock);

	 /*
	 ** Go through extents to get total size of file.
	 */

	 fcbext = FCBEXTENT;
	 totalaus = 0;
	 while (TRUE)
	 { 
	    start = (fcbblock[fcbext] << 8) | fcbblock[fcbext+1];
	    if (start == 0xFFFF) break;
	    numaus = (fcbblock[fcbext+2] << 8) | fcbblock[fcbext+3];
	    fcbext += 4;
	    totalaus += numaus;
	 }
	 printf (" %3d  ", totalaus);

	 /*
	 ** If extentlist, print extents, AU:SIZE pairs.
	 */

	 if (extentlist)
	 {
	    fcbext = FCBEXTENT;
	    while (TRUE)
	    { 
	       start = (fcbblock[fcbext] << 8) | fcbblock[fcbext+1];
	       if (start == 0xFFFF) break;
	       numaus = (fcbblock[fcbext+2] << 8) | fcbblock[fcbext+3];
	       fcbext += 4;
	       printf (" %d:%d", start, numaus);
	    }
	 }
	 k = 7;
      }
      else
	 k++;
      if (k >= 6)
      {
	 k = 0;
	 printf ("\n");
      }
   }
   printf ("\n");
   return (0);

}
