/***********************************************************************
*
* binloader.c - Binary loader for the TI 990 simulator.
*
* Changes:
*   05/29/03   DGP   Original.
*   07/02/03   DGP   Added compressed binary support.
*   12/30/03   DGP   Moved loader code to loadrec function.
*   04/12/04   DGP   Changed to allow non-80 byte records.
*   04/14/04   DGP   Ignore extended object tags; Add checksum support.
*
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>

#include "simdef.h"

extern uint16 pcreg;
extern char view[MAXVIEW][81];
extern int run;
extern uint8 memory[MEMSIZE];

int
loadrec (FILE *lfd, char *inbuf, int loadaddr, int *curraddr)
{
   static int binarymode = FALSE;
   static int wordtaglen = WORDTAGLEN;
   unsigned char *op = inbuf;
   int i, j;
   int16 cksum;

   if (*op == EOFSYM)
   {
      binarymode = FALSE;
      wordtaglen = WORDTAGLEN;
      return (EOF);
   }

   cksum = 0;
   for (i = 0; i < 80; i++)
   {
      int tmp;
      uint16 wdata;
      unsigned char otag;
      unsigned char item[16];

      otag = *op++;
      if (otag == EOR_TAG)
      {
	 return (0);
      }
      if (binarymode)
      {
	 wdata = (*op << 8) | *(op+1);
      }
      else
      {
	 cksum += otag;
	 strncpy (item, op, 4);
	 item[4] = '\0';
	 if (otag != CKSUM_TAG)
	    for (j = 0; j < 4; j++) cksum += item[j];
	 sscanf (item, "%4X", &tmp);
	 wdata = tmp & 0xFFFF;
      }
      wdata &= 0xFFFF;
#ifdef DEBUGLOADER
      if (lfd != NULL)
      {
	 fprintf (lfd, "loadaddr = %04X, curraddr = %04X, otag = %c\n",
		  loadaddr, *curraddr,
		  otag == 0x01 ? '0' : otag);
	 fprintf (lfd, "data = %04X\n", wdata & 0xFFFF);
      }
#endif

      switch (otag)
      {
      case BINIDT_TAG: /* Binary IDT */
	 binarymode = TRUE;
	 wordtaglen = BINWORDTAGLEN;
	 wdata = (*op << 8) | *(op+1);
#ifdef DEBUGLOADER
	 if (lfd != NULL)
	    fprintf (lfd, "Binary mode: wdata = %04x\n", wdata);
#endif
      case IDT_TAG:
	 op += wordtaglen-1;
	 if (!binarymode)
	    for (j = 0; j < IDTLEN; j++) cksum += *op++;
	 break;
      case RELORG_TAG:
	 wdata += loadaddr;
      case ABSORG_TAG:
	 *curraddr = wdata;
	 op += wordtaglen-1;
	 break;
      case RELDATA_TAG:
	 wdata += loadaddr;
      case ABSDATA_TAG:
	 PUTMEM0 (*curraddr, wdata);
	 *curraddr += 2;
	 op += wordtaglen-1;
	 break;
      case RELENTRY_TAG:
	 wdata += loadaddr;
      case ABSENTRY_TAG:
	 pcreg = wdata;
	 run = TRUE;
	 op += wordtaglen-1;
	 break;
      case CKSUM_TAG:
#ifdef DEBUGLOADER
         if (lfd != NULL)
	    fprintf (lfd, "cksum = %04X\n", cksum);
#endif
         cksum += (int16)wdata;
#ifdef DEBUGLOADER
         if (lfd != NULL)
	    fprintf (lfd, "cksum+wdata = %04X\n", cksum);
#endif
	 if (cksum != 0) return (-20);
	 op += wordtaglen-1;
         break;
      case RELEXTRN_TAG:
      case ABSEXTRN_TAG:
      case ABSGLOBAL_TAG:
      case RELGLOBAL_TAG:
      case RELSYMBOL_TAG:
      case ABSSYMBOL_TAG:
      case LOAD_TAG:
      case RELSREF_TAG:
      case ABSSREF_TAG:
	 op += wordtaglen-1;
	 if (!binarymode)
	    for (j = 0; j < SYMLEN; j++) cksum += *op++;
	 break;
      default: ;
	 op += wordtaglen-1;
      }
   }
   return (0);
}

int
binloader (char *file, int loadpt)
{
   FILE *fd;
   FILE *lfd = NULL;
#ifdef DUMPLOADREC
   int reclen;
#endif
   int status;
   int loadaddr = LOADADDRESS;
   int curraddr = LOADADDRESS;
   unsigned char inbuf[82];

   while (isspace (*file)) file++;
#ifdef DEBUGLOADER
   lfd = fopen ("fileload.log", "w");
   fprintf (lfd, "binloader: file = '%s', loadpd = %d(%04X)\n",
	    file, loadpt, loadpt);
#endif

   if (loadpt > 0)
   {
      loadaddr = loadpt;
      curraddr = loadpt;
   }

   if ((fd = fopen (file, "r")) == NULL)
   {
      sprintf (view[0], "binloader: open failed: %s",
	       strerror (ERRNO));
      sprintf (view[1], "filename: %s", file);
      run = FALSE;
#ifdef DEBUGLOADER
      fprintf (lfd, "error: %s\n", view[0]);
      fprintf (lfd, "       %s\n", view[1]);
      fclose (lfd);
#endif
      return (-1);
   }

   while (fgets (inbuf, 82, fd) != NULL)
   {
#ifdef DUMPLOADREC
      reclen = strlen (inbuf);
      fprintf (lfd, "Record:\n");
      HEXDUMP (lfd, inbuf, reclen, 0);
#endif
      status = loadrec (lfd, inbuf, loadaddr, &curraddr);
      if (status == EOF) break;
      if (status == -20)
      {
	 sprintf (view[0], "binloader: Checksum error");
	 sprintf (view[1], "filename: %s", file);
	 run = FALSE;
#ifdef DEBUGLOADER
	 fprintf (lfd, "error: %s\n", view[0]);
	 fprintf (lfd, "       %s\n", view[1]);
	 fclose (lfd);
#endif
	 fclose (fd);
	 return (-1);
      }
   }

#ifdef DUMPPROG
   HEXDUMP (stderr, &memory[loadaddr], curraddr - loadaddr, loadaddr);
   run = FALSE;
#endif

#ifdef DEBUGLOADER
   fclose (lfd);
#endif
   fclose (fd);

#ifdef DUMPROM
   {
      int i;
      int lp;
      int val;

      if ((fd = fopen ("ti990.rom", "wb")) == NULL)
      {
	 fprintf (stderr, "binloader: ROM dump open failed: %s",
	       strerror (ERRNO));
	 return (-1);
      }

      lp = 0xFC00;

      for (i = 0; i < 1024; i+=2)
      {
	 val = GETMEM0 (lp);
	 fputc ((val >> 8) & 0xFF, fd);
	 fputc (val & 0xFF, fd);
	 lp += 2;
      }
      fclose (fd);
   }
#endif

   return (0);
}
