module ReplayUtilities;

{*****************************************}
{
{        Pepper:  Spice Interim Editor
{        Transcript Replay Utilities
{        Richard Cohn
{        September 26, 1981
{
{*****************************************}


exports

imports RegionEditor from RegionEdit;
imports Perq_String from Perq_String;

  procedure ExitTranscript;
  function GetTrnByte:  char;
  function GetTrnWord:  integer;
  procedure GetTrnKey;
  procedure SendTrnByte (Byte:  Char);
  procedure SendTrnWord (Word: Integer);
  procedure SendKey (Key:  KeyEvent);
  procedure FlushTranscript;
  procedure CheckReplay (Where: ReplayMode);
  function InitReplay (DoReplay:  boolean):  boolean;
  function FindPos (Page, Ch:  integer):  Position;


{********************************} private {********************************}

imports IO_Unit from IO_Unit;
imports ScreenUtilities from EdScreen;
imports TextUtilities from EdText;

type
    TwoByte  = packed array [0..1] of char;
    SmallInt = 0..255;

var
    InSrch:     boolean;
    InRepl:     boolean;
    TId:        FileId;         { ID of transcript }
    TBlock:     integer;        { transcript block currently being read }
    TByte:      integer;        { transcript byte currently being read }
    TBlocks:    integer;        { no. of blocks in transcript file }
    TBytes:     integer;        { no. of bytes in transcript file's last block}
    TBuffer:    pDirBlk;


{****************************************************************}

procedure ExitTranscript;

var Ch:  char;
    gotChar:  integer;

begin { ExitTranscript }
Replay := NotReplaying;
Prompt ('*** Replay finished, suggest Quit-Update, <space> to continue ***');
repeat gotChar := IOCRead (KeyBoard,Ch)
until Ch = ' ';
CanvDflt := CanvKeybd;
IOSetModeTablet (RelTablet);
CmdPrompt := InsModePrompt;
Prompt (CmdPrompt)
end { ExitTranscript };


{****************************************************************}

function GetTrnByte:  char;

begin { GetTrnByte }
if DEBUG [4] then Status ('Enter GetTrnByte');
if Replay = NotReplaying then { shouldn't happen }
    Error ('Internal error--reading unopened transcript')
else
    if (TBlock >= TBlocks) or
        ((TBlock = TBlocks-1) and (TByte >= TBytes)) then
        begin
        GetTrnByte := NullCh;
        Replay := NotReplaying;
        ExitTranscript
        end
    else
        begin
        if TByte = 0 then 
            FSBlkRead (TId, TBlock, TBuffer);
        GetTrnByte := recast (TBuffer^.ByteBuffer[TByte], char);
        TByte := TByte + 1;
        if TByte >= 512 then
            begin
            TByte := 0;
            TBlock := TBlock + 1
            end
        end;
if DEBUG [4] then Status ('Exit GetTrnByte')
end { GetTrnByte };


{****************************************************************}

function GetTrnWord:  integer;

var
    Word:  TwoByte;

begin { GetTrnWord }
if DEBUG [4] then Status ('Enter GetTrnWord');
Word [0] := GetTrnByte;
Word [1] := GetTrnByte;
GetTrnWord := recast (Word, integer);
if DEBUG [4] then 
    StatusNumber ('Exit GetTrnWord:  word = ', ord (recast (Word, integer)))
end; { GetTrnWord }
   

{****************************************************************}

function FindPos (Page, Ch:  integer):  Position;

var P:  pChunk;
    Q:  Position;

begin { FindPos }
P := CurWin^.FilledFirst.Chunk;
while P^.OrderP < Page do
    P := P^.Next;
Q.Chunk := P;
Q.Offset := 0;
FindPos := Add (Q, Ch - Q.Chunk^.OrderC)
end; { FindPos }


{****************************************************************}

procedure GetTrnKey;

var
    LastCmd:    KeyCommand;
    Page, Ch:   integer;
    w:          integer;
    OldWin:     pTextWindow;
    FromQ:      boolean;

{********************************}

function GetTKey:  KeyEvent;

var 
    NewKey:  KeyEvent;

begin { GetTKey }
FromQ := not IsEmpty (InputQueue);
if FromQ then
    NewEvent := GetKey (InputQueue)
else
    begin
    if (TBlock >= TBlocks) or ((TBlock = TBlocks-1) and (TByte >= TBytes)) then
        begin
        ExitTranscript;
        NewKey.Ch := NullCh
        end
    else
        NewKey.Ch := GetTrnByte;
    NewKey.Cmd := CmdTranslate (NewKey.Ch);
    GetTKey := NewKey
    end
end { GetTKey };

{********************************}

begin { GetTrnKey }
if DEBUG [4] then Status ('Enter GetTranscriptKey');
if Replay = NotReplaying then { shouldn't happen }
    Error ('Internal error--shouldn''t be reading transcript')
else
    begin
    LastCmd := NewEvent.Cmd;
    NewEvent := GetTKey;
    if NewEvent.Cmd = CountCmd then
        begin
        RepeatCount := GetTrnWord;
        NewEvent := GetTKey
        end;
    if NewEvent.Cmd = PrefixKey then
        begin
        NewEvent := GetTKey;
        NewEvent.Cmd := LOR (PrefixBit, NewEvent.Cmd)
        end;
    if InSrch then
        begin
        InSrch := not (NewEvent.Cmd in [Accept, Reject]);
        if NewEvent.Cmd = Reject then
            InRepl := false;
        exit (LookForCommand)
        end
    else if InRepl then
        begin
        InRepl := not (NewEvent.Cmd in [Accept, Reject]);
        exit (LookForCommand)
        end
    else
        if LastCmd in InsCmds then
            exit (LookForCommand);
    InSrch := NewEvent.Cmd in InSrchCmds;
    InRepl := NewEvent.Cmd in InReplCmds;
    if FromQ or (CurWin = PromptWindow) then
        exit (LookForCommand);
    if NewEvent.Cmd in SelCmds then
        begin
        OldWin := CurWin;
        if SelectWindow <> nil then
            begin
            ChangeTextWindow (SelectWindow);
            UnderLine (Bufary [SelectB].First, Bufary [SelectB].Last, Erase)
            end;
        w := GetTrnWord;
        if w = 0 then
            begin
            SelectWindow := nil;
            Bufary [SelectB].First := NilPosition;
            Bufary [SelectB].Last := NilPosition
            end
        else
            begin
            SelectWindow := Winary [w];
            ChangeTextWindow (SelectWindow);
            Page := GetTrnWord;
            Ch := GetTrnWord;
            Bufary [SelectB].First := FindPos (Page, Ch);
            Page := GetTrnWord;
            Ch := GetTrnWord;
            Bufary [SelectB].Last := FindPos (Page, Ch);
            UnderLine (Bufary[SelectB].First, Bufary [SelectB].Last, ThinBlack)
            end;
        ChangeTextWindow (OldWin);
        DrawLn (SelectB)
        end;
    if NewEvent.Cmd in PosCmds then
        begin
        Page := GetTrnWord;
        Ch := GetTrnWord;
        Show (FindPos (Page, Ch), 0, CurWin^.LastLine)
        end;
    if Replay <> NotReplaying then
        CheckReplay (ReplaySingleStep);
    if Replay = NotReplaying then
        NewEvent.Cmd := NullCmd
    end; { else not Replay = NotReplaying }
if DEBUG [4] then
    begin
    StatusNumber ('Exit GetTrnKey:  cmd =', NewEvent.Cmd);
    StatusNumber ('Exit GetTrnKey:  ch =', ord (NewEvent.Ch))
    end
end { GetTrnKey };
  

{****************************************************************}

procedure SendTrnByte (Byte:  Char);

begin { SendTrnByte }
if DEBUG [4] then StatusNumber ('Enter SendTrnByte:  byte = ', ord (Byte));
if Replay = NotReplaying then
    begin
    TBuffer^.ByteBuffer[TByte] := recast (Byte, SmallInt);
    TByte := TByte + 1;
    if TByte >= 512 then
         begin
         FlushTranscript;
         TByte := 0;
         TBlock := TBlock + 1
         end
    end;
if DEBUG [4] then Status ('Exit SendTrnByte')
end; { SendTrnByte }


{********************************}

procedure SendTrnWord (Word: Integer);

var
    CharWord:  TwoByte;

begin { SendTrnWord }
if DEBUG [4] then StatusNumber ('Enter SendTrnWord', Word);
CharWord := recast (Word, TwoByte);
SendTrnByte (CharWord [0]);
SendTrnByte (CharWord [1]);
if DEBUG [4] then Status ('Exit SendTrnWord')
end; { SendTrnWord }


{****************************************************************}

procedure SendKey (Key:  KeyEvent);

var
    SendPos:     Position;
    FillOffset:  integer;
    i:           integer;

begin { SendKey }
if (Replay <> NotReplaying) or not (Key.Cmd in ReplayCmds) then
    exit (SendKey);
if DEBUG [4] then StatusNumber ('Enter SendKey:  Key = ', Key.Cmd);
FillOffset := 0;
if (CurWin <> PromptWindow) and (Key.Cmd in InsWhiteCmds) then
    with CurWin^ do
        if CurCol >= Ln[CurLine].Length then
            begin
            SendTrnByte (' ');
            SendPos := GetPosition (CurLine, CurCol);
            SendTrnWord (SendPos.Chunk^.OrderP);
            SendTrnWord (SendPos.Chunk^.OrderC + SendPos.Offset);
            FillOffset := CurCol - Ln[CurLine].Length + 1;
            for i := 2 to FillOffset do
                SendTrnByte (' ');
            SendTrnByte (chr (CloseInsert))
            end; { if CurCol >= ... }
if Key.Cmd >= #200 then
    SendTrnByte (chr (lor (#200, PrefixKey)));
if Key.Cmd = InsChar then
    SendTrnByte (Key.Ch)
else
    SendTrnByte (chr (lor (#200, Key.Cmd)));
if CurWin = PromptWindow then
    exit (SendKey);
if Key.Cmd in SelCmds then
    if SelectWindow = nil then
        SendTrnWord (0)
    else
        begin
        SendTrnWord (SelectWindow^.WinNum);
        with Bufary [SelectB] do
            begin
            if (SelectWindow = CurWin) and (FillOffset > 0) then
                begin
                if GE (First, SendPos) then
                    First := Add (First, FillOffset);
                if GE (Last, SendPos) then
                    Last := Add (Last, FillOffset)
                end;
            SendTrnWord (First.Chunk^.OrderP);
            SendTrnWord (First.Chunk^.OrderC + First.Offset);
            SendTrnWord (Last.Chunk^.OrderP);
            SendTrnWord (Last.Chunk^.OrderC + Last.Offset);
            if (SelectWindow = CurWin) and (FillOffset > 0) then
                begin
                if GE (First, SendPos) then
                    First := Add (First, -FillOffset);
                if GE (Last, SendPos) then
                    Last := Add (Last, -FillOffset)
                end;
            end
        end;
if Key.Cmd in PosCmds then
    begin
    if FillOffset = 0 then
        SendPos := GetPosition (CurWin^.CurLine, CurWin^.CurCol)
    else
        SendPos := Add (SendPos, FillOffset);
    SendTrnWord (SendPos.Chunk^.OrderP);
    SendTrnWord (SendPos.Chunk^.OrderC + SendPos.Offset)
    end;
if DEBUG [4] then Status ('Exit SendKey')
end; { SendKey }


{****************************************************************}

procedure FlushTranscript;

begin { FlushTranscript }
if DEBUG [4] then Status ('Enter FlushTranscript');
if Replay = NotReplaying then
    begin
    if TByte <> 0 then
        begin
        if (TBlock = 0) and (TId = 0) then
            begin
            TId := FSEnter('>Editor.Transcript');
            if TId = 0 then
                Error('Could not create Editor.Transcript file')
            end;
        if TId <> 0 then
            begin
            FSBlkWrite (TId, TBlock, TBuffer);
            FSClose (TId, TBlock + 1, TByte * 8)
            end
        end
    end;
if DEBUG [4] then Status ('Exit FlushTranscript')
end; { FlushTranscript }


{****************************************************************}

procedure CheckReplay (Where:  ReplayMode);

const
    IOEIOC = 1;  { read completed }

var
    gotChar: Integer;
    Ch: Char;
    Done: Boolean;
   

{********************************}

procedure ReplayPrompt;

var
    ReplayPrompt:  String;
    ChString:  String;
    Ch:  char;

begin { ReplayPrompt }
ReplayPrompt := 'Replay:  About to type ';
if NewEvent.Cmd >= #200 then
    begin
    AppendString (ReplayPrompt, 'ctrl-');
    AppendChar (ReplayPrompt, chr (PrefixKey));
    AppendChar (ReplayPrompt, ' ')
    end;
Ch := chr (LAND (#177, ord (NewEvent.Ch)));
case Where of
    ReplaySingleStep:
        begin
        if ord (Ch) <= #040 then
            begin
            case Ch of
                HELP:  ChString := '<HELP>';
                BS:    ChString := '<BACKSPACE>';
                CR:    ChString := '<RETURN>';
                LF:    ChString := '<LF>';
                TAB:   ChString := '<TAB>';
                OOPS:  ChString := '<OOPS>';
                INS:   ChString := '<INS>';
                ' ':   ChString := '<SPACE>'
                end; { case }
            AppendString (ReplayPrompt, ChString)
            end { if ord (Ch) ... }
        else if ord (Ch) = #177 then
            AppendString (ReplayPrompt, '<DEL>')
        else
            begin
            if ord (NewEvent.Ch) >= #200 then
                AppendString (ReplayPrompt, 'ctrl-');
            AppendChar (ReplayPrompt, NewEvent.Ch)
            end { else }
        end; { ReplaySingleStep }
     ReplayFree:
         begin
         ReplayPrompt := '';
         NeedPrompt := True
         end
     end; { case }
if ReplayPrompt <> '' then 
    Prompt (ReplayPrompt);
NeedPrompt := false;
ImmedPrompt := false
end { ReplayPrompt };
   
   
{********************************}

procedure ControlReplay;

begin { ControlReplay }
Done := True;
if Ch = ' ' then
    Replay := ReplaySingleStep
else if Ch = Ins then
    begin
    Replay := ReplayFree;
    Prompt (CmdPrompt);
    end
else if Ch = Del then
    begin
    Replay := NotReplaying;
    NeedPrompt := True;
    ExitTranscript
    end
else if Ch = '!' then
    DEBUG [4] := not DEBUG [4]
else
    begin
    Write (Bel);
    Done := False
    end
end; { ControlReplay }
   
   
{********************************}

begin { CheckReplay }
if DEBUG [4] then Status ('Enter CheckReplay');
gotChar := IOCRead(KeyBoard,Ch);
if gotChar = IOEIOC then
    ControlReplay;
if Where >= Replay then
    begin
    ReplayPrompt;
    repeat
        repeat gotChar := IOCRead(KeyBoard,Ch)
        until gotChar = IOEIOC;
        ControlReplay
    until Done
    end;
if Replay = NotReplaying then
    begin
    CanvDflt := CanvKeybd;
    IOSetModeTablet (RelTablet);
    CmdPrompt := InsModePrompt;
    Prompt (CmdPrompt)
    end;
if DEBUG [4] then Status ('Exit CheckReplay')
end; { CheckReplay }


{****************************************************************}

function InitReplay (DoReplay:  boolean):  boolean;

var
    TBits:  integer;

begin
InitReplay := true;
InSrch := false;
InRepl := false;
TId := 0;
TBlock := 0;
TByte := 0;
TBuffer := nil;
if DoReplay then
    begin
    Replay := ReplayFree;
    CanvDflt := CanvTrans;
    CmdPrompt := ReplPrompt;
    TID := FSLookUp ('>Editor.Transcript', TBlocks, TBits);
    TBytes := TBits div 8;
    if (TID = 0) or (TBlocks = 0) then
        begin
        InitReplay := false;
        writeln ('Could not find Editor.Transcript', BEL);
        exit (InitReplay)
        end
    end
else
    begin
    Replay := NotReplaying;
    CanvDflt := CanvKeybd;
    CmdPrompt := InsModePrompt
    end;
New (0, 256, TBuffer)
end.  { InitReplay }
